#ifndef _RHEO_FORM_ELEMENT_H
#define _RHEO_FORM_ELEMENT_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Class:form_element
NAME: @code{form_element} - bilinear form on a single element
@cindex  bilinear form
@cindex  geometrical element
@clindex basis
@clindex numbering
@clindex geo_element
SYNOPSIS:
  @noindent
  The @code{form_element} class defines functions that compute
  a bilinear form defined between two polynomial basis on a single
  geometrical element. This bilinear form is represented
  by a matrix.

  The bilinear form is designated by a string, e.g. "mass", "grad_grad", ...
  indicating the form. The form depends also of the geometrical element:
  triangle, square, tetrahedron (@pxref{geo_element iclass}).

IMPLEMENTATION NOTE:
  @noindent
  The @code{form_element} class 
  is managed by (@pxref{smart_pointer iclass}).
  This class uses a pointer on a pure virtual class @code{form_element_rep}
  while the effective code refers to the specific concrete derived classes:
  mass, grad_grad, etc.

AUTHORS:
    LMC-IMAG, 38041 Grenoble cedex 9, France
   | Pierre.Saramito@imag.fr
DATE:   7 january 2004
End:
*/

#include "rheolef/form_element_rep.h"
namespace rheolef { 

//<form_element:
class form_element : public smart_pointer<form_element_rep> {
public:

// typedefs:

    typedef size_t size_type;

// allocators:

    form_element (std::string name = "");

// modifier:

    void initialize (const space& X, const space& Y) const;

    //! optional, for scalar-weighted forms:
    void set_weight (const field& wh) const;
    void set_skip_geo_compatibility_check () const;

    //! for special forms for which coordinate-system specific weights (ie, cylindrical
    //! coordinates weights) should not be used:
    void set_use_coordinate_system_weight(bool use) const;

// accessor:

    void operator() (const geo_element& K, ublas::matrix<Float>& m) const;
    const field& get_weight() const;
};
//>form_element:
// -----------------------------------------------------------
// inlined
// -----------------------------------------------------------
inline
form_element::form_element(std::string name)
 : smart_pointer<form_element_rep> (form_element_rep::make_ptr(name))
{
}
inline
void
form_element::initialize (const space& X, const space& Y) const
{
    data().initialize (X, Y);
}
inline
void
form_element::operator() (const geo_element& K, ublas::matrix<Float>& m) const
{
    data().operator() (K, m);
}
inline
void
form_element::set_weight (const field& wh) const
{
    data().set_weight (wh);
}
inline
void
form_element::set_skip_geo_compatibility_check () const
{
    data().set_skip_geo_compatibility_check ();
}
inline
void 
form_element::set_use_coordinate_system_weight(bool use) const
{
	data().set_use_coordinate_system_weight(use);
}
inline
const field&
form_element::get_weight() const
{
  return data().get_weight();
}
}// namespace rheolef
#endif // _RHEO_FORM_ELEMENT_H
