#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <sys/klog.h>
#include <sys/reboot.h>
#include <sys/types.h>
#include <fcntl.h>
#include <ctype.h>

#include "dinstall.h"
#include "../busybox/internal.h"
#include <newt.h>
#include "lang.h"

static int check_color = 1;

static void suspend(void) {
    newtSuspend();
    raise(SIGTSTP);
    newtResume();
}

static void initScreen (void) {
    char *term;

    setenv("NEWT_MONO","1",1);
    newtInit();
    term = getenv("TERM");
    if (!term || (term && !strncmp(term,"linux",5)))
      setenv("TERM","linux-m",1);
    else
      check_color = 0;
    newtSetSuspendCallback(suspend);
    newtPushHelpLine(MSG_MAIN_KEY);
}

struct Arg {
  const char *name;
  int isflag;
  void *value;
};

int parsefile(const char *filename, struct Arg *options, 
	int hasinterspace) {
  int fd, bytes, lineend=0;
  char *start= prtbuf, *end;
  struct Arg *a;

  if ((fd = open(filename, O_RDONLY, 0)) < 0) {
    fprintf( stderr, MSG_CANNOT_OPEN_FILE, filename, strerror(errno) );
    return 1;
  }
  while ( (bytes = read(fd,prtbuf,PRTBUFSIZE-1)) != 0) {
    prtbuf[bytes]= '\0';

    /* parse line */
    while (!lineend) {
      /* skip space at the beggining of the line */
      while (isspace(*start)) start++;
      if (!(*start)) break;

      /* take the first word on the string as the option name */
      end = start;
      while (*end && !isspace(*end)) end++;
      if (!*end) lineend = 1;
      *end = '\0';

      a =options;
      while ( a->name != NULL ) {
	if (a->isflag) {
	  if (!strcmp(start, a->name)) {
	    /* the option is valid. It is a flag, switch it on */
	    *((int *)a->value)=1;
	  }
	} else {
	  int namelen=strlen(a->name);
	  if (!strncmp(start, a->name, namelen)) {
	    /* the option is valid. It expects a value */
	    if (hasinterspace) {
	      if (lineend) {
		/* If the file syntax is "name value" (hasinterspace==1) 
		 * and lineend == 1, we have found a name with no value. */
		break;
	      }
	      /* skip whitespace after the option name */
	      namelen=0;
	      start = end + 1;
	      while (isspace(*start)) start++;
	      if (!(*start)) {
		/* We have reached the end of the line, that means we have
		 * found a name with no value. */
		break;
	      }
	      end = start;
	      while (*end && !isspace(*end)) end++;
	      *end = '\0';
	      /* Skip the rest of the line */
	      lineend = 1;
	    }  
	    *((char **)a->value)=strdup(start+namelen);
	  }
	}
	a++;
      }
      start = end + 1;
    }
  }
  close(fd);
  return 0;
}
  
/* Get arguments from boot commmand line */
int readcmdline(void){
  int status;
  struct Arg bootoptions[] = {
    { "root=", 0, &bootargs.root },
    { "tecra", 1, &bootargs.istecra },
    { NULL, 0, NULL}
  };
  memset((void *)&bootargs, 0, sizeof(struct BootArgs));
  status=parsefile("/proc/cmdline",bootoptions,0);
  return status;
}

#if #cpu (m68k)
static const char *get_m68k_model( void )
  char *model = NULL;
  struct Arg archoptions[] = {
    { "Model:", 0, &model },
    { NULL, 0, NULL}
  };
  status=parsefile("/proc/hardware",archoptions,1);
  if(status) {
    fprintf( stderr, MSG_UNKNOW_M68K_TYPE );
    return NULL;
  }
  if (!model || !*model)
     fprintf( stderr, MSG_UNABLE_TO_FIND_M68K_TYPE );
  return model;
}
#endif

/* try to eject root floppy */
#if #cpu(sparc)
void
try_to_eject_root_floppy(void)
{
  if (!strncmp(bootargs.root,"/dev/fd",7))
      /* extract floppy device name */
      eject_floppy(bootargs.root);
  }
}
#endif

#ifdef INCLUDE_DINSTALL
  int dinstall_main(void) {
#else
  int main (void) {
#endif

/* Downgrade kernel verbosity */
    klogctl(8,NULL,4);

    initScreen();
    readcmdline();
#if #cpu (m68k)
    Arch2=get_m68k_model();
    if ( (strcmp(Arch2,"Atari")) && (strcmp(Arch2,"Amiga")) &&
	 (strcmp(Arch2,"Macintosh"))) {
      problemBox(MSG_UNKNOW_M68K_ARCH,MSG_PROBLEM);
      reboot(RB_AUTOBOOT);
    }
#endif

#if #cpu(sparc)
    try_to_eject_root_floppy();
#endif

    Root = NULL;
    noSwap = 0;
    notCreatedBootFloppy = 1;
    notInstalledLILO = 1;
    Archive_Dir=NULL;

    InstallationRootDevice=block_device(T_FILE("/"),NULL);
    if (!InstallationRootDevice) {
      /* net installation with no /dev/root device on nfsroot */
      problemBox(MSG_TROUBLE_ROOT_KO,MSG_PROBLEM);
      reboot(RB_AUTOBOOT);
    }

    if (check_color)
      color_or_monochrome();
    
    release_notes();

    is_root_a_floppy ();

    main_menu();

    newtFinished();


    return 0;
}
