#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>
#include <time.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <newt.h>
#include "dinstall.h"
#include "lang.h"

/*
 * Option Constants for the lists
 */
#define OPT_F    (10000)
#define OPT_D    (10000 + OPT_F)

/*
 * Button length for tz_dialog_box
 */
#undef _ButtonH
#define _ButtonH (1)

/*
 * Internal representation of the list
 */
struct list
{
  int nelem;
  char** data;
};

/*
 * Function prototypes
 */
int configure_timezone(void);
static int read_directory(const char* dirname,
		    struct list *files,
		    struct list *dirs);
static int configure_timezone_step1();
static int configure_timezone_step2();
static int tz_dialog_box(const char* text, const char* title, 
		  int height, int width,
		  struct list* files, struct list* dirs);
static int compare(const void*,const void*);

/*
 * configure_timezone_step1
 * Performs the timezone selection and setting.
 */
static int configure_timezone_step1() {
  #ifndef _TESTING_
  char target[]="/target";
  char dir_sfx[]="/usr/share/zoneinfo";
  char dest_tz[]="/target/etc/timezone";
  char dest_lt[]="/target/etc/localtime";
  #else
  char target[]="";
  char dir_sfx[]="/usr/share/zoneinfo";
  char dest_tz[]="./timezone";
  char dest_lt[]="./localtime";
  #endif
  int  dir_length=strlen(target)+strlen(dir_sfx)+3;
  char *dir=calloc(dir_length,sizeof(char));
  char *rel_dir=calloc(1,sizeof(char));
  char *timezone=(char *)NULL;
  char *localtime=(char *)NULL;
  FILE* pfile;

  char buf[4096];
  struct list files={ 0,(char **)NULL };
  struct list dirs={ 0,(char **)NULL };
  int result;
  int idx;
  int sz;
  char *substr;
  rel_dir[0]='\0';

  /*
   * Check whether the root system has been setup.
   * 
   * This check is redundant, as tzconfig is called from configure_base
   * and we checked this there. <ezanardi>
   *
   * if(require_root())
   *  return 1;
   */

  /*
   * Initialise the directory information.
   */
  strcpy(dir,target);
  strcat(dir,dir_sfx);
  dir[dir_length-2]='/';
  dir[dir_length-1]='.';
  dir[dir_length]='\0';

  for(;;)
    {
      if(read_directory(dir,&files,&dirs)!=0)
	{
	 /* 4096 - ( strlen(MSG_UNABLE_TO_OPEN1) - strnlen("%s") ) */
	  if(strlen(dir)<4098-strlen(MSG_UNABLE_TO_OPEN1))
	    sprintf(buf,MSG_UNABLE_TO_OPEN1,dir);
	  else
	    sprintf(buf,MSG_UNABLE_TO_OPEN2);
	  problemBox(buf,MSG_TZ_ERROR_TITLE);
	}
      if( (result=tz_dialog_box(MSG_TZ_TEXT,MSG_TZ_TITLE,18,66,&files,&dirs))
	  ==DLG_CANCEL)
	return -1;
      else if(result<OPT_D) 
	{
	  timezone=calloc(strlen(rel_dir)+
			  strlen(files.data[result-OPT_F])+
			  2, sizeof(char));
	  if(strlen(rel_dir)>0)
	    {
	      strcpy(timezone,rel_dir);
	      strcat(timezone+strlen(rel_dir),"/");
	      strcat(timezone+strlen(rel_dir)+1,files.data[result-OPT_F]);
	    }
	  else
	    strcpy(timezone,files.data[result-OPT_F]);
	  
	  localtime=calloc(strlen(dir_sfx)+
			   strlen(rel_dir)+
			   strlen(files.data[result-OPT_F])+
			   3,sizeof(char));
	  strcpy(localtime,dir_sfx);
	  strcat(localtime+strlen(dir_sfx),"/");
	  strcat(localtime+strlen(dir_sfx)+1,rel_dir);
	  strcat(localtime+strlen(dir_sfx)+strlen(rel_dir)+1,"/");
	  strcat(localtime+strlen(dir_sfx)+strlen(rel_dir)+2,
		 files.data[result-OPT_F]);
	  
	  /*
	   * Deletes the previous local_time, creates a symlink to the
	   * new one and writes the timezone.
	   */
	  unlink(dest_lt);
	  symlink(localtime,dest_lt);
	  pfile=fopen(dest_tz,"w");
	  fprintf(pfile,"%s\n",timezone);
	  fclose(pfile);
	  
	  free(localtime);
	  free(timezone);
	  
	  for(idx=0;idx<files.nelem;idx++)
	    free(files.data[idx]);
	  free(files.data);
	  for(idx=0;idx<dirs.nelem;idx++)
	    free(dirs.data[idx]);
	  free(dirs.data);
	  
	  return DLG_OKAY;
	}
      else
	{
	  if(result-OPT_D==0) /* It's the ".." directory. */
	    {
	      if(strlen(rel_dir)!=0) /* We are NOT in the top_level */
		{
		  /*
		   * Re-adjust rel_dir and dir.
		   */
		  if((substr=strrchr(rel_dir,'/'))==NULL)
		    sz = 0;
		  else
		    sz = strlen(rel_dir)-strlen(substr);
		  rel_dir=realloc(rel_dir,sz+1);
		  rel_dir[sz]='\0';
		  
		  /*
		   * Remeber: dir always ends with a '/'
		   */ 
		  sz=strlen(dir);
		  dir[sz-1]='\0';
		  substr=strrchr(dir,'/');
		  sz-=strlen(substr);
		  dir=realloc(dir,sz);
		  dir[sz-1]='\0';
		}
	    }
	  else
	    {
	      rel_dir=realloc(rel_dir,
			      strlen(dirs.data[result-OPT_D])+
			      strlen(rel_dir)+2);
	      if(strlen(rel_dir)>0)
		{
		  strcat(rel_dir+strlen(rel_dir),"/");
		  strcat(rel_dir+strlen(rel_dir),dirs.data[result-OPT_D]);
		}
	      else
		strcpy(rel_dir,dirs.data[result-OPT_D]);
	      
	      dir=realloc(dir,strlen(dir)+strlen(dirs.data[result-OPT_D])+2);
	      strcat(dir+strlen(dir),"/");
	      strcat(dir+strlen(dir),dirs.data[result-OPT_D]);
	    }
	  
	  
	  /*
	   * Free memory to restart the directory search.
	   */
	  for(idx=0;idx<files.nelem;idx++)
	    free(files.data[idx]);
	  free(files.data);
	  files.nelem=0;
	  for(idx=0;idx<dirs.nelem;idx++)
	    free(dirs.data[idx]);
	  free(dirs.data);
	  dirs.nelem=0;
	}
    }   
}


/*
 * tz_dialog_box
 * Dialog box for timezone selection and setting:
 * char *text         - Introductory info.
 * char *title        - Title in the dialog box
 * int height         - Height of the box
 * int width          - Width of the box.
 * struct list* files - List of the file options.
 * struct list* dirs  - List of the directory options.
 */
static int tz_dialog_box(const char* text, const char* title, 
		  int height, int width,
		  struct list* files, struct list* dirs)
{
  int result=DLG_OKAY;
  int top;
  int file_sz=0;
  int dir_sz=0;
  int *f_ans, *d_ans;
  int idx;
  char file_fmt[20], dir_fmt[20], buf[80];
  newtComponent form,answer;
  newtComponent file_lb,dir_lb;
  newtComponent txtbx,t1,t2;
  newtComponent cancel;
  int* selection;

  f_ans=calloc(files->nelem,sizeof(int));
  for(idx=0;idx<files->nelem;idx++)
    f_ans[idx]=OPT_F+idx;

  d_ans=calloc(files->nelem,sizeof(int));
  for(idx=0;idx<files->nelem;idx++)
    d_ans[idx]=OPT_D+idx;  

  newtPushHelpLine(MSG_TZ_DIALOG_KEYS);
  top = 3+strlen(text)/(width-2);
  newtOpenWindow(7, 2, 65, 18, MSG_SELECT_TZ);

  form=newtForm(NULL,NULL,0);
  txtbx=newtTextbox(1,0,width-2,top-1,NEWT_FLAG_WRAP);
  newtTextboxSetText(txtbx,text);

  /*
   * Works out the element size.
   */
  for(idx=0;idx<files->nelem;idx++)
    if(strlen(files->data[idx])>file_sz)
      file_sz=strlen(files->data[idx]);

  for(idx=0;idx<dirs->nelem;idx++)
    if(strlen(dirs->data[idx])>dir_sz)
      dir_sz=strlen(dirs->data[idx]);

  /*
   * Files list.
   */
  t1=newtTextbox(4,top-1,12,1,NEWT_FLAG_WRAP);
  newtTextboxSetText(t1,MSG_TZ);
  file_lb=newtListbox(4,top,height-top-3, NEWT_FLAG_DOBORDER | NEWT_FLAG_RETURNEXIT);
  sprintf(file_fmt,"%%-%ds",file_sz);
  for(idx=0;idx<files->nelem;idx++)
    {
      sprintf(buf,file_fmt,files->data[idx]);
      newtListboxAddEntry(file_lb,buf,&f_ans[idx]);
    }

  t2=newtTextbox(width/2+4,top-1,14,1,NEWT_FLAG_WRAP);
  newtTextboxSetText(t2,MSG_TZ_DIR);
  dir_lb=newtListbox(width/2+4,top,height-top-2, NEWT_FLAG_DOBORDER | NEWT_FLAG_RETURNEXIT);
  sprintf(dir_fmt,"%%-%ds",dir_sz);
  for(idx=0;idx<dirs->nelem;idx++)
    {
      sprintf(buf,dir_fmt,dirs->data[idx]);
      newtListboxAddEntry(dir_lb,buf,&d_ans[idx]);
    }

  cancel=newtCompactButton( width/2-3, height-_ButtonH, MSG_CANCEL);
  newtFormAddComponents(form,txtbx,t1,t2,file_lb,dir_lb,cancel,NULL);
  answer=newtRunForm(form);
  
  if(answer==cancel)
    result=DLG_CANCEL;
  else if(answer==file_lb)
    {
      selection=newtListboxGetCurrent(file_lb);
      result=(*selection);
    }
  else if(answer==dir_lb)
    {
      selection=newtListboxGetCurrent(dir_lb);
      result=(*selection);
    }
  newtFormDestroy(form);
  newtPopWindow();
  newtPopHelpLine();
  return result;
}

/*
 * read_directory
 * This will check the current directory for other directories or
 * new files and will generate and allocate the memory for both lists.
 * char *dir    - Name of the directory to process
 * struct list* - List of the files in that directory.
 * struct dirs* - List of the sub-directories in that directory.
 */
static int read_directory(const char* dir,
		   struct list* files, struct list* dirs)
{
  DIR* pdir;
  struct dirent *current;
  struct stat buffer;

  int ndir=0;
  int nfile=0;
 
  char* tmp_name;
  int   tmp_sz;
  char* path;

  char error[4096];
  /*
   * Open directory
   */
  if((pdir=opendir(dir))==NULL)
    {
      problemBox(MSG_TZ_DIROPENERROR,MSG_TZ_ERROR_TITLE);
      return -1;
    }
  
  while((current=readdir( pdir )) != NULL)
    {
      tmp_sz=strlen(current->d_name)+1;
      path=calloc(strlen(dir)+tmp_sz+1,
		  sizeof(char));
      strncpy(path,dir,strlen(dir));
      strncat(path+strlen(dir),"/",1);
      strcat(path+strlen(dir)+1,current->d_name);

      if(stat(path,&buffer)==-1)
	{
	  sprintf(error,MSG_TZ_FILESTATERROR,path);
	  problemBox(error,MSG_TZ_ERROR_TITLE);
	}
      if(S_ISDIR(buffer.st_mode))
	{
	  if(strcmp(current->d_name,"."))
	    ndir++;
	}
      else if(S_ISREG(buffer.st_mode))
	nfile++;
      
      free(path);
    }
  
  dirs->data=(char **)calloc(ndir,sizeof(char *));
  files->data=(char **)calloc(nfile,sizeof(char *));
  rewinddir( pdir );
  
  while((current=readdir( pdir )) != NULL)
    {
      tmp_sz=strlen(current->d_name)+1;
      tmp_name=calloc(tmp_sz,sizeof(char));
      strcpy(tmp_name,current->d_name);
      
      path=calloc(tmp_sz+strlen(dir)+1,sizeof(char));
      strncpy(path,dir,strlen(dir));
      strncat(path+strlen(dir),"/",1);
      strcat(path+strlen(dir)+1,tmp_name);
      
      if(stat(path,&buffer)==-1)
	{
	  sprintf(error,MSG_TZ_FILESTATERROR,tmp_name);
	  problemBox(error,MSG_TZ_ERROR_TITLE);
	}
      else
	{
	  if(S_ISDIR(buffer.st_mode))
	    {
	      if(strcmp(tmp_name,"."))
		dirs->data[(dirs->nelem)++]=tmp_name;
	    }
	  else if(S_ISREG(buffer.st_mode))
	    files->data[(files->nelem)++]=tmp_name;
	}
      free(path);
    }

  qsort((void*)files->data,files->nelem,sizeof(char*),compare);
  qsort((void*)dirs->data,dirs->nelem,sizeof(char*),compare);
  return 0;
}


/*
 * configure_timezone_step2
 * Dialog to determine whether the clock is set to GMT or no.
 */
static int configure_timezone_step2() {
  char info[]=MSG_TZ_GMT_INFO;
  char yesno_intro[]=MSG_TZ_GMT_YN_INTRO;
  char yesno_suffix[]=MSG_TZ_GMT_YN_SUFFIX;
  time_t now;
  char *buf;
  int result;

  problemBox(info,MSG_TZ_TITLE);
  time(&now);
  buf=calloc(strlen(yesno_intro)+strlen(yesno_suffix)+256,sizeof(char));
  sprintf(buf,"%s %s%s\n",yesno_intro,ctime(&now),yesno_suffix);
  result=yesNoBox(buf,MSG_TZ_TITLE);
  free(buf);

  if(result==0)
    {
      /*
       * Edit the /etc/default/rcS file to change the setting to GMT=""
       */
      #ifndef _TESTING_
      system("sed -e 's:^GMT=\"-u\":GMT=\"\":' /target/etc/default/rcS > /target/etc/default/rcS.new");
      system("mv /target/etc/default/rcS.new /target/etc/default/rcS");
      chmod("/target/etc/default/rcS",S_IRUSR|S_IWUSR|S_IXUSR|
	S_IRGRP|S_IROTH);
      #endif
    }
  
  return 0;
}

/*
 * Main timezone configuration routine.
 * The process is divided in two steps:
 * 1st we select the timezone for the system.
 * 2nd we determine whether the clock should use UTC.
 */
int configure_timezone(void)
{
  int result;
  result=configure_timezone_step1();
  if(result==0)
    configure_timezone_step2();
  else
    return -1;
  return 0;
}

int compare(const void *a, const void *b)
{
  const char** _a=(const char**)a;
  const char** _b=(const char**)b;
  
  if(**_a=='.')
    return (-1);
  else if(!(**_b)=='.')
    return(1);
  else
    return strcmp((*_a),(*_b));
}

#ifdef _TESTING_
/*
 * To test, compile using:
 * cc -o ./testing -D_TESTING_ -g tzconfig.c boxes.c -lnewt -lslang
 */
int require_root() { return 0; }
main()
{

  newtInit();
  newtCls();

  newtDrawRootText(0, 0, "TimeZone Config test program");
  newtOpenWindow(7, 2, 65, 18, "Root Window");
  newtPushHelpLine(NULL);
  
  configure_timezone();
  
  newtPopWindow();
  newtFinished();
}
#endif
