/*
 * Worldvisions Weaver Software:
 *   Copyright (C) 1997, 1998 Worldvisions Computer Technology, Inc.
 * 
 * Unified support for streams, that is, sequences of bytes that may or
 * may not be ready for read/write at any given time.
 * 
 * We provide typical read and write routines, as well as a select() function
 * for each stream and a WvStreamList::select() for multiple simultaneous
 * streams.
 */
#include "wvstream.h"
#include <sys/time.h>
#include <sys/types.h>
#include <errno.h>
#include <assert.h>


void WvStream::init()
{
    sbuf = head = NULL;
    bufsize = bufcount = 0;
    callfunc = NULL;
    userdata = NULL;
    errnum = 0;
}


WvStream::~WvStream()
{
    close();
    if (sbuf) delete[] sbuf;
}


void WvStream::close()
{
    if (fd >= 0) ::close(fd);
    fd = -1;
}


int WvStream::autoforward_callback(WvStream &s, void *userdata)
{
    WvStream &s2 = *(WvStream *)userdata;
    char buf[1024];
    size_t len;
    
    while (s.isok() && s2.isok() && s.select(0))
    {
	len = s.read(buf, sizeof(buf));
	s2.write(buf, len);
    }
    
    return 0;
}


int WvStream::getfd() const
{
    return fd;
}


bool WvStream::test_set(fd_set &r, fd_set &w, fd_set &x)
{
    return (FD_ISSET(getfd(), &r)
	    || FD_ISSET(getfd(), &w)
	    || FD_ISSET(getfd(), &x));
}


bool WvStream::isok() const
{
    return (fd != -1);
}


void WvStream::seterr(int _errnum)
{
    if (!errnum)
	errnum = _errnum;
    close();
}


void WvStream::seterr(const WvString &specialerr)
{
    if (!errnum)
    {
	errstring = specialerr;
	errnum = -1;
    }
    close();
}


int WvStream::geterr() const
{
    return errnum;
}


const char *WvStream::errstr() const
{
    if (errnum == -1)
    {
	assert(errstring.str);
	return errstring.str;
    }
    else
	return strerror(errnum);
}


void WvStream::setbuf(size_t size)
{
    if (sbuf) delete[] sbuf;
    bufsize = size;
    bufcount = 0;
    sbuf = head = new unsigned char[size];
}


size_t WvStream::read(void *buf, size_t count)
{
    if (!sbuf)
	return uread(buf, count);

    // if the buffer is empty, we need to read real data
    if (!bufcount)
	return uread(buf, count);
    
    // otherwise just read from the buffer
    size_t rlen = bufcount < count ? bufcount : count;
    memcpy(buf, head, rlen);
    head += rlen;
    bufcount -= rlen;
    return rlen;
}

#include "strutils.h"
size_t WvStream::uread(void *buf, size_t count)
{
    if (!isok() || !buf || !count) return 0;
    
    int in = ::read(getfd(), buf, count);
    
    if (in < 0 && (errno==EINTR || errno==EAGAIN))
	return 0; // interrupted

    if (in < 0 || (count && in==0))
    {
	seterr(in < 0 ? errno : 0);
	return 0;
    }

    return in;
}


size_t WvStream::uwrite(const void *buf, size_t count)
{
    if (!isok() || !buf || !count) return 0;
    
    // usually people ignore the return value of write(), so we make
    // a feeble attempt to continue even if interrupted.
    int out;
    do
	out = ::write(getfd(), buf, count);
    while (out < 0 && (errno==EINTR || errno==EAGAIN));
    
    if (out < 0 || (count && out==0))
    {
	seterr(out < 0 ? errno : 0);
	return 0;
    }
    return out;
}


// search for a particular character in a buffer.  Like strchr, but ignores
// NUL bytes.
inline static char *find_char(unsigned char *buf, size_t count, char c)
{
    while (count-- > 0)
    {
	if (*(buf++) == c)
	    return (char *)(buf-1);
    }
    return NULL;
}


// NOTE:  wait_msec is implemented wrong, but it has cleaner code this way
// and can at least handle wait vs wait forever vs wait never.
char *WvStream::getline(time_t wait_msec, char separator)
{
    char *ohead, *newline;
    size_t real_bufcount;
   
    if (!sbuf) setbuf(WVSTREAM_BUF_SIZE);
    
    // is a newline in the buffer already?
    if (!bufcount)
	head = sbuf;
    else if (head > sbuf)
    {
	newline = find_char(head, bufcount, separator);
	if (newline)
	{
	    ohead = (char *)head;
	    head = (unsigned char *)newline + 1;
	    bufcount -= newline - ohead + 1;
	    *newline = 0;
	    return ohead;
	}
	else
	{
	    memmove(sbuf, head, bufcount);
	    head = sbuf;
	}
    }
    
    ohead = (char *)head;
    
    // if we get here, we either want to wait a bit or there is data
    // available.
    while (bufcount < bufsize-1 && isok())
    {
	if (wait_msec >= 0)
	{
	    real_bufcount = bufcount;
	    bufcount = 0;  // otherwise select() will always return true!
	    if (!select(wait_msec) && isok())
	    {
		bufcount = real_bufcount;
		return NULL;
	    }
	    bufcount = real_bufcount;
	}

	bufcount += uread(head + bufcount, bufsize - 1 - bufcount);
	newline = find_char(head, bufcount, separator);
	
	// handle "EOF without newline" condition
	if (!isok() && bufcount)
	{
	    newline = (char *)head + bufcount;
	    bufcount++; // leave room for our fake newline
	}

	// okay, we have a line.  Return it.
	if (newline)
	{
	    head = (unsigned char *)newline + 1;
	    bufcount -= newline - ohead + 1;
	    *newline = 0;
	    return ohead;
	}
    }
    
    // did we fill up the buffer without finding a newline?  truncate.
    // This is ugly, especially with regard to wait_msec, but we have
    // little choice.
    if (bufcount >= bufsize-1)
    {
	unsigned char c;
	if (!wait_msec)
	    wait_msec = 1000;

	real_bufcount = bufcount;
	bufcount = 0;
	while (select(wait_msec) && uread(&c, 1) && c != separator)
	    ;
	bufcount = real_bufcount;
	
	// return the part of the string that we did get
	sbuf[bufsize-1] = 0;
	bufcount = 0;
	return ohead;
    }
    
    // we timed out or had a socket error
    return NULL;
}


void WvStream::drain()
{
    char buf[1024];
    while (select(0))
	read(buf, sizeof(buf));
}


bool WvStream::select_setup(fd_set &r, fd_set &w, fd_set &x, int &max_fd,
			    bool readable, bool writable, bool isexception)
{
    int fd;
    if (readable && !buffer_empty()) return true; // already ready
    
    fd = getfd();
    if (readable)    FD_SET(fd, &r);
    if (writable)    FD_SET(fd, &w);
    if (isexception) FD_SET(fd, &x);
    if (max_fd < fd) max_fd = fd;
    
    return false;
}


bool WvStream::select(time_t msec_timeout,
		      bool readable, bool writable, bool isexcept)
{
    int max_fd, sel;
    fd_set r, w, x;
    timeval tv;
    
    if (!readable && !writable && !isexcept)
	return false;  // why are you asking ME?
    if (readable && bufcount)
	return true;  // data already in buffer
    
    if (!isok()) return false;

    max_fd = -1;
    FD_ZERO(&r);
    FD_ZERO(&w);
    FD_ZERO(&x);
    if (isok() && select_setup(r, w, x, max_fd, readable, writable, isexcept))
	return true;
    
    tv.tv_sec = msec_timeout / 1000;
    tv.tv_usec = (msec_timeout % 1000) * 1000;
    
    sel = ::select(max_fd+1, &r, &w, &x,
		   msec_timeout >= 0 ? &tv : (timeval*)NULL);

    if (sel < 0)
    {
	if (errno!=EAGAIN && errno!=EINTR)
	    seterr(errno);
	return false;
    }

    if (!sel)
	return false;	// timed out
    
    return test_set(r, w, x);
}


const WvAddr *WvStream::src() const
{
    return NULL;
}


void WvStream::setdest(const WvAddr &)
{
    // do nothing; we ignore the destination address.
}


bool WvFile::open(const char *filename, int mode)
{
    if (fd >= 0)
	close();
    fd = ::open(filename, mode, 0666);
    if (fd < 0)
    {
	seterr(errno);
	return false;
    }
    else
	return true;
}


///////////////// WvStreamList


WvStreamList::WvStreamList()
{
    sets_valid = false;
    sure_thing = NULL;
    setcallback(dist_callback, this);
}


bool WvStreamList::isok() const
{
    return true;  // "error" condition on a list is undefined
}


bool WvStreamList::select_setup(fd_set &r, fd_set &w, fd_set &x, int &max_fd,
				bool readable, bool writable, bool isexcept)
{
    sure_thing = NULL;
    sets_valid = false;

    Iter i(*this);
    for (i.rewind(); i.cur() && i.next(); )
    {
	WvStream &s = *i.data();
	
	if (!s.isok())
	{
	    i.unlink();
	    continue;
	}
	
	if (readable && !s.buffer_empty())
	{
	    sure_thing = &s;
	    return true;
	}
	
	if (s.isok() 
	    && s.select_setup(r, w, x, max_fd, readable, writable, isexcept))
	{
	    sure_thing = &s;
	    return true;
	}
    }
    
    return false;
}


bool WvStreamList::test_set(fd_set &r, fd_set &w, fd_set &x)
{
    WvStream *s;
    
    sel_r = r;
    sel_w = w;
    sel_x = x;
    sets_valid = true;

    if (sure_thing)
	return true;
    
    Iter i(*this);
    for (i.rewind(); i.cur() && i.next(); )
    {
	s = i.data();
	if (s->test_set(r, w, x))
	{
	    sure_thing = s;
	    return true;
	}
    }
    return false;
}


WvStream *WvStreamList::select_one(int msec_timeout, bool readable,
				   bool writable, bool isexception)
{
    if (select(msec_timeout, readable, writable, isexception))
	return sure_thing;
    return NULL;
}


// distribute the callback() request to all children that select 'true'
int WvStreamList::dist_callback(WvStream &, void *userdata)
{
    WvStreamList &l = *(WvStreamList *)userdata;
    WvStream *s;

    if (!l.sets_valid)
    {
	if (l.sure_thing)
	    return l.sure_thing->callback();
	return 0;
    }
	
    WvStreamList::Iter i(l);
    for (i.rewind(); i.next(); )
    {
	s = i.data();
	if (!s->isok()) continue;
	
	if (s->test_set(l.sel_r, l.sel_w, l.sel_x)
	    && s->select(0, true, true, true))
	{
	    s->callback();
	}
    }
    
    l.sure_thing = NULL;
    l.sets_valid = false;
    return 0;
}
