/****************************************************************************/
/*                                                                          */
/*                         GNAT COMPILER COMPONENTS                         */
/*                                                                          */
/*                               A - I N I T                                */
/*                                                                          */
/*                            $Revision: 1.37 $                              */
/*                                                                          */
/*                          C Implementation File                           */
/*                                                                          */
/*    Copyright (C) 1992, 1993, 1994, 1995 Free Software Foundation, Inc.   */
/*                                                                          */
/* GNAT is free software;  you can  redistribute it  and/or modify it under */
/* terms of the  GNU General Public License as published  by the Free Soft- */
/* ware  Foundation;  either version 2,  or (at your option) any later ver- */
/* sion.  GNAT is distributed in the hope that it will be useful, but WITH- */
/* OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY */
/* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License */
/* for  more details.  You should have  received  a copy of the GNU General */
/* Public License  distributed with GNAT;  see file COPYING.  If not, write */
/* to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, */
/* MA 02111-1307, USA.                                                      */
/*                                                                          */
/* As a  special  exception,  if you  link  this file  with other  files to */
/* produce an executable,  this file does not by itself cause the resulting */
/* executable to be covered by the GNU General Public License. This except- */
/* ion does not  however invalidate  any other reasons  why the  executable */
/* file might be covered by the  GNU Public License.                        */
/*                                                                          */
/* GNAT was originally developed  by the GNAT team at  New York University. */
/* It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). */
/*                                                                          */
/****************************************************************************/

#include "config.h"
#include "a-ada.h"
#include "a-types.h"
#include "a-raise.h"

/* __gnat_initialize is called at the start of execution of an Ada program
   (the call is generated by the binder). The standard routine does nothing
   at all; the intention is that this be replaced by system specific
   code where initialization is required. */

extern struct Exception_Data constraint_error;
extern struct Exception_Data numeric_error;
extern struct Exception_Data program_error;
extern struct Exception_Data storage_error;
extern struct Exception_Data tasking_error;
extern struct Exception_Data _abort_signal;

extern void system__task_specific_data__set_gnat_exception
				(struct Exception_Data *);
extern void *system__task_specific_data__get_exc_stack_addr (void);
extern int (*system__tasking_soft_links__check_abort_status) (void);
extern void *(*system__tasking_soft_links__abort_handler) (void);
extern void  __gnat_reraise (int);

int  __gl_main_priority = -1;
int  __gl_time_slice_val = -1;
char __gl_locking_policy = ' ';
char __gl_queuing_policy = ' ';
char __gl_task_dispatching_policy = ' ';
void (*__gl_adafinal_ptr) () = 0;

void
__gnat_set_globals (main_priority, time_slice_val, locking_policy,
		    queuing_policy, task_dispatching_policy, adafinal_ptr)
     int main_priority;
     int time_slice_val;
     char locking_policy, queuing_policy, task_dispatching_policy;
     void (*adafinal_ptr) ();
{
  __gl_main_priority = main_priority;
  __gl_time_slice_val = time_slice_val;
  __gl_locking_policy = locking_policy;
  __gl_queuing_policy = queuing_policy;
  __gl_task_dispatching_policy = task_dispatching_policy;
  __gl_adafinal_ptr = adafinal_ptr;
}

#if defined (MSDOS)
#include <stdio.h>

/* This is DOS specific.  It does nothing unless run from an 
   environment that creates a gw-gnat.$$$ file.  If that file is present,
   it forces all program output to the console.  This allows the debugger
   to be run sending output to a file, but the program output still goes
   to the console. */

void
__gnat_initialize()
{
  if (access ("gw-gnat.$$$", 0) == 0)
    freopen ("CON", "w", stdout);
}

#elif defined (sgi)

/* This is SGI specific */

#include <signal.h>
#include <siginfo.h>

#define NULL 0
#define SIGADAABORT 34

static void
__gnat_error_handler (sig, sip)
     int sig;
     siginfo_t *sip;
{
  struct Exception_Data *exception;
  static int recurse = 0;

  /* If this was an explicit signal from a "kill", just resignal it unless
     it was SIGADAABORT.  */
  if (sig != SIGADAABORT && SI_FROMUSER (sip))
    {
      signal (sig, SIG_DFL);
      kill (getpid(), sig);
    }

  /* Otherwise, treat it as something we handle.  */
  switch (sig)
    {
    case SIGSEGV:
      /* If the problem was permissions, this is a constraint error.
	 Likewise if the failing address isn't maximally aligned or if 
	 we've recursed.

	 ??? Using a static variable here isn't task-safe, but it's
	 much too hard to do anything else and we're just determining
	 which exception to raise.  */
      if (sip->si_code == SEGV_ACCERR
	  || (((long) sip->si_addr) & 3) != 0
	  || recurse)
	exception = &constraint_error;
      else
	{
	  /* See if the page before the faulting page is accessable.  Do that
	     by trying to access it.  We'd like to simply try to access
	     4096 + the faulting address, but it's not guaranteed to be
	     the actual address, just to be on the same page.  */
	  recurse++;
	  ((volatile char *)
	   ((long) sip->si_addr & - getpagesize ()))[getpagesize ()];
	  exception = &storage_error;
	}
      break;

    case SIGBUS:
      /* Map all bus errors to Constraint_Error.  */
      exception = &constraint_error;
      break;

    case SIGFPE:
      /* Map all fpe errors to Constraint_Error.  */
      exception = &constraint_error;
      break;

    case SIGADAABORT:
      if ((*system__tasking_soft_links__check_abort_status) ())
	exception = &_abort_signal;
      else
	return;

      break;

    default:
      /* Everything else is a Program_Error. */
      exception = &program_error;
    }

  recurse = 0;
  __gnat_raise (exception);
}

static void
__gnat_install_handler ()
{
  stack_t ss;
  struct sigaction act;

  /* Setup signal handler to map synchronous signals to appropriate
     exceptions.  Make sure that the handler isn't interrupted by another
     signal that might cause a scheduling event! */

  act.sa_handler = __gnat_error_handler;
  act.sa_flags = SA_NODEFER + SA_RESTART + SA_SIGINFO;
  (void) sigemptyset (&act.sa_mask);

  (void) sigaction (SIGILL,  &act, NULL);
  (void) sigaction (SIGABRT, &act, NULL);
  (void) sigaction (SIGFPE,  &act, NULL);
  (void) sigaction (SIGSEGV, &act, NULL);
  (void) sigaction (SIGBUS,  &act, NULL);

  (void) sigaction (SIGADAABORT,  &act, NULL);
}

void
__gnat_initialize ()
{
  __gnat_install_handler ();
}

#elif defined (VMS)

/*
 * The prehandler actually gets control first on a condition. It swaps the
 * stack pointer and calls the handler (__gnat_error_handler).
 */
extern long __gnat_error_prehandler ();
extern char* __gnat_error_prehandler_stack;

#ifndef IN_GCC
/*
 * Conditions that don't have an Ada exception counterpart must raise
 * Non_Ada_Error.  Since this is defined in s-auxdec, it should only be
 * referenced by user programs, not the compiler or tools. Hence the
 * #ifdef IN_GCC.
 */
extern struct Exception_Data system__aux_dec__non_ada_error;
#endif

#include <stdio.h>

/*
 * Define macro symbols for the VMS conditions that become Ada exceptions.
 * Most of these are also defined in the header file ssdef.h which has not
 * yet been converted to be recoginized by Gnu C. Some, which couldn't be
 * located, are assigned names based on the DEC test suite tests which
 * raise them.
 */
#define SS$_ACCVIO 12
#define SS$_INTDIV 1156
#define SS$_HPARITH 1284
#define SS$_RESIGNAL 2328
#define SS$_CE24VRU 3253636       /* Write to unopened file ??? */
#define SS$_C980VTE 3246436       /* AST requests time slice ??? */
#define CMA$_EXIT_THREAD 4227492
#define CMA$_EXCCOPLOS 4228108
#define CMA$_ALERTED 4227460

struct descriptor_s {unsigned short len, mbz; char *adr; };

long
__gnat_error_handler (sigargs, mechargs)
     int *sigargs;
     void *mechargs;
{
  struct Exception_Data *exception;
  char message [256];

  switch (sigargs[1])
    {
    case SS$_ACCVIO:
      exception = &storage_error;
      break;
    case SS$_INTDIV:
      exception = &constraint_error;
      break;
    case SS$_HPARITH:
#ifdef IN_GCC
      return SS$_RESIGNAL; /* toplev.c handles for compiler... */
#else
      exception = &constraint_error;
#endif
      break;
    case SS$_CE24VRU:
      exception = &constraint_error;
      break;
    case SS$_C980VTE:
      exception = &program_error;
      break;
    case CMA$_EXIT_THREAD:
      return SS$_RESIGNAL;

    default:
#ifdef IN_GCC
      /* In the compiler and tools, write a message about the condition */
      {
	unsigned short outlen;
	int msg_flag = 0x000f; /* 1 bit for each of the four message parts */
	int msg_code;
	struct descriptor_s msgdesc;

	msgdesc.len = 256;
	msgdesc.mbz = 0;
	msgdesc.adr = message;

	sys$getmsg(sigargs[1], &outlen, &msgdesc, msg_flag, 0);
	message[outlen] = '0';

	fputs (message, stdout);
	return SS$_RESIGNAL;
      }
#else
      /*
       * User programs expect Non_Ada_Error to be raised, reference DEC
       * Ada test CXCONDHAN.
       */
      exception = &system__aux_dec__non_ada_error;
      break;
#endif
    }

  system__task_specific_data__set_gnat_exception (exception);
  __gnat_reraise (-1);

  /*
   * Unhandled exceptions return here to get resignaled which will normally
   * cause a traceback. Some less severe warnings apparently get handled
   * elsewhere in VMS library code (e.g. EXIT_THREAD).
   */
  return SS$_RESIGNAL;
}

static void
install_handler ()
{
  long prvhnd;
  char *c;

  c = (char *) malloc (1025);

  __gnat_error_prehandler_stack = &c[1024];

  /* __gnat_error_prehandler is an assembly function.  */
  sys$setexv (1, __gnat_error_prehandler, 3, &prvhnd);
}

void
__gnat_initialize()
{
  install_handler();
}

#elif defined(__alpha__) && defined(__osf__)

#include <signal.h>
#include <sys/siginfo.h>
#include <sys/ucontext.h>
#include <sys/types.h>

static void
__gnat_error_handler (sig, code, sc)
     int sig;
     int code;
     struct sigcontext *sc;
{
  struct Exception_Data *exception;
  void *exc_stack = system__task_specific_data__get_exc_stack_addr ();

  switch (sig)
    {
    case SIGSEGV:
      exception = &constraint_error;
      break;
    case SIGBUS:
      exception = &constraint_error;
      break;
    case SIGFPE:
      exception = &constraint_error;
      break;
    default:
      exception = &program_error;
    }

  __gnat_raise (exception);
}

static
__gnat_install_handler ()
{
  struct sigaction act;

  /* Setup signal handler to map synchronous signals to appropriate
     exceptions.  Make sure that the handler isn't interrupted by another
     signal that might cause a scheduling event! */

  act.sa_handler = __gnat_error_handler;
  act.sa_flags = SA_ONSTACK + SA_NODEFER;
  (void) sigfillset (&act.sa_mask);

  (void) sigaction (SIGILL,  &act, NULL);
  (void) sigaction (SIGABRT, &act, NULL);
  (void) sigaction (SIGFPE,  &act, NULL);
  (void) sigaction (SIGSEGV, &act, NULL);
  (void) sigaction (SIGBUS,  &act, NULL);
}

void
__gnat_initialize ()
{
  __gnat_install_handler ();
}

#elif defined (sun) && defined (__SVR4)

#include <signal.h>
#include <siginfo.h>

static void
__gnat_error_handler (sig, sip)
     int sig;
     siginfo_t *sip;
{
  struct Exception_Data *exception;
  static int recurse = 0;

  /* If this was an explicit signal from a "kill", just resignal it.  */
  if (SI_FROMUSER (sip))
    {
      signal (sig, SIG_DFL);
      kill (getpid(), sig);
    }

  /* Otherwise, treat it as something we handle.  */
  switch (sig)
    {
    case SIGSEGV:
      /* If the problem was permissions, this is a constraint error.
	 Likewise if the failing address isn't maximally aligned or if 
	 we've recursed.

	 ??? Using a static variable here isn't task-safe, but it's
	 much too hard to do anything else and we're just determining
	 which exception to raise.  */
      if (sip->si_code == SEGV_ACCERR
	  || (((long) sip->si_addr) & 3) != 0
	  || recurse)
	exception = &constraint_error;
      else
	{
	  /* See if the page before the faulting page is accessable.  Do that
	     by trying to access it.  We'd like to simply try to access
	     4096 + the faulting address, but it's not guaranteed to be
	     the actual address, just to be on the same page.  */
	  recurse++;
	  ((volatile char *)
	   ((long) sip->si_addr & - getpagesize ()))[getpagesize ()];
	  exception = &storage_error;
	}
      break;

    case SIGBUS:
      exception = &constraint_error;
      break;

    case SIGFPE:
      exception = &constraint_error;
      break;

    default:
      exception = &program_error;
    }

  recurse = 0;
  __gnat_raise (exception);
}

static void
__gnat_install_handler ()
{
  struct sigaction act;

  /* Setup signal handler to map synchronous signals to appropriate
     exceptions.  Make sure that the handler isn't interrupted by another
     signal that might cause a scheduling event! */

  act.sa_handler = __gnat_error_handler;
  act.sa_flags = SA_NODEFER | SA_RESTART | SA_SIGINFO;
  (void) sigemptyset (&act.sa_mask);

  (void) sigaction (SIGILL,  &act, NULL);
  (void) sigaction (SIGABRT, &act, NULL);
  (void) sigaction (SIGFPE,  &act, NULL);
  (void) sigaction (SIGSEGV, &act, NULL);
  (void) sigaction (SIGBUS,  &act, NULL);
}

void
__gnat_initialize ()
{
   __gnat_install_handler ();
}

#elif defined (linux)

#include <signal.h>

static void
__gnat_error_handler (sig)
     int sig;
{
  struct Exception_Data *exception;

  switch (sig)
    {
    case SIGSEGV:
      /* FIXME: we need to detect the case of a *real* SIGSEGV */
      exception = &storage_error;
      break;
    case SIGBUS:
      exception = &constraint_error;
      break;
    case SIGFPE:
      exception = &constraint_error;
      break;
    default:
      exception = &program_error;
    }

  __gnat_raise (exception);
}

static void
__gnat_install_handler ()
{
  struct sigaction act;

  /* Setup signal handler to map synchronous signals to appropriate
     exceptions.  Make sure that the handler isn't interrupted by another
     signal that might cause a scheduling event! */

  act.sa_handler = __gnat_error_handler;
  act.sa_flags = SA_NODEFER | SA_RESTART;
  (void) sigemptyset (&act.sa_mask);

  (void) sigaction (SIGILL,  &act, NULL);
  (void) sigaction (SIGABRT, &act, NULL);
  (void) sigaction (SIGFPE,  &act, NULL);
  (void) sigaction (SIGSEGV, &act, NULL);
  (void) sigaction (SIGBUS,  &act, NULL);
}

void
__gnat_initialize ()
{
   __gnat_install_handler ();
}

#else

void
__gnat_initialize ()
{
}

#endif

#ifdef _WIN32
void
__gnat_init_float ()
{
#ifdef	__i386__

  /* This is used to properly initialize the FPU on an x86 for each
     process thread. */

#define FPU_RESERVED	0xF0C0
#define	FPU_DEFAULT	0x033F

  volatile unsigned short cw;

  /* Get fp control word */
  __asm__ volatile ("fnstcw %0" : "=m" (cw) : );

  /* Do initialization masking.  */
  cw &= FPU_RESERVED;
  cw |= FPU_DEFAULT;

  /* Set the control word.  */
  __asm__ volatile ("fldcw %0" :: "m" (cw));

#endif	/* Defined __i386__ */
}
#endif
