/*
Copyright (c) 1997 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

$Id: idlparser.y,v 1.16 1997/11/08 03:23:45 janssen Exp $
*/

/* This code was contributed by Martin von Lwis */

/*
  Theory of operation:
  The stubbers invoke ParseFile, which in turn calls ParseIDLFile.
  This function first calls the parser, which generates an abstract 
  syntax tree. The flex scanner already resolves all preprocessor macros
  except #pragmas, which are passed as tokens to the parser (FIXME: this
  is not the best way of dealing with pragmas).
  The abstract syntax tree is identical to the concrete syntax, except:
  - nested type definitions as in
    struct foo{
      struct bar{long i;} j;
    };
    are inserted as a definition, i.e. bar is on one level as j.
  - enumeration values are inserted as definitions on the same level
    as the enumeration.
  - for unions, a complex switch type is inserted as a definition into
    the union
  After the parsing is finished, several passes process the tree:
  1. backlink: all names get a pointer to the definition where they are
     used. all definitions get a pointer to the environment scope
  2. resolvenames: all definition references get a pointer to the 
     corresponding definition. Errors are produced for unresolvable names.
  3. check: various consistency checks are performed (ambiguous definitions,
     type references that don't reference types, cycle checks). Also, constant
     values are computed and checked whether they are within range.
  4. setuid: all entities are assigned repository IDs.
  5. make_toplevel: non-module definitions are inserted into pseudo-modules.
  6. lift: types in nested scopes are lifted to the toplevel module, prefixing
     the name with the relative scoped.
  7. makeisl: all entities get corresponding ISL definitions. While this
     is in progress, ISL definitions of referenced entities might not be there.
  8. update: all cross-scope references are updated so that the ISL points
     to the ISL of the referenced entity.
  9. select_import: the INTERFACEs are sorted into either imported or defined
     ones
  Pending issues:
  - There is no way of updating the KnownInterfaces of ilu.bison. So, once
    everything is parsed, ilu.bison will find that some modules are missing
    and parse all included interfaces again
  - There is no way of telling ilu.bison that a union branch is default
    FigureUnionIDs will overwrite the default_arm, and search for one
    that points to iluparser_DefaultUnionArm, which is static in ilu.bison.
  - the parser is not really re-entrant. Basically, this is because the
    lexer is not re-entrant.
*/
%{
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <iluptype.h>
#include "iluidl.h"

#if (defined(WIN32) || defined(WIN16))
#include <malloc.h>	/* needed because of include of alloca */
#endif

/* tunable parameters. For 2.0a10 compatibility, all should be 0 */
#define LOCAL_TYPES_HAVE_REP_IDS  0
#define HAVE_SCOPING              1
#define HAVE_STRNDUP              0

/* Make the Bison parser produce better error messages */
#define YYERROR_VERBOSE
#include "idlparser-output.c"

  /* missing prototypes */
int yylex(void);
void FigureTypeUID(Type t);

#define ADD_PRIMITIVE_TYPES  \
  ADD_PRIMITIVE_TYPE (void)\
  ADD_PRIMITIVE_TYPE (short)\
  ADD_PRIMITIVE_TYPE (long)\
  ADD_PRIMITIVE_TYPE (long_long)\
  ADD_PRIMITIVE_TYPE (float)\
  ADD_PRIMITIVE_TYPE (double)\
  ADD_PRIMITIVE_TYPE (unsigned_short)\
  ADD_PRIMITIVE_TYPE (unsigned_long)\
  ADD_PRIMITIVE_TYPE (unsigned_long_long)\
  ADD_PRIMITIVE_TYPE (octet)\
  ADD_PRIMITIVE_TYPE (boolean)\
  ADD_PRIMITIVE_TYPE (char)\
  ADD_PRIMITIVE_TYPE (wchar)\
  ADD_PRIMITIVE_TYPE (object)\
  ADD_PRIMITIVE_TYPE (any)

#define ADD_PRIMITIVE_TYPE(x)  static IDLType the_##x##_t;
ADD_PRIMITIVE_TYPES
#undef ADD_PRIMITIVE_TYPE

static Type the_CORBA_Object;
static Interface the_ilu_module;
static Type the_ilu_CString;
static IDLType the_string_t;
static IDLType the_fixed_t;
static int next_serial=0;

list the_result=0;

%}
%union{
  refany any; /*unspecified*/
  IDLDefinition definition;
  list definition_list;
  IDLName name;
  list name_list;
  IDLValue value;
  IDLType type;
  boolean flag;
  list string_list;
  list caselabel_list;
  ArgDirection inout;
  char opname;
  IDLCase caselabel;
}
%type <definition_list> definition_list specification interface_body
%type <definition_list> export_list member_list opt_member_list
%type <definition_list> parameter_dcls param_dcl_list switch_body
%type <definition> const_dcl except_dcl module export attr_dcl op_dcl
%type <definition> interface interface_dcl interface_header forward_dcl
%type <definition> struct_type param_dcl pragma constr_type_spec union_type 
%type <definition> enum_type element_spec case type_dcl definition
%type <definition> type_declarator member enumerator
%type <name>  scoped_name simple_declarator complex_declarator 
%type <name>  array_declarator declarator
%type <name_list>    opt_raises_expr scoped_name_list enumerator_list
%type <name_list>    declarators
%type <value> const_expr or_expr xor_expr and_expr literal unary_expr
%type <value> shift_expr add_expr mult_expr primary_expr boolean_literal
%type <value> positive_const_int fixed_array_size string_literal

%type <type>  type_spec simple_type_spec base_type_spec 
%type <type>  string_type sequence_type 
%type <type>  switch_type_spec
%type <type>  template_type_spec floating_pt_type integer_type char_type
%type <type>  boolean_type any_type octet_type const_type signed_int
%type <type>  unsigned_int signed_long_int signed_short_int
%type <type>  unsigned_long_int unsigned_short_int op_type_spec 
%type <type>  param_type_spec object_type
%type <type>  signed_long_long_int unsigned_long_long_int fixed_pt_const_type
%type <type>  fixed_pt_type wide_string_type wide_char_type

%type <flag>             opt_op_attr opt_readonly
%type <inout>            param_attribute
%type <string_list>      opt_context_expr string_literal_list
%type <opname>           unary_operator
%type <caselabel_list>   case_label_list
%type <caselabel>        case_label

%token <name>   IDENT
%token <value>  BOOL_FALSE BOOL_TRUE CHAR_L FLOAT_L INTEGER_L STRING_L
%token <value>  FIXED_PT_L
%token <flag>   ONEWAY
%token <type>   VOID_T
%token <inout>  IN OUT INOUT
%token <definition>    PRAGMA_ID PRAGMA_VERSION PRAGMA_PREFIX
%token ANY_T ATTRIBUTE BOOLEAN_T CASE CHAR_T
%token CONST CONTEXT DEFAULT DOUBLE_T ENUM EXCEPTION FIXED FLOAT_T
%token INTERFACE LONG_T LSHIFT MODULE OBJECT_T NATIVE OCTET_T RAISES
%token READONLY RSHIFT SCOPE SEQUENCE SHORT_T STRING_L STRING_T STRUCT SWITCH
%token TYPEDEF UNION UNSIGNED_T VOID_T WCHAR_T WSTRING_T
%%

specification:	definition_list {the_result=$1;}

definition:	type_dcl ';'
	|	const_dcl ';'	
	|	except_dcl ';'
	|	interface ';'
	|	module ';'
	|	pragma
	;

module:	MODULE IDENT '{' definition_list '}'
	{	$$=new_definition();$$->tag=MODULEtag;
		$$->name=$2;
		$$->u.module.definitions=$4;
	}

definition_list: definition	
        {       $$=iluparser_new_list();deflist_insert($$,$1);}
	|	definition_list definition
	{	deflist_insert($1,$2);
		$$=$1;
	}
	;

interface:	interface_dcl
	| 	forward_dcl

interface_dcl:	interface_header '{' interface_body '}'
	{	$1->u.interface.definitions=$3;
		$$=$1;
	}

forward_dcl:	INTERFACE IDENT
	{	$$=new_definition();$$->tag=INTERFACEFWDtag;
		$$->name=$2;
	}

interface_header:	INTERFACE IDENT
	{	$$=new_definition();$$->tag=INTERFACEtag;
		$$->name=$2;
		$$->u.interface.bases=NULL;
		$$->u.interface.definitions=NULL;
		$$->u.interface.resolution_state=0;
	}
	|	INTERFACE IDENT ':' scoped_name_list
	{	$$=new_definition();$$->tag=INTERFACEtag;
		$$->name=$2;$$->u.interface.bases=$4;
		$$->u.interface.definitions=NULL;
		$$->u.interface.resolution_state=0;
	}
	;

interface_body:	/* empty */         {$$=iluparser_new_list();}
	|	export_list

export:	type_dcl ';'
	|	const_dcl ';'
	|	except_dcl ';'
	|	attr_dcl ';'
	|	op_dcl ';'

export_list:	export 
	{ 
		$$=iluparser_new_list();
		deflist_insert($$,$1);
	}
	|	export_list export
	{	$$=$1;deflist_insert($$,$2);
	}

scoped_name:	IDENT
	|	SCOPE IDENT		{$$=$2;$2->scope=new_name();}
	|	scoped_name SCOPE IDENT {$$=$3;$$->scope=$1;}
	;

scoped_name_list:	scoped_name         
	{	$$=iluparser_new_list();
		list_insert($$,$1);
	}
	| scoped_name_list ',' scoped_name  
	{	$$=$1;list_insert($$,$3);
	}
	;

const_dcl:	CONST const_type IDENT '=' const_expr
	{	$$=new_definition();$$->tag=CONSTtag;
		$$->name=$3;
		$$->u.constant.type=$2;
		$$->u.constant.val=$5;
		$$->u.constant.computed=FALSE;
	}

const_type:	integer_type
	|	char_type
	|	wide_char_type
	|	boolean_type
	|	octet_type
	|	floating_pt_type
	|	string_type
	|	wide_string_type
	|	fixed_pt_const_type
	|	scoped_name
	{
		$$=new_referenced($1);
	}

const_expr:	or_expr
or_expr:	xor_expr
	|	or_expr '|' xor_expr
	{	$$=new_binary($1,'|',$3);}		
	;
xor_expr:	and_expr
	|	xor_expr '^' and_expr
	{	$$=new_binary($1,'^',$3);}		
	;
and_expr:	shift_expr
	|	and_expr '&' shift_expr
	{	$$=new_binary($1,'&',$3);}		
shift_expr:	add_expr
	|	shift_expr LSHIFT add_expr
	{	$$=new_binary($1,'L',$3);}		
	|	shift_expr RSHIFT add_expr
	{	$$=new_binary($1,'R',$3);}		
add_expr:	mult_expr
	|	add_expr '+' mult_expr
	{	$$=new_binary($1,'+',$3);}		
	|	add_expr '-' mult_expr
	{	$$=new_binary($1,'-',$3);}		
mult_expr:	unary_expr
	|	mult_expr '*' unary_expr
	{	$$=new_binary($1,'*',$3);}		
	|	mult_expr '/' unary_expr
	{	$$=new_binary($1,'/',$3);}
	|	mult_expr '%' unary_expr
	{	$$=new_binary($1,'%',$3);}
unary_expr:	unary_operator primary_expr {$$=new_unary($1,$2);}
	|	primary_expr
unary_operator:	'+' 	{$$='+';}
	| '-' 		{$$='-';}
	| '~'		{$$='~';}
primary_expr:	scoped_name	
	{	$$=new_value();$$->tag=idl_named;
		$$->u.named=$1;
	}
	|	literal
	|	'(' const_expr ')' {$$=$2;}
literal:	INTEGER_L
	|	string_literal
	|	CHAR_L
	|	FIXED_PT_L
	|	FLOAT_L
	|	boolean_literal

boolean_literal:	BOOL_TRUE | BOOL_FALSE
positive_const_int:	const_expr  /* check for int, positive is done later */

string_literal: STRING_L
	|	string_literal STRING_L
	{	char *s=iluparser_Malloc(strlen($1->u.string)+
			strlen($2->u.string)+1);
		strcpy(s,$1->u.string);strcat(s,$2->u.string);
		$$=$1;
		$$->u.string=s;
	}

type_dcl:	TYPEDEF type_declarator	{$$=$2;}
	|	struct_type		
	|	union_type
	|	enum_type
	|	NATIVE simple_declarator
	{	$$=new_definition();$$->tag=TYPEtag;
		$$->name=$2;
	  	$$->u.type=new_type();$$->u.type->tag=NATIVEtag;
	}

type_declarator:	type_spec declarators
	{	$$=new_definition();$$->tag=TYPELISTtag;
		$$->u.typelist.type=$1;
		$$->u.typelist.names=$2;
	}

type_spec:	simple_type_spec
	|	constr_type_spec       
	{	$$=new_type();$$->tag=DEFINEDtag;
		$$->u.defined=$1;
		$$->name=$1->name;
	}

simple_type_spec:	base_type_spec
	|	template_type_spec
	|	scoped_name            
	{
		$$=new_referenced($1);
	}

base_type_spec:	floating_pt_type
	|	integer_type
	|	char_type
	|	wide_char_type
	|	boolean_type
	|	octet_type
	|	object_type
	|	any_type

template_type_spec:	sequence_type
	|	string_type
	|	wide_string_type
	|	fixed_pt_type

constr_type_spec:	struct_type
	|	union_type
	|	enum_type

declarators:	declarator                 
	{	$$=iluparser_new_list();
		list_insert($$,$1);
	}
	|	declarators ',' declarator
	{	$$=$1;list_insert($$,$3);}

declarator:	simple_declarator
	|	complex_declarator

simple_declarator:	IDENT
complex_declarator:	array_declarator

floating_pt_type:	FLOAT_T  {$$=the_float_t;}
	| 		DOUBLE_T {$$=the_double_t;}

/* long long types as adopted in 96-05-04 */
integer_type:	signed_int
	| 	unsigned_int

signed_int:	signed_long_int
	|	signed_short_int
	|	signed_long_long_int
signed_long_long_int:   LONG_T LONG_T   {$$=the_long_long_t;}
signed_long_int:	LONG_T		{$$=the_long_t;}
signed_short_int:	SHORT_T		{$$=the_short_t;}

unsigned_int:	unsigned_long_int
	|	unsigned_short_int
	|	unsigned_long_long_int
unsigned_long_long_int: UNSIGNED_T LONG_T LONG_T {$$=the_unsigned_long_long_t;}
unsigned_long_int:	UNSIGNED_T LONG_T	 {$$=the_unsigned_long_t;}
unsigned_short_int:	UNSIGNED_T SHORT_T	 {$$=the_unsigned_short_t;}

char_type:	CHAR_T		{$$=the_char_t;}
wide_char_type:	WCHAR_T		{$$=the_wchar_t;}
boolean_type:	BOOLEAN_T	{$$=the_boolean_t;}
octet_type:	OCTET_T		{$$=the_octet_t;}
any_type:	ANY_T		{$$=the_any_t;}
object_type:	OBJECT_T	{$$=the_object_t;}

struct_type:	STRUCT IDENT '{' member_list '}'
	{	$$=new_definition();$$->tag=TYPEtag;
		$$->name=$2;
		$$->u.type=new_type();
		$$->u.type->tag=STRUCTtag;
		$$->u.type->has_scope=TRUE;
		$$->u.type->u.structure=$4;
		$$->u.type->name=$2;
	}
member_list:	member			
	{	$$=iluparser_new_list();deflist_insert($$,$1);}
	|	member_list member
	{	deflist_insert($1,$2);$$=$1;}
member:	type_spec declarators ';'
	{	$$=new_definition();$$->tag=MEMBERLISTtag;
		$$->u.typelist.type=$1;
		$$->u.typelist.names=$2;
	}

union_type:	UNION IDENT SWITCH '(' switch_type_spec ')' '{' switch_body '}'
	{	$$=new_definition();$$->tag=TYPEtag;
		$$->name=$2;
		$$->u.type=new_type();$$->u.type->tag=UNIONtag;
		$$->u.type->has_scope=TRUE;
		$$->u.type->u._union.head=$5;
		/* make identifiers in switch_type_spec visible */
		type_insert($8,$5);
		$$->u.type->u._union.body=$8;
		$$->u.type->u._union._default=0;
	}

switch_type_spec:	integer_type
	|	char_type
	|	wide_char_type
	|	boolean_type
	|	octet_type
	|	enum_type
	{	$$=new_type();$$->tag=DEFINEDtag;$$->u.defined=$1;}
	|	scoped_name
	{	$$=new_referenced($1);
	}

switch_body:	case
	{	$$=iluparser_new_list();deflist_insert($$,$1);}
	|	switch_body case
	{	$$=$1;deflist_insert($$,$2);}

case:	case_label_list element_spec ';'
	{	$$=$2;$$->u._case.labels=$1;}

case_label_list:	case_label
	{	$$=iluparser_new_list();list_insert($$,$1);}
	|	case_label_list case_label
	{	$$=$1;list_insert($$,$2);}

case_label:	CASE const_expr ':' 
	{	$$=new_case();$$->value=$2;}
	|	DEFAULT ':'
	{	$$=new_case();}

element_spec:	type_spec declarator
	{	$$=new_definition();$$->tag=CASEtag;
		$$->name=$2;
		$$->u._case.type=$1;
	}

enum_type:	ENUM IDENT '{' enumerator_list '}'
	{	$$=new_definition();$$->tag=TYPEtag;
		$$->name=$2;
		$$->u.type=new_type();
		$$->u.type->tag=ENUMtag;
		$$->u.type->u.enumerated=$4;
	}
enumerator_list: enumerator
	{	$$=iluparser_new_list();list_insert($$,$1);}
	|	enumerator_list ',' enumerator
	{	$$=$1;list_insert($$,$3);}
enumerator:	IDENT
	{	$$=new_definition();$$->tag=ENUMVALtag;
		$$->name=$1;
		$$->u.enumval=new_value();
		$$->u.enumval->tag=idl_enum;
		$$->u.enumval->u.enumerated.type=0;
		$$->u.enumval->u.enumerated.name=$1;
	}

sequence_type:	SEQUENCE '<' simple_type_spec ',' positive_const_int '>'
	{	$$=new_type();$$->tag=SEQUENCEtag;
		$$->u.sequence.size=$5;
		$$->u.sequence.type=$3;
	}
	|	SEQUENCE '<' simple_type_spec '>'
	{	$$=new_type();$$->tag=SEQUENCEtag;
		$$->u.sequence.size=NULL;
		$$->u.sequence.type=$3;
	}

string_type:	STRING_T '<' positive_const_int '>'
	{	$$=new_type();$$->tag=STRINGtag;$$->u.stringsize=$3;}
	|	STRING_T               
	{	$$=the_string_t;}

wide_string_type:	WSTRING_T '<' positive_const_int '>'
	{	$$=new_type();$$->tag=WSTRINGtag;$$->u.stringsize=$3;}
	|		WSTRING_T
	{	$$=new_type();$$->tag=WSTRINGtag;$$->u.stringsize=0;}

array_declarator:	IDENT fixed_array_size
	{	$1->array=iluparser_new_list();list_insert($$->array,$2);
		$$=$1;
	}
	|	array_declarator fixed_array_size
	{	$$=$1;list_insert($$->array,$2);}

fixed_array_size:	'[' positive_const_int ']'	{$$=$2;}

attr_dcl:	opt_readonly ATTRIBUTE param_type_spec simple_declarator
	{	$$=new_definition();$$->tag=ATTRIBUTEtag;
		$$->name=$4;
		$$->u.attribute.readonly=$1;
		$$->u.attribute.type=$3;
		$$->u.attribute.names=0;
	}
	|	attr_dcl ',' simple_declarator
	{	if($1->tag==ATTRIBUTEtag){
			$1->tag=ATTRLISTtag;
			$1->u.attribute.names=iluparser_new_list();
			list_insert($1->u.attribute.names,$1->name);
			$1->name=0;
		}
		$$=$1;list_insert($1->u.attribute.names,$3);
	}

opt_readonly:	/*empty*/	{$$=FALSE;}
	|	READONLY	{$$=TRUE;}

except_dcl:	EXCEPTION IDENT '{' opt_member_list '}' 
	{	$$=new_definition();$$->tag=EXCEPTIONtag;
		$$->name=$2;
		$$->u.exception.members=$4;
	}
opt_member_list:	/*empty*/ {$$=0;}
	|	member_list

op_dcl: opt_op_attr op_type_spec IDENT parameter_dcls 
	opt_raises_expr opt_context_expr
	{	$$=new_definition();$$->tag=OPERATIONtag;
		$$->u.operation.oneway=$1;
		$$->u.operation.returntype=$2;
		$$->name=$3;
		$$->u.operation.parameters=$4;
		$$->u.operation.raises=$5;
		$$->u.operation.context=$6;
	}

opt_op_attr:	/*empty*/	{$$=FALSE;}
	|	ONEWAY		{$$=TRUE;}

op_type_spec:	param_type_spec
	|	VOID_T		{$$=the_void_t;}

parameter_dcls:	'(' param_dcl_list ')'	{ $$=$2;}
	|	'(' ')'			
	{	$$=iluparser_new_list();}

param_dcl_list: param_dcl		
	{ 	$$=iluparser_new_list();list_insert($$,$1);}
	|	param_dcl_list ',' param_dcl
		{ list_insert($1,$3);$$=$1;}

param_dcl:	param_attribute param_type_spec simple_declarator
	{	$$=new_definition();$$->tag=PARAMETERtag;
		$$->u.parameter.direction=$1;
		$$->u.parameter.type=$2;
		$$->name=$3;
	}

param_attribute:	IN	{$$=In;}
	|		OUT 	{$$=Out;}
	|		INOUT	{$$=InOut;}

opt_raises_expr:	/*empty*/		{$$=0;}
	|	RAISES '(' scoped_name_list ')'	{$$=$3;}

opt_context_expr:	/*empty*/			{$$=0;}
	|	CONTEXT '(' string_literal_list ')'	{$$=$3;}

string_literal_list:	string_literal
	{	$$=iluparser_new_list();list_insert($$,$1);}
	| 	string_literal_list ',' string_literal
	{	$$=$1;list_insert($$,$3);}

param_type_spec:	base_type_spec
	|	string_type
	|	wide_string_type
	|	fixed_pt_type
	|	scoped_name
	{
		$$=new_referenced($1);
	}

fixed_pt_type:	FIXED '<' positive_const_int ',' positive_const_int '>'
	{	$$=new_type();$$->tag=FIXEDtag;
		$$->u.fixed.digits=$3;
		$$->u.fixed.fraction=$5;
	}

fixed_pt_const_type:	FIXED	{$$=the_fixed_t;}

pragma:		PRAGMA_VERSION
	|	PRAGMA_ID
	|	PRAGMA_PREFIX

%%

/* Constructors */
IDLDefinition new_definition()
{
  IDLDefinition result;
  result=(IDLDefinition)iluparser_Malloc(sizeof(struct idl_definition));
  result->tag=NILtag;
  result->name=0;
  result->isl=0;
  result->env=0;
  result->prefix=0;
  result->id=0;
  result->version=0;
  return result;
}

IDLType new_type()
{
  IDLType result;
  result=(IDLType)iluparser_Malloc(sizeof(struct idl_type));
  result->tag=NULLTYPEtag;
  result->has_scope=FALSE;
  result->isl=NULL;
  result->name=NULL;
  result->anon_def=NULL;
  return result;
}

IDLName new_name()
{
  IDLName result=(IDLName)iluparser_Malloc(sizeof(struct idl_name));
  next_serial++;
  result->serial=next_serial;
  result->file=NULL;
  result->line=0;
  result->env=NULL;
  result->scope=NULL;
  result->name=NULL;
  result->lifted=NULL;
  result->value=NULL;
  result->array=NULL;
  result->no_ordering=0;
  return result;
}

IDLValue new_value()
{
  IDLValue result;
  result=(IDLValue)iluparser_Malloc(sizeof(struct idl_value));
  result->tag=idl_void;
  result->value=NULL;
  result->isl=NULL;
  return result;
}

IDLCase new_case()
{
  IDLCase result;
  result=(IDLCase)iluparser_Malloc(sizeof(struct idl_case));
  result->value=0;
  result->isl=0;
  return result;
}

/*********************** complex constructors ***************************/

IDLType new_referenced(IDLName n)
{		
  IDLType result=new_type();
  result->tag=REFERENCEDtag;
  result->u.referenced.name=n;
  result->u.referenced.val=0;
  return result;
}

IDLValue new_unary(char opname,IDLValue val)
{
  IDLValue result=new_value();
  result->tag=idl_unary;
  result->u.unary.operator=opname;
  result->u.unary.arg=val;
  return result;
}

IDLValue new_binary(IDLValue val1,char opname,IDLValue val2)
{
  IDLValue result=new_value();
  result->tag=idl_binary;
  result->u.binary.operator=opname;
  result->u.binary.arg1=val1;
  result->u.binary.arg2=val2;
  return result;
}

static boolean
is_anonymous(IDLType t)
{
  switch(t->tag){
  case FIXEDtag:
    /* this is not supported, yet */
  case WSTRINGtag:
    /* if there was an ilu.WString type, this would not be anonymous */
  case ARRAYtag:
  case SEQUENCEtag:
    return t->name==NULL;
  case STRINGtag:
    /* can use ilu.CString */
    if(t->u.stringsize)return t->name==NULL;
    return t->name==NULL;
  default:
    break;
  }
  return FALSE;
}

static void
declare_type(refany name,refany rock)
{
  refany *r=rock;
  IDLType t,t1;
  IDLDefinition d=new_definition();
  d->tag=TYPEtag;
  d->name=name;
  t=r[0];
  if(d->name->array){
    t1=new_type();
    t1->tag=ARRAYtag;
    t1->name=name;
    t1->u.array.type=t;
    t1->u.array.dimensions=d->name->array;
    d->name->array=0;
    t=t1;
  }else{
    /* this is a possible alias definition */
    if(is_anonymous(t))
      t->name=name;
    else{
      t1=new_type();
      t1->tag=ALIAStag;
      t1->name=name;
      t1->u.alias=t;
      t=t1;
    }
  }
  d->u.type=t;
  list_insert(r[1],d);
}
	
static void
declare_member(refany name,refany rock)
{
  refany *r=rock;
  IDLDefinition d=new_definition();
  d->tag=MEMBERtag;
  d->name=name;
  d->u.member=r[0];
  list_insert(r[1],d);
}

static void
declare_attribute(refany name,refany rock)
{
  refany *r=rock;
  IDLDefinition a=r[0];
  IDLDefinition d=new_definition();
  d->tag=ATTRIBUTEtag;
  d->name=name;
  d->u.attribute.type=a->u.attribute.type;
  d->u.attribute.readonly=a->u.attribute.readonly;
  d->u.attribute.names=0;
  list_insert(r[1],d);
}

void
type_insert(list l,IDLType t)
{
  if(t->tag!=DEFINEDtag)return;
  /* typedef struct foo{long bar;} foobar,baz; */
  deflist_insert(l,t->u.defined);
}

static void
declare_enum(refany e,refany rock)
{
  refany *r=rock;
  IDLDefinition d=e;
  d->u.enumval->u.enumerated.type=r[0];
  list_insert(r[1],e);
}

void deflist_insert(list l,IDLDefinition d)
{
  refany rock[2];
  switch(d->tag){
  case TYPELISTtag:
    type_insert(l,d->u.typelist.type);
    rock[0]=d->u.typelist.type;
    rock[1]=l;
    list_enumerate(d->u.typelist.names,declare_type,rock);
    break;
  case MEMBERLISTtag:
    type_insert(l,d->u.typelist.type);
    rock[0]=d->u.typelist.type;
    rock[1]=l;
    list_enumerate(d->u.typelist.names,declare_member,rock);
    break;
  case ATTRLISTtag:
    type_insert(l,d->u.attribute.type);
    rock[0]=d;
    rock[1]=l;
    list_enumerate(d->u.attribute.names,declare_attribute,rock);
    break;
  case CASEtag:
    type_insert(l,d->u._case.type);
    list_insert(l,d);
    break;
  case TYPEtag:
    if(d->u.type->tag==ENUMtag){
      /* insert enum values as definitions */
      rock[0]=d->u.type;
      rock[1]=l;
      list_enumerate(d->u.type->u.enumerated,declare_enum,rock);
    }
  default:
    list_insert(l,d);
  }
}

static Type new_Type(void);
static TypeDescription new_TypeDescription(void);
static Name new_Name(void);

static void
primitive_setuid(Type T)
{
  if(!T->uid){
    char buf[100];
    FigureTypeUID(T);
    /* FIXME: work around FigureTypeUid setting the mark */
    sprintf(buf,"(%s)",type_name(T));
    name_set_lang_name(T->name,"parser:full",buf);
  }
}

static boolean
find_ilu_type(refany object,refany string)
{
  Type t=object;
  return strcmp(name_base_name(t->name),string)==0;
}

void init_types()
{
	IDLType v;
	Type t;
	TypeDescription d;
#define ADD_PRIMITIVE_TYPE(n)\
	v=the_##n##_t= new_type();\
	v->tag=BASICtag;\
	v->u.basic=idl_##n;

  ADD_PRIMITIVE_TYPES;

  /* need to get ilu before making a call to FigureTypeUID */
  the_ilu_module = GetInterface("ilu",0);

#undef ADD_PRIMITIVE_TYPE
#define ADD_PRIMITIVE_TYPE(n,str,tk)\
        v=the_##n##_t;\
	v->isl=t=new_Type();\
	t->description=d=new_TypeDescription();\
	d->type=tk;\
	t->builtIn=TRUE;\
	name_set_base_name(t->name,str);\
	primitive_setuid(t);

  ADD_PRIMITIVE_TYPE (short, "shortinteger", shortinteger_Type);
  ADD_PRIMITIVE_TYPE (long, "integer", integer_Type);
  ADD_PRIMITIVE_TYPE (long_long, "longinteger", longinteger_Type);
  ADD_PRIMITIVE_TYPE (unsigned_short, "shortcardinal", shortcardinal_Type);
  ADD_PRIMITIVE_TYPE (unsigned_long, "cardinal", cardinal_Type);
  ADD_PRIMITIVE_TYPE (unsigned_long_long, "longcardinal", longcardinal_Type);
  ADD_PRIMITIVE_TYPE (double, "real", real_Type);
  ADD_PRIMITIVE_TYPE (float, "shortreal", shortreal_Type);
  ADD_PRIMITIVE_TYPE (octet, "byte", byte_Type);
  ADD_PRIMITIVE_TYPE (boolean, "boolean", boolean_Type);
  ADD_PRIMITIVE_TYPE (char, "shortcharacter", shortcharacter_Type);
  ADD_PRIMITIVE_TYPE (wchar, "character", character_Type);
  ADD_PRIMITIVE_TYPE (any, "pickle", pickle_Type);

  /* not supported by IDL
  ADD_PRIMITIVE_TYPE ( "longreal", longreal_Type);
  */

  /* look for ilu.CORBA-Object */
  the_CORBA_Object = list_find(the_ilu_module->classes,
			       find_ilu_type,"CORBA-Object");
  the_ilu_CString = list_find(the_ilu_module->types,find_ilu_type,"CString");
  /* import this type */
  t=new_Type();
  t->name=new_Name();
  name_set_base_name(t->name,name_base_name(the_CORBA_Object->name));
  name_set_lang_name(t->name,"import","CORBA-Object");
  t->supertype=the_CORBA_Object;
  t->importInterfaceName = "ilu";
  t->interface=the_CORBA_Object->interface;
  t->uid=the_CORBA_Object->uid;
  the_CORBA_Object=t;
  the_object_t = new_type();
  the_object_t->tag=BASICtag;
  the_object_t->u.basic=idl_object;
  the_object_t->isl=the_CORBA_Object;

  the_string_t=new_type();
  the_string_t->tag=STRINGtag;
  the_string_t->u.stringsize=0;
  the_string_t->isl=t=new_Type();
  /* make this type an imported ilu.CString */
  t->name=new_Name();
  name_set_base_name(t->name,name_base_name(the_ilu_CString->name));
  name_set_lang_name(t->name,"import","CString");
  t->supertype=the_ilu_CString;
  t->importInterfaceName="ilu";
  t->interface=the_ilu_CString->interface;
  t->uid=the_ilu_CString->uid;

  the_fixed_t=new_type();
  the_fixed_t->tag=FIXEDtag;
  the_fixed_t->u.fixed.digits=0;
  the_fixed_t->u.fixed.fraction=0;
  /* FIXME: ISL for fixed types */
}

int idlerror(char *s)
{
#ifdef YYERROR_VERBOSE
  extern char *idltext;
  fprintf(stderr,"%s:%ld:parse error before '%s'\n%s\n",
	  idlcurrentfile(),(long)idlcurrentline(),idltext,s);
#else
  fprintf(stderr,"%s:%ld:%s\n",idlcurrentfile(),idlcurrentline(),s);
#endif
  return 0;
}

static void
name_warning(IDLName n,char *s)
{
  if(n)
    fprintf(stderr,"%s:%ld:%s:%s\n",n->file,(long)n->line,n->name,s);
  else
    fprintf(stderr,"<no line>:%s\n",s);
}

static void
name_error(IDLName n,char *s)
{
  name_warning(n,s);
  exit(1);
}

static char*
new_anon_type()
{
  static int anon_counter=0;
  return aprintf("AnonType-%d-",++anon_counter);
}

/********* Print functions **************************************/
static void 
name_print(FILE *f,IDLName n)
{
  if(n->scope){
    name_print(f,n->scope);
    fprintf(f,"::");
  }else if(n->name)
    fprintf(f,"%s",n->name);
  else
    fprintf(f,"::");
}

void definition_print(FILE* f,IDLDefinition d)
{
  char *n="unknown definition";
  if(d->env)
    definition_print(f,d->env);
  if(d->name && d->name->name)
    n=d->name->name;
  fprintf(f,"::%s",n);
}

/********* Converting to ISL ************************************/

/* copied from ilu.bison */
static Name new_Name(void)
{
  Name new = (Name) iluparser_Malloc (sizeof(struct name_s));
  new->base_name = NULL;
  new->langnames = (set) new_list();
  return (new);
}

static Exception new_Exception(void)
{
  Exception new = (Exception) iluparser_Malloc (sizeof(struct ilu_exception_s));
  new->name = new_Name();
#if HAVE_SCOPING
  new->scoping = iluparser_new_list();
#else
  new->scoping = NULL;
#endif
  new->type = NULL;
  new->valueOptional = FALSE;
  new->refs = new_list();
  new->def = 0;
  new->id = 0;
  new->builtIn = FALSE;
  new->interface = NULL;
  new->importInterfaceName = NULL;
  new->import = NULL;
  new->marked = FALSE;
  new->corba_rep_id = NULL;
  new->doc_string = NULL;
  return (new);
}

static Type new_Type(void)
{
  Type new = (Type) iluparser_Malloc (sizeof(struct ilu_type_s));
#if HAVE_SCOPING
  new->scoping = iluparser_new_list();
#else
  new->scoping = NULL;
#endif
  new->name = (Name) new_Name();
  new->description = NULL;
  new->supertype = NULL;
  new->refs = new_list();
  new->def = 0;
  new->builtIn = FALSE;
  new->importInterfaceName = NULL;
  new->interface = NULL;
  new->cached_des = NULL;
  new->marked = FALSE;
  new->uid = NULL;
  return (new);
}

static TypeDescription new_TypeDescription(void)
{
  TypeDescription new = (TypeDescription) iluparser_Malloc(sizeof(struct ilu_typedes_s));
  new->type = invalid_Type;
  memset((char *) new, 0, sizeof(struct ilu_typedes_s));
  return (new);
}

static Procedure new_Procedure(void)
{
  Procedure new = (Procedure) iluparser_Malloc (sizeof(struct ilu_proc_s));
  new->name = new_Name();
  new->object = NULL;
  new->arguments = NULL;
  new->returnType = NULL;
  new->returnOptional = FALSE;
  new->functional = FALSE;
  new->asynch = FALSE;
  new->exceptions = NULL;
  new->def = 0;
  new->id = -1;
  new->interface = NULL;
  new->marked = FALSE;
  /* added MvL */
  new->authentication_type = NULL;
  new->doc_string = NULL;
  return (new);
}

static Argument new_Argument(void)
{
  Argument new = (Argument) iluparser_Malloc (sizeof(struct ilu_argument_s));
  new->name = new_Name();
  new->type = NULL;
  new->values = NULL;
  new->def = 0;
  new->sibling = FALSE;
  new->direction = In;
  return (new);
}

/* end of copy */

static char*
underscore2hyphen(char *s)
{
  char *result=ilu_strdup(s);
  for(s=result;*s;s++)
    if(*s=='_')*s='-';
  return result;
}

/************************* Name lookup ***************************/

static IDLDefinition
up_find_definition(IDLDefinition d,enum idldefinition_tag t)
{
  while(d && d->tag!=t)
    d=d->env;
  return d;
}

static IDLDefinition
toplevel_module(IDLDefinition d)
{
  if(!d->env)return d;
  return toplevel_module(d->env);
}

static boolean
cmp_name(refany def,refany name)
{
  boolean result;
  IDLDefinition d=def;
  /* pragma prefix does not have a name */
  if(!d->name)return FALSE;
  if(d->name->scope)return FALSE;
  /* catch cases where it doesn't define something */
  switch(d->tag){
  case PRAGMA_PREFIXtag:
  case PRAGMA_IDtag:
  case PRAGMA_VERSIONtag:
  case INTERFACEFWDtag:
    return FALSE;
  default:
    break;
  }
  /* IDL is case-insensitive */
  result=ilu_strcasecmp(d->name->name,name)==0;
  /* check for consistent spelling */
  if(result && strcmp(d->name->name,name)!=0)
    name_error(d->name,aprintf(" is also spelled as '%s'.",name));
  return result;
}

struct lookup_base_s{
  char *name;
  list toplevel;
  IDLDefinition result;
  IDLDefinition def;
};

static IDLDefinition lookup_name(IDLDefinition d,char* n,list toplevel);

static void
lookup_base(refany name,refany rock)
{
  struct lookup_base_s *r=rock;
  IDLName n=name;
  IDLDefinition found=lookup_name(n->value,r->name,r->toplevel);
  if(r->result && found && r->result!=found){
    name_error(r->def->name,aprintf("ambiguous resolution for %s",r->name));
    return;
  }
  if(found)
    r->result=found;
}

static IDLDefinition
lookup_name(IDLDefinition d,char* n,list toplevel)
{
  IDLDefinition result=0;
  switch(d->tag){
  case MODULEtag:
    result=list_find(d->u.module.definitions,cmp_name,n);
    break;
  case INTERFACEtag:
    result=list_find(d->u.interface.definitions,cmp_name,n);
    if(!result){
      /* search bases */
      struct lookup_base_s s;
      s.name=n;
      s.toplevel=toplevel;
      s.result=0;
      s.def=d;
      list_enumerate(d->u.interface.bases,lookup_base,&s);
      result=s.result;
    }
    break;
  case OPERATIONtag:
    result=list_find(d->u.operation.parameters,cmp_name,n);
    break;
  case EXCEPTIONtag:
    result=list_find(d->u.exception.members,cmp_name,n);
    break;
  case TYPEtag:
    switch(d->u.type->tag){
    case STRUCTtag:
      result=list_find(d->u.type->u.structure,cmp_name,n);
      break;
    case UNIONtag:
      result=list_find(d->u.type->u._union.body,cmp_name,n);
      break;
    default:
      break;
    }
    break;
  /* inside the other definitions, there are no more names */
  default:
    break;
  }
  if(!result)
    if(!d->env)
      /* not found, need toplevel lookup */
      return list_find(toplevel,cmp_name,n);
    else 
      return lookup_name(d->env,n,toplevel);
  return result;
}

static boolean
find_imported(refany imported,refany name)
{
  return strcmp(((Imported)imported)->name,name)==0;
}

static void
add_import(IDLDefinition m,char *name,char *filename)
{
  Interface I=m->isl;
  Imported imp;
  assert(I);
  /* importing self? */
  if(strcmp(name,m->name->name)==0)
    return;
  if(!I->imports)
    I->imports=iluparser_new_list();
  if(list_find(I->imports,find_imported,name))
    return;
  imp=(Imported)iluparser_Malloc(sizeof(*imp));
  imp->name=name;
  imp->filename=filename;
  list_insert(I->imports,imp);
}

#if HAVE_SCOPING
static void
definition_setscoping(IDLDefinition d,list l)
{
  list_push(l,d->name->name);
  while(d->env){
    d=d->env;
    list_push(l,d->name->name);
  }
}
#endif

/************************** module reopening pass ******************/

static void list_insert1(refany i,refany l)
{
  list_insert(l,i);
}

static void find_duplicate_modules(refany def,refany l)
{
  IDLDefinition d=def;
  if(d->tag==MODULEtag){
    IDLDefinition first=list_find(l,cmp_name,d->name->name);
    if(first && first->tag==MODULEtag)
      /* insert definitions of def into first */
      list_enumerate(d->u.module.definitions,list_insert1,
		     first->u.module.definitions);
    else
      list_insert(l,d);
  }
  else
    /* add this definition */
    list_insert(l,d);
}

static list reopen_modules(list alt);

static void 
module_reopen_modules(refany def,refany rock)
{
  IDLDefinition d=def;
  if(d->tag!=MODULEtag)return;
  d->u.module.definitions=reopen_modules(d->u.module.definitions);
}
  

static list 
reopen_modules(list alt)
{
  list neu=iluparser_new_list();
  list_enumerate(alt,find_duplicate_modules,neu);
  list_enumerate(neu,module_reopen_modules,0);
  /* delete the old list, leaving the items in the new list */
  list_clear(alt,0);
  return neu;
}

/************************** backlink pass **************************/

static void definition_backlink(refany def,refany env);

static void
name_backlink(refany name,refany rock)
{
  IDLName n=name;
  while(n){
    n->env=rock;
    n=n->scope;
  }
}

static void
value_backlink(refany val,refany rock)
{
  IDLValue v=val;
  switch(v->tag){
  case idl_named:
    name_backlink(v->u.named,rock);
    break;
  case idl_unary:
    value_backlink(v->u.unary.arg,rock);
    break;
  case idl_binary:
    value_backlink(v->u.binary.arg1,rock);
    value_backlink(v->u.binary.arg2,rock);
    break;
  default:
    /* nothing to do for other value */
    break;
  }
}

static void
type_backlink(refany type,refany rock)
{
  IDLType t=type;
  switch(t->tag){
  case NULLTYPEtag:
  case BASICtag: 
  case NATIVEtag:
    break;
  case REFERENCEDtag:
    name_backlink(t->u.referenced.name,rock);
    break;
  case ALIAStag:
    type_backlink(t->u.alias,rock);
    break;
  case SEQUENCEtag:
    type_backlink(t->u.sequence.type,rock);
    if(t->u.sequence.size)
      value_backlink(t->u.sequence.size,rock);
    break;
  case STRUCTtag:
    list_enumerate(t->u.structure,definition_backlink,rock);
    break;
  case ENUMtag:
    break;
  case UNIONtag:
    type_backlink(t->u._union.head,rock);
    list_enumerate(t->u._union.body,definition_backlink,rock);
    break;
  case ARRAYtag:
    type_backlink(t->u.array.type,rock);
    list_enumerate(t->u.array.dimensions,value_backlink,rock);
    break;
  case WSTRINGtag:
  case STRINGtag:
    if(t->u.stringsize)
      value_backlink(t->u.stringsize,rock);
    break;
  case FIXEDtag:
    if(t->u.fixed.digits){
      value_backlink(t->u.fixed.digits,rock);
      value_backlink(t->u.fixed.fraction,rock);
    }
    break;
  case DEFINEDtag:
    definition_backlink(t->u.defined,rock);
    break;
  }
}

static void
case_backlink(refany Case,refany rock)
{
  IDLCase c=Case;
  IDLDefinition d=rock;
  c->env=d;
  if(c->value)
    value_backlink(c->value,d->env);
}

static void
definition_backlink(refany def,refany env)
{
  IDLDefinition d=def;
  d->env=env;
  if(d->name){
    d->prefix=env?aprintf("%s/%s",d->env->prefix,d->name->name):d->name->name;
    d->name->lifted=underscore2hyphen(d->name->name);
  }
  switch(d->tag){
    /* those should not happen */
  case NILtag:case TYPELISTtag:case MEMBERLISTtag:case ATTRLISTtag:
    break;
    /* those don't have nested items */
  case INTERFACEFWDtag:
    break;
  case MODULEtag:
    list_enumerate(d->u.module.definitions,definition_backlink,d);
    break;
  case INTERFACEtag:
    list_enumerate(d->u.interface.definitions,definition_backlink,d);
    /* base names are used in the environment of the interface */
    list_enumerate(d->u.interface.bases,name_backlink,d->env);
    break;
  case EXCEPTIONtag:
    list_enumerate(d->u.exception.members,definition_backlink,d);
    break;
  case CONSTtag:
    type_backlink(d->u.constant.type,d->env);
    value_backlink(d->u.constant.val,d->env);
    break;
  case ENUMVALtag:
    break;
  case OPERATIONtag:
    type_backlink(d->u.operation.returntype,d);
    list_enumerate(d->u.operation.parameters,definition_backlink,d);
    list_enumerate(d->u.operation.raises,name_backlink,d);
    if(d->u.operation.context)
      name_warning(d->name,"Context not supported");
    break;
  case ATTRIBUTEtag:
    type_backlink(d->u.attribute.type,d);
    break;
  case PARAMETERtag:
    type_backlink(d->u.parameter.type,d);
    break;
  case TYPEtag:
    if(d->u.type->has_scope)
      type_backlink(d->u.type,d);
    else
      type_backlink(d->u.type,d->env);
    break;
  case MEMBERtag:
    /* structure fields don't define a scope */
    type_backlink(d->u.member,d->env);
    break;
  case CASEtag:
    type_backlink(d->u._case.type,d->env);
    list_enumerate(d->u._case.labels,case_backlink,d);
    break;
  case PRAGMA_IDtag:
  case PRAGMA_VERSIONtag:
    name_backlink(d->name,env);
    break;
  case PRAGMA_PREFIXtag:
    break;
  }
}

/************************* resolvenames pass *************************/
static void definition_resolvenames(refany def,refany rock);

static boolean
find_fwd(refany def,refany name)
{
  IDLDefinition d=def;
  IDLName n=name;
  return d->tag==INTERFACEFWDtag && strcmp(n->name,d->name->name)==0;
}

static void
name_resolvenames(refany name,refany rock)
{
  IDLName n=name;
  if(n->scope){
    /* scoped name */
    if(n->scope->name){
      /* relative scope */
      name_resolvenames(n->scope,rock);
      n->value=lookup_name(n->scope->value,n->name,rock);
    }else
      /* absolute scope */
      n->value=list_find(rock,cmp_name,n->name);
  }else if(n->env)
    /* unscoped name used inside a scope */
    n->value=lookup_name(n->env,n->name,rock);
  else
    /* unscoped name used on the toplevel */
    n->value=list_find(rock,cmp_name,n->name);
  if(!n->value)
    name_error(n,"undefined");
  if(n->value->name->serial > n->serial && !n->no_ordering){
    /* this the value is an interface definition,
       a forward declaration would be enough.
       If the interface is used as a base interface,
       the caller will check it */
    if(n->value->tag==INTERFACEtag){
      IDLDefinition fwd,mod;
      mod=n->value->env;
      /* the interface is better in a module, if it is in something at all */
      if(mod && mod->tag!=MODULEtag)
	name_error(n,"Internal error, please report");
      if(mod)
	fwd=list_find(mod->u.module.definitions,find_fwd,n->value->name);
      else
	fwd=list_find(rock,find_fwd,n->value->name);
      if(!fwd)
	name_error(n,"missing forward declaration");
      if(fwd->name->serial > n->serial)
	name_error(n,"used before first forward declaration");
    }else
      /* the name is defined too late */
      name_error(n,"used before definition");
  }
}

static void
value_resolvenames(refany val,refany rock)
{
  IDLValue v=val;
  switch(v->tag){
  case idl_named:
    name_resolvenames(v->u.named,rock);
    break;
  case idl_unary:
    value_resolvenames(v->u.unary.arg,rock);
    break;
  case idl_binary:
    value_resolvenames(v->u.binary.arg1,rock);
    value_resolvenames(v->u.binary.arg2,rock);
    break;
  default:
    break;
  }
}

static void
type_resolvenames(refany type,refany rock)
{
  IDLType t=type;
  switch(t->tag){
  case NULLTYPEtag:
  case BASICtag:
  case NATIVEtag:
    break;
  case REFERENCEDtag:
    name_resolvenames(t->u.referenced.name,rock);
    break;
  case ALIAStag:
    type_resolvenames(t->u.alias,rock);
    break;
  case SEQUENCEtag:
    type_resolvenames(t->u.sequence.type,rock);
    if(t->u.sequence.size)
      value_resolvenames(t->u.sequence.size,rock);
    break;
  case STRUCTtag:
    list_enumerate(t->u.structure,definition_resolvenames,rock);
    break;
  case ENUMtag:
    break;
  case WSTRINGtag:
  case STRINGtag:
    if(t->u.stringsize)
      value_resolvenames(t->u.stringsize,rock);
    break;
  case FIXEDtag:
    if(t->u.fixed.digits){
      value_resolvenames(t->u.fixed.digits,rock);
      value_resolvenames(t->u.fixed.fraction,rock);
    }
    break;
  case UNIONtag:
    type_resolvenames(t->u._union.head,rock);
    list_enumerate(t->u._union.body,definition_resolvenames,rock);
    break;
  case ARRAYtag:
    type_resolvenames(t->u.array.type,rock);
    list_enumerate(t->u.array.dimensions,value_resolvenames,rock);
    break;
  case DEFINEDtag:
    definition_resolvenames(t->u.defined,rock);
    break;
  }
}

static void
case_resolvenames(refany Case,refany rock)
{
  IDLCase c=Case;
  if(c->value)
    value_resolvenames(c->value,rock);
}

static void
interface_resolvenames(refany name,refany rock)
{
  IDLName n=name;
  IDLDefinition d;
  name_resolvenames(n,rock);
  d=n->value;
  if(d->tag!=INTERFACEtag)
    name_error(n,"does not specify an interface");
  definition_resolvenames(d,rock);
}
 
static void
pragma_setversionid(IDLDefinition d,char *value,int do_id)
{
  switch(d->tag){
  case TYPEtag:
  case INTERFACEtag:
  case EXCEPTIONtag:
  case CONSTtag:
    if(do_id){
      if(d->id)
	name_error(d->name,"Duplicate ID");
      else d->id=value;
    }else{
      if(d->version)
	name_error(d->name,"Duplicate version");
      else d->version=value;
    }
    if(d->id && d->version)
      name_error(d->name,"Both repository ID and version assigned\n");
    break;
  case NILtag:case OPERATIONtag:case MODULEtag:case MEMBERtag:
  case PARAMETERtag:case ATTRIBUTEtag:
  case TYPELISTtag:case ATTRLISTtag:case MEMBERLISTtag:
  case CASEtag:case INTERFACEFWDtag:case ENUMVALtag:
  case PRAGMA_IDtag:case PRAGMA_VERSIONtag:case PRAGMA_PREFIXtag:
    name_error(d->name,do_id ? "Invalid assignment of repository ID":
	       "Invalid assignment of repository version");
  }
}

static void
definition_resolvenames(refany def,refany rock)
{
  IDLDefinition d=def;
  switch(d->tag){
    /* those should not happen */
  case NILtag:case TYPELISTtag:case MEMBERLISTtag:case ATTRLISTtag:
    break;
  case MODULEtag:
    list_enumerate(d->u.module.definitions,definition_resolvenames,rock);
    break;
  case INTERFACEtag:
    /* bases first, because local definitions might refer to bases */
    switch(d->u.interface.resolution_state){
    case 0:
      /* normal operation, resolve */
      d->u.interface.resolution_state=1;
      list_enumerate(d->u.interface.bases,interface_resolvenames,rock);
      d->u.interface.resolution_state=2;
      /* local definitions */
      list_enumerate(d->u.interface.definitions,definition_resolvenames,rock);
      break;
    case 1:
      /* oops, nested call */
      name_error(d->name,"Cycle in base interface resolution");
      break;
    default:break;
    }
    break;
  case CONSTtag:
    type_resolvenames(d->u.constant.type,rock);
    value_resolvenames(d->u.constant.val,rock);
    break;
  case ENUMVALtag:
    break;
  case OPERATIONtag:
    type_resolvenames(d->u.operation.returntype,rock);
    list_enumerate(d->u.operation.parameters,definition_resolvenames,rock);
    /* list_enumerate allows for NULL lists */
    list_enumerate(d->u.operation.raises,name_resolvenames,rock);
    break;
  case ATTRIBUTEtag:
    type_resolvenames(d->u.attribute.type,rock);
    break;
  case EXCEPTIONtag:
    list_enumerate(d->u.exception.members,definition_resolvenames,rock);
    break;
  case MEMBERtag:
    type_resolvenames(d->u.member,rock);
    break;
  case TYPEtag:
    type_resolvenames(d->u.type,rock);
    break;
  case PARAMETERtag:
    type_resolvenames(d->u.parameter.type,rock);
    break;
  case CASEtag:
    type_resolvenames(d->u._case.type,rock);
    list_enumerate(d->u._case.labels,case_resolvenames,rock);
    break;
  case INTERFACEFWDtag:
    /* The corresponding definition has to be an interface,
       and it must be defined in the same scope */
    if(d->env)
      d->name->value=lookup_name(d->env,d->name->name,rock);
    else
      d->name->value=list_find(rock,cmp_name,d->name->name);
    if(!d->name->value ||
       d->name->value->env!=d->env)
      name_error(d->name,"no interface for forward declaration");
    else if(d->name->value->tag!=INTERFACEtag)
      name_error(d->name,"corresponding definition is not an interface");
    break;
  case PRAGMA_PREFIXtag:
    break;
  case PRAGMA_VERSIONtag:
    name_resolvenames(d->name,rock);
    pragma_setversionid(d->name->value,d->u.pragma,FALSE);
    break;
  case PRAGMA_IDtag:
    name_resolvenames(d->name,rock);
    pragma_setversionid(d->name->value,d->u.pragma,TRUE);
    break;
  }
}

/************************* check pass        *************************/

static boolean 
identity(refany v1,refany v2)
{
  return v1==v2;
}

static IDLValue value_compute(IDLValue v,list start,IDLName n,IDLType target);
static void definition_check(refany def,refany rock);
static IDLType type_compute(IDLType t);
static void type_check(IDLType t,IDLName n);

static boolean
isint(enum idltk t)
{
  return t==idl_int || t==idl_octet
    || t==idl_short || t==idl_long || t==idl_long_long 
    || t==idl_unsigned_short || t==idl_unsigned_long || 
    t==idl_unsigned_long_long 
    ;
}

static boolean
isfloat(enum idltk t)
{
  return t==idl_float || t==idl_double;
}

static IDLValue
value_coerce(IDLValue v,IDLType t,IDLName n)
{
  IDLValue v1;
  if(t->tag==BASICtag && v->tag==t->u.basic)
    return v;
  if(t->tag==ENUMtag){
    if(v->tag!=idl_enum)
      name_error(n,"value is not enumerated"); 
    if(type_compute(v->u.enumerated.type)!=t)
      name_error(n,"value is of wrong enumeration");
    return v;
  }
  if(t->tag==STRINGtag || t->tag==WSTRINGtag){
    if(v->tag!=idl_string)
      name_error(n,"value is not a string");
    return v;
  }
  if(t->tag!=BASICtag)
    name_error(n,"Conversion into complex type not possible");
  if(isint(t->u.basic)){
    boolean ok;
    int I=v->u.INT;
    if(!isint(v->tag))
      name_error(n,"Value is not an integral type");
    switch(t->u.basic){
    case idl_octet:            ok= (      0 <= I) && (I<=0xFF);  break;
    case idl_short:            ok= (-0x8000 <= I) && (I<=0x7FFF);break;
    case idl_unsigned_short:   ok= (      0 <= I) && (I<=0xFFFF);break;
    case idl_long_long: /* FIXME: cannot check long long */
    case idl_long:             ok= 1;break;
    case idl_unsigned_long_long: /* FIXME: cannot check long long */
    case idl_unsigned_long:    ok= I>=0;break;
    default:
      name_error(0,"Unknown basic type");
    }
    if(!ok)
      name_error(n,"Value out of range");
    v1=new_value();
    *v1=*v;
    v1->tag=t->u.basic;
    return v1;
  }
  if(isfloat(t->u.basic)){
    if(!isfloat(v->tag))
      name_error(n,"Value is not floating point");
    /* FIXME: range check */
    v1=new_value();
    *v1=*v;
    v1->tag=t->u.basic;
    return v1;
  }
  if(t->u.basic==idl_boolean){
    if(v->tag!=idl_boolean)
      name_error(n,"Value is not boolean");
    return v;
  }
  if(t->u.basic==idl_char || t->u.basic==idl_wchar){
    if(v->tag!=idl_char)
      name_error(n,"Value is not a character");
    return v;
  }
  if(t->u.basic==idl_any)
    name_error(n,"Constants of type any are not allowed");
  name_error(n,"Unsupported type for constants");
  return 0;
}

static IDLValue
definition_compute(IDLDefinition d,list start,IDLName n)
{
  if(d->tag==ENUMVALtag)
    return d->u.enumval;
  if(d->u.constant.computed)
    return d->u.constant.val;
  if(list_find(start,identity,d))
    name_error(d->name,"cycle in computation of value");
  if(d->tag!=CONSTtag)
    name_error(n,aprintf("%s is not a constant",d->name->name));
  list_insert(start,d);
  d->u.constant.val=value_compute(d->u.constant.val,start,
				  d->name,type_compute(d->u.constant.type));
  d->u.constant.computed=TRUE;
  return d->u.constant.val;
}

static IDLValue
value_compute(IDLValue v,list start,IDLName n,IDLType target)
{
  IDLDefinition named;
  IDLValue v1,v2;
  if(v->value)
    return value_coerce(v->value,target,n);
  switch(v->tag){
  case idl_named:
    named=v->u.named->value;
    v->value=definition_compute(named,start,named->name);
    return value_coerce(v->value,target,n);
  case idl_unary:
    v1=value_compute(v->u.unary.arg,start,n,target);
    if(isint(v1->tag)){
      v->tag=v1->tag;
      switch(v->u.unary.operator){
      case '+':v->u.INT = v1->u.INT;break;
      case '-':v->u.INT =-v1->u.INT;break;
      case '~':v->u.INT =~v1->u.INT;break;
      default:
	name_error(n,"Invalid unary integer operator");
      }
    }else if(isfloat(v1->tag)){
      char op=v->u.unary.operator;
      *v=*v1;
      v->value=0;
      switch(op){
      case '+':break;
      case '-':
	v->u.FLOAT.val=-v->u.FLOAT.val;
	v->u.FLOAT.sign=-v->u.FLOAT.sign;
	break;
      default:
	name_error(n,"Invalid unary float operator");
      }
    }else
      name_error(n,"Invalid operand for unary operator");
    return value_coerce(v,target,n);
  case idl_binary:
    v1=value_compute(v->u.binary.arg1,start,n,target);
    v2=value_compute(v->u.binary.arg2,start,n,target);
    if(isint(v1->tag)){
      v->tag=v1->tag;
      switch(v->u.binary.operator){
	/* FIXME: integer overflow must be detected */
      case '+':v->u.INT = v1->u.INT + v2->u.INT;break;
      case '-':v->u.INT = v1->u.INT - v2->u.INT;break;
      case '*':v->u.INT = v1->u.INT * v2->u.INT;break;
      case '/':v->u.INT = v1->u.INT / v2->u.INT;break;
      case '%':v->u.INT = v1->u.INT % v2->u.INT;break;
      case 'L':v->u.INT = v1->u.INT <<v2->u.INT;break;
      case 'R':v->u.INT = v1->u.INT >>v2->u.INT;break;
      case '&':v->u.INT = v1->u.INT & v2->u.INT;break;
      case '^':v->u.INT = v1->u.INT ^ v2->u.INT;break;
      case '|':v->u.INT = v1->u.INT | v2->u.INT;break;
      default:
	name_error(n,"Invalid binary integer operator");
      }
    }else if(isfloat(v1->tag)){
      char op=v->u.binary.operator;
      v->tag=v1->tag;
      /* literal integral parts not supported here */
      v->u.FLOAT.integer=v->u.FLOAT.fraction=0;
      switch(op){
      case '+':v->u.FLOAT.val = v1->u.FLOAT.val + v2->u.FLOAT.val;break;
      case '-':v->u.FLOAT.val = v1->u.FLOAT.val - v2->u.FLOAT.val;break;
      case '*':v->u.FLOAT.val = v1->u.FLOAT.val * v2->u.FLOAT.val;break;
      case '/':v->u.FLOAT.val = v1->u.FLOAT.val / v2->u.FLOAT.val;break;
      default:
	name_error(n,"Invalid floating point binary operator");
      }
    }else
      name_error(n,"Invalid arguments for binary operators");
    return value_coerce(v,target,n);
  default:
    /* primitive values */
    return value_coerce(v,target,n);
  }
  return 0;
}

static IDLType
type_compute(IDLType t)
{
  switch(t->tag){
    /* those identify themselves */
  case NULLTYPEtag:case BASICtag:case SEQUENCEtag:case STRINGtag:
  case STRUCTtag:case ARRAYtag:case ENUMtag:case UNIONtag:case WSTRINGtag:
  case FIXEDtag:case NATIVEtag:
    return t;
    /* those is one of the above after one level of indirection */
  case DEFINEDtag:
    return type_compute(t->u.defined->u.type);
  case ALIAStag:
    return type_compute(t->u.alias);
  case REFERENCEDtag:
    if(t->u.referenced.val)
      return type_compute(t->u.referenced.val);
    type_check(t,t->u.referenced.name);
    assert(t->u.referenced.val);
    return type_compute(t->u.referenced.val);
  }
  return 0;
}

static boolean
definition_isdefinition(IDLDefinition d)
{
  switch(d->tag){
  case PRAGMA_VERSIONtag:case PRAGMA_IDtag:case PRAGMA_PREFIXtag:
  case INTERFACEFWDtag:
    return FALSE;
  default:
    return TRUE;
  }
}

static void
duplicate_check(refany def1,refany def)
{
  IDLDefinition d1=def1,d=def;
  /* identical definitions, no problem */
  if(def1==def)return;
  /* one or the other does not really define anything */
  if(!definition_isdefinition(d) || !definition_isdefinition(d1))
    return;
  /* different names, no problem */
  if(!cmp_name(d,d1->name->name))return;
  /* for same names, only complain if def is after def1 */
  if(d->name->line<d1->name->line)return;
  name_error(d->name,aprintf("already defined in line %d\n",d1->name->line));
}

static void
dimensions_check(refany val,refany name)
{
  IDLValue v=val;
  list l;
  *v=*value_compute(val,l=iluparser_new_list(),name,the_long_t);
  if(v->u.INT<=0)
    name_error(name,"dimension is not positive");
  /* free temporary list */
  list_clear(l,FALSE);
  iluparser_Free(l);
}

static IDLType
type_resolve(IDLType t,list l,IDLName n)
{
  IDLDefinition d;
  if(t->tag!=REFERENCEDtag)return t;
  if(t->u.referenced.val)return t->u.referenced.val;
  d=t->u.referenced.name->value;
  if(d->tag==INTERFACEtag)
    /* referenced interface */
    return t;
  if(d->tag!=TYPEtag){
    name_error(n,aprintf("%s does not specify a type",d->name->name));
    return 0;
  }
  if(list_find(l,identity,d))
    name_error(n,"cycle in type resolution");
  list_insert(l,d);
  return d->u.type=type_resolve(d->u.type,l,n);
}

static void
type_check(IDLType t,IDLName n)
{
  list l=0;
  switch(t->tag){
  case NULLTYPEtag:
    /* this should not happen */
  case BASICtag:case ENUMtag:case NATIVEtag:
    /* nothing to do */
  case DEFINEDtag:
    /* those will be checked separately */
    break;
  case ALIAStag:
    type_check(t->u.alias,n);
    break;
  case REFERENCEDtag:
    if(t->u.referenced.val)break;
    t->u.referenced.val=type_resolve(t,l=iluparser_new_list(),n);
    break;
  case SEQUENCEtag:
    type_check(t->u.sequence.type,n);
    if(t->u.sequence.size){
      t->u.sequence.size=
	value_compute(t->u.sequence.size,l=iluparser_new_list(),n,the_long_t);
      if(t->u.sequence.size->u.INT<=0)
	name_error(n,"sequence size is not positive");
    }
    break;
  case STRUCTtag:
    list_enumerate(t->u.structure,definition_check,0);
    break;
  case WSTRINGtag:
  case STRINGtag:
    if(t->u.stringsize){
      t->u.stringsize=
	value_compute(t->u.stringsize,l=iluparser_new_list(),n,the_long_t);
      if(t->u.stringsize->u.INT<=0)
	name_error(n,"string size is not positive");
    }
    break;
  case ARRAYtag:
    type_check(t->u.array.type,n);
    list_enumerate(t->u.array.dimensions,dimensions_check,n);
    break;
  case FIXEDtag:
    if(t->u.fixed.digits){
      list l1;
      t->u.fixed.digits=
	value_compute(t->u.fixed.digits,l=iluparser_new_list(),n,the_long_t);
      t->u.fixed.fraction=
	value_compute(t->u.fixed.fraction,l1=iluparser_new_list(),
		      n,the_long_t);
      if(t->u.fixed.digits->u.INT<=0)
	name_error(n,"number of digits not positive");
      list_clear(l1,FALSE);
      iluparser_Free(l1);
      /* l will be cleared below */
    }
    break;
  case UNIONtag:
    type_check(t->u._union.head,n);
    list_enumerate(t->u._union.body,definition_check,0);
    break;
  }
  if(l){
    /* free temporary list */
    list_clear(l,FALSE);
    iluparser_Free(l);
  }
}

static void
caselabel_check(refany label,refany Union)
{
  IDLCase c=label;
  IDLDefinition u=Union;
  IDLName n=c->env->name;
  list l;
  if(c->value){
    c->value=value_compute(c->value,l=iluparser_new_list(),
			   n,type_compute(u->u.type->u._union.head));
    list_clear(l,FALSE);
    iluparser_Free(l);
  }else{ /*default*/
      IDLName n1=u->u.type->u._union._default;
      if(n1 && n!=n1)
	name_error(n,aprintf("default branch already assigned to %s",
			     n1->name));
      else
	u->u.type->u._union._default=n;
  }
}

/* compare location of base interface definition with this interface */
static void
definition_precedes(refany base,refany derived)
{
  IDLName d=derived;
  IDLName b=base;
  if(b->value->name->serial >= d->serial)
    name_error(d,"base interface must precede derived one");
}

/* merge an operation in l, complain if it is already there */
static void
collect_operations1(refany def,refany l)
{
  IDLDefinition d=def;
  IDLDefinition d1;
  if(d->tag!=OPERATIONtag)return;
  if((d1=list_find(l,cmp_name,d->name->name)))
    name_error(d->name,aprintf("When defining interface %s, "
			       "operation conflicts with interface %s\n",
			       /*FIXME: find name for base interface */
			       "with multiple interfaces",
			       d1->env->name->name));
  else
    list_insert(l,d);
}
			       
/* merge the definitions of one base interface into l */
static void
collect_operations(refany base,refany l)
{
  IDLName b=base;
  list_enumerate(b->value->u.interface.definitions,collect_operations1,l);
}

static void 
definition_check(refany def,refany rock)
{
  IDLDefinition d=def;
  list l=0;
  list_enumerate(rock,duplicate_check,d);
  switch(d->tag){
  case NILtag:case TYPELISTtag:case MEMBERLISTtag:case ATTRLISTtag:
    /* this should not happen */
    break;
  case PRAGMA_IDtag:case PRAGMA_VERSIONtag:case PRAGMA_PREFIXtag:
    /* nothing to do */
    break;
  case MODULEtag:
    list_enumerate(d->u.module.definitions,definition_check,
		   d->u.module.definitions);
    break;
  case INTERFACEtag:
    /* base interfaces must appear before derived */
    list_enumerate(d->u.interface.bases,definition_precedes,d->name);
    /* base interfaces must not share operation names */
    l=iluparser_new_list();
    list_enumerate(d->u.interface.bases,collect_operations,l);
    /* recurse */
    list_enumerate(d->u.interface.definitions,definition_check,
		   d->u.interface.definitions);
    break;
  case INTERFACEFWDtag:
    break;
  case EXCEPTIONtag:
    /* FIXME: check members */
    break;
  case CONSTtag:
    /* FIXME: type_check(d->u.constant*/
    definition_compute(d,l=iluparser_new_list(),d->name);
    break;
  case TYPEtag:
    type_check(d->u.type,d->name);
    break;
  case OPERATIONtag:
    break;
  case PARAMETERtag:
    type_check(d->u.parameter.type,d->name);
    break;
  case CASEtag:
    type_check(d->u._case.type,d->name);
    list_enumerate(d->u._case.labels,caselabel_check,d->env);
    break;
  case ATTRIBUTEtag:
    type_check(d->u.attribute.type,d->name);
    break;
  case MEMBERtag:
    type_check(d->u.member,d->name);
    break;
  case ENUMVALtag:
    break;
  }
  if(l){
    /* free list */
    list_clear(l,FALSE);
    iluparser_Free(l);
  }
}

/************************* setuid pass       *************************/

static char*
makeid(IDLDefinition d,char *prefix,char *version)
{
  return aprintf("IDL:%s%s%s:%s",
		 prefix?prefix:"",prefix?"/":"",
		 d->prefix,version?version:"1.0");
}

static void definition_setuid(refany def,refany cur);

static void
type_setuid(IDLType t,char **prefix)
{
  switch(t->tag){
    /* those don't have anything with uids inside */
  case NULLTYPEtag:
  case BASICtag:
  case REFERENCEDtag:
  case ENUMtag:
  case ALIAStag:
  case NATIVEtag:
    break;
    /* FIXME: anonymous types don't get a repository id, yet */
  case ARRAYtag:
  case SEQUENCEtag:
  case STRINGtag:
  case WSTRINGtag:
  case FIXEDtag:
    break;
  case UNIONtag:
    type_setuid(t->u._union.head,prefix);
    list_enumerate(t->u._union.body,definition_setuid,prefix);
    break;
  case STRUCTtag:
    list_enumerate(t->u.structure,definition_setuid,prefix);
    break;
  case DEFINEDtag:
    definition_setuid(t->u.defined,prefix);
    break;
  }
}

static void
definition_setuid(refany def,refany cur)
{
  char *newprefix=*(char**)cur;
  IDLDefinition d=def;
  switch(d->tag){
  case NILtag:case TYPELISTtag:case ATTRLISTtag:case MEMBERLISTtag:
    name_error(d->name,"Unexpected case in definition_setuid");
    break;
    /* those don't have anything with uids inside */
  case INTERFACEFWDtag:case PARAMETERtag:case MEMBERtag:case CONSTtag:
  case ENUMVALtag:case CASEtag:    

    break;
  case MODULEtag:
    list_enumerate(d->u.module.definitions,definition_setuid,&newprefix);
    break;
  case PRAGMA_PREFIXtag:
    *(char**)cur=d->u.pragma;
    break;
  case INTERFACEtag:
    list_enumerate(d->u.interface.definitions,definition_setuid,&newprefix);
    break;
  case TYPEtag:
    type_setuid(d->u.type,&newprefix);
    break;
  case EXCEPTIONtag:
    list_enumerate(d->u.exception.members,definition_setuid,&newprefix);
    break;
    /* those don't have repository ids */
  case OPERATIONtag:
  case ATTRIBUTEtag:
  case PRAGMA_IDtag:
  case PRAGMA_VERSIONtag:
    return;
  }
  if(!d->id)
    d->id=makeid(d,*(char**)cur,d->version);
}

/************************* maketoplevel      *************************/

static boolean
find_nonmodules(refany def,refany rock)
{
  IDLDefinition d=def;
  IDLDefinition r=rock;
  switch(d->tag){
    /* those are OK on toplevel */
  case PRAGMA_IDtag:case PRAGMA_VERSIONtag:case PRAGMA_PREFIXtag:
  case MODULEtag:
    return FALSE;
  default:
    return r ? (d->name->file==r->name->file):1;
  }
}

static void
make_toplevel(struct idl_parse *P)
{
  IDLDefinition d,m;
  char *s,*name;
  do{
    d=list_find(P->definitions,find_nonmodules,0);
    if(!d)break;
    m=new_definition();
    m->tag=MODULEtag;
    m->name=new_name();
    m->name->name=name=iluparser_basename(d->name->file);
    /* make sure this is a valid identifier */
    while(*name && !isalpha(*name))
      name++;
    if(!*name)
      name_error(d->name,"source file contains no letters");
    /* if it doesn't start with a letter, copy it */
    if(name!=m->name->name){
      s=m->name->name;
      m->name->name=ilu_strdup(name);
      iluparser_Free(s);
    }
    /* find the first invalid character */
    for(s=m->name->name;*s && (isalnum(*s) || *s=='_');s++)
      ;
    if(*s)*s='\0';
    m->name->lifted=underscore2hyphen(m->name->name);
    m->name->file=d->name->file;
    name_warning(m->name,"warning: introducing module");
    m->u.module.definitions=iluparser_new_list();
    list_insert(P->definitions,m);
    while(d){
      list_insert(m->u.module.definitions,d);
      definition_backlink(d,m);
      list_remove(P->definitions,d);
      d=list_find(P->definitions,find_nonmodules,m);
    }
  }while(1);
}

/************************* lift pass         *************************/

struct lifter{
  char *prefix;
  list lifted;
};

static void
lift_prefix(refany def,refany lift)
{
  IDLDefinition d=def;
  struct lifter *l=lift;
  d->name->lifted=aprintf("%s-%s",l->prefix,d->name->lifted);
  list_insert(l->lifted,d);
}

static void
definition_lift(refany def,refany rock)
{
  IDLDefinition d=def;
  struct lifter l;
  boolean do_lift=FALSE;
  if(!d->name)
    return;
  l.prefix=underscore2hyphen(d->name->name);
  l.lifted=iluparser_new_list();
  switch(d->tag){
  case MODULEtag:
    /* don't lift names directly inside the toplevel module */
    list_enumerate(d->u.module.definitions,definition_lift,d->env?&l:0);
    do_lift=1;
    break;
  case INTERFACEtag:
    list_enumerate(d->u.interface.definitions,definition_lift,&l);
    do_lift=1;
    break;
  case EXCEPTIONtag:
    do_lift=1;
    list_enumerate(d->u.exception.members,definition_lift,&l);
    break;
  case TYPEtag:
    do_lift=1;
    switch(d->u.type->tag){
    case STRUCTtag:
      list_enumerate(d->u.type->u.structure,definition_lift,&l);
      break;
    case UNIONtag:
      list_enumerate(d->u.type->u._union.body,definition_lift,&l);
      break;
    default:break;
    }
    break;
  case CONSTtag:
    do_lift=1;
    break;
  default:
    break;
  }
  /* lift myself as well */
  if(do_lift)
    list_insert(l.lifted,d);
  if(rock)
    list_enumerate(l.lifted,lift_prefix,rock);
  iluparser_Free(l.prefix);
  /* free list */
  list_clear(l.lifted,FALSE);
  iluparser_Free(l.lifted);
}

/************************* makeisl pass      *************************/

static void module_makeisl(IDLDefinition d);
static void interface_makeisl(IDLDefinition d);
static void operation_makeisl(IDLDefinition d);
static void typedef_makeisl(IDLDefinition d);
static void member_makeisl(IDLDefinition d);
static void parameter_makeisl(IDLDefinition d);
static void exception_makeisl(IDLDefinition d);
static void case_makeisl(IDLDefinition d);
static void const_makeisl(IDLDefinition d);
static void attribute_makeisl(IDLDefinition d);
static void value_makeisl(IDLValue v);
static void type_makeisl(IDLType p,IDLDefinition env);

static void
definition_makeisl(refany definition,refany rock)
{
  IDLDefinition d=definition;
  switch(d->tag){
    /* those should not happen */
  case NILtag:case TYPELISTtag:case MEMBERLISTtag:case ATTRLISTtag:
    break;
    /* those don't have ISL */
  case INTERFACEFWDtag:
    break;
  case MODULEtag:
    module_makeisl(d);
    break;
  case INTERFACEtag:
    interface_makeisl(d);
    break;
  case EXCEPTIONtag:
    exception_makeisl(d);
    break;
  case CONSTtag:
    const_makeisl(d);
    break;
  case ENUMVALtag:
    break;
  case OPERATIONtag:
    operation_makeisl(d);
    break;
  case ATTRIBUTEtag:
    attribute_makeisl(d);
    break;
  case TYPEtag:
    typedef_makeisl(d);
    break;
  case MEMBERtag:
    member_makeisl(d);
    break;
  case CASEtag:
    case_makeisl(d);
    break;
  case PARAMETERtag:
    parameter_makeisl(d);
    break;
  case PRAGMA_PREFIXtag:
  case PRAGMA_VERSIONtag:
  case PRAGMA_IDtag:
    break;
  }
}

static void
module_makeisl(IDLDefinition d)
{
  Interface new;
  if(d->env==NULL){
    /* toplevel module */
    d->isl=new=(Interface)iluparser_Malloc(sizeof(struct ilu_interface_s));
    new->name=new_Name();
    name_set_base_name(new->name,d->name->lifted);
    new->def=d->name->line;
    new->types=(set)iluparser_new_list();
    new->classes=(set)iluparser_new_list();
    new->imports=iluparser_new_list();    
    /* make sure the module knows about the ilu INTERFACE */
    add_import(d,"ilu",NULL);
    new->exceptions=(set)iluparser_new_list();
    new->constants=(set)iluparser_new_list();
    new->brand=NULL;
    new->def=0;
  }
  if(d->u.module.definitions)
    list_enumerate(d->u.module.definitions,definition_makeisl,d);
}

static void
interface_makeisl(IDLDefinition d)
{
  TypeDescription new;
  Type isl;
  new=new_TypeDescription();
  new->type=object_Type;
  new->structuredDes.object=iluparser_Malloc(sizeof(struct ilu_class_s));
  new->structuredDes.object->brand = NULL;
  new->structuredDes.object->corba_rep_id = d->id;
  new->structuredDes.object->collectible = FALSE;
  /* IDL interfaces are always optional */
  new->structuredDes.object->optional = TRUE;
  new->structuredDes.object->singleton = NULL;
  new->structuredDes.object->superclasses = NULL;
  new->structuredDes.object->authentication = NULL;
  new->structuredDes.object->methods = NULL;
  new->structuredDes.object->doc_string = NULL;

  d->isl=isl=new_Type();
  isl->def=d->name->line;
#if HAVE_SCOPING
  definition_setscoping(d,isl->scoping);
#endif

  isl->description=new;
  isl->interface=toplevel_module(d)->isl;
  /* ISL has the types in reverse order. In order to generate the
     same stub code, items are inserted at front */
  list_push(isl->interface->types,isl);
  /* FixupInterface will add them to the classes list later on
  list_push(isl->interface->classes,isl);*/
  name_set_base_name(isl->name,d->name->lifted);
  
  assert(d->u.interface.definitions);
  list_enumerate(d->u.interface.definitions,definition_makeisl,d);
} 

static void
operation_makeisl(IDLDefinition d)
{
  Class C;
  Procedure new=new_Procedure();
  assert(d->name->scope==NULL);
  d->isl=new;
  name_set_base_name(new->name,d->name->lifted);
  new->def=d->name->line;
  /* IDL operations are never functional */
  new->functional = FALSE;
  new->asynch = d->u.operation.oneway;
  new->interface = toplevel_module(d)->isl;
  /* add parameters */
  new->arguments=iluparser_new_list();
  list_enumerate(d->u.operation.parameters,definition_makeisl,0);
  type_makeisl(d->u.operation.returntype,d);
  /* link with interface/CLASS */
  new->object = up_find_definition(d,INTERFACEtag)->isl;
  C=new->object->description->structuredDes.object;
  if(!C->methods)
    C->methods=iluparser_new_list();
  list_insert(C->methods,new);
  /* FIXME: context */
}

static void
fill_array_dimensions(refany val,refany list)
{
  IDLValue v=val;
  assert(v->tag==idl_long);
  /* Hmm. Casting integers to pointers */
  list_insert(list,(refany)(v->u.INT));
}

static void
enum_makeisl(refany def,refany list)
{
  IDLDefinition d=def;
  EnumField f=iluparser_Malloc(sizeof(struct enumerationField_s));
  f->name=d->name->lifted;
  f->id=-1;
  list_insert(list,f);
}

static void
typedef_makeisl(IDLDefinition d)
{
  Type T;
  TypeDescription D;
  IDLType t;
  list l;
  t=d->u.type;
  d->isl=T=new_Type();
  if(!t->isl)
    t->isl=d->isl;
  T->description=D=new_TypeDescription();
  assert(d->name->name);
  assert(d->name->array==0);
  name_set_base_name(T->name,d->name->lifted);
#if HAVE_SCOPING
  definition_setscoping(d,T->scoping);
#endif
  T->def = d->name->line;
  T->interface = toplevel_module(d)->isl;
  list_push(T->interface->types,T);
  switch(t->tag){
    /* those should not happen */
  case NULLTYPEtag:case REFERENCEDtag:case BASICtag:
    break;
  case STRUCTtag:
    D->type=record_Type;
    D->structuredDes.record.fields=iluparser_new_list();
    D->structuredDes.record.extensible=0;
    D->structuredDes.record.supertype=(Type)0;
    list_enumerate(t->u.structure,definition_makeisl,0);
    break;
  case ENUMtag:
    D->type=enumeration_Type;
    D->structuredDes.enumeration=l=iluparser_new_list();
    list_enumerate(t->u.enumerated,enum_makeisl,l);
    break;
  case UNIONtag:
    D->type=union_Type;
    D->structuredDes.uniond.discriminator_type=0;
    D->structuredDes.uniond.types=iluparser_new_list();
    D->structuredDes.uniond.default_arm=NULL;
    /* In CORBA, all values of the switch type are valid,
       even if they are not specified in the cases - 
       unless there is a default arm */
    D->structuredDes.uniond.others_allowed=TRUE;
    list_enumerate(t->u._union.body,definition_makeisl,0);
    break;
  case ARRAYtag:
    D->type=array_Type;
    D->structuredDes.array.type=0; /* update later */
    D->structuredDes.array.dimensions=l=iluparser_new_list();
    list_enumerate(t->u.array.dimensions,fill_array_dimensions,l);
    D->structuredDes.array.optional=FALSE;
    type_makeisl(t->u.array.type,d);
    /* for the dimensions, no special isl is necessary */
    break;
  case STRINGtag:
    if(t->u.stringsize){
      D->type=sequence_Type;
      D->structuredDes.sequence.type=the_char_t->isl;
      D->structuredDes.sequence.optional=TRUE;
      D->structuredDes.sequence.limit=t->u.stringsize->u.INT;
    }else{
      /* alias for ilu.CString */
      T->description=NULL;
      iluparser_Free(D);
      T->supertype=t->isl;
      /*name_set_lang_name(T->name,"import","CString");*/
    }
    break;
  case WSTRINGtag:
    D->type=sequence_Type;
    D->structuredDes.sequence.type=the_wchar_t->isl;
    D->structuredDes.sequence.optional=TRUE;
    if(t->u.stringsize)
      D->structuredDes.sequence.limit=t->u.stringsize->u.INT;
    else
      D->structuredDes.sequence.limit=0;
    break;
  case SEQUENCEtag:
    D->type=sequence_Type;
    D->structuredDes.sequence.type=0; /* update later */
    D->structuredDes.sequence.optional=TRUE;
    if(t->u.sequence.size)
      D->structuredDes.sequence.limit=t->u.sequence.size->u.INT;
    else
      D->structuredDes.sequence.limit=0;
    type_makeisl(t->u.sequence.type,d);
    break;
  case ALIAStag:
    T->description=NULL;
    iluparser_Free(D);
    /* need to make isl for referenced types as well
       if(is_anonymous(t->u.alias)) */
    type_makeisl(t->u.alias,d);
    break;
  case DEFINEDtag:
    /* this is an alias type */
    T->description=NULL;
    iluparser_Free(D);
    definition_makeisl(t->u.defined,0);
    break;
  case FIXEDtag:
    /* FIXME: implement */
    name_error(d->name,"fixed point types not supported, yet");
    break;
  case NATIVEtag:
    name_error(d->name,"native types not supported, yet");
    break;
  }
}

static void 
member_makeisl(IDLDefinition d)
{
  Argument A=new_Argument();
  Type T;
  name_set_base_name(A->name,d->name->lifted);
  A->def=d->name->line;
  assert(d->isl==0);
  d->isl=A;
  switch(d->env->tag){
  case TYPEtag:
    /* record type */
    T=d->env->isl;
    break;
  case EXCEPTIONtag:
    /* pseudo record for exceptions */
    T=((Exception)d->env->isl)->type;
    break;
  default:
    name_error(d->name,"unexpected scope for member");
  }
  list_insert(T->description->structuredDes.record.fields,A);
  type_makeisl(d->u.type,d);
}

static void
parameter_makeisl(IDLDefinition d)
{
  Argument A=new_Argument();
  Procedure P=d->env->isl;

  A->direction=d->u.parameter.direction;
  name_set_base_name(A->name,d->name->lifted);
  A->def=d->name->line;
  type_makeisl(d->u.parameter.type,d);
  /* this might not succeed if the type is referenced */
  list_insert(P->arguments,A);
  d->isl=A;
}

/* anonymous type */
static void
anon_type(IDLType p,IDLDefinition env)
{
  IDLDefinition d=new_definition();
  d->name=p->name=new_name();
  /* name and lifted name are identical (no scoping, no underscores) */
  d->name->lifted=d->name->name=new_anon_type();
  d->name->file=env->name->file;
  d->name->line=env->name->line;
  d->tag=TYPEtag;
  d->env=env;
  d->u.type=p;
  p->anon_def=d;
  definition_makeisl(d,0);
  p->isl=d->isl;
}

static void
type_makeisl(IDLType p,IDLDefinition env)
{
  IDLDefinition d;
  Type T;
  if(p->isl)return;
  switch(p->tag){
    /* those should not happen */
  case NULLTYPEtag:case STRUCTtag:case ENUMtag:case UNIONtag:
    name_error(d->name,"Unexpected type");
    break;
  case BASICtag:
    /* no isl needs to be make here */
    break;
  case REFERENCEDtag:
    /* have to wait until after makeisl to locate the correct isl
       if this is an imported type, make the subtype */
    if(toplevel_module(env)->isl
       !=toplevel_module(p->u.referenced.name->value)->isl){
      Interface I;
      d=p->u.referenced.name->value;
      p->isl=T=new_Type();
      name_set_base_name(T->name, d->name->lifted);
      name_set_lang_name(T->name, "import", d->name->lifted);
      I=toplevel_module(env)->isl;
      list_push(I->types,T);
      /* supertype, importInterfaceName and interface added later */
    }
    break;
  case ALIAStag:
    type_makeisl(p->u.alias,env);
    break;
  case DEFINEDtag:
    /* nothing to do here: the struct/union definition 
       is in the defition list */
    break;
  case SEQUENCEtag:
    if(!p->name)
      anon_type(p,env);
    type_makeisl(p->u.sequence.type,env);
    break;
  case ARRAYtag:
    if(!p->name)
      anon_type(p,env);
    type_makeisl(p->u.array.type,env);
    break;
  case WSTRINGtag:
    anon_type(p,env);
    type_makeisl(the_wchar_t,env);
    break;
  case STRINGtag:
    if(!p->name && p->u.stringsize){
      anon_type(p,env);
      type_makeisl(the_char_t,env);
    }
    break;
  case NATIVEtag:
    break;
  case FIXEDtag:
    name_error(env->name,"fixed types not supported");
    break;
  }
}

static void
exception_makeisl(IDLDefinition d)
{
  Exception E=new_Exception();
  d->isl=E;
  name_set_base_name(E->name,d->name->lifted);
#if HAVE_SCOPING
  definition_setscoping(d,E->scoping);
#endif
  E->def=d->name->line;
  E->interface=toplevel_module(d)->isl;
  list_insert(E->interface->exceptions,E);
  /* if this has parameters, set type to a RECORD */
  if(d->u.exception.members){
    Type T;
    TypeDescription D;
    int i;
    E->type=T=new_Type();
    T->description=D=new_TypeDescription();
    name_set_base_name(T->name,aprintf("ilu--prefix-idlExceptionType-%s",
				       d->name->lifted));
#ifdef HAVE_SCOPING
    T->scoping=new_list();
    for (i=0;i<list_size(E->scoping)-1;i++)
      list_insert(T->scoping,list_ref(E->scoping,i));
    list_insert(T->scoping,name_base_name(T->name));
#endif
    T->def=d->name->line;
    T->interface=toplevel_module(d)->isl;
    list_push(T->interface->types,T);
    D->type=record_Type;
    D->structuredDes.record.fields=iluparser_new_list();
    D->structuredDes.record.extensible=0;
    D->structuredDes.record.supertype=(Type)0;
    list_enumerate(d->u.exception.members,definition_makeisl,0);
  }
}

static void
caselabel_makeisl(refany Case,refany arg)
{
  IDLCase c=Case;
  Argument A=arg;
  if(c->value){
    value_makeisl(c->value);
    c->isl=c->value->isl;
    list_insert(A->values,c->isl);
  }else{
    /* default: */
    IDLDefinition d=c->env->env;
    Type T;
    assert(d && d->tag==TYPEtag && d->u.type->tag==UNIONtag);
    T=d->isl;
    T->description->structuredDes.uniond.default_arm=A;
    T->description->structuredDes.uniond.others_allowed=FALSE;
  }
}

static void
case_makeisl(IDLDefinition d)
{
  Type T;
  Argument A=d->isl=new_Argument();
  name_set_base_name(A->name,d->name->lifted);
  A->values=iluparser_new_list();
  A->def=d->name->line;
  type_makeisl(d->u._case.type,d);
  /* insert labels */
  A->values=iluparser_new_list();
  list_enumerate(d->u._case.labels,caselabel_makeisl,A);
  /* insert case into union */
  T=d->env->isl;
  list_insert(T->description->structuredDes.uniond.types,A);
}

static void
value_makeisl(IDLValue v)
{
  ConstantValue V;
  boolean isint=FALSE,issigned=FALSE;
  if(v->isl)return;
  V=v->isl=iluparser_Malloc(sizeof(struct ilu_constantvalue_s));
  switch(v->tag){
    /* those should not happen */
  case idl_void:case idl_unary:case idl_binary:case idl_named:case idl_any:
  case idl_object:
    name_error(0,"unexpected value in makeisl");
    break;
    /* in the ISL, they are all integer_Type */
  case idl_short:
  case idl_long:
  case idl_long_long:
    isint=TRUE;issigned=TRUE;break;
  case idl_octet:
  case idl_unsigned_short:
  case idl_unsigned_long:
  case idl_unsigned_long_long:
    isint=TRUE;break;
  case idl_string:
    V->type=shortcharacter_Type;
    V->val.s=v->u.string;
    break;
  case idl_enum:
    V->type=shortcharacter_Type;
    V->val.s=v->u.enumerated.name->lifted;
    break;
  case idl_float:case idl_double:
    V->type=real_Type;
    if(v->u.FLOAT.integer || v->u.FLOAT.fraction){
      V->val.r.sign=v->u.FLOAT.sign;
      V->val.r.value=v->u.FLOAT.integer;
      V->val.r.fraction=v->u.FLOAT.fraction;
      V->val.r.exponent=v->u.FLOAT.exponent;
    }else{
      /* FIXME: What is the precision for double? */
      char *buf=aprintf("%.17E",v->u.FLOAT.val);
      char *p,*p1;
      p=buf;
      if(*p=='-'){
	V->val.r.sign=-1;
	p++;
      }else V->val.r.sign=1;
      p1=strchr(p,'.');
      V->val.r.value=strndup(p,p1-p);
      p=p1+1;
      p1=strchr(p,'E');
      V->val.r.fraction=strndup(p,p1-p);
      V->val.r.exponent=strtol(p1+1,0,10);
      free(buf);
    }
    break;
  case idl_char:
    /* FIXME: should use character arm */
    V->type=shortinteger_Type;
    V->val.i.sign=1;
    V->val.i.value=v->u.CHAR;
    break;
  case idl_wchar:
    V->type=shortinteger_Type;
    V->val.i.sign=1;
    V->val.i.value=v->u.CHAR;
    break;
  case idl_boolean:
    V->type=boolean_Type;
    V->val.b=v->u.BOOL;
    break;
  case idl_int:
    name_error(0,"Untyped integer");
    break;
  }
  if(isint){
    V->type=integer_Type;
    if(issigned){
      V->val.i.sign=(v->u.INT<0)?-1:1;
      V->val.i.value=(v->u.INT<0)?-v->u.INT:v->u.INT;
    }else{
      V->val.i.sign=1;
      V->val.i.value=v->u.INT;
    }
  }
}

static void
const_makeisl(IDLDefinition d)
{
  Constant C=iluparser_Malloc(sizeof(struct ilu_constant_s));
  d->isl=C;
  C->name=new_Name();
#if HAVE_SCOPING
  C->scoping = iluparser_new_list();
  definition_setscoping(d,C->scoping);
#endif
  name_set_base_name(C->name,d->name->lifted);
  C->def=d->name->line;
  C->interface=toplevel_module(d)->isl;
  list_insert(C->interface->constants,C);
  C->type=0;
  C->importInterfaceName=0;
  C->import=0;
  value_makeisl(d->u.constant.val);
  C->value=d->u.constant.val->isl;
}

static Procedure
attr_get_set(IDLDefinition d,boolean get)
{
  /* see operation_makeisl */
  Class C;
  Procedure new=new_Procedure();
  name_set_base_name(new->name,aprintf("ilu--prefix-idlAttribute--%s-%s",
				       get?"get":"set",d->name->lifted));
  new->def=d->name->line;
  new->functional=FALSE;
  new->asynch=FALSE;
  new->interface=toplevel_module(d)->isl;
  new->arguments=iluparser_new_list();
  if(get){
    /* add return type later */
  }else{
    Argument A=new_Argument();
    name_set_base_name(A->name,"value");
    A->direction=In;
    A->def=d->name->line;
    /* add parameter type later */
    list_insert(new->arguments,A);
  }
  type_makeisl(d->u.attribute.type,d);
  new->object = up_find_definition(d,INTERFACEtag)->isl;
  C=new->object->description->structuredDes.object;
  if(!C->methods)C->methods=iluparser_new_list();
  list_insert(C->methods,new);
  return new;
}
static void
attribute_makeisl(IDLDefinition d)
{
  /* make pseudo-operations */
  d->isl=attr_get_set(d,TRUE);
  if(!d->u.attribute.readonly)
    d->u.attribute.set=attr_get_set(d,FALSE);
}

/************************* update pass       *************************/

static void definition_update(refany def,refany rock);
static void typedef_update(IDLDefinition d);

static void
interface_insertsuper(refany name,refany def)
{
  IDLName n=name;
  IDLDefinition d=def;
  Type T=d->isl;
  Class C=T->description->structuredDes.object;
  IDLDefinition super=n->value;

  assert(super);
  if(toplevel_module(d)==toplevel_module(super)){
    /* insert ISL supertype into ISL superclasses list */
    list_insert(C->superclasses,super->isl);
  }else{
    /* make subtype */
    IDLDefinition m1,m2;
    Type T=new_Type();
    name_set_base_name(T->name,super->name->lifted);
    name_set_lang_name(T->name,"import",super->name->lifted);
    m1=toplevel_module(d);
    m2=toplevel_module(super);
    T->importInterfaceName=m2->name->lifted;
    /* import INTERFACE of supertype into this INTERFACE */
    add_import(m1,m2->name->lifted,m2->name->file);
    list_insert(((Interface)m1->isl)->types,T);
    list_insert(C->superclasses,T);
  }
}

static void
method_assignid(refany method,refany rock)
{
  Procedure p=method;
  int* pid=rock;
  p->id= ++*pid;
}

static void
interface_update(IDLDefinition d)
{
  Type T;
  Class C;
  int id=0;
  T=d->isl;
  C=T->description->structuredDes.object;
  assert(C->superclasses==NULL);
  C->superclasses=iluparser_new_list();
  if(d->u.interface.bases){
    list_enumerate(d->u.interface.bases,interface_insertsuper,d);
  }else{
    /* implicit superclass ilu.CORBA-Object */
    list_insert(C->superclasses,the_CORBA_Object);
  }
  list_enumerate(d->u.interface.definitions,definition_update,0);
  /* FIXME: should really call ilu.bison->AssignMethodIDs here */
  list_enumerate(C->methods,method_assignid,&id);
  /* assign repository IDs */
}

static void
type_update(IDLType t)
{
  IDLDefinition d;
  Type T;
  switch(t->tag){
    /* those should not happen */
  case NULLTYPEtag:case STRUCTtag:case ENUMtag:case UNIONtag:case NATIVEtag:
    name_error(0,"unexpected case in type_update");
    break;
    /* nothing to be done */
  case BASICtag:
  case WSTRINGtag:
  case STRINGtag:
    break;
  case REFERENCEDtag:
    {
      IDLDefinition m1,m2;
      d=t->u.referenced.name->value;
      assert(d && (d->tag==TYPEtag ||d->tag==INTERFACEtag));
      if(d->tag==TYPEtag)
	typedef_update(d);
      m1=toplevel_module(t->u.referenced.name->env);
      m2=toplevel_module(t->u.referenced.name->value);
      if(m1!=m2){
	/* imported type */
	T=t->isl;
	T->supertype=d->isl;
	T->importInterfaceName=m2->name->lifted;
	T->interface=m2->isl;
	add_import(m1,m2->name->lifted,m2->name->file);
      }else
	t->isl=d->isl;
    }
    break;
  case ALIAStag:
    if(t->isl)break;
    type_update(t->u.alias);
    t->isl=t->u.alias->isl;
    break;
  case DEFINEDtag:
    definition_update(t->u.defined,0);
    t->isl=t->u.defined->isl;
    break;
  case SEQUENCEtag:
  case ARRAYtag:
    if(t->anon_def)
      definition_update(t->anon_def,0);
    break;
  case FIXEDtag:
    break;
  }
}

static void
parameter_update(refany param)
{
  IDLDefinition p=param;
  Argument A=p->isl;
  assert(p->tag==PARAMETERtag);
  assert(A);
  type_update(p->u.parameter.type);
  A->type=p->u.parameter.type->isl;
}

static void
exc_update(refany exc,refany list)
{
  IDLName n=exc;
  IDLDefinition d=n->value;
  IDLDefinition m1,m2;

  assert(d && d->isl);
  m1=toplevel_module(n->env);
  m2=toplevel_module(d);
  if(m1!=m2)
    add_import(m1,m2->name->lifted,m2->name->file);
  list_insert(list,d->isl);
}

static void
operation_update(IDLDefinition d)
{
  Procedure P=d->isl;

  assert(d->u.operation.returntype);
  type_update(d->u.operation.returntype);
  assert(P);
  P->returnType=d->u.operation.returntype->isl;
  list_enumerate(d->u.operation.parameters,definition_update,0);
  if(d->u.operation.raises){
    P->exceptions=iluparser_new_list();
    list_enumerate(d->u.operation.raises,exc_update,P->exceptions);
  }
  /* FIXME: context */
}

static void
typedef_update(IDLDefinition d)
{
  IDLType t=d->u.type;
  Type T=d->isl;
  /* The Type's uid indicates that this typedef is uptodate */
  if(T->uid)return;
  /* will be overwritten below #if idl2isl compatible*/
  T->uid=d->id;
  switch(t->tag){
    /* those should not happen */
  case NULLTYPEtag:case REFERENCEDtag:case NATIVEtag:
    name_error(d->name,"Unexpected case in typedef_update");
    break;
  case WSTRINGtag:
  case STRINGtag:
  case BASICtag:
    /* this sets the uid for the basic type */
    type_update(t);
    break;
  case SEQUENCEtag:
    type_update(t->u.sequence.type);
    assert(t->u.sequence.type->isl);
    T->description->structuredDes.sequence.type=t->u.sequence.type->isl;
    break;
  case STRUCTtag:
    list_enumerate(t->u.structure,definition_update,0);
    break;
  case ENUMtag:
    break;
  case UNIONtag:
    type_update(t->u._union.head);
    T->description->structuredDes.uniond.discriminator_type=
      t->u._union.head->isl;
    list_enumerate(t->u._union.body,definition_update,0);
    break;
  case ARRAYtag:
    type_update(t->u.array.type);
    assert(t->u.array.type->isl);
    T->description->structuredDes.array.type=t->u.array.type->isl;
    break;
  case FIXEDtag:
    break;
  case ALIAStag:
    type_update(t->u.alias);
    T->supertype=t->u.alias->isl;
    break;
  case DEFINEDtag:
    definition_update(t->u.defined,0);
    assert(t->u.defined->isl);
    T->supertype=t->u.defined->isl;
    break;
  }
#if LOCAL_TYPES_HAVE_REP_IDS
#else
  T->uid=0;
  if(t->tag==SEQUENCEtag || t->tag==ARRAYtag 
     || t->tag==WSTRINGtag || (t->tag==STRINGtag && t->u.stringsize!=0))
    /* typedef sequence<t>, typedef long */
    FigureTypeUID(T);
  else
    /* struct, union */
    T->uid=d->id;
#endif
}

static void
member_update(IDLDefinition d)
{
  Argument A=d->isl;
  type_update(d->u.type);
  assert(d->u.type->isl);
  A->type=d->u.type->isl;
}

static void
exception_update(IDLDefinition d)
{
  Exception E=d->isl;
  Type T=E->type;
  list_enumerate(d->u.exception.members,definition_update,0);
  E->corba_rep_id=d->id;
  if(T)
    T->uid=E->corba_rep_id;
}

static void
const_update(IDLDefinition d)
{
  Constant C=d->isl;
  type_update(d->u.constant.type);
  C->type=d->u.constant.type->isl;
}

static void
attribute_update(IDLDefinition d)
{
  Procedure P;
  type_update(d->u.attribute.type);
  P=d->isl;
  P->returnType=d->u.attribute.type->isl;
  if(!d->u.attribute.readonly){
    Argument A;
    P=d->u.attribute.set;
    A=list_car(P->arguments);
    A->type=d->u.attribute.type->isl;
  }
}

static void
case_update(IDLDefinition d)
{
  Argument A=d->isl;
  type_update(d->u._case.type);
  A->type=d->u._case.type->isl;
}

static void
definition_update(refany def,refany rock)
{
  IDLDefinition d=def;
  switch(d->tag){
    /* those should not happen */
  case NILtag:case TYPELISTtag:case MEMBERLISTtag:case ATTRLISTtag:
    break;
  case MODULEtag:
    list_enumerate(d->u.module.definitions,definition_update,rock);
    break;
  case INTERFACEtag:
    interface_update(d);
    break;
  case CONSTtag:
    const_update(d);
    break;
  case ENUMVALtag:
    break;
  case EXCEPTIONtag:
    exception_update(d);
    break;
  case OPERATIONtag:
    operation_update(d);
    break;
  case ATTRIBUTEtag:
    attribute_update(d);
    break;
  case PARAMETERtag:
    parameter_update(d);
    break;
  case TYPEtag:
    typedef_update(d);
    break;
  case MEMBERtag:
    member_update(d);
    break;
  case CASEtag:
    case_update(d);
    break;
    /* nothing to do */
  case INTERFACEFWDtag:
  case PRAGMA_IDtag:
  case PRAGMA_PREFIXtag:
  case PRAGMA_VERSIONtag:
    break;
  }
}

static void
select_imported(refany def,refany parse)
{
  struct idl_parse *p=parse;
  IDLDefinition d=def;
  if(d->tag!=MODULEtag)
    /* #pragma */
    return;
  if(strcmp(d->name->file,p->file)==0)
    list_insert(p->defined_interfaces,d->isl);
  else
    list_insert(p->imported_interfaces,d->isl);
}

static void
IDL2ISL(struct idl_parse *P)
{
  char *cur;
  /* join modules for re-opening */
  P->definitions=reopen_modules(P->definitions);
  /* backlink, toplevel has no parent */
  list_enumerate(P->definitions,definition_backlink,0);
  /* resolve all names */
  list_enumerate(P->definitions,definition_resolvenames,P->definitions);
  /* perform consistency checks, compute constants */
  list_enumerate(P->definitions,definition_check,P->definitions);
  /* assign repository IDs */
  cur=0;
  list_enumerate(P->definitions,definition_setuid,&cur);
  /* put everything in a module */
  make_toplevel(P);
  /* lift local types etc. to module level */
  list_enumerate(P->definitions,definition_lift,0);
  /* attach an ISL structure everywhere */
  list_enumerate(P->definitions,definition_makeisl,0);
  /* update ISL */
  list_enumerate(P->definitions,definition_update,0);
  /* select modules of toplevel source file */
  list_enumerate(P->definitions,select_imported,P);
}

list ParseIDLFile (struct idl_parse* P)
{
  int stat;
  extern FILE* idlin;

  extern int idl_flex_debug;
  static int initial=1;
  /* default: debugging is off */
  idl_flex_debug=0;
  if(initial){
    init_types();
    initial=0;
  }else{
    fprintf(stderr,"ParseIDLFile invoked twice (for %s)\n",P->file);
    return 0;
  }
  if ((idlin = fopen(P->file, "r")) == NULL)
    {
      fprintf (stderr, "ParseFile:  Error opening file \"%s\" for read.\n", P->file);
      return (NULL);
    }
  idlsetinitialfile(P->file);
  stat = idlparse();

  if(stat)return 0;

  P->defined_interfaces=iluparser_new_list();
  P->imported_interfaces=iluparser_new_list();
  P->definitions=the_result;
  IDL2ISL(P);

  return P->defined_interfaces;
}

#if HAVE_STRNDUP
#else
char* strndup(char* src,int n)
{
  char *result=iluparser_Malloc(n+1);
  strncpy(result,src,n);
  return result;
}
#endif

/* this should go into pathname.c */
char*
iluparser_basename(char* fullname)
{
  char *result,*h;
#if defined(WIN32) || defined(WIN16)
  for(result=fullname;(h=strpbrk(result,"/\\:"));result=h+1)
    /*nothing*/;
#elif defined(macintosh)
  for(result=fullname;(h=strchr(result,":"));result=h+1)
    ;
#else
  for(result=fullname;(h=strchr(result,'/'));result=h+1)
    ;
#endif
  return ilu_strdup(result);
}
