/***********************************************************************/
/* MOUSE.C - THE mouse handling                                        */
/* This file contains all commands that can be assigned to function    */
/* keys or typed on the command line.                                  */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1997 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                 Email:             M.Hessling@qut.edu.au
 * PO Box 203                    Phone:                    +617 3802 0800
 * Bellara                       http://www.gu.edu.au/gext/the/markh.html
 * QLD 4507                      **** Maintainer PDCurses & REXX/SQL ****
 * Australia                     ************* Author of THE ************
 */

/*
$Id: mouse.c 2.1 1995/06/24 16:29:25 MH Rel MH $
*/

#include <the.h>
#include <proto.h>

/*
 * Following #defines to cater for those platforms that don't
 * have mouse definitions in <curses.h>
 */
#if !defined(BUTTON_SHIFT)
# define BUTTON_SHIFT 0
#endif
#if !defined(BUTTON_CONTROL)
# define BUTTON_CONTROL 0
#endif
#if !defined(BUTTON_ALT)
# define BUTTON_ALT 0
#endif
#if !defined(BUTTON_PRESSED)
# define BUTTON_PRESSED 0
#endif
#if !defined(BUTTON_RELEASED)
# define BUTTON_RELEASED 0
#endif
#if !defined(BUTTON_MOVED)
# define BUTTON_MOVED 0
#endif
#if !defined(BUTTON_DOUBLE_CLICKED)
# define BUTTON_DOUBLE_CLICKED 0
#endif

/*
 * Button masks
 */
#define MOUSE_MODIFIER_MASK(x)  ((x) & 0xE000)
#define MOUSE_ACTION_MASK(x)    ((x) & 0x1C00)
#define MOUSE_BUTTON_MASK(x)    ((x) & 0x0380)
#define MOUSE_WINDOW_MASK(x)    ((x) & 0x007F)
/*
 * Button modifiers
 */
#define MOUSE_MODIFIER_OFFSET   13
#define MOUSE_NORMAL            0
#define MOUSE_SHIFT             ((BUTTON_SHIFT >> 3)   << MOUSE_MODIFIER_OFFSET)
#define MOUSE_CONTROL           ((BUTTON_CONTROL >> 3) << MOUSE_MODIFIER_OFFSET)
#define MOUSE_ALT               ((BUTTON_ALT >> 3)     << MOUSE_MODIFIER_OFFSET)
/*
 * Button actions
 */
#define MOUSE_ACTION_OFFSET     10
#define MOUSE_PRESS             (BUTTON_PRESSED << MOUSE_ACTION_OFFSET)
#define MOUSE_RELEASE           (BUTTON_RELEASED << MOUSE_ACTION_OFFSET)
#define MOUSE_DRAG              (BUTTON_MOVED << MOUSE_ACTION_OFFSET)
#define MOUSE_DOUBLE_CLICK      (BUTTON_DOUBLE_CLICKED << MOUSE_ACTION_OFFSET)
/*
 * Button numbers
 */
#define MOUSE_BUTTON_OFFSET     7
#define MOUSE_LEFT              (1 << MOUSE_BUTTON_OFFSET)
#define MOUSE_MIDDLE            (2 << MOUSE_BUTTON_OFFSET)
#define MOUSE_RIGHT             (3 << MOUSE_BUTTON_OFFSET)

#define MOUSE_INFO_TO_KEY(w,b,ba,bm) ((w)|(b<<MOUSE_BUTTON_OFFSET)|(ba<<MOUSE_ACTION_OFFSET)|((bm>>3)<<MOUSE_MODIFIER_OFFSET))

static CHARTYPE *button_names[] =
 {(CHARTYPE *)"-button 0-",(CHARTYPE *)"LB",(CHARTYPE *)"MB",(CHARTYPE *)"RB"};

static CHARTYPE *button_modifier_names[] =
 {(CHARTYPE *)"",(CHARTYPE *)"S-",(CHARTYPE *)"C-",(CHARTYPE *)"?",(CHARTYPE *)"A-"};

static CHARTYPE *button_action_names[] =
 {(CHARTYPE *)"R",(CHARTYPE *)"P",(CHARTYPE *)"C",(CHARTYPE *)"2",(CHARTYPE *)"3",(CHARTYPE *)"D"};

#if defined(MOUSE_SUPPORT_ENABLED)   /* set in "the.h" */

/***********************************************************************/
#ifdef HAVE_PROTO
short THEMouse(CHARTYPE *params)
#else
short THEMouse(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 int w=0;
 CHARTYPE scrn=0;
 short rc=RC_OK;
 int curr_button_action=0;
 int curr_button_modifier=0;
 int curr_button=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("mouse.c:  THEMouse");
#endif
 which_window_is_mouse_in(&scrn,&w);
 if (w == (-1)) /* shouldn't happen! */
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
 rc = get_mouse_info(&curr_button,&curr_button_action,&curr_button_modifier);
 if (rc != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 rc = execute_mouse_commands(MOUSE_INFO_TO_KEY(w,curr_button,curr_button_action,curr_button_modifier));
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef HAVE_PROTO
void which_window_is_mouse_in(CHARTYPE *scrn,int *w)
#else
void which_window_is_mouse_in(scrn,w)
CHARTYPE *scrn;
int *w;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE display_screens;
 extern WINDOW *statarea,*divider;
 extern bool horizontal;
/*--------------------------- local data ------------------------------*/
 CHARTYPE i=0;
 int j=0;
 int y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("mouse.c:  which_window_is_mouse_in");
#endif
 for (i=0;i<display_screens;i++)
   {
    for (j=0;j<VIEW_WINDOWS;j++)
      {
       if (screen[i].win[j] != (WINDOW *)NULL)
         {
          wmouse_position(screen[i].win[j],&y,&x);
          if (y != (-1)
          &&  x != (-1))
            {
             *scrn = i;
             *w = j;
#ifdef TRACE
             trace_return();
#endif
             return;
            }
         }
      }
   }
/*---------------------------------------------------------------------*/
/* To get here, the mouse is NOT in any of the view windows; is it in  */
/* the status line ?                                                   */
/*---------------------------------------------------------------------*/
 wmouse_position(statarea,&y,&x);
 if (y != (-1)
 &&  x != (-1))
   {
    *w = WINDOW_STATAREA;
    *scrn = current_screen;
#ifdef TRACE
    trace_return();
#endif
    return;
   }
/*---------------------------------------------------------------------*/
/* To get here, the mouse is NOT in any of the view windows; or the    */
/* status line. Is it in the DIVIDER window ?                          */
/*---------------------------------------------------------------------*/
 if (display_screens > 1
 &&  !horizontal)
   {
    wmouse_position(divider,&y,&x);
    if (y != (-1)
    &&  x != (-1))
      {
       *w = WINDOW_DIVIDER;
       *scrn = current_screen;
#ifdef TRACE
       trace_return();
#endif
       return;
      }
   }
/*---------------------------------------------------------------------*/
/* To get here, the mouse is NOT in ANY window. Return an error.       */
/*---------------------------------------------------------------------*/
 *w = (-1);
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
short get_mouse_info(int *button,int *button_action,int *button_modifier)
#else
short get_mouse_info(button,button_action,button_modifier)
int *button,*button_action,*button_modifier;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("mouse.c:  get_mouse_info");
#endif
 request_mouse_pos();
 if (A_BUTTON_CHANGED)
   {
    if (BUTTON_CHANGED(1))
       *button = 1;
    else
       if (BUTTON_CHANGED(2))
          *button = 2;
       else
          if (BUTTON_CHANGED(3))
             *button = 3;
          else
            {
#ifdef TRACE
             trace_return();
#endif
             return(RC_OK);
            }
    if (BUTTON_STATUS(*button) & BUTTON_SHIFT)
       *button_modifier = BUTTON_SHIFT;
    else
       if (BUTTON_STATUS(*button) & BUTTON_CONTROL)
          *button_modifier = BUTTON_CONTROL;
       else
          if (BUTTON_STATUS(*button) & BUTTON_ALT)
             *button_modifier = BUTTON_ALT;
          else
             *button_modifier = 0;
    if (MOUSE_MOVED)
       *button_action = BUTTON_MOVED;
    else
       *button_action = BUTTON_STATUS(*button) & BUTTON_ACTION_MASK;
   }
 else
   {
    *button = *button_action = *button_modifier = 0;
    rc = RC_INVALID_OPERAND;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef HAVE_PROTO
void initialise_mouse_commands(void)
#else
void initialise_mouse_commands()
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern DEFINE *first_mouse_define;
 extern DEFINE *last_mouse_define;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("mouse.c:   initialise_mouse_commands");
#endif

/*
 * Default mouse actions in FILEAREA
 */
 add_define(&first_mouse_define,&last_mouse_define,
            WINDOW_FILEAREA|MOUSE_LEFT|MOUSE_PRESS|MOUSE_NORMAL,
            (CHARTYPE *)"CURSOR MOUSE",FALSE);
 add_define(&first_mouse_define,&last_mouse_define,
            WINDOW_FILEAREA|MOUSE_LEFT|MOUSE_PRESS|MOUSE_SHIFT,
            (CHARTYPE *)"CURSOR MOUSE#RESET BLOCK#MARK LINE",FALSE);
 add_define(&first_mouse_define,&last_mouse_define,
            WINDOW_FILEAREA|MOUSE_LEFT|MOUSE_PRESS|MOUSE_CONTROL,
            (CHARTYPE *)"CURSOR MOUSE#RESET BLOCK#MARK BOX",FALSE);
 add_define(&first_mouse_define,&last_mouse_define,
            WINDOW_FILEAREA|MOUSE_LEFT|MOUSE_DRAG|MOUSE_SHIFT,
            (CHARTYPE *)"CURSOR MOUSE#MARK LINE",FALSE);
 add_define(&first_mouse_define,&last_mouse_define,
            WINDOW_FILEAREA|MOUSE_LEFT|MOUSE_DRAG|MOUSE_CONTROL,
            (CHARTYPE *)"CURSOR MOUSE#MARK BOX",FALSE);
 add_define(&first_mouse_define,&last_mouse_define,
            WINDOW_FILEAREA|MOUSE_RIGHT|MOUSE_PRESS|MOUSE_SHIFT,
            (CHARTYPE *)"CURSOR MOUSE#MARK LINE",FALSE);
 add_define(&first_mouse_define,&last_mouse_define,
            WINDOW_FILEAREA|MOUSE_RIGHT|MOUSE_PRESS|MOUSE_CONTROL,
            (CHARTYPE *)"CURSOR MOUSE#MARK BOX",FALSE);
 add_define(&first_mouse_define,&last_mouse_define,
            WINDOW_FILEAREA|MOUSE_RIGHT|MOUSE_PRESS|MOUSE_NORMAL,
            (CHARTYPE *)"CURSOR MOUSE#SOS MAKECURR",FALSE);
 add_define(&first_mouse_define,&last_mouse_define,
            WINDOW_FILEAREA|MOUSE_LEFT|MOUSE_DRAG|MOUSE_SHIFT,
            (CHARTYPE *)"CURSOR MOUSE#MARK LINE",FALSE);
 add_define(&first_mouse_define,&last_mouse_define,
            WINDOW_FILEAREA|MOUSE_LEFT|MOUSE_DRAG|MOUSE_CONTROL,
            (CHARTYPE *)"CURSOR MOUSE#MARK BOX",FALSE);
 add_define(&first_mouse_define,&last_mouse_define,
            WINDOW_FILEAREA|MOUSE_LEFT|MOUSE_DOUBLE_CLICK|MOUSE_NORMAL,
            (CHARTYPE *)"CURSOR MOUSE#SOS EDIT",FALSE);
/*
 * Default mouse actions in PREFIX area
 */
 add_define(&first_mouse_define,&last_mouse_define,
            WINDOW_PREFIX|MOUSE_LEFT|MOUSE_PRESS|MOUSE_NORMAL,
            (CHARTYPE *)"CURSOR MOUSE",FALSE);
 add_define(&first_mouse_define,&last_mouse_define,
            WINDOW_PREFIX|MOUSE_RIGHT|MOUSE_PRESS|MOUSE_NORMAL,
            (CHARTYPE *)"CURSOR MOUSE#SOS MAKECURR",FALSE);
 add_define(&first_mouse_define,&last_mouse_define,
            WINDOW_PREFIX|MOUSE_LEFT|MOUSE_DOUBLE_CLICK|MOUSE_NORMAL,
            (CHARTYPE *)"CURSOR MOUSE#SOS EDIT",FALSE);
/*
 * Default mouse actions in COMMAND line
 */
 add_define(&first_mouse_define,&last_mouse_define,
            WINDOW_COMMAND|MOUSE_LEFT|MOUSE_PRESS|MOUSE_NORMAL,
            (CHARTYPE *)"CURSOR MOUSE",FALSE);
/*
 * Default mouse actions in STATAREA
 */
 add_define(&first_mouse_define,&last_mouse_define,
            WINDOW_STATAREA|MOUSE_LEFT|MOUSE_PRESS|MOUSE_NORMAL,
            (CHARTYPE *)"STATUS",FALSE);
/*
 * Default mouse actions in IDLINE
 */
 add_define(&first_mouse_define,&last_mouse_define,
            WINDOW_IDLINE|MOUSE_LEFT|MOUSE_PRESS|MOUSE_NORMAL,
            (CHARTYPE *)"XEDIT",FALSE);
 add_define(&first_mouse_define,&last_mouse_define,
            WINDOW_IDLINE|MOUSE_RIGHT|MOUSE_PRESS|MOUSE_NORMAL,
            (CHARTYPE *)"XEDIT -",FALSE);
/*
 * Default mouse actions in DIVIDER
 */
 add_define(&first_mouse_define,&last_mouse_define,
            WINDOW_DIVIDER|MOUSE_LEFT|MOUSE_PRESS|MOUSE_NORMAL,
            (CHARTYPE *)"SCREEN 1",FALSE);

#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
int mouse_info_to_key(int w, int button, int button_action, int button_modifier)
#else
int mouse_info_to_key(w,button,button_action,button_modifier)
int w,button,button_action,button_modifier;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("mouse.c:   mouse_info_to_key");
#endif

#ifdef TRACE
 trace_return();
#endif
 return(MOUSE_INFO_TO_KEY(w,button,button_action,button_modifier));
}
#endif
/***********************************************************************/
#ifdef HAVE_PROTO
CHARTYPE *mouse_key_number_to_name(int key_number,CHARTYPE *key_name)
#else
CHARTYPE *mouse_key_number_to_name(key_number,key_name)
int key_number;
CHARTYPE *key_name;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern AREAS valid_areas[ATTR_MAX];
/*--------------------------- local data ------------------------------*/
 register int i=0;
 int w=0,b=0,ba=0,bm=0;
 CHARTYPE *win_name=(CHARTYPE *)"*** unknown ***";
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("mouse.c:   mouse_key_number_to_name");
#endif
 w = MOUSE_WINDOW_MASK(key_number);
 b = (MOUSE_BUTTON_MASK(key_number)>>MOUSE_BUTTON_OFFSET);
 ba = (MOUSE_ACTION_MASK(key_number)>>MOUSE_ACTION_OFFSET);
 bm = (MOUSE_MODIFIER_MASK(key_number)>>MOUSE_MODIFIER_OFFSET);
 for (i=0;i<ATTR_MAX;i++)
  {
   if (w == valid_areas[i].area_window
   &&  valid_areas[i].actual_window)
     {
      win_name = valid_areas[i].area;
      break;
     }
  }
 sprintf((DEFCHAR *)key_name,"%s%s%s in %s",button_modifier_names[bm],button_action_names[ba],
         button_names[b],win_name);
#ifdef TRACE
 trace_return();
#endif
 return(key_name);
}
/***********************************************************************/
#ifdef HAVE_PROTO
int find_mouse_key_value(CHARTYPE *mnemonic,CHARTYPE *win_name)
#else
int find_mouse_key_value(mnemonic,win_name)
CHARTYPE *mnemonic,*win_name;
#endif
/***********************************************************************/
/*   Function: find the matching key value for the supplied key name   */
/* Parameters:                                                         */
/*   mnemonic: the key name to be matched                              */
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern AREAS valid_areas[ATTR_MAX];
/*--------------------------- local data ------------------------------*/
 register short i=0;
 int w=(-1),key=0,len=0;
 int b=0,ba=0,bm=0;
 CHARTYPE tmp_buf[6];
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("mouse.c:   find_mouse_key_value");
#endif
/*---------------------------------------------------------------------*/
/* Parse the mnemonic for a valid mouse key definition...              */
/*---------------------------------------------------------------------*/
 len = strlen((DEFCHAR *)mnemonic);
 if (len == 3)
   {
    strcpy((DEFCHAR *)tmp_buf,"N-");
    strcat((DEFCHAR *)tmp_buf,(DEFCHAR *)mnemonic);
   }
 else
   {
    if (len == 5)
      {
       strcpy((DEFCHAR *)tmp_buf,(DEFCHAR *)mnemonic);
      }
    else
      {
       display_error(1,mnemonic,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(-1);
      }
   }
 if (tmp_buf[1] != '-'
 || ( tmp_buf[4] != 'B'
   && tmp_buf[4] != 'b'))
   {
    display_error(1,mnemonic,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(-1);
   }
 /*
  * Validate button modifier
  */
 switch(tmp_buf[0])
   {
    case 'N':
    case 'n':
         bm = 0;
         break;
    case 'S':
    case 's':
         bm = MOUSE_SHIFT;
         break;
    case 'C':
    case 'c':
         bm = MOUSE_CONTROL;
         break;
    case 'A':
    case 'a':
         bm = MOUSE_ALT;
         break;
    default:
         display_error(1,mnemonic,FALSE);
#ifdef TRACE
         trace_return();
#endif
         return(-1);
         break;
   }
 /*
  * Validate button action
  */
 switch(tmp_buf[2])
   {
    case 'P':
    case 'p':
         ba = MOUSE_PRESS;
         break;
    case 'R':
    case 'r':
         ba = MOUSE_RELEASE;
         break;
    case '2':
         ba = MOUSE_DOUBLE_CLICK;
         break;
    case 'D':
    case 'd':
         ba = MOUSE_DRAG;
         break;
    default:
         display_error(1,mnemonic,FALSE);
#ifdef TRACE
         trace_return();
#endif
         return(-1);
         break;
   }
 /*
  * Validate button number
  */
 switch(tmp_buf[3])
   {
    case 'L':
    case 'l':
         b = MOUSE_LEFT;
         break;
    case 'R':
    case 'r':
         b = MOUSE_RIGHT;
         break;
    case 'M':
    case 'm':
         b = MOUSE_MIDDLE;
         break;
    default:
         display_error(1,mnemonic,FALSE);
#ifdef TRACE
         trace_return();
#endif
         return(-1);
         break;
   }
/*---------------------------------------------------------------------*/
/* Find a valid window name for win_name...                            */
/*---------------------------------------------------------------------*/
 for (i=0;i<ATTR_MAX;i++)
   {
    if (equal(valid_areas[i].area,win_name,valid_areas[i].area_min_len))
      {
       w = valid_areas[i].area_window;
       break;
      }
   }
 if (w == (-1))
   {
    display_error(1,win_name,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(-1);
   }
 key = w|b|ba|bm;
#ifdef TRACE
 trace_return();
#endif
 return(key);
}
#if defined(HAVE_SB_INIT)
/***********************************************************************/
#ifdef HAVE_PROTO
short ScrollbarHorz(CHARTYPE *params)
#else
short ScrollbarHorz(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 int cur=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("mouse.c:   ScrollbarHorz");
#endif
/*---------------------------------------------------------------------*/
/* Parse the mnemonic for a valid mouse key definition...              */
/*---------------------------------------------------------------------*/
 sb_get_horz(NULL,NULL,&cur);
 if (cur < CURRENT_VIEW->verify_col)
    rc = Left((CHARTYPE*)"FULL");
 else
    rc = Right((CHARTYPE*)"FULL");
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef HAVE_PROTO
short ScrollbarVert(CHARTYPE *params)
#else
short ScrollbarVert(params)
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 int cur=0;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("mouse.c:   ScrollbarVert");
#endif
/*---------------------------------------------------------------------*/
/* Parse the mnemonic for a valid mouse key definition...              */
/*---------------------------------------------------------------------*/
 sb_get_vert(NULL,NULL,&cur);
 rc = execute_makecurr((LINETYPE)cur);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
#endif
