/* Dvorak Keyboard Driver for Sun Console */

#include <sys/types.h>
#include <sys/file.h>
#include <sundev/kbio.h>
#include <sundev/kbd.h>
#include <stdio.h>

typedef struct {
  char *normal_num;
  char *normal_top;
  char *normal_mid;
  char *normal_bot;

  char *shift_num;
  char *shift_top;
  char *shift_mid;
  char *shift_bot;

  char *caps_num;
  char *caps_top;
  char *caps_mid;
  char *caps_bot;

  char *ctrl_num;
  char *ctrl_top;
  char *ctrl_mid;
  char *ctrl_bot;
  } KbdDescription;

KbdDescription sun3_normal_keys = {
  "\0331234567890-=`",
  "qwertyuiop[]",
  "asdfghjkl;'",
  "zxcvbnm,./" , /* normal */
  "\033!@#$%^&*()_+~",
  "QWERTYUIOP{}",
  "ASDFGHJKL:\"",
  "ZXCVBNM<>?" , /* shift */
  "\0331234567890-=`",
  "QWERTYUIOP[]",
  "ASDFGHJKL;'",
  "ZXCVBNM,./" , /* caps */
  "\0331@345^7890-=`",
  "QWERTYUIOP[]",
  "ASDFGHJKL;'",
  "ZXCVBNM,._"  /* control */
  };

KbdDescription sun3_dvorak_keys = {
  "\0331234567890`=/",
  "',.pyfgcrl[]",
  "aoeuidhtns-",
  ";qjkxbmwvz", /* normal */
  "\033!@#$%^&*()~+?",
  "\"<>PYFGCRL{}",
  "AOEUIDHTNS_",
  ":QJKXBMWVZ", /* shift */
  "\0331234567890`=/",
  "',.PYFGCRL[]",
  "AOEUIDHTNS-",
  ";QJKXBMWVZ", /* caps */
  "\0331@345^7890`=/",
  "',.PYFGCRL[]",
  "AOEUIDHTNS_",
  ";QJKXBMWVZ"  /* control */
  };

typedef struct {
  int num;
  int top;
  int mid;
  int bot;
  } ScanKeyPositions;

/* Places on the sun3 keyboard where the keys that we hack begin */
ScanKeyPositions sun3_scans = {29, 54, 77, 100};


do_set_key(fd, table, slot, value)
int fd, table, slot, value;
{
struct kiockey keyinfo;

keyinfo.kio_tablemask = table;
keyinfo.kio_station = slot;
keyinfo.kio_entry = value;

if (ioctl(fd, KIOCSETKEY, &keyinfo) < 0) {
  perror("KIOCSETKEY");
  fprintf(stderr, "Can't change key %d in table %d", slot, table);
  exit(1); 
  }

/* For debugging, check keys */
#if 0
  if (ioctl(fd, KIOCGETKEY, &keyinfo) < 0) {
    perror("Can't get key %d in table %d", slot, table);
    exit(1); }
  if (keyinfo.kio_entry == value)
    printf("%c", value);
  else printf("\n'%c' -> '%c'\n", keyinfo.kio_entry, value);
#endif
}

set_key_row(fd, start, table, string)
/* Set a row of keys starting with scan code START from string */
int fd,start,table;
char *string;
{
char c;
int i;
int len = strlen(string);

for (i = 0; i < len; i++) {
  c = string[i];		/* Get char from spec */
  if (table == CTRLMASK) {
    if (c >= '@' && c <= '_') /* Chars that can be controlified */
      c = c & 0x1F;		/* Do that, otherwise leave it alone */
    }
  do_set_key(fd, table, start++, c); 
  }
}

/* Set all keys on 3 rows */
set_key_table(fd, thePositions, theMap)
int fd;
ScanKeyPositions *thePositions;
KbdDescription *theMap;
{
set_key_row(fd, thePositions->num, 0, theMap->normal_num);
set_key_row(fd, thePositions->top, 0, theMap->normal_top);
set_key_row(fd, thePositions->mid, 0, theMap->normal_mid);
set_key_row(fd, thePositions->bot, 0, theMap->normal_bot);

set_key_row(fd, thePositions->num, SHIFTMASK, theMap->shift_num);
set_key_row(fd, thePositions->top, SHIFTMASK, theMap->shift_top);
set_key_row(fd, thePositions->mid, SHIFTMASK, theMap->shift_mid);
set_key_row(fd, thePositions->bot, SHIFTMASK, theMap->shift_bot);

set_key_row(fd, thePositions->num, CAPSMASK, theMap->caps_num);
set_key_row(fd, thePositions->top, CAPSMASK, theMap->caps_top);
set_key_row(fd, thePositions->mid, CAPSMASK, theMap->caps_mid);
set_key_row(fd, thePositions->bot, CAPSMASK, theMap->caps_bot);

set_key_row(fd, thePositions->top, CTRLMASK, theMap->ctrl_top);
set_key_row(fd, thePositions->top, CTRLMASK, theMap->ctrl_top);
set_key_row(fd, thePositions->mid, CTRLMASK, theMap->ctrl_mid);
set_key_row(fd, thePositions->bot, CTRLMASK, theMap->ctrl_bot);
}

/* Tell about how to use the program */
usage()
{
  printf("Usage: 'dvorak on' or 'dvorak off'\n");
  exit (1); 
}

xerror(str)
char *str;
{ 
  perror(str);
  exit(1);
}

int main(argc, argv)
int argc;
char *argv[];
{
int fd,type;
int dvorak;			/* true if going to dvorak */
KbdDescription *theDescription;
ScanKeyPositions *thePositions;

if (argc!=2) usage();

if (strcmp(argv[1],"on") == 0)
  dvorak = 1;
else 
if (strcmp(argv[1],"off") == 0)
  dvorak = 0;
else 
  usage();

/* Open the keyboard device to hack on it */

if ((fd = open("/dev/kbd",O_RDONLY,0)) < 0)
  xerror("Can't open /dev/kbd");

/* Check that the keyboard is the right type.  They have different slots
   so we could lose badly if it isn't the right one */

if (ioctl(fd, KIOCTYPE, &type) < 0)
  xerror("Can't determine keyboard type");

switch (type) {
  case KB_SUN3:
  case KB_SUN4:
    thePositions = &sun3_scans;
    if (dvorak)
      theDescription = &sun3_dvorak_keys;
    else
      theDescription = &sun3_normal_keys;
    break;

  default:
    fprintf(stderr, "Unknown keyboard type %d\n", type);
    exit(1);
  }

/* Now do the work */
set_key_table(fd, thePositions, theDescription);

if (dvorak) 
  printf("Now dvorak\n");
else 
  printf("Now qwerty\n");

if (close(fd) < 0)
  xerror("Can't close /dev/kbd");
}
