/* XQF - Quake server browser and launcher
 * Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>	/* sprintf */
#include <stdlib.h>	/* strtol */
#include <sys/time.h>	/* FD_SETSIZE */
#include <sys/types.h>	/* FD_SETSIZE */

#include <gtk/gtk.h>

#include "xqf.h"
#include "pref.h"
#include "dialogs.h"
#include "skin.h"
#include "utils.h"
#include "qrun.h"


struct user_info user;


int 	default_rate = 2500;
int 	default_cl_nodelta = FALSE;
int 	default_cl_predict = TRUE;
int 	default_noaim = FALSE;
int 	default_windowed_mouse = FALSE;
char 	*default_name = NULL;
char 	*default_team = NULL;
char 	*default_skin = NULL;
int 	default_top_color = 0;
int 	default_bottom_color = 0;
int 	default_noskins = 0;
char 	*default_quake_dir = NULL;
char	*default_qw_cmd = NULL;
char 	*default_q2_dir = NULL;
char	*default_q2_cmd = NULL;
int	default_terminate = 0;
int 	default_b_switch = 0;
int 	default_w_switch = 0;
int	default_save_lists = FALSE;

int	filter_retries = 3;
int	filter_ping = 2000;
int 	filter_not_full = 0;
int 	filter_not_empty = 0;
int	filter_no_cheats = 0;
int	filter_no_password = 0;

int     maxretries = 3;
int     maxsimultaneous = 20;

int	filter_prefs_changed = FALSE;


static  GtkWidget *rate_entry;
static  GtkWidget *cl_nodelta_check_button;
static  GtkWidget *cl_predict_check_button;
static  GtkWidget *noaim_check_button;
static  GtkWidget *windowed_mouse_check_button;
static  GtkWidget *name_entry;
static  GtkWidget *team_entry;
static  GtkWidget *skin_preview = NULL;
static  GtkWidget *skin_entry;
static  GtkWidget *top_color_button;
static  GtkWidget *bottom_color_button;
static  GtkWidget *noskins_option_menu;
static  GtkWidget *quake_dir_entry;
static 	GtkWidget *qw_cmd_entry;
static  GtkWidget *q2_dir_entry;
static 	GtkWidget *q2_cmd_entry;
static  GtkWidget *terminate_check_button;
static  GtkWidget *w_switch_option_menu;
static  GtkWidget *b_switch_option_menu;
static  GtkWidget *save_lists_check_button;

static  GtkWidget *filter_retries_entry;
static  GtkWidget *filter_ping_entry;
static  GtkWidget *filter_not_full_check_button;
static  GtkWidget *filter_not_empty_check_button;
static  GtkWidget *filter_no_cheats_check_button;
static  GtkWidget *filter_no_password_check_button;

static  GtkWidget *maxretries_entry;
static  GtkWidget *maxsimultaneous_entry;

static 	guchar *skin_data = NULL;
static	GtkWidget *skin_list = NULL;
static	GtkWidget *color_menu = NULL;


static void get_new_defaults (void) {
  int i;

  if (default_name) g_free (default_name);
  default_name = strdup_strip (gtk_entry_get_text (GTK_ENTRY (name_entry)));

  if (default_team) g_free (default_team);
  default_team = strdup_strip (gtk_entry_get_text (GTK_ENTRY (team_entry)));

  default_rate = strtol (gtk_entry_get_text (GTK_ENTRY (rate_entry)), NULL, 10);
  default_cl_nodelta = GTK_TOGGLE_BUTTON (cl_nodelta_check_button)->active;
  default_cl_predict = GTK_TOGGLE_BUTTON (cl_predict_check_button)->active;
  default_noaim = 1 - GTK_TOGGLE_BUTTON (noaim_check_button)->active;

  if (default_qw_cmd) g_free (default_qw_cmd);
  default_qw_cmd = strdup_strip (gtk_entry_get_text (GTK_ENTRY (qw_cmd_entry)));

  if (default_quake_dir) g_free (default_quake_dir);
  default_quake_dir = strdup_strip (gtk_entry_get_text (
                                               GTK_ENTRY (quake_dir_entry)));

  if (default_q2_cmd) g_free (default_q2_cmd);
  default_q2_cmd = strdup_strip (gtk_entry_get_text (GTK_ENTRY (q2_cmd_entry)));

  if (default_q2_dir) g_free (default_q2_dir);
  default_q2_dir = strdup_strip (gtk_entry_get_text (GTK_ENTRY (q2_dir_entry)));

  default_windowed_mouse = 
                       GTK_TOGGLE_BUTTON (windowed_mouse_check_button)->active;

  default_terminate = GTK_TOGGLE_BUTTON (terminate_check_button)->active;

  default_save_lists = GTK_TOGGLE_BUTTON (save_lists_check_button)->active;

  i = strtol (gtk_entry_get_text (GTK_ENTRY (filter_retries_entry)), NULL, 10);
  if (filter_retries != i) {
    filter_retries = i;
    filter_prefs_changed = TRUE;

    if (filter_retries > MAX_RETRIES)
      filter_retries = MAX_RETRIES;
    if (filter_retries < 0)
      filter_retries = 0;
  }

  i = strtol (gtk_entry_get_text (GTK_ENTRY (filter_ping_entry)), NULL, 10);
  if (filter_ping != i) {
    filter_ping = i;
    filter_prefs_changed = TRUE;

    if (filter_ping > MAX_PING)
      filter_ping = MAX_PING;
    if (filter_ping < 0)
      filter_ping = 0;
  }

  i = GTK_TOGGLE_BUTTON (filter_not_full_check_button)->active;
  if (filter_not_full != i) {
    filter_not_full = i;
    filter_prefs_changed = TRUE;
  }

  i = GTK_TOGGLE_BUTTON (filter_not_empty_check_button)->active;
  if (filter_not_empty != i) {
    filter_not_empty = i;
    filter_prefs_changed = TRUE;
  }

  i = GTK_TOGGLE_BUTTON (filter_no_cheats_check_button)->active;
  if (filter_no_cheats != i) {
    filter_no_cheats = i;
    filter_prefs_changed = TRUE;
  }

  i = GTK_TOGGLE_BUTTON (filter_no_password_check_button)->active;
  if (filter_no_password != i) {
    filter_no_password  = i;
    filter_prefs_changed = TRUE;
  }

  maxretries = 
          strtol (gtk_entry_get_text (GTK_ENTRY (maxretries_entry)), NULL, 10);
  if (maxretries > MAX_RETRIES)
    maxretries = MAX_RETRIES;
  if (maxretries < 1)
    maxretries = 1;

  maxsimultaneous = 
     strtol (gtk_entry_get_text (GTK_ENTRY (maxsimultaneous_entry)), NULL, 10);
  if (maxsimultaneous > FD_SETSIZE)
    maxsimultaneous = FD_SETSIZE;
  if (maxsimultaneous < 1)
    maxsimultaneous = 1;
  
  /*
   * things already set by callbacks:
   *   - default_top_color, default_bottom_color
   *   - default_b_switch, default_w_switch
   *   - default_skin
   *   - default_noskins
   */
}


void update_skins (void) {
  GList *list;
  char *path;

  if (skin_preview == NULL)
    return;			/* no configuration window */

  gtk_list_clear_items (GTK_LIST (skin_list), 0, -1);

  if (skin_data) {
    g_free (skin_data);
    skin_data = NULL;
  }

  path = file_in_dir (default_quake_dir, "qw/skins");
  list = build_skin_list (path);
  g_free (path);

  if (list) {
    /*  
     *  gtk_list_insert_items automaticaly selects the first list item 
     *	if selection mode is GTK_SELECTION_BROWSE
     *  So, drop GTK_SELECTION_BROWSE mode in time of list filling
     */

    gtk_list_set_selection_mode (GTK_LIST (skin_list), GTK_SELECTION_SINGLE);
    gtk_list_prepend_items (GTK_LIST (skin_list), list);
    gtk_list_set_selection_mode (GTK_LIST (skin_list), GTK_SELECTION_BROWSE);

    skin_data = get_skin (default_skin);
  }

  draw_skin (skin_preview, skin_data);
}


static void qw_dir_entry_activate_callback (GtkWidget *widget, gpointer data) {

  if (default_quake_dir)
    g_free (default_quake_dir);

  default_quake_dir = strdup_strip (gtk_entry_get_text (
                                                 GTK_ENTRY (quake_dir_entry)));
  update_skins ();
}


static void skin_list_selection_changed_callback (GtkWidget *widget, gpointer data) {
  GList	*selected;
  GList *item_widget_list;
  char *str;

  selected = GTK_LIST (widget) -> selection;
  if (selected) {
    item_widget_list = gtk_container_children (GTK_CONTAINER (selected->data));
    gtk_label_get (GTK_LABEL (item_widget_list->data), &str);

    if (default_skin) g_free (default_skin);
    default_skin = strdup_strip (str);

    gtk_entry_set_text (GTK_ENTRY (skin_entry), default_skin);

    if (skin_data) g_free (skin_data);
    skin_data = get_skin (default_skin);
    draw_skin (skin_preview, skin_data);
  }
}


static void skin_entry_activate_callback (GtkWidget *widget, gpointer data) {
  GList *selected;

  if (default_skin) g_free (default_skin);
  default_skin = strdup_strip (gtk_entry_get_text (GTK_ENTRY (skin_entry)));

  selected = GTK_LIST (skin_list) -> selection;
  if (selected)
    gtk_list_unselect_child (GTK_LIST (skin_list), GTK_WIDGET (selected->data));

  if (skin_data) g_free (skin_data);
  skin_data = get_skin (default_skin);
  draw_skin (skin_preview, skin_data);
}


static GtkWidget *color_button_event_widget = NULL;


static void set_player_color (GtkWidget *widget, int i) {
  if (color_button_event_widget == top_color_button) {
    if (default_top_color != i) {
      default_top_color = i;
      set_bg_color (top_color_button, default_top_color);
      draw_skin (skin_preview, skin_data);
    }
    return;
  }

  if (color_button_event_widget == bottom_color_button) {
    if (default_bottom_color != i) {
      default_bottom_color = i;
      set_bg_color (bottom_color_button, default_bottom_color);
      draw_skin (skin_preview, skin_data);
    }
    return;
  }
}


static int color_button_event_callback (GtkWidget *widget, GdkEvent *event) {
  GdkEventButton *bevent; 

  if (event->type == GDK_BUTTON_PRESS) {
    bevent = (GdkEventButton *) event; 
    color_button_event_widget = widget;

    if (color_menu == NULL)
      color_menu = create_color_menu (set_player_color);

    gtk_menu_popup (GTK_MENU (color_menu), NULL, NULL, NULL, NULL,
	                                         bevent->button, bevent->time);
    return TRUE;
  }
  return FALSE;
}


GtkWidget *skin_box_create (void) {
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *scrolled;

  /* skin preview  */

  hbox = gtk_hbox_new (FALSE, 6);
  gtk_container_border_width (GTK_CONTAINER (hbox), 6);

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
  gtk_box_pack_start (GTK_BOX (hbox), frame, FALSE, FALSE, 0);

  skin_preview = gtk_preview_new (GTK_PREVIEW_COLOR);
  gtk_preview_size (GTK_PREVIEW (skin_preview), 320, 200);
  gtk_container_add (GTK_CONTAINER (frame), skin_preview);
  gtk_widget_show (skin_preview);

  gtk_widget_show (frame);

  vbox = gtk_vbox_new (FALSE, 6);
  gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);

  /* skin entry */

  skin_entry = gtk_entry_new_with_max_length (32);
  gtk_widget_set_usize (skin_entry, 96, -1);
  if (default_skin) {
    gtk_entry_set_text (GTK_ENTRY (skin_entry), default_skin);
    gtk_entry_set_position (GTK_ENTRY (skin_entry), 0);
  }
  gtk_signal_connect (GTK_OBJECT (skin_entry), "activate",
		      GTK_SIGNAL_FUNC (skin_entry_activate_callback), NULL);
  gtk_box_pack_start (GTK_BOX (vbox), skin_entry, FALSE, FALSE, 0);
  gtk_widget_show (skin_entry);

  /* skin list */

  scrolled = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_set_usize (scrolled, 96, -1);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled), 
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX (vbox), scrolled, TRUE, TRUE, 0);

  skin_list = gtk_list_new ();
  gtk_list_set_selection_mode (GTK_LIST (skin_list), GTK_SELECTION_BROWSE);
  gtk_signal_connect (GTK_OBJECT (skin_list), "selection_changed",
		      GTK_SIGNAL_FUNC (skin_list_selection_changed_callback), NULL);
  gtk_container_add (GTK_CONTAINER (scrolled), skin_list);
  gtk_widget_show (skin_list);

  gtk_widget_show (scrolled);

  /* top and bottom colors */

  table = gtk_table_new (2, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 4);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  gtk_box_pack_end (GTK_BOX (vbox), table, FALSE, FALSE, 2);

  /* top (shirt) color */

  label = gtk_label_new ("Top");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show (label);

  top_color_button = gtk_button_new_with_label (" ");
  gtk_widget_set_usize (GTK_WIDGET (top_color_button), 40, -1);
  gtk_signal_connect (GTK_OBJECT(top_color_button), "event",
                          GTK_SIGNAL_FUNC (color_button_event_callback), NULL);
  gtk_table_attach_defaults (GTK_TABLE (table), top_color_button, 0, 1, 0, 1);
  set_bg_color (top_color_button, default_top_color);
  gtk_widget_show (top_color_button);

  /* bottom (pants) color */

  label = gtk_label_new ("Bottom");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show (label);

  bottom_color_button = gtk_button_new_with_label (" ");
  gtk_widget_set_usize (GTK_WIDGET (bottom_color_button), 40, -1);
  gtk_signal_connect (GTK_OBJECT(bottom_color_button), "event",
		          GTK_SIGNAL_FUNC (color_button_event_callback), NULL);
  gtk_table_attach_defaults (GTK_TABLE (table), bottom_color_button, 0, 1, 1, 2);
  set_bg_color (bottom_color_button, default_bottom_color);
  gtk_widget_show (bottom_color_button);

  gtk_widget_show (table);

  gtk_widget_show (vbox);

  gtk_widget_show (hbox);

  return hbox;
}


static GtkWidget *player_profile_page (void) {
  GtkWidget *page_vbox;
  GtkWidget *label;
  GtkWidget *frame;
  GtkWidget *hbox;
  GtkWidget *hbox2;
  GtkWidget *skin;

  page_vbox = gtk_vbox_new (FALSE, 8);
  gtk_container_border_width (GTK_CONTAINER (page_vbox), 8);

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (page_vbox), frame, FALSE, FALSE, 0);

  hbox = gtk_hbox_new (TRUE, 64);
  gtk_container_border_width (GTK_CONTAINER (hbox), 6);
  gtk_container_add (GTK_CONTAINER (frame), hbox);

  /* player name */

  hbox2 = gtk_hbox_new (FALSE, 6);
  gtk_box_pack_start (GTK_BOX (hbox), hbox2, TRUE, TRUE, 0);

  label = gtk_label_new ("Name");
  gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  name_entry = gtk_entry_new_with_max_length (32);
  gtk_widget_set_usize (GTK_WIDGET (name_entry), 96, -1);
  if (default_name) {
    gtk_entry_set_text (GTK_ENTRY (name_entry), default_name);
    gtk_entry_set_position (GTK_ENTRY (name_entry), 0);
  }
  gtk_box_pack_end (GTK_BOX (hbox2), name_entry, FALSE, FALSE, 0);
  gtk_widget_show (name_entry);

  gtk_widget_show (hbox2);

  /* player team */

  hbox2 = gtk_hbox_new (FALSE, 6);
  gtk_box_pack_start (GTK_BOX (hbox), hbox2, TRUE, TRUE, 0);

  label = gtk_label_new ("QW Team");
  gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 0);
  gtk_widget_show (label);

  team_entry = gtk_entry_new_with_max_length (32);
  gtk_widget_set_usize (GTK_WIDGET (team_entry), 96, -1);
  if (default_team) {
    gtk_entry_set_text (GTK_ENTRY (team_entry), default_team);
    gtk_entry_set_position (GTK_ENTRY (team_entry), 0);
  }
  gtk_box_pack_end (GTK_BOX (hbox2), team_entry, FALSE, FALSE, 0);
  gtk_widget_show (team_entry);

  gtk_widget_show (hbox2);

  gtk_widget_show (hbox);
  gtk_widget_show (frame);

  /* skin */

  frame = gtk_frame_new (" QW Skin ");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (page_vbox), frame, FALSE, FALSE, 0);

  skin = skin_box_create ();
  gtk_container_add (GTK_CONTAINER (frame), skin);

  gtk_widget_show (frame);

  gtk_widget_show (page_vbox);

  return page_vbox;
}


static char *wb_switch_labels[9] = {
  "--- Anything ---",
  "Axe",
  "Shotgun",
  "Super Shotgun",
  "Nailgun",
  "Super Nailgun",
  "Grenade Launcher",
  "Rocket Launcher",
  "ThunderBolt"
};


static void set_w_switch_callback (GtkWidget *widget, int i) {
  default_w_switch = i;
}


static void set_b_switch_callback (GtkWidget *widget, int i) {
  default_b_switch = i;
}


static GtkWidget *create_wb_switch_menu (void (*callback) (GtkWidget *, int)) {
  GtkWidget *menu;
  GtkWidget *menu_item;
  int i;

  menu = gtk_menu_new ();

  for (i = 0; i < 9; i++) {
    menu_item = gtk_menu_item_new_with_label (wb_switch_labels[i]);
    gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
			GTK_SIGNAL_FUNC (callback), (gpointer) i);
    gtk_menu_append (GTK_MENU (menu), menu_item);
    gtk_widget_show (menu_item);
  }

  return menu;
}


static void noskins_option_menu_callback (GtkWidget *widget, int i) {
  default_noskins = i;
}


static GtkWidget *create_noskins_menu (void) {
  GtkWidget *menu;
  GtkWidget *menu_item;

  menu = gtk_menu_new ();

  menu_item = gtk_menu_item_new_with_label ("Use skins");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
                 GTK_SIGNAL_FUNC (noskins_option_menu_callback), (gpointer) 0);
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  menu_item = gtk_menu_item_new_with_label ("Don't use skins");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
                 GTK_SIGNAL_FUNC (noskins_option_menu_callback), (gpointer) 1);
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  menu_item = gtk_menu_item_new_with_label ("Don't download new skins");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
                 GTK_SIGNAL_FUNC (noskins_option_menu_callback), (gpointer) 2);
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  return menu;
}


static GtkWidget *gameplay_options_page (void) {
  GtkWidget *page_vbox;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *frame;
  GtkWidget *hbox;
  GtkWidget *vbox;
  GtkWidget *option_menu;
  GtkWidget *alignment;
  char str[32];

  page_vbox = gtk_vbox_new (FALSE, 8);
  gtk_container_border_width (GTK_CONTAINER (page_vbox), 8);

  /* Network Options */

  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (page_vbox), frame, FALSE, FALSE, 0);

  table = gtk_table_new (2, 3, FALSE);
  gtk_container_border_width (GTK_CONTAINER (table), 6);
  gtk_table_set_row_spacings (GTK_TABLE (table), 4);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  gtk_table_set_col_spacing (GTK_TABLE (table), 0, 40);
  gtk_container_add (GTK_CONTAINER (frame), table);

  /* Skins */

  option_menu = gtk_option_menu_new ();
  gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), 
                                                       create_noskins_menu ());
  gtk_option_menu_set_history (GTK_OPTION_MENU (option_menu), default_noskins);
  gtk_table_attach_defaults (GTK_TABLE (table), option_menu, 0, 1, 0, 1);
  gtk_widget_show (option_menu);

  /* rate */

  label = gtk_label_new ("rate");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 0, 1);
  gtk_widget_show (label);

  rate_entry = gtk_entry_new_with_max_length (5);
  sprintf (str, "%d", default_rate);
  gtk_entry_set_text (GTK_ENTRY (rate_entry), str);
  gtk_widget_set_usize (GTK_WIDGET (rate_entry), 48, -1);
  gtk_table_attach_defaults (GTK_TABLE (table), rate_entry, 2, 3, 0, 1);
  gtk_widget_show (rate_entry);

  /* cl_nodelta */

  cl_nodelta_check_button = gtk_check_button_new_with_label ("cl_nodelta");
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (cl_nodelta_check_button), 
                                                           default_cl_nodelta);
  gtk_table_attach_defaults (GTK_TABLE (table), cl_nodelta_check_button, 1, 3, 1, 2);
  gtk_widget_show (cl_nodelta_check_button);

  /* cl_predict_players (cl_predict in Q2) */

  cl_predict_check_button = gtk_check_button_new_with_label ("cl_predict_players");
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (cl_predict_check_button), 
                                                           default_cl_predict);
  gtk_table_attach_defaults (GTK_TABLE (table), cl_predict_check_button, 0, 1, 1, 2);
  gtk_widget_show (cl_predict_check_button);

  gtk_widget_show (table);
  gtk_widget_show (frame);

  /* QW specific features */

  frame = gtk_frame_new (" QuakeWorld ");
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (page_vbox), frame, FALSE, FALSE, 0);

  vbox = gtk_vbox_new (FALSE, 8);
  gtk_container_border_width (GTK_CONTAINER (vbox), 6);
  gtk_container_add (GTK_CONTAINER (frame), vbox);

  /* w_switch & b_switch */

  table = gtk_table_new (2, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 8);
  gtk_table_set_col_spacings (GTK_TABLE (table), 4);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 0);

  /* w_switch */

  label = gtk_label_new ("The highest weapon\n"
                         "that Quake should switch to\n"
                         "upon a weapon pickup");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 0, 1);
  gtk_widget_show (label);

  alignment = gtk_alignment_new (1.0, 0.5, 0.0, 0.0);
  gtk_table_attach_defaults (GTK_TABLE (table), alignment, 1, 2, 0, 1);

  option_menu = gtk_option_menu_new ();
  gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), 
                                create_wb_switch_menu (set_w_switch_callback));
  gtk_option_menu_set_history (GTK_OPTION_MENU (option_menu), default_w_switch);
  gtk_container_add (GTK_CONTAINER (alignment), option_menu);
  gtk_widget_show (option_menu);

  gtk_widget_show (alignment);

  /* b_switch */

  label = gtk_label_new ("The highest weapon\n"
                         "that Quake should switch to\n"
                         "upon a backpack pickup");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 1, 2);
  gtk_widget_show (label);

  alignment = gtk_alignment_new (1.0, 0.5, 0.0, 0.0);
  gtk_table_attach_defaults (GTK_TABLE (table), alignment, 1, 2, 1, 2);

  option_menu = gtk_option_menu_new ();
  gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), 
                                create_wb_switch_menu (set_b_switch_callback));
  gtk_option_menu_set_history (GTK_OPTION_MENU (option_menu), default_b_switch);
  gtk_container_add (GTK_CONTAINER (alignment), option_menu);
  gtk_widget_show (option_menu);

  gtk_widget_show (alignment);

  gtk_widget_show (table);

  /* noaim */

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

  noaim_check_button = gtk_check_button_new_with_label ("Autoaiming");
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (noaim_check_button), 
                                                           1 - default_noaim);
  gtk_box_pack_start (GTK_BOX (hbox), noaim_check_button, FALSE, FALSE, 0);
  gtk_widget_show (noaim_check_button);

  gtk_widget_show (hbox);

  gtk_widget_show (vbox);
  gtk_widget_show (frame);

  gtk_widget_show (page_vbox);

  return page_vbox;
}


static GtkWidget *filter_options_page (void) {
  GtkWidget *alignment;
  GtkWidget *vbox;
  GtkWidget *table;
  GtkWidget *label;
  char str[32];

  alignment = gtk_alignment_new (0.5, 0.5, 0.0, 0.0);
  gtk_container_border_width (GTK_CONTAINER (alignment), 8);

  vbox = gtk_vbox_new (FALSE, 6);
  gtk_container_add (GTK_CONTAINER (alignment), vbox);

  label = gtk_label_new ("Server would pass filter if:");
  gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 10);
  gtk_widget_show (label);

  table = gtk_table_new (6, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 4);
  gtk_table_set_col_spacings (GTK_TABLE (table), 8);
  gtk_box_pack_start (GTK_BOX (vbox), table, TRUE, FALSE, 0);

  /* max ping */

  label = gtk_label_new ("ping is less than");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show (label);

  filter_ping_entry = gtk_entry_new_with_max_length (4);
  sprintf (str, "%d", filter_ping);
  gtk_entry_set_text (GTK_ENTRY (filter_ping_entry), str);
  gtk_widget_set_usize (filter_ping_entry, 48, -1);
  gtk_table_attach_defaults (GTK_TABLE (table), filter_ping_entry, 1, 2, 0, 1);
  gtk_widget_show (filter_ping_entry);

  /* max timeouts */

  label = gtk_label_new ("the number of retires is fewer than");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show (label);

  filter_retries_entry = gtk_entry_new_with_max_length (1);
  sprintf (str, "%d", filter_retries);
  gtk_entry_set_text (GTK_ENTRY (filter_retries_entry), str);
  gtk_widget_set_usize (filter_retries_entry, 48, -1);
  gtk_table_attach_defaults (GTK_TABLE (table), filter_retries_entry, 1, 2, 1, 2);
  gtk_widget_show (filter_retries_entry);

  /* not full */

  filter_not_full_check_button = 
                            gtk_check_button_new_with_label ("it is not full");
  gtk_toggle_button_set_state (
            GTK_TOGGLE_BUTTON (filter_not_full_check_button), filter_not_full);
  gtk_table_attach_defaults (GTK_TABLE (table), filter_not_full_check_button, 
                                                                   0, 2, 2, 3);
  gtk_widget_show (filter_not_full_check_button);

  /* not empty */

  filter_not_empty_check_button = 
                           gtk_check_button_new_with_label ("it is not empty");
  gtk_toggle_button_set_state (
          GTK_TOGGLE_BUTTON (filter_not_empty_check_button), filter_not_empty);
  gtk_table_attach_defaults (GTK_TABLE (table), filter_not_empty_check_button, 
                                                                   0, 2, 3, 4);
  gtk_widget_show (filter_not_empty_check_button);

  /* no cheats */

  filter_no_cheats_check_button = 
                    gtk_check_button_new_with_label ("cheats are not allowed");
  gtk_toggle_button_set_state (
          GTK_TOGGLE_BUTTON (filter_no_cheats_check_button), filter_no_cheats);
  gtk_table_attach_defaults (GTK_TABLE (table), filter_no_cheats_check_button, 
                                                                   0, 2, 4, 5);
  gtk_widget_show (filter_no_cheats_check_button);

  /* no password */

  filter_no_password_check_button = 
                      gtk_check_button_new_with_label ("no password required");
  gtk_toggle_button_set_state (
      GTK_TOGGLE_BUTTON (filter_no_password_check_button), filter_no_password);
  gtk_table_attach_defaults (GTK_TABLE (table), filter_no_password_check_button, 
                                                                   0, 2, 5, 6);
  gtk_widget_show (filter_no_password_check_button);

  gtk_widget_show (table);
  gtk_widget_show (vbox);
  gtk_widget_show (alignment);

  return alignment;
}


static GtkWidget *configuration_page (void) {
  GtkWidget *page_vbox;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *hbox;
  char str[32];

  page_vbox = gtk_vbox_new (FALSE, 4);
  gtk_container_border_width (GTK_CONTAINER (page_vbox), 8);

  /* QuakeWorld frame */

  frame = gtk_frame_new (" QuakeWorld ");
  gtk_box_pack_start (GTK_BOX (page_vbox), frame, FALSE, FALSE, 0);

  table = gtk_table_new (2, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 4);
  gtk_table_set_col_spacings (GTK_TABLE (table), 8);
  gtk_container_border_width (GTK_CONTAINER (table), 6);
  gtk_container_add (GTK_CONTAINER (frame), table);

  /* QW command line */

  label = gtk_label_new ("Command Line");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show (label);

  qw_cmd_entry = gtk_entry_new ();
  if (default_qw_cmd) {
    gtk_entry_set_text (GTK_ENTRY (qw_cmd_entry), default_qw_cmd);
    gtk_entry_set_position (GTK_ENTRY (qw_cmd_entry), 0);
  }
  gtk_table_attach_defaults (GTK_TABLE (table), qw_cmd_entry, 1, 2, 0, 1);
  gtk_widget_show (qw_cmd_entry);

  /* QW working directory */

  label = gtk_label_new ("Working Directory");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show (label);

  quake_dir_entry = gtk_entry_new ();
  if (default_quake_dir) {
    gtk_entry_set_text (GTK_ENTRY (quake_dir_entry), default_quake_dir);
    gtk_entry_set_position (GTK_ENTRY (quake_dir_entry), 0);
  }
  gtk_table_attach_defaults (GTK_TABLE (table), quake_dir_entry, 1, 2, 1, 2);
  gtk_signal_connect (GTK_OBJECT (quake_dir_entry), "activate",
		      GTK_SIGNAL_FUNC (qw_dir_entry_activate_callback), NULL);
  gtk_widget_show (quake_dir_entry);

  gtk_widget_show (table);
  gtk_widget_show (frame);

  /* Quake II frame */

  frame = gtk_frame_new (" Quake II ");
  gtk_box_pack_start (GTK_BOX (page_vbox), frame, FALSE, FALSE, 0);

  table = gtk_table_new (2, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 4);
  gtk_table_set_col_spacings (GTK_TABLE (table), 8);
  gtk_container_border_width (GTK_CONTAINER (table), 6);
  gtk_container_add (GTK_CONTAINER (frame), table);

  /* Q2 command line */

  label = gtk_label_new ("Command Line");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show (label);

  q2_cmd_entry = gtk_entry_new ();
  if (default_q2_cmd) {
    gtk_entry_set_text (GTK_ENTRY (q2_cmd_entry), default_q2_cmd);
    gtk_entry_set_position (GTK_ENTRY (q2_cmd_entry), 0);
  }
  gtk_table_attach_defaults (GTK_TABLE (table), q2_cmd_entry, 1, 2, 0, 1);
  gtk_widget_show (q2_cmd_entry);

  /* Q2 working directory */

  label = gtk_label_new ("Working Directory");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show (label);

  q2_dir_entry = gtk_entry_new ();
  if (default_q2_dir) {
    gtk_entry_set_text (GTK_ENTRY (q2_dir_entry), default_q2_dir);
    gtk_entry_set_position (GTK_ENTRY (q2_dir_entry), 0);
  }
  gtk_table_attach_defaults (GTK_TABLE (table), q2_dir_entry, 1, 2, 1, 2);
  gtk_widget_show (q2_dir_entry);

  gtk_widget_show (table);
  gtk_widget_show (frame);

  /* _windowed_mouse */

  hbox = gtk_hbox_new (FALSE, 6);
  gtk_box_pack_start (GTK_BOX (page_vbox), hbox, FALSE, FALSE, 0);

  windowed_mouse_check_button = 
                  gtk_check_button_new_with_label ("Client should grab mouse");
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (windowed_mouse_check_button), 
                                                       default_windowed_mouse);
  gtk_box_pack_start (GTK_BOX (hbox), windowed_mouse_check_button, FALSE, FALSE, 0);
  gtk_widget_show (windowed_mouse_check_button);

  gtk_widget_show (hbox);

  /* Terminate */

  hbox = gtk_hbox_new (FALSE, 6);
  gtk_box_pack_start (GTK_BOX (page_vbox), hbox, FALSE, FALSE, 0);

  terminate_check_button = 
        gtk_check_button_new_with_label ("Terminate XQF when launching Quake");
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (terminate_check_button), 
                                                            default_terminate);
  gtk_box_pack_start (GTK_BOX (hbox), terminate_check_button, FALSE, FALSE, 0);
  gtk_widget_show (terminate_check_button);

  gtk_widget_show (hbox);

  /* Save master lists on exit */

  hbox = gtk_hbox_new (FALSE, 6);
  gtk_box_pack_start (GTK_BOX (page_vbox), hbox, FALSE, FALSE, 0);

  save_lists_check_button = 
                 gtk_check_button_new_with_label ("Save server lists on exit");
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (save_lists_check_button), 
                                                           default_save_lists);
  gtk_box_pack_start (GTK_BOX (hbox), save_lists_check_button, FALSE, FALSE, 0);
  gtk_widget_show (save_lists_check_button);

  gtk_widget_show (hbox);

  /* QStat preferences -- maxsimultaneous & maxretries */

  frame = gtk_frame_new (" QStat ");
  gtk_box_pack_start (GTK_BOX (page_vbox), frame, FALSE, FALSE, 0);

  table = gtk_table_new (2, 2, FALSE);
  gtk_table_set_row_spacings (GTK_TABLE (table), 4);
  gtk_table_set_col_spacings (GTK_TABLE (table), 48);
  gtk_container_border_width (GTK_CONTAINER (table), 6);
  gtk_container_add (GTK_CONTAINER (frame), table);

  /* maxsimultaneous */

  label = gtk_label_new ("Number of simultaneous servers to query");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show (label);

  maxsimultaneous_entry = gtk_entry_new_with_max_length (4);
  sprintf (str, "%d", maxsimultaneous);
  gtk_entry_set_text (GTK_ENTRY (maxsimultaneous_entry), str);
  gtk_table_attach_defaults (GTK_TABLE (table), maxsimultaneous_entry, 1, 2, 0, 1);
  gtk_widget_set_usize (GTK_WIDGET (maxsimultaneous_entry), 32, -1);
  gtk_widget_show (maxsimultaneous_entry);

  /* maxretries */

  label = gtk_label_new ("Number of retries");
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
  gtk_widget_show (label);

  maxretries_entry = gtk_entry_new_with_max_length (2);
  sprintf (str, "%d", maxretries);
  gtk_entry_set_text (GTK_ENTRY (maxretries_entry), str);
  gtk_table_attach_defaults (GTK_TABLE (table), maxretries_entry, 1, 2, 1, 2);
  gtk_widget_set_usize (GTK_WIDGET (maxretries_entry), 32, -1);
  gtk_widget_show (maxretries_entry);

  gtk_widget_show (table);
  gtk_widget_show (frame);

  gtk_widget_show (page_vbox);

  return page_vbox;
}


static void preferences_window_destroy (GtkWidget *widget, gpointer data) {

  get_new_defaults ();

  if (color_menu) {
    gtk_widget_destroy (color_menu);
    color_menu = NULL;
  }

  skin_preview = NULL;

  if (skin_data) { 
    g_free (skin_data);
    skin_data = NULL; 
  }

  gtk_main_quit ();
}


void preferences_dialog (int page_num) {
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *label;
  GtkWidget *notebook;
  GtkWidget *page;
  GtkWidget *button;
  GtkWidget *window;

  window = gtk_window_new (GTK_WINDOW_DIALOG);
  gtk_window_set_title (GTK_WINDOW (window), "XQF: Preferences");
  gtk_window_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER);
  gtk_signal_connect (GTK_OBJECT (window), "delete_event",
		      GTK_SIGNAL_FUNC (window_delete_event_callback), NULL);
  gtk_signal_connect (GTK_OBJECT (window), "destroy",
		      GTK_SIGNAL_FUNC (preferences_window_destroy), NULL);
  gtk_grab_add (GTK_WIDGET (window));

  register_window (window);

  gtk_widget_realize (window); 
  allocate_quake_player_colors (window->window);

  vbox = gtk_vbox_new (FALSE, 8);
  gtk_container_border_width (GTK_CONTAINER (vbox), 8);
  gtk_container_add (GTK_CONTAINER (window), vbox);
         
  /*
   *  Notebook
   */

  notebook = gtk_notebook_new ();
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
  gtk_box_pack_start (GTK_BOX (vbox), notebook, FALSE, FALSE, 0);

  page = player_profile_page ();
  label = gtk_label_new (" Player Profile ");
  gtk_widget_show (label);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, label);

  page = gameplay_options_page ();
  label = gtk_label_new (" Gameplay ");
  gtk_widget_show (label);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, label);

  page = filter_options_page ();
  label = gtk_label_new (" Filtering ");
  gtk_widget_show (label);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, label);

  page = configuration_page ();
  label = gtk_label_new (" Options ");
  gtk_widget_show (label);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, label);

  gtk_notebook_set_page (GTK_NOTEBOOK (notebook), page_num);

  gtk_widget_show (notebook);

  /* 
   *  Buttons at the bottom
   */

  hbox = gtk_hbox_new (FALSE, 8);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

  button = gtk_button_new_with_label ("Ok");
  gtk_widget_set_usize (button, 96, -1);
  gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 16);
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                    GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (window));
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  gtk_widget_show (hbox);

  gtk_widget_show (vbox);

  gtk_widget_show (window);

  update_skins ();
  filter_prefs_changed = FALSE;

  gtk_main ();

  unregister_window (window);
}


void user_fix_defaults (void) {
  if (!default_qw_cmd)
    default_qw_cmd = g_strdup ("qwcl.x11");
  if (!default_q2_cmd)
    default_q2_cmd = g_strdup ("quake2");
  if (!default_name)
    default_name = g_strdup (user.name);

  if (default_top_color < 0 || default_top_color > 13)
    default_top_color = 0;

  if (default_bottom_color < 0 || default_top_color > 13)
    default_bottom_color = 0;
}


