/* XQF - Quake server browser and launcher
 * Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>
#include <ctype.h>	/* is...() */
#include <string.h>	/* strcmp */
#include <sys/stat.h>	/* stat. mkdir */
#include <unistd.h>	/* stat, mkdir */
#include <sys/types.h>	/* mkdir */
#include <fcntl.h>	/* mkdir */

#include "xqf.h"
#include "pref.h"
#include "utils.h"
#include "rc.h"


static FILE *rc;
static int token = -1;
static char *token_str = NULL;
static int token_int = 0;
static char *tptr = NULL;
static int state = STATE_SPACE;

static char *rcfilename = NULL;
static int line;
static int pos;


static struct keyword  keywords[] = {
  { "name",	KEYWORD_STRING,	&default_name },
  { "team",	KEYWORD_STRING,	&default_team },
  { "skin",	KEYWORD_STRING,	&default_skin },
  { "top",	KEYWORD_INT,	&default_top_color },
  { "bottom",	KEYWORD_INT,	&default_bottom_color },

  { "retries", 	KEYWORD_INT, 	&filter_retries }, 
  { "ping",  	KEYWORD_INT,	&filter_ping },
  { "notfull",  KEYWORD_INT,	&filter_not_full },
  { "notempty", KEYWORD_INT,	&filter_not_empty },
  { "nocheats", KEYWORD_INT,	&filter_no_cheats },
  { "nopasswd", KEYWORD_INT,	&filter_no_password },

  { "noskins",	KEYWORD_INT,	&default_noskins },
  { "rate", 	KEYWORD_INT,	&default_rate },
  { "cl_nodelta", KEYWORD_INT,	&default_cl_nodelta },
  { "cl_predict", KEYWORD_INT,	&default_cl_predict },
  { "noaim",	KEYWORD_INT,    &default_noaim },
  { "w_switch",	KEYWORD_INT,    &default_w_switch },
  { "b_switch",	KEYWORD_INT,    &default_b_switch },

  { "qw_dir",   KEYWORD_STRING, &default_quake_dir },
  { "qw_cmd",	KEYWORD_STRING, &default_qw_cmd },
  { "q2_dir",   KEYWORD_STRING, &default_q2_dir },
  { "q2_cmd",	KEYWORD_STRING, &default_q2_cmd },
  { "grabmouse", KEYWORD_INT,   &default_windowed_mouse },
  { "terminate", KEYWORD_INT,   &default_terminate },
  { "savelists", KEYWORD_INT,    &default_save_lists },
  { "maxsimultaneous", KEYWORD_INT,   &maxsimultaneous },
  { "maxretries", KEYWORD_INT,   &maxretries },

  /* compatibility with old versions */

  { "dir",      KEYWORD_STRING, &default_quake_dir },
  { "cmd",	KEYWORD_STRING, &default_qw_cmd },
  { "cl_predict_players", KEYWORD_INT,	&default_cl_predict },

  { NULL, 	0, 		NULL }
};


static void unexpected_char_error (char c) {
  fprintf (stderr, "Unexpected character: ");
  fprintf (stderr, (isprint (c))? "\'%c\'" : "\\%03o", c);
  fprintf (stderr, " in file %s[line:%d,pos:%d]\n", rcfilename, line, pos);
}


static void unexpected_eof_error (void) {
  fprintf (stderr, "Unexpected end of line in file %s[line:%d,pos:%d]\n", 
           rcfilename, line, pos);
}


static void syntax_error (void) {
  fprintf (stderr, "Syntax error in file %s[line:%d,pos:%d]\n", 
           rcfilename, line, pos);
}


static void unexpected_token_error (int need_token) {
  fprintf (stderr, "Syntax error in file %s[line:%d,pos:%d]\n",
                                                   rcfilename, line, pos);
  fprintf (stderr, "Skipping to the end of the line...\n");
}


static inline int rc_getc (FILE *rc) {
  pos++;
  return getc (rc);
}


static inline int rc_ungetc (char c, FILE *rc) {
  pos--;
  return ungetc (c, rc);
}


static int rc_open (char *filename) {
  rc = fopen (filename, "r");
  if (rc == NULL)
    return -1;

  token_str = g_malloc (BUFFER_SIZE);
  tptr = token_str;
  state = STATE_SPACE;
  token_int = 0;
  line = pos = 1;
  rcfilename = g_strdup (filename);
  return 0;
}


static void rc_close (void) {
  if (rc) {
    fclose (rc);
    rc = NULL;
  }

  if (token_str) {
    g_free (token_str);
    token_str = NULL;
  }

  if (rcfilename) {
    g_free (rcfilename);
    rcfilename = NULL;
  }
}


int rc_next_token (void) {
  int res;
  int c;
  int num, i;
  int sign;

  if (!rc)
    return TOKEN_EOF;

  if ((c = rc_getc (rc)) == EOF) {
    rc_close ();
    return TOKEN_EOF;
  }

  tptr = token_str;

  while (1) {

    switch (state) {

    case STATE_SPACE:
      if (c == ' ' || c == '\r' || c == '\t')
	break;

      if (isdigit (c)) {
	sign = 1;
	token_int = c - '0';
	state = STATE_INT;
	break;
      }

      if (isalnum (c) || c == '_') {
	*tptr++ = c;
	state = STATE_TOKEN;
	break;
      }

      switch (c) {

      case '\n':
	line++;
	pos = 1;
	return TOKEN_EOL;
	break;

      case ',':
	return TOKEN_COMMA;
	break;

      case '\"':
	state = STATE_STRING;
	break;

      case '-':
	sign = -1;
	token_int = 0;
	state = STATE_INT;
	break;
	
      case '#':
	state = STATE_COMMENT;
	break;

      default:
	unexpected_char_error (c);
	break;

      }	/* switch */
      
      break;

    case STATE_COMMENT:
      if (c == '\n') {
	line++;
	pos = 1;
	state = STATE_SPACE;
	return TOKEN_EOL;
      }
      break;

    case STATE_INT:
      if (!isdigit (c)) {
	rc_ungetc (c, rc);
	token_int = token_int * sign;
	state = STATE_SPACE;
	return TOKEN_INT;
      }

      token_int = token_int*10 + c - '0';
      break;

    case STATE_TOKEN:
      if (!isalnum (c) && c != '_') {
	rc_ungetc (c, rc);
	*tptr = '\0';
	state = STATE_SPACE;
	return TOKEN_KEYWORD;
      }

      *tptr++ = c;
      break;
      
    case STATE_STRING:
      if (c == '\"') {
	*tptr = '\0';
	state = STATE_SPACE;
	return TOKEN_STRING;
      }

      if (c == '\n') {
	unexpected_eof_error ();
	rc_ungetc (c, rc);
	*tptr = '\0';
	state = STATE_SPACE;
	return TOKEN_STRING;
      }

      if (c == '\\') {
	if ((c = rc_getc (rc)) == EOF) {
	  *tptr = '\0';
	  rc_close ();
	  return TOKEN_STRING;
	}

	if (c >= '0' && c <='7') {
	  for (i=0, num=0; i < 3; i++) {
	    num <<= 3;
	    num |= c - '0';
	    c = rc_getc (rc);
	    if (c < '0' || c > '7')
	      break;
	  }

	  *tptr++ = num;

	  if (c == EOF) {
	    *tptr = '\0';
	    rc_close ();
	    return TOKEN_STRING;
	  }
	  
	  continue;
	} /* if (c >= '0' && c <='7') */

	switch (c) {

	case 'n':
	  *tptr++ = '\n';
	  break;

	case 'r':
	  *tptr++ = '\r';
	  break;

	case 't':
	  *tptr++ = '\t';
	  break;

	default:
	  *tptr++ = c;
	  break;

	} /* switch (c) */
	break;
	
      } /* if (c == '\\') */

      *tptr++ = c;
      break;

    } /* switch (state) */

    c = rc_getc (rc);

  } /* while (1) */

}


static void rc_skip_to_eol (void) {
  while (token != TOKEN_EOL && token != TOKEN_EOF) {
    token = rc_next_token ();
  }
}


static int rc_expect_token (int need_token) {
  if ((token = rc_next_token ()) == need_token)
    return TRUE;

  unexpected_token_error (need_token);
  rc_skip_to_eol ();
  return FALSE;
}


int rc_parse (void) {
  char *fn;
  struct keyword *kw;
  char **str_val;
  int res;

  fn = file_in_dir (user.rcdir, RC_FILE);
  rc_open (fn);
  g_free (fn);

  if(!rc)
    return -1;

  while ((token = rc_next_token ()) != TOKEN_EOF) {

    switch (token) {

    case TOKEN_KEYWORD:
      for (kw = keywords; kw->name; kw++) {
	if (strcmp (token_str, kw->name) == 0) {

	  switch (kw->required) {
	  case KEYWORD_INT:
	    if (rc_expect_token (TOKEN_INT))
	      *((int*) kw->target) = token_int;
	    break;

	  case KEYWORD_STRING:
	    if (rc_expect_token (TOKEN_STRING)) {
	      str_val = (char**) kw->target;
	      if (*str_val)
		g_free (*str_val);
	      *str_val = strdup_strip (token_str);
	    }
	    break;
	  }

	  break;
	}
      }
      break;

    case TOKEN_EOL:
    case TOKEN_EOF:
      break;

    default:
      syntax_error ();
      rc_skip_to_eol ();
      break;

    } /* switch */
  } /* while */

  rc_close ();

  /* compatibility with old versions */

  if (default_w_switch < 0) default_w_switch = 0;
  if (default_b_switch < 0) default_b_switch = 0;

  return 0;
}


int rc_save (void) {
  char *fn;
  FILE *f;

  fn = file_in_dir (user.rcdir, RC_FILE);
  f = fopen (fn, "w");
  g_free (fn);

  if (!f)
    return -1;

  fprintf (f, "name\t\t\"%s\"\n", (default_name)? default_name : "");
  fprintf (f, "team\t\t\"%s\"\n", (default_team)? default_team : "");
  fprintf (f, "skin\t\t\"%s\"\n", (default_skin)? default_skin : "");
  fprintf (f, "top\t\t%d\n", default_top_color);
  fprintf (f, "bottom\t\t%d\n", default_bottom_color);

  fprintf (f, "retries\t\t%d\n", filter_retries);
  fprintf (f, "ping\t\t%d\n", filter_ping);
  fprintf (f, "notfull\t\t%d\n", filter_not_full);
  fprintf (f, "notempty\t%d\n", filter_not_empty);
  fprintf (f, "nocheats\t%d\n", filter_no_cheats);
  fprintf (f, "nopasswd\t%d\n", filter_no_password);

  fprintf (f, "noskins\t\t%d\n", default_noskins);
  fprintf (f, "rate\t\t%d\n", default_rate);
  fprintf (f, "cl_nodelta\t%d\n", default_cl_nodelta);
  fprintf (f, "cl_predict\t%d\n", default_cl_predict);
  fprintf (f, "noaim\t\t%d\n", default_noaim);
  fprintf (f, "w_switch\t%d\n", default_w_switch);
  fprintf (f, "b_switch\t%d\n", default_b_switch);

  fprintf (f, "qw_dir\t\t\"%s\"\n", (default_quake_dir)? default_quake_dir : "");
  fprintf (f, "qw_cmd\t\t\"%s\"\n", (default_qw_cmd)? default_qw_cmd : "");
  fprintf (f, "q2_dir\t\t\"%s\"\n", (default_q2_dir)? default_q2_dir : "");
  fprintf (f, "q2_cmd\t\t\"%s\"\n", (default_q2_cmd)? default_q2_cmd : "");
  fprintf (f, "grabmouse\t%d\n", default_windowed_mouse);
  fprintf (f, "terminate\t%d\n", default_terminate);
  fprintf (f, "savelists\t%d\n", default_save_lists);
  fprintf (f, "maxsimultaneous\t%d\n", maxsimultaneous);
  fprintf (f, "maxretries\t%d\n", maxretries);

  fclose (f);
  return 0;
}


int rc_check_dir (void) {
  struct stat st_buf;

  if (stat (user.rcdir, &st_buf) == -1) {
    return mkdir (user.rcdir, 0755);
  }
  else {
    if (!S_ISDIR(st_buf.st_mode)) {
      fprintf (stderr, "%s is not a directory\n", user.rcdir);
      return -1;
    }
  }

  return 0;
}
