# ScrollWin.w -- ScrolledWindow widget
# Bert Bos <bert@let.rug.nl>
# Version 1.1 (see README for history)
# 
# $Id: ScrollWin.w,v 1.1 1996-09-25 09:24:06+02 mho Exp $

@class XfwfScrolledWindow (XfwfBoard)  @file = ScrollWin

@ The ScrolledWindow widget is a composite widget composed of two
Scrollbars and a Board within a Frame, and presumably a
grandgrandchild which is a child of the Board. The grandgrandchild is
called the `controlled widget' (CW). Usually, the controlled widget is
larger than the Board, and its origin will have negative x and y
coordinates. It is moved about inside the Board by the ScrolledWindow,
in reaction to the user clicking on the scrollbars. The controlled
widget is therefore clipped by the Board.

The ScrolledWindow provides a callback, but most application will not
need it, since the ScrolledWindow already moves the CW. The callback
is invoked {\em after} the CW is moved.

@exports

@ Scrolling may be done using different policies:

@ * |XfwfAutoScrolling|: scrolling is completely done by the widget.
    The controlled widget is moved inside the viewport.

@ * |XfwfUserScrolling|: the scrolling has to be done inside the
    scrollCallback, i.e. the controlled widget should be moved inside
    the viewport. The ConfigureNotify events are traced and the
    scrollbars are set appropriately.

@ * |XfwfVirtualScrolling|: the scrolling and the setting of the
    scrollbars have to be done inside the scrollCallback.

	@def XfwfAutoScrolling    = 0
	@def XfwfUserScrolling    = 1
	@def XfwfVirtualScrolling = 2

@ |XfwfSetScrolledWindow| is used to set the scrollbars of the widget.
|(x,y)| is the start of the viewport, |(w,h)| the (virtual) size of the
controlled widget and |(sx,sy)| the thumb size, if sx and sy are between
0.0 and 1.0. It is not used to move the window!

@proc XfwfSetScrolledWindow($, int x, int y, int w, int h,
			   float sx, float sy)
{
    if (! XtIsSubclass($, xfwfScrolledWindowWidgetClass))
	XtError("XfwfSetScrolledWindow called with incorrect widget type");

    if ($CW) {
	/* adjust scrollbars: x, y, wd, ht must be between 0.0 and 1.0 */
	float fw = (w == 0.0    ? 1.0 : (float) $hView / (float) w);
	float fh = (h == 0.0    ? 1.0 : (float) $vView / (float) h);
	float fx = (w == $hView ? 0.0 : (float) x      / (float) (w - $hView));
	float fy = (h == $vView ? 0.0 : (float) y      / (float) (h - $vView));
	fw = max(0.0, fw); fw = min(fw, 1.0);
	fh = max(0.0, fh); fh = min(fh, 1.0);
	fx = max(0.0, fx); fx = min(fx, 1.0);
	fy = max(0.0, fy); fy = min(fy, 1.0);
	/* controlled widget (virtual) size */
	$hMaximum = w;
	$vMaximum = h;
	$hValue   = x;
	$vValue   = y;
	/* set scrollbars */
	XfwfSetScrollbar($hscroll, fx, (0.0 <= sx && sx <= 1.0) ? sx : fw);
	XfwfSetScrollbar($vscroll, fy, (0.0 <= sy && sy <= 1.0) ? sy : fh);
    } else {
	XfwfSetScrollbar($hscroll, 0.0, 1.0);
	XfwfSetScrollbar($vscroll, 0.0, 1.0);
    }
}

@public

@ The window itself will not need a highlight border.

	@var highlightThickness = 0

@ Decide, if the the scrolled window should be included in the
keyboard traversal.

	@var Boolean traverseToChild = TRUE

@ Which keys should be allowed for keyboard traversal to the child.

	@var int traversalKeys = XfwfTraverseKeyAll

@ The width of the scrollbars can be set with the |scrollbarWidth|
resource. The default is 22 pixels.

	@var Dimension scrollbarWidth = 14

@ The scrollbars and the frame have the same shadow frame. (The board
has no border or frame.) The width of both is set with a single
resource: |shadowWidth|. The style of the shadow cannot be changed. It
will always be |XfwfSunken|.

	@var Dimension shadowWidth = 2

@ It is possible to switch off the display of the scrollbars, with the
following two resources. The scrollbars are visible by default.

	@var Boolean hideVScrollbar = False
	@var Boolean hideHScrollbar = False

@ |scrollingPolicy| specifies whether the scrolling should be done by the
widget itself or inside |scrollCallback|.

	@var int scrollingPolicy = XfwfAutoScrolling

@ |vIncrement| is the amount by which the scrollbar is moved when the
vertical scrollbar arrows are pressed and |vPageIncrement| is the same
for the area above and below the thumb.

	@var int vIncrement     = 1
	@var int vPageIncrement = 0

@ The same for horizontal scrolling.

	@var int hIncrement     = 1
	@var int hPageIncrement = 0

@ It should be possible to set a child window to a initial position. Usually
the inserted child will be moved to |(0,0)|. If you want another initial
position use |(XtNhInit, XtNvInit)|.

	@var int hInit = 0
	@var int vInit = 0

@ The callback is passed a pointer to an |XfwfScrollInfo| structure,
which is defined as |struct { XfwfSReason reason; XfwfSFlags flags;
float vpos, vsize, hpos, hsize; }| The |reason| can be:

\item{-} |XfwfSUp|: if the user clicked (or holds) the up arrow.
\item{-} |XfwfSLeft|: ditto left arrow.
\item{-} |XfwfSDown|: ditto down arrow.
\item{-} |XfwfSRight|: ditto right arrow.
\item{-} |XfwfSPageUp|: if the user clicked (or holds) area above thumb.
\item{-} |XfwfSPageDown|: ditto area below thumb.
\item{-} |XfwfSPageLeft|: ditto area left of thumb.
\item{-} |XfwfSPageRight|: ditto area right of thumb.
\item{-} |XfwfSTop|: if the user requests to scroll all the way up.
\item{-} |XfwfSBottom|: same all the way down.
\item{-} |XfwfSLeftSide|: same to the left side.
\item{-} |XfwfSRightSide|: same to the right side.
\item{-} |XfwfSDrag|: if the user drags the thumb.
\item{-} |XfwfSMove|: if the user stops dragging the thumb.

@ The callback is called when an action on the scrollbar was done.

	@var <Callback> XtCallbackList scrollCallback = NULL

@ For virtual scrolling it is nice to trace the ConfigureNotify events
delivered to |$board| to resize the controlled widget appropriately.
The |call_data| is a XRectangle with the values generated by
|compute_inside($board)|.

	@var <Callback> XtCallbackList resizeCallback = NULL

@ To set the scrolled window to a particular position, the
scrollResponse function must be used. The function has the type
of a callback function and it must be retrieved with a call to
|XtGetValues|.

	@var XtCallbackProc scrollResponse = scroll_response

@private

@ The ScrolledWindow widget has exactly three children, one grandchild
and one grandgrandchild. For convenience they are stored in private
variables as well as in the |children| field.

	@var Widget vscroll
	@var Widget hscroll
	@var Widget frame
	@var Widget board
	@var Widget CW

@ |hView, vView| is the inner size of the board widget for
easier access. It is not needed before a controlled widget is added.
The sizes are computed automatically by the event handler |configure|.

	@var int hView
	@var int vView

@ |vMaximum, hMaximum| is the (virtual) vertical size of the controlled
widget. |vValue, hValue| is the current (virtual) vertical position of
the controlled widget.

	@var int vMaximum
	@var int vValue
	@var int hMaximum
	@var int hValue

@methods

@ The |initialize| method creates the three child widgets and the
grandchild and sets the CW to NULL. The resources of the children are
set to fixed values and some of the ScrolledWindow's resources are
copied. The board widget will automatically be set to the correct size
by the frame widget.

@proc initialize
{
    /* initialize private variables */
    $CW = NULL; $frame = $board = $vscroll = $hscroll = NULL;
    $vMaximum = $vValue = $hMaximum = $hValue = $hView = $vView = 0;

    $frame = XtVaCreateManagedWidget
	("_frame", xfwfEnforcerWidgetClass, $,
	 XtNframeType, XfwfSunken, XtNframeWidth, $shadowWidth,
	 XtNborderWidth, 0, XtNhighlightThickness, 2,
	 XtNbackground, $background_pixel,
	 XtNtraversalOn, $traverseToChild,
	 XtNtraversalKeys, $traversalKeys,
	 NULL);
    $board = XtVaCreateManagedWidget
	("_board", xfwfBoardWidgetClass, $frame, XtNframeWidth, 0,
	 XtNborderWidth, 0, XtNhighlightThickness, 0,
	 XtNbackground, $background_pixel,
	 XtNtraversalOn, False, NULL);
    $vscroll = XtVaCreateWidget
	("_vscroll", xfwfScrollbarWidgetClass, $, XtNvertical, TRUE,
	 XtNframeWidth, $shadowWidth, XtNframeType, XfwfSunken, XtNborderWidth, 0,
	 XtNbackground, $background_pixel, XtNthumbColor, $background_pixel,
	 XtNtraversalOn, False, XtNhighlightThickness, 0, NULL);
    $hscroll = XtVaCreateWidget
	("_hscroll", xfwfScrollbarWidgetClass, $, XtNvertical, FALSE,
	 XtNframeWidth, $shadowWidth, XtNframeType, XfwfSunken, XtNborderWidth, 0,
	 XtNbackground, $background_pixel, XtNthumbColor, $background_pixel,
	 XtNtraversalOn, False, XtNhighlightThickness, 0, NULL);
    if (! $hideVScrollbar) XtManageChild($vscroll);
    if (! $hideHScrollbar) XtManageChild($hscroll);
    compute_sizes($);
    XtAddCallback($vscroll, XtNscrollCallback, scroll_callback, $);
    XtAddCallback($hscroll, XtNscrollCallback, scroll_callback, $);
    if ($scrollResponse != scroll_response) {
	$scrollResponse = scroll_response;
	XtWarning("scrollResponse resource may only be queried, not set");
    }
}

@ Changes in the resources cause the scrolled window to recompute the
layout of its three children.

The |scrollResponse| resource may not be changed.

@proc set_values
{
    if ($traversalKeys != $old$traversalKeys) {
	XtVaSetValues($frame, XtNtraversalKeys, $traversalKeys, NULL);
    }
    if ($background_pixel != $old$background_pixel) {
	XtVaSetValues($frame, XtNbackground, $background_pixel, NULL);
	XtVaSetValues($board, XtNbackground, $background_pixel, NULL);
	XtVaSetValues($hscroll, XtNthumbColor, $background_pixel,
				XtNbackground, $background_pixel, NULL);
	XtVaSetValues($vscroll, XtNthumbColor, $background_pixel,
				XtNbackground, $background_pixel, NULL);
    }
    if ($old$hideVScrollbar && ! $hideVScrollbar) {
	XtManageChild($vscroll);
    } else if (! $old$hideVScrollbar && $hideVScrollbar) {
	XtUnmanageChild($vscroll);
    }
    if ($old$hideHScrollbar && ! $hideHScrollbar) {
	XtManageChild($hscroll);
    } else if (! $old$hideHScrollbar && $hideHScrollbar) {
	XtUnmanageChild($hscroll);
    }
    if ($old$traverseToChild != $traverseToChild) {
	XtVaSetValues($frame, XtNtraversalOn, $traverseToChild, NULL);
	XtVaSetValues($frame, XtNpropagateTarget, ($traverseToChild ? $CW : NULL), NULL);
    }
    if (   $old$scrollbarWidth != $scrollbarWidth
	|| $old$shadowWidth    != $shadowWidth
	|| $old$hideHScrollbar != $hideHScrollbar
	|| $old$hideVScrollbar != $hideVScrollbar)
	compute_sizes($);
    if ($scrollingPolicy != $old$scrollingPolicy) {
	$scrollingPolicy  = $old$scrollingPolicy;
	XtWarning("scrollingPolicy resource may only set by XtVaCreateWidget");
    }
    if ($scrollResponse != $old$scrollResponse) {
	$scrollResponse  = $old$scrollResponse;
	XtWarning("scrollResponse resource may only be queried, not set");
    }
    return False;
}

@ When the scrolled window is resized, the scrollbars and the frame
should be repositioned. Also, the thumbs in the scrollbars will have
to change size, but for that purpose there is an event handler that
will intercept changes to the configuration of the CW or the board.
The event handler is installed when the CW is inserted (in the
|insert_child| method) and removed again when the child is destroyed
(in the |CW_killed| callback routine).

@proc resize
{
    compute_sizes($);
}

@ The |insert_child| method performs some tricks to ensure that the
three proper children are added normally, that the fourth is passed on
to the board and that any other children are refused.

When the CW is added as a child of the board, the scrollbars are
initialized to the current position and size of the CW relative to the
board.

@proc insert_child
{
    Position boardx, boardy, gx, gy;
    Dimension boardwd, boardht, gwd, ght;
    Boolean dummy;

    if ( !$hscroll ) { /* The horizontal scrollbar is the last regular child */
	#insert_child(child);
    } else if ($CW == NULL) {
	$CW = child;
	$child$parent = $board;
	XtAddCallback(child, XtNdestroyCallback, CW_killed, $);
	#insert_child(child);
	XtAddEventHandler(child,  StructureNotifyMask, False, configure, $);
	XtAddEventHandler($board, StructureNotifyMask, False, configure, $);
	#compute_inside($board, &boardx, &boardy, &boardwd, &boardht);
	XtVaGetValues(child, XtNwidth, &gwd, XtNheight, &ght, NULL);
	gx = $hValue = $hInit;
	gy = $vValue = $vInit;
	if ( !$hInit )
	    gx = gwd <= boardwd ? 0 : max($hValue, boardwd - gwd);
	if ( !$vInit )
	    gy = ght <= boardht ? 0 : max($vValue, boardht - ght);
	XtMoveWidget(child, gx, gy);
	/* propagate key events from frame to child widget */
	if ($traverseToChild)
	    XtVaSetValues($frame, XtNpropagateTarget, $CW, NULL);
	configure($, $, NULL, &dummy);
    } else {
	char s[500];
	(void) sprintf(s, "Cannot add <%s>, %s <%s> already has child <%s>\n",
		       XtName(child), "ScrolledWindow", XtName($), XtName($CW));
	XtWarning(s);
    }
}

@ The method |compute_inside| is re-defined. The method now returns
the area of the "view port".

@proc compute_inside
{
    #compute_inside(($board ? $board : $), x, y, w, h);
}

@ When the scrolled window widget is destroyed, the event handler is
removed (if it was installed).

@proc destroy
{
    if ($CW != NULL) {
	XtRemoveEventHandler($CW, StructureNotifyMask, False,
			     configure, $);
	XtRemoveEventHandler($board, StructureNotifyMask, False,
			     configure, $);
    }
}

@ When the ScrolledWindow widget is configured from outside, this is
done through a call to the |scroll_response| method.

@proc scroll_response(Widget w, XtPointer client_data, XtPointer call_data)
{
    Widget $ = (Widget) client_data;
    XfwfScrollInfo *info = (XfwfScrollInfo *) call_data;
    XfwfScrollInfo new;
    Position boardx, boardy, gx, gy, minx, miny;
    Dimension boardwd, boardht, gwd, ght;

    #compute_inside($board, &boardx, &boardy, &boardwd, &boardht);
    XtVaGetValues($CW, XtNx, &gx, XtNy, &gy, XtNwidth, &gwd,
		  XtNheight, &ght, NULL);
    minx = gwd <= boardwd ? 0 : boardwd - gwd;
    miny = ght <= boardht ? 0 : boardht - ght;

    if (info->flags & XFWF_VPOS) gy = info->vpos * miny;
    if (info->flags & XFWF_HPOS) gx = info->hpos * minx;

    XtMoveWidget($CW, gx, gy);

    if (info->reason != XfwfSNotify && $scrollingPolicy == XfwfAutoScrolling) {
	new.reason = XfwfSNotify;
	new.flags = info->flags & (XFWF_VPOS | XFWF_HPOS);
	new.hpos = info->hpos;
	new.vpos = info->vpos;
	XtCallCallbackList($, $scrollCallback, info);
    }
}
    
@utilities

@ |compute_sizes| is used by |initialize|, |resize| and |set_values|
to configure the children.

@proc compute_sizes($)
{
    Dimension selfw, selfh;
    Position  selfx, selfy;

    #compute_inside($, &selfx, &selfy, &selfw, &selfh);

    /* 2 is a small gap between scrollbar and frame */
    XtConfigureWidget($vscroll,
		      selfx + selfw - $scrollbarWidth,
		      2,
		      $scrollbarWidth,
		      ($hideHScrollbar ? selfh : selfh - 2 - $scrollbarWidth) - 4,
		      0);
    XtConfigureWidget($hscroll,
		      2,
		      selfy + selfh - $scrollbarWidth,
		      ($hideVScrollbar ? selfw : selfw - 2 - $scrollbarWidth)  - 4,
		      $scrollbarWidth,
		      0);
    XtConfigureWidget($frame,
		      selfx,
		      selfy,
		      ($hideVScrollbar ? selfw : selfw - 2 - $scrollbarWidth),
		      ($hideHScrollbar ? selfh : selfh - 2 - $scrollbarWidth),
		      0);
    if ($CW != NULL) XtMoveWidget($CW, $hInit, $vInit);
}

@ When the user interacts with the scrollbar, this callback is called.
In response, the ScrolledWindow moves its CW and -- if it was not a
Notify message -- then calls its own callbacks.

@proc scroll_callback(Widget w, XtPointer client_data, XtPointer call_data)
{
    /* this widget */
    Widget $ = (Widget) client_data;
    /* scroll info of scrollbar and own new scroll info */
    XfwfScrollInfo new, *info = (XfwfScrollInfo*)call_data;
    /* amount for paging, maximal position, current position */
    int h_page = (0<$hPageIncrement && $hPageIncrement<$hView) ? $hPageIncrement : $hView;
    int v_page = (0<$vPageIncrement && $vPageIncrement<$vView) ? $vPageIncrement : $vView;
    int xmax   = max(0, $hMaximum - $hView);
    int ymax   = max(0, $vMaximum - $vView);
    int gx     = $hValue;
    int gy     = $vValue;
    int sb     = 0;
    /* adjust gx,gy according to the scrollbar */
    switch (info->reason) {
    case XfwfSUp:        gy = max(gy - $vIncrement, 0);    sb = XFWF_VERTICAL;   break;
    case XfwfSDown:      gy = min(gy + $vIncrement, ymax); sb = XFWF_VERTICAL;   break;
    case XfwfSLeft:      gx = max(gx - $hIncrement, 0);    sb = XFWF_HORIZONTAL; break;
    case XfwfSRight:     gx = min(gx + $hIncrement, xmax); sb = XFWF_HORIZONTAL; break;
    case XfwfSPageUp:    gy = max(gy - v_page, 0);         sb = XFWF_VERTICAL;   break;
    case XfwfSPageDown:  gy = min(gy + v_page, ymax);      sb = XFWF_VERTICAL;   break;
    case XfwfSPageLeft:  gx = max(gx - h_page, 0);         sb = XFWF_HORIZONTAL; break;
    case XfwfSPageRight: gx = min(gx + h_page, xmax);      sb = XFWF_HORIZONTAL; break;
    case XfwfSTop:       gy = 0;                           sb = XFWF_VERTICAL;   break;
    case XfwfSBottom:    gy = ymax;                        sb = XFWF_VERTICAL;   break;
    case XfwfSLeftSide:  gx = 0;                           sb = XFWF_HORIZONTAL; break;
    case XfwfSRightSide: gx = xmax;                        sb = XFWF_HORIZONTAL; break;
    case XfwfSMove:
    case XfwfSDrag:
	if (w == $vscroll) {
	    gy = info->vpos * ymax; sb = XFWF_VERTICAL;
	} else {
	    gx = info->hpos * xmax; sb = XFWF_HORIZONTAL; 
	}
	break;
    default:
	break;
    }
    /* for auto scrolling the widget is moved immediately */
    if ($scrollingPolicy == XfwfAutoScrolling) {
	XtMoveWidget($CW, -gx, -gy);
	$hValue = gx;
	$vValue = gy;
    }
    /* call callback with information */
    if (info->reason != XfwfSNotify) {
	new.reason = info->reason;
	new.flags  = XFWF_VPOS | XFWF_HPOS | XFWF_HSIZE | XFWF_VSIZE | sb;
	new.hpos   = xmax == 0 ? 0.0 : (float)gx/xmax;
	new.vpos   = ymax == 0 ? 0.0 : (float)gy/ymax;
	new.hsize  = $hMaximum == 0 ? 1.0 : max((float)$hView/$hMaximum, 1.0);
	new.vsize  = $vMaximum == 0 ? 1.0 : max((float)$vView/$vMaximum, 1.0);
	new.gx     = gx;
	new.gy     = gy;
	XtCallCallbackList($, $scrollCallback, &new);
    }
}

@ When the CW is destroyed, the private variable |CW|
must be set back to |NULL|. The event handler that was installed to
intercept changes to the board and the CW must also be
removed.  The scrollbars are set to their `resting position'.

@proc CW_killed(Widget w, XtPointer client_data, XtPointer call_data)
{
    Widget $ = (Widget) client_data;

    /* don't propagate key events from frame to child widget anymore */
    XtVaSetValues($frame, XtNpropagateTarget, NULL, NULL);
    XtRemoveEventHandler(w, StructureNotifyMask, False, configure, $);
    XtRemoveEventHandler($board, StructureNotifyMask, False, configure, $);
    /* initialize private variables and reset scrollbars */
    $CW = NULL;
    $vMaximum = $vValue = $hMaximum = $hValue = $hView = $vView = 0;
    XfwfSetScrollbar($vscroll, 0.0, 1.0);
    XfwfSetScrollbar($hscroll, 0.0, 1.0);
}

@ The |configure| routine is called when the CW or the board receive
ConfigureNotify events (and a few others, which are not useful).  It
is also called by |insert_child|, when the CW is first created.  The
changed size of these widgets results in different sizes for the
sliders in the scrollbars.

@proc configure(Widget w, XtPointer client_data, XEvent *event, Boolean *cont)
{
    /* geometries of controlled widget and of board */
    Position  gx, gy;         Dimension gwd, ght;
    Position  boardx, boardy; Dimension boardwd, boardht;
    /* scrolled window widget */
    Widget $ = (Widget)client_data;

    /* skip during destruction */
    if ($being_destroyed)
	return;
    if (w == $board) {
	/* adjust viewport size */
	XRectangle bounds;
	#compute_inside($board, &boardx, &boardy, &boardwd, &boardht);
	$hView = boardwd;
	$vView = boardht;
	/* call notify callback */
	bounds.x = boardx; bounds.width  = boardwd;
	bounds.y = boardy; bounds.height = boardht;
	XtCallCallbackList($, $resizeCallback, (XtPointer)(&bounds));
    }
    /* adjust scrollbars for changed geometry */
    if ($scrollingPolicy != XfwfVirtualScrolling) {
	/* get board and controlled widget sizes */
	XtVaGetValues($CW, XtNx,     &gx,  XtNy,      &gy,
		           XtNwidth, &gwd, XtNheight, &ght, NULL);
	/* store new size and position of controlled widget */
	XfwfSetScrolledWindow($, -gx, -gy, gwd, ght, -1, -1);
    }
}

@imports

@ <stdio.h> is needed for error and debugging purposes, <scroll.h> for
the scroll notify structures. The widgets are parts of this composite widget.
<Enforcer.h> is included as an export to make the traversal key defines
available.

@incl <stdio.h>
@incl <Board.h>
@incl <Scrollbar.h>
@incl <scroll.h>

@exports

@incl <Enforcer.h>
