;;; mew-xface-mule.el -- show X-Face in Mew message buffer for Emacs, MULE

;;; Copyright (C) 1997 KORIYAMA Naohiro
;;; Author: KORIYAMA Naohiro <kory@ba2.so-net.or.jp>
;;; Version: 0.05
;;; Created: 1997/10/24
;;; Revised: 1997/11/02
;;; Keywords: mew, X-Face, bitmap, Emacs, MULE

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Emacs; see the file COPYING.  If not, write to the
;;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;;; Boston, MA 02111-1307, USA.

;;; x-face-mule version 0.05 is for Mew version 1.92 or higher

;;; [USAGE]
;;; 1. "bitmap-mule" package and related packages or
;;;    "bitmap.tar.gz" (etl version bitmap.el) are required.
;;;    also "compface.tar.gz" (uncompface) is required.
;;; 2. add the following in your .emacs
;;;    (if window-system
;;;      (progn
;;;        ;; From: (default)
;;;        (setq mew-opt-highlight-x-face-type 'from)
;;;        ;; X-Face:
;;;        ;(setq mew-opt-highlight-x-face-type 'xface)
;;;        (setq mew-opt-highlight-x-face t)
;;;        (require 'mew-xface-mule)))
;;; 3. that's all!
;;;
;;; [CUSTOMIZATION]
;;; 1. If you don't want to show X-Face at "From:" field, 
;;;    add the following in your .emacs
;;;    (setq mew-opt-highlight-x-face-type 'xface)
;;;    and you can show X-Face at "X-Face:" field.
;;; 2. If you want to change hilight color for X-Face,
;;;    add like the following in your .emacs
;;;    (setq mew-highlight-x-face-color "Red") ;;; foreground color
;;;    (setq mew-highlight-x-face--bgcolor "Green") ;;; background color
;;;
;;; [USER FUNCTION]
;;; (mew-toggle-x-face-type) toggle show method. from->xface->nil->from...
;;;
;;; [TODO]
;;; 1. show multiple X-Face.
;;; 2. cool code!
;;; 3. and many many things
;;;
;;; [THANKS TO]
;;; OKUNISHI Fujikazu <fuji0924@mbox.kyoto-inet.or.jp>
;;; Yuuichi Teranishi <teranisi@isl.ntt.co.jp>
;;; TSUMURA Tomoaki <tsumura@kuis.kyoto-u.ac.jp>
;;; 

;;; Code:
(require 'bitmap)

(defvar mew-prog-uncompface "uncompface")

(defvar mew-opt-highlight-x-face-type 'from
  "*Where to show X-Face.
'from at From: field, 'xface at X-Face: field.")

(setq mew-opt-highlight-x-face-function 
	  (function
	   (lambda (beg end)
		 (interactive)
		 (if (and window-system mew-opt-highlight-x-face)
		   (cond
		    ((memq mew-opt-highlight-x-face-type '(from xface))
		     (mew-x-face-decode-message-header beg end)))))))


;;
;; define face for X-Face
;;
(defvar mew-highlight-x-face nil)
(defvar mew-highlight-x-face-color      "Black")
(defvar mew-highlight-x-face-bgcolor    "White")
(defun mew-x-face-make-face ()
  (set 'mew-highlight-x-face 'mew-highlight-x-face)
  (copy-face 'default mew-highlight-x-face)
  (set-face-foreground mew-highlight-x-face mew-highlight-x-face-color)
  (set-face-background mew-highlight-x-face mew-highlight-x-face-bgcolor))
(if mew-highlight-x-face ;; make face for x-face if not defined.
    ()
  (mew-x-face-make-face))

;; functions

;;; toggle x-face-type
(defun mew-toggle-x-face-type ()
  (interactive)
  (cond ((equal mew-opt-highlight-x-face-type 'from)
		 (message "Show X-Face at X-Face:")
		 (setq mew-opt-highlight-x-face-type 'xface))
		((equal mew-opt-highlight-x-face-type 'xface)
		 (message "Don't Show X-Face")
		 (setq mew-opt-highlight-x-face-type nil))
		(t
		 (message "Show X-Face at From:")
		 (setq mew-opt-highlight-x-face-type 'from)))
  (mew-summary-display))

;;
;; originate from 'x-face-decode-message-header () in x-face-mule.el
;;
(defun mew-x-face-convert-xface-to-icon (string)
  "decode xface string to UN|X ICON."
  (save-excursion
    (let ((tmp-buffer (get-buffer-create "*xface-tmp*"))
	  ret-string)
      (set-buffer tmp-buffer)
      (insert string)
      (call-process-region (point-min) (point-max)
			   mew-prog-uncompface t t nil)
      (setq ret-string (buffer-substring (point-min) (point-max)))
      (kill-buffer tmp-buffer)
      ret-string
      )))

(defun mew-x-face-convert-vector-to-rectangle (vector)
  "make xface rectangle from vector."
    (let ((ret nil)
	  line i k k+6)
      (setq k 0)
      (setq i 0)
      (while (< i 3)
	(setq line "")
	(setq k (* i 6)  k+6 (+ k 6))
	(while (< k k+6)
	  (setq line (concat line (bitmap-compose (aref vector k))))
	  (setq k (1+ k))
	  )
	(setq ret (append ret (list line)))
	(setq i (1+ i)))
      ret
      ))

(defun mew-x-face-convert-icon-to-rectangle (icon)
  "decode UN|X ICON to rectangle."
  (save-excursion 
    (let ((tmp-buffer (get-buffer-create "*xface-tmp*"))
	  i temp cmp k k+6)
      (set-buffer tmp-buffer)
      (insert icon)
      (goto-char (point-min))
      (search-forward "0x" nil t)
      (setq cmp (make-vector 18 nil))
      (setq i 0)
      (while (< i 48)
	(setq k (* (/ i 16) 6))
	(setq k+6 (+ k 6))
	(while (< k k+6)
	  (setq temp (buffer-substring (point)
				       (+ (point) 2)))
	  (aset cmp k (concat (aref cmp k) temp))
	  (setq k (1+ k))
	  (setq temp (buffer-substring (+ (point) 2)
				       (+ (point) 4)))
	  (aset cmp k (concat (aref cmp k) temp))
	  (setq k (1+ k))
	  (search-forward "0x" nil t))
	(setq i (1+ i))
	)
      (kill-buffer tmp-buffer)
      (mew-x-face-convert-vector-to-rectangle cmp))))

(defun mew-x-face-insert-at-point (rectangle)
  "insert x-face rectangle and overlay its face."
  (let ((lines rectangle)
	(insertcolumn (current-column))
	(first t)
	beg-point)
    (while lines
      (or first
	  (progn
	   (forward-line 1)
	   (or (bolp) (insert ?\n))
	   (if (fboundp 'move-to-column-strictly)
	       (move-to-column-strictly insertcolumn t) ;; XEmacs only ??
	       (move-to-column insertcolumn t)) ))      ;; text Emacs
      (setq first nil)
      (setq beg-point (point))
      (insert (car lines))
      (overlay-put (make-overlay beg-point (point)) 'face mew-highlight-x-face)
      (setq lines (cdr lines)))))

(defun mew-x-face-allocate-lines (beg end height)
  "allocate new lines according to mew-opt-highlight-x-face-type.
returns the begin-point of the xface rectangle."
  (let (begin-point n)
    (cond
     ((eq mew-opt-highlight-x-face-type 'from)
      (goto-char beg)
      (if (re-search-forward  "^From:" end t)
	  (progn
	    (beginning-of-line)
	    (insert "     ")
	    (setq begin-point (point))
		(insert "\n     \n")
		(setq n 0)
	    (while (< n (- height 1)) 
		  (insert "     \n     \n     \n")
		  (setq n (1+ n)))
	    )))
     ((eq mew-opt-highlight-x-face-type 'xface)
      (insert "X-Face: ")
      (setq begin-point (point))
	  (setq n 0)
	  (while (< n height)
		(insert "        \n        \n        \n")
		(setq n (1+ n)))
      ))
    begin-point ;; return value.
    ))
        
(defun mew-x-face-decode-message-header (beg end)
  (let ((buffer-read-only nil) xface faces faces-s first)
	(setq geometry (mew-analyze-x-face-geometry beg end))
	(setq xp (string-match "x" geometry))
	(setq M (string-to-number (substring geometry 0 xp)))
	(setq N (string-to-number (substring geometry (1+ xp))))
	(goto-char beg)
	(setq n 0)
	(while (< n N)
	  (setq m 0)
	  (while (< m M)
		(re-search-forward
		 "^X-Face: *\\(.*\\(\n[ \t].*\\)*\\)\n" end t)
		(setq faces-s (cons 
		   (buffer-substring (match-beginning 1) (match-end 1))
		   faces-s
		   ))
		(delete-region (match-beginning 0) (match-end 0))
		(setq m (1+ m)))
	  (setq faces (append faces faces-s))
	  (setq faces-s nil)
	  (setq n (1+ n)))
	  ;;
	  (setq first t)
	  (let (begin-point m n)
		(setq n 0)
		(while (< n N)
		  (setq m 0)
		  (while (< m M)
			(setq xface 
			  (mew-x-face-convert-icon-to-rectangle
			   (mew-x-face-convert-xface-to-icon (car faces))))
			(if first ;; allocate lines for face field
				(setq begin-point (mew-x-face-allocate-lines beg end N)))
			(goto-char begin-point)
			(mew-x-face-insert-at-point xface)
			(setq faces (cdr faces))
			(setq first nil)
			(setq m (1+ m)))
		  (setq n (1+ n))
		  (if (< n N)
			  (progn 
				(end-of-line 2) 
				(setq begin-point (point))))
		  ))))

(defun mew-analyze-x-face-geometry (beg end)
  (goto-char beg)
  (if (re-search-forward 
	   "^X-Face-Type: geometry=[0-9]+x[0-9]+" end t)
	  (let ((sp (+ 22 (match-beginning 0))) (ep (match-end 0)))
		(buffer-substring sp ep))
	(let ((i 0))
	  (while (re-search-forward
			  "^X-Face: *\\(.*\\(\n[ \t].*\\)*\\)\n" end t)
		(setq i (1+ i)))
	  (concat i "x1"))))

(provide 'mew-xface-mule)

;;; mew-xface-mule.el ends here
