cor.test <- function(x, y, alternative = "two.sided", method = "pearson") 
{
  CHOICES <- c("two.sided", "less", "greater")
  alternative <- CHOICES[pmatch(alternative, CHOICES)]
  if (length(alternative) > 1 || is.na(alternative)) 
    stop("alternative must be \"two.sided\", \"less\" or \"greater\"")

  CHOICES <- c("pearson", "kendall", "spearman")
  method <- CHOICES[pmatch(method, CHOICES)]
  if (length(method) > 1 || is.na(method)) 
    stop("method must be \"pearson\", \"kendall\" or \"spearman\"")

  DNAME <- paste(deparse(substitute(x)), "and", deparse(substitute(y)))

  if (length (x) != length (y))
    stop ("x and y must have the same length")
  OK <- complete.cases(x, y)
  x <- x[OK]
  n <- length(x)
  if (n < 3)
    stop("not enough finite observations")
  else
    y <- y[OK]

  NVAL <- 0

  if (method == "pearson") {
    method <- "Pearson's product-moment correlation"
    names(NVAL) <- "correlation"
    r <- cor(x, y)
    ESTIMATE <- r
    names(ESTIMATE) <- "cor"
    PARAMETER <- n - 2
    names(PARAMETER) <- "df"
    STATISTIC <- sqrt(PARAMETER) * r / sqrt(1 - r^2)
    names(STATISTIC) <- "t"
    p <- pt(STATISTIC, PARAMETER)
  }
  else {
    if (method == "kendall") {
      method <- "Kendall's rank correlation tau"
      names(NVAL) <- "tau"
      x <- rank(x)
      y <- rank(y)
      ESTIMATE <- cor(c(sign(outer(x, x, "-"))),
		      c(sign(outer(y, y, "-"))))
      names(ESTIMATE) <- "tau"
      STATISTIC <- ESTIMATE / sqrt((4 * n + 10) / (9 * n * (n-1)))
    }
    else {
      method <- "Spearman's rank correlation rho"
      names(NVAL) <- "rho"
      ESTIMATE <- cor(rank(x), rank(y))
      names(ESTIMATE) <- "rho"
      STATISTIC <- sqrt(n-1) * (ESTIMATE - 6 / (n^3 - n))
    }
    PARAMETER <- NULL
    names(STATISTIC) <- "z"
    p <- pnorm(STATISTIC)
  }

  PVAL <- switch(alternative,
		 "less" = p,
		 "greater" = 1 - p,
		 "two.sided" = 2 * min (p, 1 - p))

  structure(list(statistic = STATISTIC,
		 parameter = PARAMETER,
		 p.value = PVAL,
		 estimate = ESTIMATE,
		 null.value = NVAL,
		 alternative = alternative,
		 method = method,
		 data.name = DNAME),
	    class = "htest")
}
