/*
   Copyright (C) 1996, 1997  Ulric Eriksson <ulric@edu.stockholm.se>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the Licence, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>

#include "xfonts.h"	/* from common */
#include "embed.h"
#ifdef XPM
#include <X11/xpm.h>
#endif

#include "TableP.h"

#define offset(field) XtOffsetOf(TableRec, table.field)
static XtResource resources[] = {
	{
		XtNtableMaxRow,		/* name */
		XtCTableMaxRow,		/* class */
		XtRInt,			/* type */
		sizeof(int),		/* size */
		offset(max_row),	/* offset */
		XtRImmediate,		/* default_type */
		(XtPointer)1000		/* default_addr */
	}, {
		XtNtableMaxCol,
		XtCTableMaxCol,
		XtRInt,
		sizeof(int),
		offset(max_col),
		XtRImmediate,
		(XtPointer)1000
	}, {
		XtNtableProtRow,
		XtCTableProtRow,
		XtRInt,
		sizeof(int),
		offset(prot_row),
		XtRImmediate,
		(XtPointer)1,
	}, {
		XtNtableProtCol,
		XtCTableProtCol,
		XtRInt,
		sizeof(int),
		offset(prot_col),
		XtRImmediate,
		(XtPointer)1,
	}, {
		XtNtableTopRow,
		XtCTableTopRow,
		XtRInt,
		sizeof(int),
		offset(top_row),
		XtRImmediate,
		(XtPointer)1
	}, {
		XtNtableTopCol,
		XtCTableTopCol,
		XtRInt,
		sizeof(int),
		offset(top_col),
		XtRImmediate,
		(XtPointer)1
	}, {
		XtNtableSelectTopRow,
		XtCTableSelectTopRow,
		XtRInt,
		sizeof(int),
		offset(sel_top_row),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNtableSelectBottomRow,
		XtCTableSelectBottomRow,
		XtRInt,
		sizeof(int),
		offset(sel_bottom_row),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNtableSelectLeftCol,
		XtCTableSelectLeftCol,
		XtRInt,
		sizeof(int),
		offset(sel_left_col),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNtableSelectRightCol,
		XtCTableSelectRightCol,
		XtRInt,
		sizeof(int),
		offset(sel_right_col),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNtablePointRow,
		XtCTablePointRow,
		XtRInt,
		sizeof(int),
		offset(point_row),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNtablePointCol,
		XtCTablePointCol,
		XtRInt,
		sizeof(int),
		offset(point_col),
		XtRImmediate,
		(XtPointer)0
	}, {
		XtNtableDefaultWidth,
		XtCTableDefaultWidth,
		XtRInt,
		sizeof(int),
		offset(default_width),
		XtRImmediate,
		(XtPointer)80
	}, {
		XtNtableDefaultHeight,
		XtCTableDefaultHeight,
		XtRInt,
		sizeof(int),
		offset(default_height),
		XtRImmediate,
		(XtPointer)20
	}, {
		XtNtableColWidth,
		XtCTableColWidth,
		XtRPointer,
		sizeof(XtPointer),
		offset(col_width),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNtableRowHeight,
		XtCTableRowHeight,
		XtRPointer,
		sizeof(XtPointer),
		offset(row_height),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNtableType,
		XtCTableType,
		XtRPointer,
		sizeof(XtPointer),
		offset(type),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNtableText,
		XtCTableText,
		XtRPointer,
		sizeof(XtPointer),
		offset(text),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNtableFormat,
		XtCTableFormat,
		XtRPointer,
		sizeof(XtPointer),
		offset(format),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNtableGC,
		XtCTableGC,
		XtRPointer,
		sizeof(XtPointer),
		offset(gc),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNtableAdjHoriz,
		XtCTableAdjHoriz,
		XtRPointer,
		sizeof(XtPointer),
		offset(adj_horiz),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNtableAdjVert,
		XtCTableAdjVert,
		XtRPointer,
		sizeof(XtPointer),
		offset(adj_vert),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNtableBackground,
		XtCTableBackground,
		XtRPixel,
		sizeof(Pixel),
		offset(background),
		XtRPixel,
		XtDefaultBackground
	}, {
		XtNtableForeground,
		XtCTableForeground,
		XtRPixel,
		sizeof(Pixel),
		offset(foreground),
		XtRPixel,
		XtDefaultForeground
	}, {
		XtNtableGrid,
		XtCTableGrid,
		XtRPixel,
		sizeof(Pixel),
		offset(grid),
		XtRPixel,
		XtDefaultForeground
	}, {
		XtNtableCursor,
		XtCTableCursor,
		XtRPixel,
		sizeof(Pixel),
		offset(cursor),
		XtRPixel,
		XtDefaultForeground
	}, {
		XtNtableHighlight,
		XtCTableHighlight,
		XtRPixel,
		sizeof(Pixel),
		offset(highlight),
		XtRPixel,
		XtDefaultForeground
	}, {
		XtNtableData,
		XtCTableData,
		XtRPointer,
		sizeof(XtPointer),
		offset(data),
		XtRImmediate,
		(XtPointer)NULL
	}, {
		XtNtableRedisplay,
		XtCTableRedisplay,
		XtRBoolean,
		sizeof(Boolean),
		offset(redisplay),
		XtRImmediate,
		(XtPointer)False
	}, {
		XtNtableGridLines,
		XtCTableGridLines,
		XtRBoolean,
		sizeof(Boolean),
		offset(grid_lines),
		XtRImmediate,
		(XtPointer)True
	}, {
		XtNtableVisibleCursor,
		XtCTableVisibleCursor,
		XtRBoolean,
		sizeof(Boolean),
		offset(visible_cursor),
		XtRImmediate,
		(XtPointer)False
	}
};
#undef offset

/* methods */
static void Redisplay();
static void Realize();
static void Destroy();
static Boolean SetValues();

/* actions */
static void TableAction(Widget, XEvent *, String *, Cardinal *);

static XtActionsRec actions[] =
{
	{"table", TableAction},
};

/* translations */
static char translations[] =
"<Key>:		table()		\n";

TableClassRec tableClassRec = {
  { /* core fields */
    /* superclass		*/	(WidgetClass) &widgetClassRec,
    /* class_name		*/	"Table",
    /* widget_size		*/	sizeof(TableRec),
    /* class_initialize		*/	NULL,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	NULL,
    /* initialize_hook		*/	NULL,
    /* realize			*/	Realize,
    /* actions			*/	actions,
    /* num_actions		*/	XtNumber(actions),
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	NULL,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	translations,
    /* query_geometry		*/	XtInheritQueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
    /* extension		*/	NULL
  },
  { /* table fields */
    /* empty			*/	0
  }
};

WidgetClass tableWidgetClass = (WidgetClass)&tableClassRec;


/* supporting code copied from window.c */

static void TableAction(Widget w, XEvent *event, String *params, Cardinal *n)
{
	;	/* application should create its own actions */
}

static int inblock(Widget w, int r, int c)
{
	TableWidget tw = (TableWidget)w;

	return ((r >= tw->table.sel_top_row) &&
		(r <= tw->table.sel_bottom_row) &&
		(c >= tw->table.sel_left_col) &&
		(c <= tw->table.sel_right_col));
}

static int ret_type(Widget w, int row, int col)
{
	TableWidget tw = (TableWidget) w;
	if (tw->table.type)
		return (*tw->table.type)(tw->table.data, row, col);
	return TABLE_TEXT;
}

static char *ret_pvalue(Widget w, int row, int col)
{
	TableWidget tw = (TableWidget) w;
	if (tw->table.text)
		return (*tw->table.text)(tw->table.data, row, col);
	else return NULL;
}

static unsigned long ret_format(Widget w, int row, int col)
{
	TableWidget tw = (TableWidget) w;
	if (tw->table.format)
		return (*tw->table.format)(tw->table.data, row, col);
	return HELVETICA | SIZE_10;
}

static unsigned long ret_font(Widget w, int row, int col)
{
	return ret_format(w, row, col) & (SIZE_MASK | ITALIC | BOLD | FONT_MASK);
}

static unsigned long ret_color(Widget w, int row, int col)
{
	return (ret_format(w, row, col) & COLOR_MASK) >> COLOR_SHIFT;
}

static Dimension cell_width(TableWidget tw, int col)
{
	Dimension wi;

	if (tw->table.col_width) {
		wi = (*tw->table.col_width)(tw->table.data, col);
		return wi;
	}
	return tw->table.default_width;
}

static Dimension cell_height(TableWidget tw, int row)
{
	Dimension h;
	if (tw->table.row_height) {
		h = (*tw->table.row_height)(tw->table.data, row);
		return h;
	}
	return tw->table.default_height;
}


static void draw_cell(TableWidget tw, Drawable scribble,
			int row, int col, int x_base, int y_base)
{
	Widget w = (Widget) tw;
	char *p;
	int font_index, color_index;
	Font font;
	unsigned long color;
	int x_pos, y_pos;
	unsigned int text_height, text_width;
	unsigned long format;
	int cw, ch;

	if (row > tw->table.max_row || col > tw->table.max_col) return;

	if (ret_type(w, row, col) == TABLE_EMBED) {
		char *p = ret_pvalue(w, row, col);
		embed_draw(scribble, x_base, y_base, p);
		return;
	}

	if (ret_type(w, row, col) == TABLE_PIXMAP) {
		XtWarning("The Table widget does not yet display pixmaps");
		return;
	}

	font_index = ret_font(w, row, col);
	font = get_font(XtDisplay(w), font_index);
	XSetFont(XtDisplay(w), tw->table.cell_gc, font);

	color_index = ret_color(w, row, col);
	color = get_color(XtDisplay(w), color_index);
	XSetForeground(XtDisplay(w), tw->table.cell_gc, color);

	p = ret_pvalue(w, row, col);
	if (p == NULL) p = "";

#if 1	/* this seems to be necessary to keep X from crashing */
	if (strlen(p) > 1000) p[1000] = '\0';
#endif

	text_height = font_height(XtDisplay(w), font_index);
	text_width = XTextWidth(font_struct(XtDisplay(w), font_index),
				p, strlen(p));
	format = ret_format(w, row, col);
	cw = cell_width(tw, col);
	ch = cell_height(tw, row);
	switch (format & HADJ_MASK) {
	case HADJ_CENTER:
		x_pos = (cw-text_width) / 2;
		break;
	case HADJ_RIGHT:
		x_pos = cw-text_width-5;
		break;
	default:	/* HADJ_LEFT */
		x_pos = 5;
	}

	switch (format & VADJ_MASK) {
	case VADJ_BOTTOM:
		y_pos = ch-5;
		break;
	case VADJ_TOP:
		y_pos = text_height+font_descent(XtDisplay(w), font_index);
		break;
	default:	/* VADJ_CENTER */
		y_pos = (ch+text_height)/2
			-font_descent(XtDisplay(w), font_index);
	}

	XDrawString(XtDisplay(w), scribble, tw->table.cell_gc,
		x_base+x_pos, y_base+y_pos,
		p, strlen(p));
	if (tw->table.grid_lines) {
		XDrawLine(XtDisplay(w), scribble, tw->table.grid_gc,
			x_base, y_base+ch,
			x_base+cw, y_base+ch);
		XDrawLine(XtDisplay(w), scribble, tw->table.grid_gc,
			x_base+cw, y_base+ch,
			x_base+cw, y_base);
	}
	/* these lines are from the format */
	if (format & BORDER_LEFT)
		XDrawLine(XtDisplay(w), scribble, tw->table.grid_gc,
			x_base+1, y_base+1,
			x_base+1, y_base+ch-1);
	if (format & BORDER_RIGHT)
		XDrawLine(XtDisplay(w), scribble, tw->table.grid_gc,
			x_base+cw-1, y_base+1,
			x_base+cw-1, y_base+ch-1);
	if (format & BORDER_TOP)
		XDrawLine(XtDisplay(w), scribble, tw->table.grid_gc,
			x_base+1, y_base+1,
			x_base+cw-1, y_base+1);
	if (format & BORDER_BOTTOM)
		XDrawLine(XtDisplay(w), scribble, tw->table.grid_gc,
			x_base+1, y_base+ch-1,
			x_base+cw-1, y_base+ch-1);
	if (inblock(w, row, col)) {
		XFillRectangle(XtDisplay(w), scribble, tw->table.cursor_gc,
			x_base, y_base,
			cw, ch);
	}
}

static void cell_row(TableWidget tw, Drawable scribble,
			int width, int y_base, int i)
{
	int j, x_base;
	x_base = 0;

	for (j = 1; (j < tw->table.prot_col) && (x_base < width); j++) {
		draw_cell(tw, scribble, i, j, x_base, y_base);
		x_base += cell_width(tw, j);
	}
	for (j = tw->table.top_col; x_base < width; j++) {
		draw_cell(tw, scribble, i, j, x_base, y_base);
		x_base += cell_width(tw, j);
	}
}

static void draw_table(TableWidget tw, Drawable d)
{
	int y_base = 0;
	int width = tw->core.width;
	int height = tw->core.height;
	int i;

	for (i = 1; (i < tw->table.prot_row) && (y_base < height); i++) {
		cell_row(tw, d, width, y_base, i);
		y_base += cell_height(tw, i);
	}
	for (i = tw->table.top_row; y_base < height; i++) {
		cell_row(tw, d, width, y_base, i);
		y_base += cell_height(tw, i);
	}
}

/* create, draw and return a pixmap of the stage at time now.
   Caller must free */
/* Strategy:
   1. Clear the window, or create a new empty Pixmap
   2. Paint a rectangle for the block, if there is one (coords != 0)
   3. Draw the grid
   4. Draw the data
   5. Draw the cursor using XOR, if one is desired (coords != 0)
   The purpose of using XOR is to make sure the cursor is visible
   on any background. Technicolor is a potential side effect.
*/
Pixmap table_pixmap(TableWidget tw)
{
	Dimension width, height;
	Pixmap scribble;

	width = tw->core.width;
	height = tw->core.height;

	if (width > 2000 || height > 2000) return None;

	scribble = XCreatePixmap(XtDisplay(tw), XtWindow(tw),
		width, height, tw->core.depth);
	XFillRectangle(XtDisplay(tw), scribble, tw->table.clear_gc,
		0, 0, width, height);

	draw_table(tw, scribble);
	return scribble;
}


static GC get_gc(Widget w, unsigned long fg, unsigned long bg)
{
	unsigned long valuemask = 0;
	XGCValues values;
	GC gc = XCreateGC(XtDisplay(w), XtWindow(w),
				valuemask, &values);
	XSetForeground(XtDisplay(w), gc, fg);
	XSetBackground(XtDisplay(w), gc, bg);
	return gc;
}

#define superclass (&coreClassRec)
static void Realize(Widget w, XtValueMask *valueMask,
		XSetWindowAttributes *attributes)
{
	TableWidget tw = (TableWidget) w;
	unsigned long fg, bg;

	(*superclass->core_class.realize) (w, valueMask, attributes);
	fg = BlackPixelOfScreen(XtScreen(w));
	bg = tw->core.background_pixel;
	tw->table.clear_gc = get_gc(w, bg, fg);
	tw->table.cell_gc = get_gc(w, fg, bg);
	tw->table.grid_gc = get_gc(w, fg, bg);
	tw->table.cursor_gc = get_gc(w, fg^bg, 0);
	XSetFunction(XtDisplay(w), tw->table.cursor_gc, GXxor);
	XSetLineAttributes(XtDisplay(w), tw->table.cursor_gc,
			2, LineSolid, CapButt, JoinMiter);
}

static void Destroy(Widget w)
{
	TableWidget tw = (TableWidget) w;

	XFreeGC(XtDisplay(w), tw->table.clear_gc);
	XFreeGC(XtDisplay(w), tw->table.cell_gc);
	XFreeGC(XtDisplay(w), tw->table.grid_gc);
	XFreeGC(XtDisplay(w), tw->table.cursor_gc);
}

static int cell_next_row(TableWidget w, int row)
{
        if (row+1 == w->table.prot_row) return w->table.top_row;
        return row+1;
}

static int cell_next_col(TableWidget w, int col)
{
        if (col+1 == w->table.prot_col) return w->table.top_col;
        return col+1;
}

static int cell_prev_row(TableWidget w, int row)
{
        if (row == w->table.top_row) return w->table.prot_row-1;
        return row-1;
}

static int cell_prev_col(TableWidget w, int col)
{
        if (col == w->table.top_col) return w->table.prot_col-1;
        return col-1;
}


void table_cell2coords(TableWidget tw,
			int cell_row, int cell_col,
			int *cell_x, int *cell_y)
{
        int i;
        *cell_y = 0;
        for (i = 1; i < tw->table.prot_row; i++)
                *cell_y += cell_height(tw, i);

        while (cell_row < tw->table.top_row) {
                *cell_y -= cell_height(tw, cell_row);
                cell_row = cell_next_row(tw, cell_row);
        }
        while (cell_row > tw->table.top_row) {
                cell_row = cell_prev_row(tw, cell_row);
                *cell_y += cell_height(tw, cell_row);
        }
        *cell_x = 0;
        for (i = 1; i < tw->table.prot_col; i++)
                *cell_x += cell_width(tw, i);

        while (cell_col < tw->table.top_col) {
                *cell_x -= cell_width(tw, cell_col);
                cell_col = cell_next_col(tw, cell_col);
        }
        while (cell_col > tw->table.top_col) {
                cell_col = cell_prev_col(tw, cell_col);
                *cell_x += cell_width(tw, cell_col);
        }
}

void table_coords2cell(TableWidget tw,
			int *cur_row, int *cur_col,
			int cur_x, int cur_y)
{
        int prot_x = 0, prot_y = 0, i;

        for (i = 1; i < tw->table.prot_col; i++)
              cur_x -= cell_width(tw, i);
        for (i = 1; i < tw->table.prot_row; i++)
              cur_y -= cell_height(tw, i);

        *cur_row = tw->table.top_row;
        *cur_col = tw->table.top_col;
        while (cur_y < prot_y) {
                cur_y += cell_height(tw, *cur_row);
                (*cur_row) = cell_prev_row(tw, *cur_row);
        }
        while (cur_y > cell_height(tw, *cur_row)) {
                cur_y -= cell_height(tw, *cur_row);
                (*cur_row) = cell_next_row(tw, *cur_row);
        }
        while (cur_x < prot_x) {
                cur_x += cell_width(tw, *cur_col);
                (*cur_col) = cell_prev_col(tw, *cur_col);
        }
        while (cur_x > cell_width(tw, *cur_col)) {
                cur_x -= cell_width(tw, *cur_col);
                (*cur_col) = cell_next_col(tw, *cur_col);
        }

}

static Boolean move_top(TableWidget tw)
{
	Boolean pr_scr_flag = False;
	int cur_x, cur_y, prot_x = 0, prot_y = 0, i;
	unsigned int width, height;

        for (i = 1; i < tw->table.prot_col; i++)
                prot_x += cell_width(tw, i);
        for (i = 1; i < tw->table.prot_row; i++)
                prot_y += cell_height(tw, i);

        /* Figure out how big the window is */
	width = tw->core.width;
	height = tw->core.height;

        table_cell2coords(tw,
                        tw->table.point_row, tw->table.point_col,
                        &cur_x, &cur_y);
        /* this isn't efficient, but it will work */
	while (cur_y < prot_y) {
		tw->table.top_row--;
		cur_y += cell_height(tw, tw->table.top_row);
		pr_scr_flag = TRUE;
	}
	while (cur_y + cell_height(tw, tw->table.point_row) > height) {
		cur_y -= cell_height(tw, tw->table.top_row);
		tw->table.top_row++;
		pr_scr_flag = TRUE;
	}
	while (cur_x < prot_x) {
		tw->table.top_col--;
		cur_x += cell_width(tw, tw->table.top_col);
		pr_scr_flag = TRUE;
	}
	while (cur_x + cell_width(tw, tw->table.point_col) > width) {
		cur_x -= cell_width(tw, tw->table.top_col);
		tw->table.top_col++;
		pr_scr_flag = TRUE;
	}
	return pr_scr_flag;
}

static void toggle_cursor(TableWidget tw)
{
	int cur_x, cur_y;

	table_cell2coords(tw,
		tw->table.point_row, tw->table.point_col,
		&cur_x, &cur_y);
	XDrawRectangle(XtDisplay((Widget)tw), XtWindow((Widget)tw),
		tw->table.cursor_gc, cur_x+3, cur_y+3,
		cell_width(tw, tw->table.point_col)-5,
		cell_height(tw, tw->table.point_row)-5);
}

static void Redisplay(Widget w, XExposeEvent *event)
{
	Pixmap scribble;
	GC gc;
	unsigned long valuemask = 0;
	XGCValues values;
	TableWidget tw = (TableWidget) w;

	if (!XtIsRealized(w)) return;	/* but that doesn't work */
	if (!XtIsManaged(w)) return;	/* what about this */
	scribble = table_pixmap(tw);
	if (scribble == None) return;
	gc = XCreateGC(XtDisplay(w), XtWindow(w),
			valuemask, &values);
	XCopyArea(XtDisplay(w), scribble, XtWindow(w),
		gc, 0, 0, tw->core.width, tw->core.height, 0, 0);
	XFreePixmap(XtDisplay(w), scribble);
	XFreeGC(XtDisplay(w), gc);

	/* draw the cursor, perhaps? */
	if (tw->table.visible_cursor) {
		toggle_cursor(tw);
	}
}


static Boolean SetValues(Widget current, Widget request, Widget new)
{
	TableWidget curtw = (TableWidget) current;
	TableWidget newtw = (TableWidget) new;
	Boolean do_redisplay = False;

	do_redisplay |= (curtw->table.sel_top_row != newtw->table.sel_top_row
		|| curtw->table.sel_bottom_row != newtw->table.sel_bottom_row
		|| curtw->table.sel_left_col != newtw->table.sel_left_col
		|| curtw->table.sel_right_col != newtw->table.sel_right_col);

	if (newtw->table.visible_cursor)
		do_redisplay |= move_top(newtw);

	if (newtw->table.redisplay) {
		do_redisplay = True;
		newtw->table.redisplay = False;
	}
#if 1
/* A dilemma: if Xt handles the redisplaying, it also takes care of
   compressing expose events. On the other hand, it clears the window
   which makes it flicker. We'll redisplay ourselves and try to
   make it more efficient later. */
	/* can't let Xt handle this because it flickers */
	if (do_redisplay) {
		Redisplay((Widget)newtw, NULL);
		do_redisplay = False;
	} else 
#endif
	if (!do_redisplay) {
		if (curtw->table.visible_cursor) {
			toggle_cursor(curtw);
		}
		if (newtw->table.visible_cursor) {
			toggle_cursor(newtw);
		}
	}

	return do_redisplay;
}

