/*
   Egon Animator
   Copyright (C) 1997  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * forminput.c
 */

#include <stdio.h>
#include <string.h>
#include <X11/keysym.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>

#include "../siod/siod.h"
#include "../common/cmalloc.h"

extern void center(Widget, Widget);	/* from dialogs.c */

#define ABORT	0
#define	DONE	1
#define	WAITING	2

#define TEXT 1
#define OKBUTTON 2
#define CANCELBUTTON 3
#define MENU 4

typedef struct s_control {
	int type;
	Widget w;
	char *name;
	struct s_control *next, *prev;
} control;

static control *c_list;

static Widget top;
static XtAppContext app;

static Widget form_shell, form_dialog;
static Widget vert = NULL, horiz = NULL, lastw = NULL, menu = NULL;

static int status;

static void CBform_done(Widget w, XtPointer client_data, XtPointer call_data)
{
	status = DONE;
}

static void CBform_cancel(Widget w, XtPointer client_data, XtPointer call_data)
{
	status = ABORT;
}

#define ANYWHERE 0
#define LEFT 1
#define RIGHT 2

static Widget place(Widget w, Widget vert, Widget horiz, int chain)
{
	if (vert)
		XtVaSetValues(w,
			XtNfromVert, vert, NULL);
	if (horiz)
		XtVaSetValues(w,
			XtNfromHoriz, horiz, NULL);
	switch (chain) {
	case LEFT:
		XtVaSetValues(w,
			XtNtop, XawChainTop,
			XtNbottom, XawChainTop,
			XtNleft, XawChainLeft,
			XtNright, XawChainLeft, NULL);
		break;
	case RIGHT:
		XtVaSetValues(w,
			XtNtop, XawChainTop,
			XtNbottom, XawChainTop,
			XtNleft, XawChainLeft,	/* this is not a bug */
			XtNright, XawChainRight, NULL);
		break;
	default:	/* nothing to do */
		break;
	}
	return w;
}

static LISP form_begin()
{
	vert = NULL, horiz = NULL;
	c_list = NULL;
	form_shell = XtVaCreatePopupShell("form_shell",
		transientShellWidgetClass, top, NULL);
	form_dialog = XtVaCreateManagedWidget("form_dialog",
		formWidgetClass, form_shell, NULL);
	return NIL;
}

static LISP form_label(LISP text)
{
	char *label = get_c_string(text);
	Widget w = lastw = XtVaCreateManagedWidget("form_label",
		labelWidgetClass, form_dialog,
		XtNlabel, label, NULL);
	horiz = place(w, vert, horiz, LEFT);
	return NIL;
}

static control *new_control(int type, char *name)
{
	control *c = (control *)cmalloc(sizeof(control));
	if (!c_list) {
		c_list = c;
		c->next = c;
		c->prev = c;
	} else {
		c->prev = c_list->prev;
		c->next = c_list;
		c_list->prev->next = c;
		c_list->prev = c;
	}
	c->type = type;
	c->name = cstrdup(name);
	return c;
}

static LISP form_text(LISP n)
{
	char *name = get_c_string(n);
	control *c = new_control(TEXT, name);
	c->w = lastw = XtVaCreateManagedWidget(name,
		asciiTextWidgetClass, form_dialog,
		XtNeditType, XawtextEdit,
		XtNdisplayCaret, False, NULL);

	XtOverrideTranslations(lastw,
		XtParseTranslationTable(
			"<Key>Return:	form-done()\n\
			<Key>Escape:	form-cancel()\n\
			:<Key>Tab:	form-next()\n\
			Ctrl<Key>n:	form-next()\n\
			Ctrl<Key>p:	form-previous()\n\
			<Btn1Down>:	form-select()"));

	horiz = place(c->w, vert, horiz, RIGHT);
	return NIL;
}

static LISP form_menu(LISP n)
{
	char *name = get_c_string(n);
	control *c = new_control(MENU, name);
	c->w = lastw = XtVaCreateManagedWidget(name,
		menuButtonWidgetClass, form_dialog,
		XtNlabel, name, NULL);
	menu = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, c->w, NULL);

	horiz = place(c->w, vert, horiz, RIGHT);
	return NIL;
}

static void menu_select(Widget w, XtPointer client_data, XtPointer call_data)
{
	String p = (String)client_data;
	XtVaSetValues(XtParent(XtParent(w)),
		XtNlabel, p, NULL);
}

static LISP form_menuentry(LISP n)
{
	char *name = get_c_string(n);
	Widget entry = XtVaCreateManagedWidget(name,
		smeBSBObjectClass, menu, NULL);
	XtAddCallback(entry,
		XtNcallback, menu_select, name);
	return NIL;
}

static control *find_widget(Widget w)
{
	control *c = c_list;
	if (!c) return NULL;
	do {
		if (c->w == w) return c;
		c = c->next;
	} while (c != c_list);
	return NULL;
}

static Widget last_textwidget()
{
	control *c = c_list->prev;

	if (!c) return NULL;
	do {
		if (c->type == TEXT) return c->w;
		c = c->prev;
	} while (c != c_list);
	return NULL;
}

static Widget first_textwidget()
{
	control *c = c_list;

	if (!c) return NULL;
	do {
		if (c->type == TEXT) return c->w;
		c = c->next;
	} while (c != c_list);
	return NULL;
}

static Widget previous_textwidget(Widget w)
{
	control *c, *c1 = find_widget(w);
	if (c1 == NULL) return NULL;
	c = c1->prev;

	do {
		if (c->type == TEXT) return c->w;
		c = c->prev;
	} while (c != c1);

	return NULL;
}

static Widget next_textwidget(Widget w)
{
	control *c, *c1 = find_widget(w);
	if (c1 == NULL) return NULL;
	c = c1->next;

	do {
		if (c->type == TEXT) return c->w;
		c = c->next;
	} while (c != c1);

	return NULL;
}

static void done_action(Widget w, XEvent *event, String *params, Cardinal *n)
{
	status = DONE;
}

static void cancel_action(Widget w, XEvent *event, String *params, Cardinal *n)
{
	status = ABORT;
}

static void set_focus(Widget w)
{
	control *c = c_list;

	if (w == NULL) return;
	do {
		if (c->type == TEXT)
			XtVaSetValues(c->w, XtNdisplayCaret, False, NULL);
		c = c->next;
	} while (c != c_list);
	XtSetKeyboardFocus(form_shell, w);
	XtVaSetValues(w, XtNdisplayCaret, True, NULL);
}

static void next_action(Widget w, XEvent *event, String *params, Cardinal *n)
{
	Widget w1 = next_textwidget(w);
	if (w1 == NULL) w1 = first_textwidget();
	set_focus(w1);
}

static void previous_action(Widget w, XEvent *event, String *params, Cardinal *n)
{
	Widget w1 = previous_textwidget(w);
	if (w1 == NULL) w1 = last_textwidget();
	set_focus(w1);
}

static void focus_action(Widget w, XEvent *event, String *params, Cardinal *n)
{
	set_focus(w);
}

static LISP form_okbutton(LISP text)
{
	char *name = get_c_string(text);
	Widget w = lastw = XtVaCreateManagedWidget(name,
		commandWidgetClass, form_dialog,
		XtNlabel, name, NULL);
	XtAddCallback(w, XtNcallback, CBform_done, NULL);
	horiz = place(w, vert, horiz, LEFT);
	return NIL;
}

static LISP form_cancelbutton(LISP text)
{
	char *name = get_c_string(text);
	Widget w = lastw = XtVaCreateManagedWidget(name,
		commandWidgetClass, form_dialog,
		XtNlabel, name, NULL);
	XtAddCallback(w, XtNcallback, CBform_cancel, NULL);
	horiz = place(w, vert, horiz, LEFT);
	return NIL;
}

static LISP form_property(LISP name, LISP value)
{
	char *n = get_c_string(name);
	char *tv;
	long lv;

	if (!lastw) err("Last widget is NULL", NIL);

	if (FLONUMP(value)) {
		lv = get_c_long(value);
		XtVaSetValues(lastw, n, lv, NULL);
	} else {
		tv = get_c_string(value);
		XtVaSetValues(lastw, n, tv, NULL);
	}

	return NIL;
}

static LISP form_newline()
{
	vert = horiz;
	horiz = NULL;
	return NIL;
}

static LISP form_end()
{
	control *c;
	LISP result;
	Widget w = first_textwidget();

	XtRealizeWidget(form_shell);
	XtUnrealizeWidget(form_shell);
	center(top, form_shell);
	XtPopup(form_shell, XtGrabNonexclusive);

	XWarpPointer(XtDisplay(form_shell), None, XtWindow(form_shell),
		0, 0, 0, 0, 0, 0);
	set_focus(w);
	/* and the loop */
	status = WAITING;
	while (status == WAITING) {
		XEvent event_return;

		XtAppNextEvent(app, &event_return);
		XtDispatchEvent(&event_return);
	}
	XtPopdown(form_shell);
	/* collect values and free resources */
	result = NIL;
	c = c_list;
	if (!c) return NIL;
	do {
		String value;
		switch (c->type) {
		case TEXT:
			XtVaGetValues(c->w, XtNstring, &value, NULL);
			break;
		case MENU:
			XtVaGetValues(c->w, XtNlabel, &value, NULL);
			break;
		}
		result = cons(cons(strcons(strlen(c->name), c->name),
				   strcons(strlen(value), value)),
			      result);
		c = c->next;
	} while (c != c_list);
	XtDestroyWidget(form_shell);
	lastw = NULL;
	if (status == ABORT) return NIL;
	return result;
}

static XtActionsRec actions[] =
{
	{"form-done", done_action},
	{"form-cancel", cancel_action},
	{"form-next", next_action},
	{"form-previous", previous_action},
	{"form-select", focus_action}
};

void init_form(Widget topLevel)
{
	XtAppContext app_context = XtWidgetToApplicationContext(topLevel);

	XtAppAddActions(app_context, actions, XtNumber(actions));

	top = topLevel;
	app = app_context;
	init_subr_0("form-begin", form_begin);
	init_subr_1("form-label", form_label);
	init_subr_1("form-text", form_text);
	init_subr_1("form-menu", form_menu);
	init_subr_1("form-menuentry", form_menuentry);
	init_subr_1("form-okbutton", form_okbutton);
	init_subr_1("form-cancelbutton", form_cancelbutton);
	init_subr_2("form-property", form_property);
	init_subr_0("form-newline", form_newline);
	init_subr_0("form-end", form_end);
}
