/*
 * Create and destroy user menus and all widgets in them. All widgets
 * are labels or pushbuttons; they are faster than text buttons. Whenever
 * the user presses a button with text in it, it is overlaid with a Text
 * button. For editing and input into the Text button, see useredit.c.
 *
 *	all_files_served()		all files read from server?
 *	destroy_user_popup()		remove user popup
 *	create_user_popup()		create user popup
 */

#include <stdio.h>
#include <unistd.h>
#include <time.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef MIPS
#include <stdlib.h>
#endif
#include <pwd.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/LabelP.h>
#include <Xm/LabelG.h>
#include <Xm/PushBP.h>
#include <Xm/PushBG.h>
#include <Xm/ToggleB.h>
#include <Xm/ScrolledW.h>
#include <Xm/Text.h>
#include <Xm/Protocols.h>
#include "cal.h"

enum Columns { C_EN_MONTH=0, C_EN_WEEK, C_EN_DAY, C_EN_YROV, C_EN_ALARM,
		C_GROUP, C_NAME, C_EN_SRV, C_PATH, C_SERVER, NCOLUMNS };

static void create_user_rows(), edit_user_button(), got_user_text(),
	    draw_row(), delete_callback(), sort_callback(),
	    cancel_callback(), done_callback(), list_callback(), got_text();
#ifdef MIPS
extern struct passwd *getpwnam();
#endif

extern Display		*display;	/* everybody uses the same server */
extern GC		gc;		/* everybody uses this context */
extern Pixel		color[NCOLS];	/* colors: COL_* */
extern Pixmap		pixmap[NPICS];	/* common symbols */
extern Widget		mainwindow;	/* popup menus hang off main window */
extern struct config	config;		/* global configuration data */
extern struct plist	*mainlist;	/* list of all schedule entries */
extern struct mainmenu	mainmenu;	/* all important main window widgets */
extern struct user	*user;		/* user list (from file_r.c) */
extern int		nusers;		/* number of users in user list */

static struct user	*saved_user;	/* saved user[] while editing */
static int		*saved_nusers;	/* saved nusers while editing */
static BOOL		have_shell;	/* message popup exists if TRUE */
static Widget		shell;		/* popup menu shell */
static Widget		delete;		/* delete button, for desensitizing */
static Widget		info;		/* info line for error messages */
static Widget		textwidget;	/* if editing, text widget; else 0 */
static int		have_nrows;	/* # of table widget rows allocated */
static int		xedit, yedit;	/* if editing, column/row */
static int		ycurr;		/* current row, 0=none, 1=1st user...*/
static Widget		ulist;		/* user list RowColumn widget */
static Widget	(*utable)[NCOLUMNS+1];	/* all widgets in user list table */
					/* [0][*] is title row, */
					/* [*][NCOLUMNS] is form widget */

/*
 * return TRUE id all files are read from servers. In this case, buttons like
 * Sync or only-owner-can-write have no function and can be grayed out.
 */

BOOL all_files_served()
{
	int u;

	for (u=nusers-1; u >= 0; u++)
		if (!user[u].fromserver)
			return(FALSE);
	return(TRUE);
}


/*
 * Return FALSE if the user menu is up. While it is up, plan is not connected
 * to network servers, so any appointment editing is perilous and will probably
 * be lost. Also put up an error popup and bring the usermenu to the front.
 */

BOOL can_edit_appts()
{
	if (have_shell) {
		create_error_popup(shell, 0,
			"Please finish specifying the file list and the\n"
			"network connections before editing appointments.");
		return(FALSE);
	}
	return(TRUE);
}


/*
 * copy user list <src> to user list <tar>. If <tar> is non-null, it is deleted
 * first. If <src> is null, <tar> will end up empty. This is used for copying
 * user to save_user when editing begins, and to copy save_user to user when
 * Cancel is pressed. The sizes are also copied but are not otherwise checked.
 */

static void copyuser(tar, ntar, src, nsrc)
	register struct user	**tar, *src;	/* lists to copy */
	int			*ntar, nsrc;	/* list sizes to copy */
{
	register struct user	*u;		/* target scan pointer */
	register int		i;		/* target counter */

	if (*tar) {
		for (u=*tar, i=*ntar; i; i--, u++) {
			if (u->name)	free(u->name);
			if (u->path)	free(u->path);
			if (u->server)	free(u->server);
#ifdef PLANGROK
			form_delete(u->form);
			dbase_delete(u->dbase);
			u->grok  = FALSE;
			u->form  = 0;
			u->dbase = 0;
#endif
		}
		free(*tar);
	}
	*tar = 0;
	if (src) {
		if (!(u = *tar = malloc(nsrc * sizeof(struct user))))
			fatal("no memory");
		for (i=0; i < nsrc; i++, u++, src++) {
			*u = *src;
			u->name   = mystrdup(u->name);
			u->path   = mystrdup(u->path);
			u->server = mystrdup(u->server);
		}
	}
	*ntar = nsrc;
}


/*
 * destroy the popup. Remove it from the screen, and destroy its widgets.
 * Redraw the week menu if there is one. If reading fails, something may
 * be wrong with the servers; don't remove the popup in this case. Before
 * saving or reading, check if any file is served that wasn't before, and
 * do the attach and offer to copy the file to the server first.
 */

static BOOL accept_new_list()
{
	register struct entry	*ep;
	register struct user	*up;
	register int		u, i, nlocal, nremote;
	char			*msg;

	for (u=nusers-1; u >= 0; u--)
		if (!user[u].prev_fromserver && user[u].fromserver)
			break;
	if (u < 0)
		return(TRUE);
	if (msg = attach_to_network()) {
		create_error_popup(shell, 0, msg);
		free(msg);
		return(FALSE);
	}
	for (up=user, u=0; u < nusers; u++, up++) {
		if (!up->fromserver || up->prev_fromserver)
			continue;
		nremote = open_and_count_rows_on_server(u);
		nlocal  = 0;
		ep = mainlist->entry;
		for (i=mainlist->nentries-1; i >= 0; i--, ep++)
			nlocal += !ep->user && !u ||
				  !strcmp(ep->user, up->name);
		if (!nlocal)
			continue;
		if (!nremote) {
			ep = mainlist->entry;
			for (i=mainlist->nentries-1; i >= 0; i--, ep++)
				if (!ep->user && !u ||
				    !strcmp(ep->user, up->name)) {
					ep->id   = 0;
					ep->file = up->file_id;
					server_entry_op(ep, 'w');
				}
			create_error_popup(mainwindow, 0,
				"File %s on server %s did not\nexist. "
				"Created one and initialized with\n"
				"currently loaded appointments.",
				up->name, up->server);
			continue;

		} else if (!u) {
			char source[512], target[518];
			strcpy(source, resolve_tilde(DB_PUB_PATH));
			sprintf(target, "%s.old", source);
			unlink(target);
			if (!copyfile(source, target)) {
				create_error_popup(shell, 0,
					"Failed to back up %s,\n"
					"cannot write to %s",
					source, target);
				return(FALSE);
			}
			create_error_popup(mainwindow, 0,
				"You have %d appointments here and\n"
				"%d on server %s. Saved %s\n"
				"to %s and discarded old\n"
				"appointments in %s.",
				nlocal, nremote, up->server,
				source, target, source);
		} else if (nlocal != nremote)
			create_error_popup(mainwindow, 0,
				"File %s has %d appointments in %s\n"
				"and %d on server %s. Switching to "
				"server and ignoring\nfile %s.",
				up->name, nlocal, up->path,
				nremote, up->server, up->path);
	}
	mainlist->modified = TRUE;
	write_mainlist();
	if (!read_mainlist())
		return(FALSE);
	update_all_listmenus();
	redraw_all_views();
	resynchronize_daemon();
	return(TRUE);
}


void destroy_user_popup(accept)
	BOOL		accept;		/* true=Done, false=Cancel */
{
	if (!have_shell)
		return;
	edit_user_button(FALSE, 0, 0);
	if (accept) {
		if (!accept_new_list())
			return;
		copyuser(&saved_user, &saved_nusers, 0, 0);
		mainlist->modified = TRUE;
		write_mainlist();
	} else
		copyuser(&user, &nusers, saved_user, saved_nusers);

	XtPopdown(shell);
	XTDESTROYWIDGET(shell);
	have_shell = FALSE;
}


/*
 * create a user popup as a separate application shell.
 */

void create_user_popup()
{
	Widget			form, scroll, w;
	Arg			args[15];
	int			n;
	Atom			closewindow;

	if (have_shell)
		return;

	destroy_all_listmenus();/* appointment editing is now locked */
	write_mainlist();	/* last chance to write modified foreign apts*/

	n = 0;
	XtSetArg(args[n], XmNdeleteResponse,	XmUNMAP);		n++;
	XtSetArg(args[n], XmNiconic,		False);			n++;
	shell = XtAppCreateShell("File List", "plan",
			applicationShellWidgetClass, display, args, n);
#	ifdef EDITRES
	XtAddEventHandler(shell, (EventMask)0, TRUE, 
 			_XEditResCheckMessages, NULL);
#	endif
	set_icon(shell, 1);
	form = XtCreateWidget("userform", xmFormWidgetClass,
			shell, NULL, 0);
	XtAddCallback(form, XmNhelpCallback, help_callback, (XtPointer)"user");

							/*-- buttons --*/
	n = 0;
	XtSetArg(args[n], XmNbottomAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNbottomOffset,	8);			n++;
	XtSetArg(args[n], XmNleftAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNleftOffset,	8);			n++;
	XtSetArg(args[n], XmNwidth,		80);			n++;
	XtSetArg(args[n], XmNsensitive,		False);			n++;
	delete = w = XtCreateManagedWidget("Delete", xmPushButtonWidgetClass,
			form, args, n);
	XtAddCallback(w, XmNactivateCallback, delete_callback, (XtPointer)0);
	XtAddCallback(w, XmNhelpCallback,     help_callback,   (XtPointer)
								"user_delete");

	n = 0;
	XtSetArg(args[n], XmNbottomAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNbottomOffset,	8);			n++;
	XtSetArg(args[n], XmNleftAttachment,	XmATTACH_WIDGET);	n++;
	XtSetArg(args[n], XmNleftWidget,	w);			n++;
	XtSetArg(args[n], XmNleftOffset,	8);			n++;
	XtSetArg(args[n], XmNwidth,		80);			n++;
	w = XtCreateManagedWidget("Sort", xmPushButtonWidgetClass,
			form, args, n);
	XtAddCallback(w, XmNactivateCallback, sort_callback, (XtPointer)0);
	XtAddCallback(w, XmNhelpCallback,     help_callback, (XtPointer)
								"user_sort");
	n = 0;
	XtSetArg(args[n], XmNbottomAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNbottomOffset,	8);			n++;
	XtSetArg(args[n], XmNrightAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNrightOffset,	8);			n++;
	XtSetArg(args[n], XmNwidth,		80);			n++;
	w = XtCreateManagedWidget("Done", xmPushButtonWidgetClass,
			form, args, n);
	XtAddCallback(w, XmNactivateCallback, done_callback, (XtPointer)0);
	XtAddCallback(w, XmNhelpCallback,     help_callback, (XtPointer)
								"user_done");

	n = 0;
	XtSetArg(args[n], XmNbottomAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNbottomOffset,	8);			n++;
	XtSetArg(args[n], XmNrightAttachment,	XmATTACH_WIDGET);	n++;
	XtSetArg(args[n], XmNrightWidget,	w);			n++;
	XtSetArg(args[n], XmNrightOffset,	8);			n++;
	XtSetArg(args[n], XmNwidth,		80);			n++;
	w = XtCreateManagedWidget("Help", xmPushButtonWidgetClass,
			form, args, n);
	XtAddCallback(w, XmNactivateCallback, help_callback, (XtPointer)
								"user");
	XtAddCallback(w, XmNhelpCallback,     help_callback, (XtPointer)
								"user");
	n = 0;
	XtSetArg(args[n], XmNbottomAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNbottomOffset,	8);			n++;
	XtSetArg(args[n], XmNrightAttachment,	XmATTACH_WIDGET);	n++;
	XtSetArg(args[n], XmNrightWidget,	w);			n++;
	XtSetArg(args[n], XmNrightOffset,	8);			n++;
	XtSetArg(args[n], XmNwidth,		80);			n++;
	w = XtCreateManagedWidget("Cancel", xmPushButtonWidgetClass,
			form, args, n);
	XtAddCallback(w, XmNactivateCallback, cancel_callback, (XtPointer)0);
	XtAddCallback(w, XmNhelpCallback,     help_callback, (XtPointer)
								"user_done");

							/*-- infotext -- */
	n = 0;
	XtSetArg(args[n], XmNleftAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNleftOffset,	8);			n++;
	XtSetArg(args[n], XmNrightAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNrightOffset,	8);			n++;
	XtSetArg(args[n], XmNbottomAttachment,	XmATTACH_WIDGET);	n++;
	XtSetArg(args[n], XmNbottomWidget,	w);			n++;
	XtSetArg(args[n], XmNbottomOffset,	8);			n++;
	XtSetArg(args[n], XmNalignment,         XmALIGNMENT_BEGINNING);	n++;
	info = XtCreateManagedWidget(" ", xmLabelGadgetClass,
			form, args, n);

							/*-- scroll --*/
	n = 0;
	XtSetArg(args[n], XmNtopAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNtopOffset,		8);			n++;
	XtSetArg(args[n], XmNbottomAttachment,	XmATTACH_WIDGET);	n++;
	XtSetArg(args[n], XmNbottomWidget,	info);			n++;
	XtSetArg(args[n], XmNbottomOffset,	8);			n++;
	XtSetArg(args[n], XmNleftAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNleftOffset,	8);			n++;
	XtSetArg(args[n], XmNrightAttachment,	XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNrightOffset,	8);			n++;
	XtSetArg(args[n], XmNwidth,		810);			n++;
	XtSetArg(args[n], XmNheight,		220);			n++;
	XtSetArg(args[n], XmNscrollingPolicy,	XmAUTOMATIC);		n++;
	scroll = XtCreateWidget("uscroll", xmScrolledWindowWidgetClass,
			form, args, n);
	XtAddCallback(scroll, XmNhelpCallback, help_callback, (XtPointer)
								"user");

	n = 0;
	ulist = XtCreateManagedWidget("ulist", xmBulletinBoardWidgetClass,
			scroll, args, n);

	create_user_rows();	/* have_shell must be FALSE here */

	XtManageChild(scroll);
	XtManageChild(form);
	XtPopup(shell, XtGrabNone);

	closewindow = XmInternAtom(display, "WM_DELETE_WINDOW", False);
	XmAddWMProtocolCallback(shell, closewindow,
					done_callback, (XtPointer)shell);
	have_shell = TRUE;
	for (n=0; n < nusers; n++)
		user[n].prev_fromserver = user[n].fromserver;

	copyuser(&saved_user, &saved_nusers, user, nusers);
}


/*
 * makes sure there are enough widget rows for schedule entries. Also makes
 * sure that there aren't too many, for speed reasons. Allocate one extra
 * widget row for the title at the top. All the text buttons are
 * label widgets. For performance reasons, they are overlaid by a text
 * widget when pressed.
 * No text is printed into the buttons yet, this is done later by draw_users().
 */

static short cell_x    [NCOLUMNS];
static short cell_xs   [NCOLUMNS];
static char *cell_name [NCOLUMNS] = { "Month", "Week", "Day", "YrOv", "Alarm",
				      "Group", "Name", "Server", "Local path",
				      "Server Host"};
static char *cell_help [NCOLUMNS] = { "user_enable", "user_enable",
				      "user_enable", "user_enable",
				      "user_enable", "user_color", "user_name",
				      "user_home", "user_home", "user_home" };

static void create_user_rows()
{
	int			nrows = nusers+5 - nusers%5;
	int			x, y, xo;
	Arg			args[20];
	int			n;
	int			align;
	BOOL			shadow;
	String			callbk;
	char			*name;
	WidgetClass		class;

	if (!cell_x[0])
		for (x=4, n=0; n < NCOLUMNS; x+=cell_xs[n++]) {
			cell_x[n]  = x;
			cell_xs[n] = n == C_NAME || n == C_SERVER ? 120 :
				     n == C_PATH		  ? 220 : 40;
		}
	if (!have_shell)				/* check # of rows: */
		have_nrows = 0;
	if (nrows <= have_nrows)
		return;

	n = (nrows+1) * (NCOLUMNS+1) * sizeof(Widget *);
	if (utable && !(utable = (Widget (*)[])realloc(utable, n)) ||
	   !utable && !(utable = (Widget (*)[])malloc(n)))
		fatal("no memory");

	XtUnmanageChild(ulist);
	for (y=have_nrows; y <= nrows; y++) {
	    n = 0;
	    XtSetArg(args[n], XmNtopAttachment,		XmATTACH_FORM);   n++;
	    XtSetArg(args[n], XmNheight,		30);		  n++;
	    XtSetArg(args[n], XmNy,			10 + 30 * y);	  n++;
	    XtSetArg(args[n], XmNleftAttachment,	XmATTACH_FORM);   n++;
	    utable[y][NCOLUMNS] = XtCreateWidget("userrowform",
	   				 xmFormWidgetClass, ulist, args, n);
	    for (x=0; x < NCOLUMNS; x++) {
		xo = n = 0;
		if (y) {
			switch(x) {
			  case C_EN_MONTH:
			  case C_EN_WEEK:
			  case C_EN_DAY:
			  case C_EN_YROV:
			  case C_EN_ALARM:
			  case C_EN_SRV:
				shadow = FALSE;
				class  = xmToggleButtonWidgetClass;
				callbk = XmNvalueChangedCallback;
				align  = XmALIGNMENT_CENTER;
				xo     = 12;
				XtSetArg(args[n], XmNselectColor,
							color[COL_TOGGLE]);n++;
				break;
			  case C_GROUP:
				shadow = TRUE;
				class  = xmPushButtonWidgetClass;
				callbk = XmNactivateCallback;
				align  = XmALIGNMENT_CENTER;
				break;
			  case C_NAME:
			  case C_PATH:
			  case C_SERVER:
				shadow = TRUE;
				class  = xmPushButtonWidgetClass;
				callbk = XmNactivateCallback;
				align  = XmALIGNMENT_BEGINNING;
			}
			name   = " ";
		} else {
			shadow = FALSE;
			class  = xmLabelWidgetClass;
			align  = XmALIGNMENT_CENTER;
			name   = cell_name[x];
		}
		XtSetArg(args[n], XmNx,			cell_x[x]+xo);  n++;
		XtSetArg(args[n], XmNwidth,		cell_xs[x]-xo);	n++;
		XtSetArg(args[n], XmNtopAttachment,	XmATTACH_FORM); n++;
		XtSetArg(args[n], XmNbottomAttachment,	XmATTACH_FORM);	n++;
		/* needed by LessTif, doesn't work with Metrolink Linux Motif
		XtSetArg(args[n], XmNleftAttachment,	XmATTACH_SELF); n++;
		XtSetArg(args[n], XmNrightAttachment,	XmATTACH_SELF); n++;
		*/
		XtSetArg(args[n], XmNalignment,         align);		n++;
		XtSetArg(args[n], XmNrecomputeSize,	False);		n++;
		XtSetArg(args[n], XmNtraversalOn,	True);		n++;
		XtSetArg(args[n], XmNhighlightThickness,0);		n++;
		XtSetArg(args[n], XmNshadowThickness,	shadow);	n++;
		utable[y][x] = XtCreateManagedWidget(name, class,
				utable[y][NCOLUMNS], args, n);
		if (y)
			XtAddCallback(utable[y][x], callbk,
				list_callback, (XtPointer)(x + y * NCOLUMNS));
		XtAddCallback(utable[y][x], XmNhelpCallback, help_callback,
						(XtPointer)cell_help[x]);
	    }
	    XtManageChild(utable[y][NCOLUMNS]);
	}
	for (y=have_nrows; y <= nrows; y++)
		draw_row(y);
	have_nrows = nrows;
	XtManageChild(ulist);
}


/*-------------------------------------------------- editing ----------------*/
/*
 * return a default host name for row y (1=own)
 */

static char *default_server(buf, bufsz, y)
	char		*buf;		/* name buffer, static wastes space */
	int		bufsz;		/* size of buffer */
	int		y;		/* for which row, 1=own */
{
	return(y == 1 ? gethostname(buf, bufsz)	? "localhost"
						: buf
		      : user[0].server		? user[0].server
						: "localhost");
}


/*
 * turn a text label into a Text button, to allow user input. This is done
 * by simply installing a text widget on top of the label widget. The proper
 * user name or home dir is put into the text widget. The previously edited
 * button is un-edited.
 */

static void edit_user_button(doedit, x, y)
	BOOL			doedit;		/* TRUE=edit, FALSE=unedit */
	int			x;		/* column, 0..NCOLUMNS-1* */
	int			y;		/* row, y=0: title */
{
	register struct user	*u;
	Arg			args[15];
	int			n;
	char			*text, buf[1024];

	if (textwidget) {
		BOOL free_string = TRUE;
		char *string = XmTextGetString(textwidget);
		while (*string == ' ' || *string == '\t') string++;
		if (!*string) {
			free_string = FALSE;
			switch(xedit) {
			  case C_NAME: {
				struct passwd *pw = getpwuid(getuid());
				string = pw ? pw->pw_name : getenv("LOGNAME");
				break; }
			  case C_PATH:
			  	break;
			  case C_SERVER:
				string = default_server(buf, sizeof(buf), y);
			}
		}
		got_user_text(xedit, yedit,
				string && *string ? string : "unknown");
		if (free_string)
			XtFree(string);
		XtDestroyWidget(textwidget);
		if (yedit && yedit <= nusers)
			user[yedit-1].suspend_m =
			user[yedit-1].suspend_w =
			user[yedit-1].suspend_o =
			user[yedit-1].suspend_d = 0;
		draw_row(yedit);
		create_user_rows();
	}
	textwidget = 0;
	if (!doedit)
		return;

	if (y > nusers+1)
		y = nusers+1;
	u = &user[y-1];
	n = 0;
	XtSetArg(args[n], XmNx,			10 + cell_x[x]);	n++;
	XtSetArg(args[n], XmNy,			10 + 30*y);		n++;
	XtSetArg(args[n], XmNwidth,		cell_xs[x]);		n++;
	XtSetArg(args[n], XmNheight,		30);			n++;
	XtSetArg(args[n], XmNrecomputeSize,	False);			n++;
	XtSetArg(args[n], XmNpendingDelete,	True);			n++;
	XtSetArg(args[n], XmNhighlightThickness,0);			n++;
	XtSetArg(args[n], XmNshadowThickness,	1);			n++;
	XtSetArg(args[n], XmNbackground,	color[COL_TEXTBACK]);	n++;
	textwidget = XtCreateManagedWidget("text", xmTextWidgetClass,
						ulist, args, n);
	XtAddCallback(textwidget, XmNactivateCallback, got_text,
						(XtPointer)(x + y * NCOLUMNS));
	XtAddCallback(textwidget, XmNhelpCallback, help_callback,
						(XtPointer)cell_help[x]);
	XmProcessTraversal(textwidget, XmTRAVERSE_CURRENT);

	text =  y > nusers    ? ""	  :
		x == C_NAME   ? u->name   :
		x == C_PATH   ? u->path   :
		x == C_SERVER ? u->server : "";
	print_text_button(textwidget, "%s", text);
	xedit = x;
	yedit = y;
}


/*
 * the user has entered text into a name, path, or server text button.
 * Check & store.
 */

static void got_user_text(x, y, string)
	int			x;		/* column, 0..NCOLUMNS-1* */
	int			y;		/* row, y=0: title */
	char			*string;	/* text entered by user */
{
	struct passwd		*pw;		/* for searching home dirs */
	register struct user	*u;
	char			buf[100], *p;
	int			i;

	if (!y--)
		return;
	while (*string == ' ' || *string == '\t')
		string++;
	while ((i = strlen(string)) && string[i-1] == ' ')
		string[i-1] = 0;
	if (!*string)
		return;
	for (i=0; i < nusers; i++) {
		if (x == C_NAME && i != y && !strcmp(string, user[i].name)) {
			create_error_popup(shell, 0,
				"Name is not unique, rejected");
			return;
		}
		if (x == C_PATH && i != y && !strcmp(string, user[i].path)) {
			create_error_popup(shell, 0,
				"Path is not unique, rejected");
			return;
		}
	}
	if (y >= nusers) {
		int n = ++nusers * sizeof(struct user);
		if (user && !(user = (struct user *)realloc(user, n)) ||
		   !user && !(user = (struct user *)malloc(n)))
			fatal("no memory");
		y = nusers-1;
		u = user + y;
		memset(u, 0, sizeof(struct user));
		u->file_id = -1;
		u->fromserver = user[nusers > 2].fromserver;
	}
	u = user+y;
	u->modified = TRUE;
	switch(x) {
	  case C_NAME:
		if (u->name)
			free(u->name);
		u->name = mystrdup(string);
		for (p=u->name; *p; p++)	/* netplan can't mkdir */
			if (*p == '/')
				*p = '_';
		break;

	  case C_PATH:
		if (u->path)
			free(u->path);
		u->path = mystrdup(string);
		u->fromserver = FALSE;
		if (*u->path && access(u->path, F_OK))
			print_button(info, "%s: %s", u->path,
				errno==ENOENT ? "new file" : "cannot access");
		break;

	  case C_SERVER:
		if (u->server)
			free(u->server);
		u->server = mystrdup(string);
		u->fromserver = TRUE;
	}
							/* default path? */
	if (u->name && *u->name && !u->path) {
		if (pw = getpwnam(u->name))
			u->path = mystrdup(pw->pw_dir);
		else {
			sprintf(buf, "%s/%s", LIB, u->name);
			print_button(info, "no user \"%s\"", u->name);
			u->path = mystrdup(buf);
		}
	}
							/* default server? */
	if (u->name && *u->name && !u->server)
		u->server = mystrdup(default_server(buf, sizeof(buf), y));
}


/*
 * draw all buttons of row y. y must be > 0 because 0 is the title row.
 * If y is > nusers, the row is blanked.
 */

static void draw_row(y)
	int			y;
{
	Arg			arg;
	register struct user	*u = &user[y-1];
	int			col;

	if (y < 1)
		return;
	if (y <= nusers) {					/* valid row */
		XtSetArg(arg, XmNset, !u->suspend_m);
		XtSetValues (utable[y][C_EN_MONTH], &arg, 1);
		XtSetArg(arg, XmNset, !u->suspend_w);
		XtSetValues (utable[y][C_EN_WEEK],  &arg, 1);
		XtSetArg(arg, XmNset, !u->suspend_d);
		XtSetValues (utable[y][C_EN_DAY],   &arg, 1);
		XtSetArg(arg, XmNset, !u->suspend_o);
		XtSetValues (utable[y][C_EN_YROV],  &arg, 1);
		XtSetArg(arg, XmNset,  u->alarm);
		XtSetValues (utable[y][C_EN_ALARM], &arg, 1);
		XtSetArg(arg, XmNset,  u->fromserver);
		XtSetValues (utable[y][C_EN_SRV],   &arg, 1);
		XtSetArg(arg, XmNsensitive, !u->fromserver);
		XtSetValues(utable[y][C_PATH],      &arg, 1);
		XtSetArg(arg, XmNsensitive, u->fromserver);
		XtSetValues(utable[y][C_SERVER],    &arg, 1);
		if (y == 1) {
			print_button(utable[y][C_GROUP], "own");
			col = COL_BACK;
		} else {
			print_button(utable[y][C_GROUP], "%d", u->color);
			col = COL_WUSER_0 + u->color;
		}
		XtSetArg(arg, XmNbackground, color[col]);
		XtSetValues (utable[y][C_GROUP],  &arg, 1);
		print_button(utable[y][C_NAME],   "%s", u->name ? u->name:"?");
		print_button(utable[y][C_PATH],   "%s", u->path ? u->path:"?");
		print_button(utable[y][C_SERVER], "%s", u->server ?
								u->server:"?");
	} else {
		XtSetArg(arg, XmNset, 0);			/* blank row */
		XtSetValues (utable[y][C_EN_MONTH], &arg, 1);
		XtSetValues (utable[y][C_EN_WEEK],  &arg, 1);
		XtSetValues (utable[y][C_EN_DAY],   &arg, 1);
		XtSetValues (utable[y][C_EN_YROV],  &arg, 1);
		XtSetValues (utable[y][C_EN_ALARM], &arg, 1);
		XtSetValues (utable[y][C_EN_SRV],   &arg, 1);
		XtSetArg(arg, XmNbackground, color[COL_BACK]);
		XtSetValues (utable[y][C_GROUP], &arg, 1);
		print_button(utable[y][C_GROUP],  " ");
		print_button(utable[y][C_NAME],   " ");
		print_button(utable[y][C_PATH],   " ");
		print_button(utable[y][C_SERVER], " ");
	}
}



/*-------------------------------------------------- callbacks --------------*/
/*
 * Delete, Add-all, Sort, and Done buttons
 * All of these routines are direct X callbacks.
 */

/*ARGSUSED*/
static void delete_callback(widget, item, data)
	Widget				widget;
	int				item;
	XmToggleButtonCallbackStruct	*data;
{
	int				n;
	Arg				args;

	if (ycurr == 1)
		print_button(info, "Cannot delete own file");

	else if (ycurr && ycurr <= nusers) {
		edit_user_button(FALSE, 0, 0);
		for (n=ycurr-1; n < nusers-1; n++)
			user[n] = user[n+1];
		n = --nusers * sizeof(struct user);
		if (n && !(user = (struct user *)realloc(user, n)))
			fatal("no memory");
		for (n=1; n <= have_nrows; n++)
			draw_row(n);
	}
	if (ycurr <= 1) {
		XtSetArg(args, XmNsensitive, 0);
		XtSetValues(delete, &args, 1);
	}
}


#if defined(ULTRIX) || defined(MIPS)
		/* this means Ultrix 4.2A. If Ultrix 4.3 complains about */
		/* a missing const, change the following definition. */
#define CONST
#else
#define CONST const
#endif

static int compare(u, v) register CONST void *u, *v; {
	return(  ((struct user *)u)->color == ((struct user *)v)->color
	? strcmp(((struct user *)u)->name,    ((struct user *)v)->name)
	:        ((struct user *)u)->color -  ((struct user *)v)->color); }

/*ARGSUSED*/
static void sort_callback(widget, item, data)
	Widget				widget;
	int				item;
	XmToggleButtonCallbackStruct	*data;
{
	Arg				args;
	int				n;

	if (nusers > 1) {
		edit_user_button(FALSE, 0, 0);
		XtSetArg(args, XmNsensitive, 0);
		XtSetValues(delete, &args, 1);
		qsort(user+1, nusers-1, sizeof(struct user), compare);
		for (n=1; n <= nusers; n++)
			draw_row(n);
	}
}


/*ARGSUSED*/
static void cancel_callback(widget, item, data)
	Widget				widget;
	int				item;
	XmToggleButtonCallbackStruct	*data;
{
	destroy_user_popup(FALSE);
}


/*ARGSUSED*/
static void done_callback(widget, item, data)
	Widget				widget;
	int				item;
	XmToggleButtonCallbackStruct	*data;
{
	destroy_user_popup(TRUE);
}


/*
 * one of the buttons in the list was pressed. Remember that the line 0 is
 * the title, and line 1 is reserved for the user's own appointment file.
 */

/*ARGSUSED*/
static void list_callback(widget, item, data)
	Widget				widget;
	int				item;
	XmToggleButtonCallbackStruct	*data;
{
	int				x = item % NCOLUMNS;
	int				y = item / NCOLUMNS;
	Arg				arg;

	if (y > nusers) {					/* new entry */
		if (x != C_NAME) {
			print_button(info, "Enter a name first");
			return;
		}
		ycurr = 0;
		edit_user_button(TRUE, x, nusers+1);
	} else {						/* old entry */
		ycurr = y;
		switch(x) {
		  case C_EN_MONTH:
			user[y-1].suspend_m = !data->set;
			draw_calendar(NULL);
			break;

		  case C_EN_WEEK:
			user[y-1].suspend_w = !data->set;
			draw_week_calendar(NULL);
			break;

		  case C_EN_DAY:
			user[y-1].suspend_d = !data->set;
			draw_day_calendar(NULL);
			break;

		  case C_EN_YROV:
			user[y-1].suspend_o = !data->set;
			draw_yov_calendar(NULL);
			break;

		  case C_EN_ALARM:
			user[y-1].alarm = data->set;
			break;

		  case C_EN_SRV:
			user[y-1].fromserver = data->set;
			draw_row(y);
			break;

		  case C_GROUP:
			if (y == 1) {
				print_button(info, "Cannot change own group");
				return;
			}
			user[y-1].color++;
			user[y-1].color &= 7;
			XtSetArg(arg, XmNbackground,
				color[COL_WUSER_0 + user[y-1].color]);
			XtSetValues(widget, &arg, 1);
			print_button(widget, "%d", user[y-1].color);
			break;

		  case C_PATH:
			if (y == 1) {
				print_button(info, "Cannot change own path");
				return;
			}
		  case C_NAME:
		  case C_SERVER:
			edit_user_button(TRUE, x, y);
		}
	}
	XtSetArg(arg, XmNsensitive, ycurr > 0);
	XtSetValues(delete, &arg, 1);
	print_button(info, " ");
}


/*
 * the user pressed Return in a text entry button
 */

/*ARGSUSED*/
static void got_text(widget, item, data)
	Widget				widget;
	int				item;
	XmToggleButtonCallbackStruct	*data;
{
	int				x = item % NCOLUMNS;
	int				y = item / NCOLUMNS;

	edit_user_button(FALSE, 0, 0);
}
