/*
 * Copyright (c) 1990,1991 Regents of The University of Michigan.
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appears in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of The University
 * of Michigan not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission. This software is supplied as is without expressed or
 * implied warranties of any kind.
 *
 *	Research Systems Unix Group
 *	The University of Michigan
 *	c/o Mike Clark
 *	535 W. William Street
 *	Ann Arbor, Michigan
 *	+1-313-763-0525
 *	netatalk@itd.umich.edu
 */

#ifndef _ATALK_ADOUBLE_H
#define _ATALK_ADOUBLE_H

#include <unistd.h>
#include <fcntl.h>
#include <sys/cdefs.h>
#include <sys/types.h>
#include <netatalk/endian.h>

#ifdef _POSIX_MAPPED_FILES
#include <sys/mman.h>
#endif

/*
 * AppleDouble entry IDs.
 */
#define ADEID_DFORK	1
#define ADEID_RFORK	2
#define ADEID_NAME	3
#define ADEID_COMMENT	4
#define ADEID_ICONBW	5
#define ADEID_ICONCOL	6
#define ADEID_FILEI	7
#define ADEID_FINDERI	9

#define ADEID_MAX	10

/*
 * AppleDouble entry default offsets.
 * The layout looks like this:
 *
 *	  255	  200		  16	  32		  N
 *	|  NAME	|    COMMENT	| FILEI	|    FINDERI	| RFORK	|
 */
#define ADEDOFF_RFORK	589
#define ADEDOFF_NAME	86
#define ADEDOFF_COMMENT	341
#define ADEDOFF_FILEI	541
#define ADEDOFF_FINDERI	557

#define ADEDLEN_RFORK	0
#define ADEDLEN_NAME	0
#define ADEDLEN_COMMENT	0
#define ADEDLEN_FILEI	16
#define ADEDLEN_FINDERI	32

#define AD_MAGIC	0x00051607
#define AD_VERSION	0x00010000

#define AD_DATASZ	589

/*
 * The header of the AppleDouble Header File looks like this:
 *
 *	NAME			SIZE
 *	====			====
 *	Magic			4
 *	Version			4
 *	Home File System	16
 *	Number of Entries	2
 *	Entry Descriptors for each entry:
 *		Entry ID	4
 *		Offset		4
 *		Length		4
 */

struct ad_entry {
    u_int32_t	ade_off;
    u_int32_t	ade_len;
};

struct ad_fd {
    int		 adf_fd;
    u_int32_t	 adf_off;
    int		 adf_flags;
    struct flock adf_lock;
};

struct adouble {
    u_int32_t		ad_magic;
    u_int32_t		ad_version;
    char		ad_homefs[ 16 ];
    struct ad_entry	ad_eid[ ADEID_MAX ];
    char		ad_data[ AD_DATASZ ];
    struct ad_fd	ad_df, ad_hf;
};

#define ADFLAGS_DF	(1<<0)
#define ADFLAGS_HF	(1<<1)
#define ADFLAGS_DIR	(1<<2)

/* lock flags */
#define ADLOCK_RD       F_RDLCK
#define ADLOCK_WR       F_WRLCK
#define ADLOCK_CLR      F_UNLCK

#define ad_dfileno(ad)		((ad)->ad_df.adf_fd)
#define ad_dlock(ad)            ((ad)->ad_df.adf_lock)
#define ad_hfileno(ad)		((ad)->ad_hf.adf_fd)
#define ad_hlock(ad)            ((ad)->ad_hf.adf_lock)
#define ad_getversion(ad)	((ad)->ad_version)
#define ad_gethomefs(ad)	((ad)->ad_homefs);
#define ad_sethomefs(ad,buf) \
	(bcopy((buf),(ad)->ad_homefs,sizeof((ad)->ad_homefs)))
#define ad_getentrylen(ad,eid)	((ad)->ad_eid[(eid)].ade_len)
#define ad_setentrylen(ad,eid,len) \
	((ad)->ad_eid[(eid)].ade_len = (len))
#define ad_entry(ad,eid)	(((ad)->ad_eid[(eid)].ade_off + \
	(ad)->ad_eid[(eid)].ade_len > AD_DATASZ ) ? 0 : \
	(caddr_t)(ad)->ad_data + (ad)->ad_eid[(eid)].ade_off)
#define ad_getoflags(ad,adf)	(((adf)&ADFLAGS_HF) ? \
	(ad)->ad_hf.adf_flags : (ad)->ad_df.adf_flags)

/* ad_flush.c */
extern int ad_flush __P((struct adouble *, int));

/* ad_lock.c */
extern int ad_lock __P((struct adouble *, const u_int32_t /*eid*/,
			const int /*type*/, const off_t /*offset*/,
			const int /*endflag*/, const off_t /*len*/));
extern int ad_tmplock __P((struct adouble *, const u_int32_t /*eid*/,
			   const int /*type*/, const off_t /*offset*/,
			   const int /*end*/, const off_t /*len*/));

/* ad_open.c */
extern char *ad_path __P((char *, int));
extern int ad_mode __P((char *, int));
extern int ad_mkdir __P((char *, int));
extern int ad_open __P((char *, int, int, int, struct adouble *));
extern int ad_refresh __P((struct adouble *));

/* ad_read.c/ad_write.c */
extern int ad_read __P((struct adouble *, const u_int32_t, 
			const u_int32_t, char *, const size_t));
extern int ad_write __P((struct adouble *, const u_int32_t, u_int32_t,
			 const int, const char *, const size_t));
extern int ad_dtruncate __P((struct adouble *, const size_t));
extern int ad_rtruncate __P((struct adouble *, const size_t));

/* ad_size.c */
extern off_t ad_size __P((struct adouble *, const u_int32_t ));

/* ad_mmap.c */
extern void *ad_mmapread __P((struct adouble *, const u_int32_t,
			      const off_t, const size_t));
extern void *ad_mmapwrite __P((struct adouble *, const u_int32_t,
			       const off_t, const int, const size_t));
#define ad_munmap(buf, len)  (munmap((buf), (len)))

#endif
