/* aumix.c: adjust /dev/mixer   copyright (c) 1993, 1996, 1997 the authors

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
   or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
   ************************************************************************** */

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>
#ifdef GPM
#include <gpm.h>
#else
#include "gpm-xterm.h"
#endif
#ifdef NCURSES
#include <ncurses.h>
#else
#include <curses.h>
#endif

#define AUMIX_VERSION "1.8"
#define SYSTEM_AUMIXRC "/etc/aumixrc"
#ifdef NCURSES
#define TITLE_COLOR 1
#define TITLE_COLOR2 2
#define HANDLE_COLOR 3
#define TRACK_COLOR 4
#define REC_LED_COLOR 5
#define PLAY_LED_COLOR 6
#define ACTIVE_COLOR 7
#define INACTIVE_COLOR 8
#define INCREMENT 4
#define XOFFSET 9
#define YOFFSET 0
#else				/* these are defined in ncurses.h */
#define TRUE 1
#define FALSE 0
#endif

void RefreshAllSettings(void);
void InitMixer(int minor);
void Usage(void);
void SetShowNoninter(int dev);
void LoadSettings(void);
void SaveSettings(void);
int MouseHandler(Gpm_Event * event, void *data);
#ifdef NCURSES
void InitColors(void);
void DrawHandles(int dev);
void Inter(void);
void AdjustLevel(int dev, int incr, int setlevel);
void SwitchRP(int dev);
void DrawRP(int dev);
void InitScreen(void);
void AdjustBalance(int dev, int incr, int setlevel);
void DrawLevelBalMode(int dev, int mode);
void RedrawBalance(int dev);
void EraseLevel(int dev);
void CloseScreen(void);
void AboutBox(void);
void KeysBox(void);
void ToggleMute(void);
#endif

#ifdef NCURSES
WINDOW *w_about;
WINDOW *w_keys;
WINDOW *w_panel;
#endif
FILE *setfile;
char setfile_name[256];
short setfile_opened = FALSE, setfile_write_perm = FALSE, setfile_read_perm = FALSE;
int devmask = 0, recmask = 0, recsrc = 0, stereodevs = 0, mixer_fd;
int level[SOUND_MIXER_NRDEVICES];	/* store levels for muting */
int mutestate = FALSE, opened = FALSE, interactive = FALSE;
char *dev_name[SOUND_MIXER_NRDEVICES] = SOUND_DEVICE_NAMES;
char *dev_label[SOUND_MIXER_NRDEVICES] = SOUND_DEVICE_LABELS;

/* An index into this array gives the device number for the
   corresponding command-line option. */
char *moptindx = "vbtswplmcxWrio123";
char *mopt;

int main(int argc, char *const argv[], const char *optstring)
{
    Gpm_Connect conn;
    int optn, i;
    InitMixer(0);
    if (argc <= 1) {
	interactive = TRUE;
    }
/* get command-line options */
    while (1) {
/*
   from the BSD 4.3 getopt(3) man page:
   It is also possible to handle digits as option letters.  This allows
   getopt() to be used with programs that expect a number (``-3'') as an
   option.  This practice is wrong, and should not be used in any current
   development.  It is provided for backward compatibility only.

   >:-)
 */
	optn = getopt(argc, argv, "dhLqSI:v:b:t:s:w:p:l:m:c:x:W:r:i:o:1:2:3:");
	if (optn == -1)
	    break;
	if ((mopt = strchr(moptindx, optn))) {
	    SetShowNoninter(mopt - moptindx);
	} else
	    switch (optn) {
	    case 'q':
		optarg = "q";
		for (i = 0; i < SOUND_MIXER_NRDEVICES; i++) {
		    if ((1 << i) & devmask)
			SetShowNoninter(i);
		}
		break;
	    case 'd':
		InitMixer(1);
		break;
	    case 'S':		/* save to file */
		SaveSettings();
		break;
	    case 'L':		/* load from file */
		LoadSettings();
		break;
	    case 'I':
		interactive = TRUE;
		break;
	    default:
		Usage();
	    }
    }
    if (!interactive)
	exit(0);
#ifdef NCURSES
    initscr();
    leaveok(stdscr, TRUE);
    keypad(stdscr, TRUE);
    cbreak();
    noecho();
    InitColors();
    InitScreen();
#endif
    conn.eventMask = ~0;
    conn.defaultMask = GPM_MOVE | GPM_HARD;
    conn.maxMod = ~0;
    conn.minMod = 0;
    i = Gpm_Open(&conn, 0);
    gpm_handler = MouseHandler;
#ifdef NCURSES
    Inter();
#endif
    Gpm_Close();
    close(mixer_fd);
#ifdef NCURSES
    CloseScreen();
    if (interactive) {
	if (setfile_opened)
	    fclose(setfile);
    }
#endif
    return 0;
}

void LoadSettings(void)
{
/* read settings from file */
    char tmpstring[80], recplay;
    int tmp, i, j = TRUE, left, right;
    short try_etc = FALSE;
    if (getenv("HOME")) {
	sprintf(setfile_name, "%s/.aumixrc", (char *) getenv("HOME"));
    } else {
	try_etc = TRUE;
    }
    if (!try_etc)
	setfile = fopen(setfile_name, "r");
    if (try_etc || !setfile) {
/* couldn't open ~/.aumixrc, so try /etc/aumixrc */
	sprintf(setfile_name, SYSTEM_AUMIXRC);
	setfile = fopen(setfile_name, "r");
    }
    if (!setfile) {
#ifdef NCURSES
	CloseScreen();
#endif
	perror("aumix: could not open $HOME/.aumixrc or /etc/aumixrc");
	exit(1);
    }
    for (j = TRUE; (!(j == EOF) && !(j == 0)); j = fscanf(setfile, "%[^:]:%3u:%3u:%1c\n", tmpstring, &left, &right, &recplay)) {
	for (i = 0; i < SOUND_MIXER_NRDEVICES; i++) {
/* cycle through device names, looking for matches */
	    if (!strcmp(tmpstring, dev_name[i])) {
		left = (left > 100) ? 100 : left;
		left = (left < 0) ? 0 : left;
		right = (right > 100) ? 100 : right;
		right = (right < 0) ? 0 : right;
		tmp = left + (right << 8);
		if ((ioctl(mixer_fd, MIXER_WRITE(i), &tmp) == -1) && !interactive) {
		    printf("%s not set\n", dev_name[i]);
		} else {
		    if (!interactive)
			printf("%s set to %i, %i", dev_name[i], left, right);
		    if ((1 << i) & recmask) {
			recsrc = (recplay == 'R') ? recsrc | (1 << i) : recsrc &
			    ~(1 << i);
			if (ioctl(mixer_fd, SOUND_MIXER_WRITE_RECSRC, &recsrc) == -1) {
			    perror("SOUND_MIXER_WRITE_RECSRC");
			    exit(-1);
			} else {
			    if (!interactive)
				printf(", %c", recplay);
			}
		    }
		    if (!interactive)
			printf("\n");
		}
	    }
	}
    }
    fclose(setfile);
#ifdef NCURSES
    if (interactive)
	RefreshAllSettings();
#endif
}

void SaveSettings(void)
{
/* save to file */
    int tmp, i;
    if (getenv("HOME")) {
	sprintf(setfile_name, "%s/.aumixrc", (char *) getenv("HOME"));
    } else {
#ifdef NCURSES
	CloseScreen();
#endif
	printf("aumix: problem opening ~/.aumixrc: $HOME not set\n");
	exit(1);
    }
    setfile = fopen(setfile_name, "w");
    if (!setfile) {
#ifdef NCURSES
	CloseScreen();
#endif
	perror("aumix: error opening .aumixrc");
	exit(1);
    }
    if (ioctl(mixer_fd, SOUND_MIXER_READ_RECSRC, &recsrc) == -1) {
	perror("SOUND_MIXER_READ_RECSRC");
#ifdef NCURSES
	CloseScreen();
#endif
	fclose(setfile);
	exit(-1);
    }
    for (i = 0; i < SOUND_MIXER_NRDEVICES; i++) {
	if ((1 << i) & devmask) {
	    if (ioctl(mixer_fd, MIXER_READ(i), &tmp) == -1) {
		perror("MIXER_READ");
		exit(-1);
	    }
	    fprintf(setfile, "%s:%i:%i:%c\n", dev_name[i], (tmp & 0xFF), ((tmp >> 8) & 0xFF), ((1 << i) & recsrc ? 'R' : 'P'));
	}
    }
    fclose(setfile);
}

void InitMixer(int minor)
{
    if (minor) {
	if ((mixer_fd = open("/dev/mixer1", O_RDWR)) < 0) {
	    fprintf(stderr, "aumix: error opening /dev/mixer1.\n");
	    exit(1);
	}
    } else {
	if ((mixer_fd = open("/dev/mixer", O_RDWR)) < 0) {
	    fprintf(stderr, "aumix: error opening /dev/mixer.\n");
	    exit(1);
	}
    }
    if (ioctl(mixer_fd, SOUND_MIXER_READ_DEVMASK, &devmask) == -1) {
	perror("SOUND_MIXER_READ_DEVMASK");
	exit(-1);
    }
    if (ioctl(mixer_fd, SOUND_MIXER_READ_RECMASK, &recmask) == -1) {
	perror("SOUND_MIXER_READ_RECMASK");
	exit(-1);
    }
    if (ioctl(mixer_fd, SOUND_MIXER_READ_RECSRC, &recsrc) == -1) {
	perror("SOUND_MIXER_READ_RECSRC");
	exit(-1);
    }
    if (ioctl(mixer_fd, SOUND_MIXER_READ_STEREODEVS, &stereodevs) == -1) {
	perror("SOUND_MIXER_READ_STEREODEVS");
	exit(-1);
    }
    if (!devmask) {
	fprintf(stderr, "No device found.\n");
	exit(-1);
    }
}

void SetShowNoninter(int dev)
{
/* change or display settings from the command line */
    char *devstr;
    int tmp;
    if ((*optarg == 'q') || (*optarg == ' ')) {
	devstr = dev_name[dev];
	if (ioctl(mixer_fd, MIXER_READ(dev), &tmp) == -1) {
	    perror("MIXER_READ");
	    exit(-1);
	}
	printf("%s %i, %i", dev_name[dev], (tmp & 0xFF), ((tmp >> 8) & 0xFF));
	if ((1 << (dev)) & recmask) {
	    if (ioctl(mixer_fd, SOUND_MIXER_READ_RECSRC, &recsrc) == -1) {
		perror("SOUND_MIXER_READ_RECSRC");
		exit(-1);
	    }
	    printf(", %c", ((1 << dev) & recsrc ? 'R' : 'P'));
	}
	printf("\n");
	return;
    } else {
	tmp = atoi(optarg);
	tmp = (tmp > 100) ? 100 : tmp;
	tmp = (tmp < 0) ? 0 : 257 * tmp;
	if (ioctl(mixer_fd, MIXER_WRITE(dev), &tmp) == -1) {
	    perror("MIXER_WRITE");
	    exit(-1);
	}
	return;
    }
}

#ifdef NCURSES
void InitScreen()
{
    int i;
    w_panel = newwin(LINES - 1, COLS, 0, 0);
    wclear(w_panel);
    wattrset(w_panel, COLOR_PAIR(TITLE_COLOR));
    wmove(w_panel, 2, 1);
    waddstr(w_panel, "uit   ");
    wmove(w_panel, 3, 1);
    waddstr(w_panel, "oad   ");
    wmove(w_panel, 4, 1);
    waddstr(w_panel, "ave   ");
    wmove(w_panel, 5, 1);
    waddstr(w_panel, "eys   ");
    wmove(w_panel, 6, 1);
    waddstr(w_panel, "bout  ");
    wmove(w_panel, 7, 1);
    waddstr(w_panel, "efresh");
    wmove(w_panel, 8, 1);
    waddstr(w_panel, "ute   ");
    wattrset(w_panel, COLOR_PAIR(TITLE_COLOR) | A_UNDERLINE);
    wmove(w_panel, 0, 0);
    waddstr(w_panel, " aumix ");
    wattrset(w_panel, COLOR_PAIR(TITLE_COLOR2) | A_BOLD);
    wmove(w_panel, 2, 0);
    waddstr(w_panel, "Q");
    wmove(w_panel, 3, 0);
    waddstr(w_panel, "L");
    wmove(w_panel, 4, 0);
    waddstr(w_panel, "S");
    wmove(w_panel, 5, 0);
    waddstr(w_panel, "K");
    wmove(w_panel, 6, 0);
    waddstr(w_panel, "A");
    wmove(w_panel, 7, 0);
    waddstr(w_panel, "R");
    wmove(w_panel, 8, 0);
    waddstr(w_panel, "M");
    wattrset(w_panel, COLOR_PAIR(INACTIVE_COLOR));
    if (YOFFSET + SOUND_MIXER_NRDEVICES <= LINES) {
	wmove(w_panel, YOFFSET + SOUND_MIXER_NRDEVICES, XOFFSET + 1);
	waddstr(w_panel, "0         Level        100          L        Balance         R");
    }
    for (i = 0; i < SOUND_MIXER_NRDEVICES; i++) {
	if ((1 << i) & devmask) {
	    wattrset(w_panel, COLOR_PAIR(INACTIVE_COLOR) | A_BOLD);
	} else
	    wattrset(w_panel, COLOR_PAIR(INACTIVE_COLOR) | A_DIM);
	/* draw control labels */
	wmove(w_panel, YOFFSET + i, XOFFSET + 28);
	waddstr(w_panel, dev_label[i]);
    }
    refresh();
    wrefresh(w_panel);
    RefreshAllSettings();
}

void AboutBox(void)
{
    int key, temp;
    w_about = newwin(LINES - 2, COLS, 0, 0);
    wclear(w_about);
    wattrset(w_about, COLOR_PAIR(INACTIVE_COLOR));
    wmove(w_about, 2, 0);
    wprintw(w_about, "aumix version %s\n", AUMIX_VERSION);
    wprintw(w_about, "compiled for sound driver version %i\n", SOUND_VERSION);
#ifdef OSS_GETVERSION
    wprintw(w_about, "running under sound driver version %i\n", ioctl(mixer_fd, OSS_GETVERSION, temp));
#else
    wprintw(w_about, "cannot determine version of running sound driver\n");
#endif
    wprintw(w_about, "color%sdetected\n", (has_colors())? " " : " not ");
    wprintw(w_about, "\nPress a key to resume.");
    wrefresh(w_about);
    key = Gpm_Getch();
    InitScreen();
}

void KeysBox(void)
{
    int key;
    w_keys = newwin(LINES, COLS, 0, 0);
    wclear(w_keys);
    wattrset(w_keys, COLOR_PAIR(TITLE_COLOR2));
    wmove(w_keys, 2, 0);
    waddstr(w_keys, "Page Arrows   \n");
    waddstr(w_keys, "Tab Enter < > \n");
    waddstr(w_keys, "+ - [ ] Arrows\n");
    waddstr(w_keys, "Space         \n");
    wattrset(w_keys, COLOR_PAIR(INACTIVE_COLOR));
    wprintw(w_keys, "\nPress a key to resume.");
    wmove(w_keys, 1, 0);
    wprintw(w_keys, "Key            Function");
    wattrset(w_keys, COLOR_PAIR(TITLE_COLOR) | A_BOLD);
    wmove(w_keys, 2, 15);
    waddstr(w_keys, "Device   \n");
    wmove(w_keys, 3, 15);
    waddstr(w_keys, "Level/Bal\n");
    wmove(w_keys, 4, 15);
    waddstr(w_keys, "Adjust   \n");
    wmove(w_keys, 5, 15);
    waddstr(w_keys, "Rec/Play \n");
    wrefresh(w_keys);
    key = Gpm_Getch();
    InitScreen();
}

void RefreshAllSettings(void)
{
    int i;
    for (i = 0; i < SOUND_MIXER_NRDEVICES; i++) {
	if ((1 << i) & devmask) {
	    EraseLevel(i);
	    RedrawBalance(i);
	    wattrset(w_panel, COLOR_PAIR(HANDLE_COLOR));
	    DrawHandles(i);
	    /* print record/play indicators */
	    if ((1 << (i)) & recmask)
		DrawRP(i);
	}
    }
    wrefresh(w_panel);
}

void ToggleMute(void)
{
    int dev;
    if (mutestate) {
/* restore from array */
	for (dev = 0; dev < SOUND_MIXER_NRDEVICES; dev++) {
	    if (ioctl(mixer_fd, MIXER_WRITE(dev), &level[dev]) == -1) {
		perror("MIXER_WRITE");
		exit(-1);
	    }
	}
    } else {
/* store levels in array, set them to zero */
	for (dev = 0; dev < SOUND_MIXER_NRDEVICES; dev++) {
	    if ((1 << dev) & devmask) {
		if (ioctl(mixer_fd, MIXER_READ(dev), &level[dev]) == -1) {
		    perror("MIXER_READ");
		    exit(-1);
		}
		if (ioctl(mixer_fd, MIXER_WRITE(dev), 0) == -1) {
		    perror("MIXER_WRITE");
		    exit(-1);
		}
	    }
	}
    }
    mutestate = !mutestate;
    RefreshAllSettings();
    wattrset(w_panel, COLOR_PAIR(INACTIVE_COLOR));
    wmove(w_panel, 1, 1);
    wprintw(w_panel, mutestate ? "muted" : "     ");
    wrefresh(w_panel);
}

void InitColors(void)
{
    start_color();
    init_pair(TITLE_COLOR, COLOR_CYAN, COLOR_BLUE);
    init_pair(TITLE_COLOR2, COLOR_RED, COLOR_BLUE);
    init_pair(HANDLE_COLOR, COLOR_CYAN, COLOR_BLUE);
    init_pair(TRACK_COLOR, COLOR_BLUE, COLOR_BLACK);
    init_pair(REC_LED_COLOR, COLOR_RED, COLOR_BLACK);
    init_pair(PLAY_LED_COLOR, COLOR_GREEN, COLOR_BLACK);
    init_pair(ACTIVE_COLOR, COLOR_YELLOW, COLOR_RED);
    init_pair(INACTIVE_COLOR, COLOR_WHITE, COLOR_BLACK);
}

void DrawHandles(int dev)
{
    int temp;
    if (ioctl(mixer_fd, MIXER_READ(dev), &temp) == -1) {
	CloseScreen();
	perror("MIXER_READ");
	exit(-1);
    }
    EraseLevel(dev);
    wattrset(w_panel, COLOR_PAIR(HANDLE_COLOR) | ((has_colors())? A_BOLD : A_REVERSE));
    wmove(w_panel, YOFFSET + dev, XOFFSET + 1 + ((temp & 0x7F) + 2) / 4);
    waddch(w_panel, 'O');
    wmove(w_panel, YOFFSET + dev, XOFFSET + 1 + ((temp & 0x7F) + 2) / 4);
}

void Inter(void)
{
    int key, incr, dir, dev, levelbalmode;
/* find first existing device */
    for (dev = 0; !(devmask & (1 << dev)); dev++);
/* highlight device label */
    wattrset(w_panel, COLOR_PAIR(ACTIVE_COLOR) | ((has_colors())? A_BOLD : A_REVERSE));
    wmove(w_panel, YOFFSET + dev, XOFFSET + 28);
    waddstr(w_panel, dev_label[dev]);
    DrawHandles(dev);
    DrawLevelBalMode(dev, 0);
    wrefresh(w_panel);
    levelbalmode = 0;
    while ((key = Gpm_Getch()) != EOF) {
	incr = 0;
	dir = 0;
	switch (key) {
	case '.':		/* no need to shift */
	case ',':
	case '<':
	case '>':
	case 9:		/* tab */
	case 10:		/* enter */
	    levelbalmode = !levelbalmode;
	    wattrset(w_panel, COLOR_PAIR(INACTIVE_COLOR));
	    wmove(w_panel, YOFFSET + dev, XOFFSET + 27);
	    waddstr(w_panel, "       ");
	    wattrset(w_panel, COLOR_PAIR(ACTIVE_COLOR) | ((has_colors())? A_BOLD : A_REVERSE));
	    wmove(w_panel, YOFFSET + dev, XOFFSET + 28);
	    waddstr(w_panel, dev_label[dev]);
	    DrawLevelBalMode(dev, levelbalmode);
	    break;
	case 27:		/* Esc */
	case 4:		/* Ctrl-D */
	case 'q':
	case 'Q':
	    return;
	    dir = -1;
	case KEY_UP:
	case KEY_PPAGE:	/* PgUp */
	    dir = -1;
	case KEY_DOWN:
	case KEY_NPAGE:	/* PgDn */
	    if (!dir)		/* if not PgUp or < pressed */
		dir = 1;
	    /* un-highlight current label */
	    DrawHandles(dev);
	    wattrset(w_panel, COLOR_PAIR(INACTIVE_COLOR));
	    wmove(w_panel, YOFFSET + dev, XOFFSET + 27);
	    waddstr(w_panel, "       ");
	    wattrset(w_panel, COLOR_PAIR(INACTIVE_COLOR) | A_BOLD);
	    wmove(w_panel, YOFFSET + dev, XOFFSET + 28);
	    waddstr(w_panel, dev_label[dev]);
/* switch to next existing device */
	    do {
		if (dir == 1) {
		    dev++;
		    if (dev > SOUND_MIXER_NRDEVICES - 1)
			dev = 0;
		} else {
		    dev--;
		    if (dev < 0)
			dev = SOUND_MIXER_NRDEVICES - 1;
		}
	    }
	    while (!((1 << dev) & devmask));
	    /* highlight new device label */
	    wattrset(w_panel, COLOR_PAIR(ACTIVE_COLOR) | ((has_colors())? A_BOLD : A_REVERSE));
	    wmove(w_panel, YOFFSET + dev, XOFFSET + 28);
	    waddstr(w_panel, dev_label[dev]);
	    DrawHandles(dev);
	    DrawLevelBalMode(dev, levelbalmode);
	    break;
	case ' ':
	    SwitchRP(dev);
	    break;
	case '[':
	    if ((levelbalmode) && ((1 << dev) & stereodevs))
		AdjustBalance(dev, 0, 0);
	    else
		AdjustLevel(dev, 0, 0);
	    break;
	case ']':
	    if ((levelbalmode) && ((1 << dev) & stereodevs))
		AdjustBalance(dev, 0, 100);
	    else
		AdjustLevel(dev, 0, 100);
	    break;
	case '+':
	case KEY_RIGHT:
	    incr = INCREMENT;
	case '-':
	case KEY_LEFT:
	    if (!incr)		/* '+'/up/right not pressed */
		incr = -INCREMENT;
	    if ((levelbalmode) && ((1 << dev) & stereodevs))
		AdjustBalance(dev, incr, -1);
	    else
		AdjustLevel(dev, incr, -1);
	    break;
	case 'K':
	case 'k':
	    KeysBox();
	    break;
	case 'L':
	case 'l':
	    LoadSettings();
	    break;
	case 'R':
	case 'r':
	    RefreshAllSettings();
	    break;
	case 'S':
	case 's':
	    SaveSettings();
	    break;
	case 'A':
	case 'a':
	    AboutBox();
	    break;
	case 'M':
	case 'm':
	    ToggleMute();
	    break;
	}
	wrefresh(w_panel);
    }
}

void DrawLevelBalMode(int dev, int mode)
/* arrow to show whether keyboard commands will adjust level or balance */
{
    wattrset(w_panel, COLOR_PAIR(ACTIVE_COLOR) | ((has_colors())? A_BOLD : A_REVERSE));
    if ((mode) && ((1 << dev) & stereodevs)) {
	wmove(w_panel, YOFFSET + dev, XOFFSET + 33);
	waddch(w_panel, '>');
	wmove(w_panel, YOFFSET + dev, XOFFSET + 33);
    } else {
	wmove(w_panel, YOFFSET + dev, XOFFSET + 27);
	waddch(w_panel, '<');
	wmove(w_panel, YOFFSET + dev, XOFFSET + 27);
    }
}

void AdjustLevel(int dev, int incr, int setlevel)
/*
 *  dev: device to adjust
 *  incr: signed percentage to increase (decrease) level, ignored unless
 *      setlevel = -1
 *  setlevel: level to set directly, or -1 to increment
 */
{
    int balset, max, left, right, temp, left_orig, right_orig;
    if (!((1 << dev) & devmask) || (dev > SOUND_MIXER_NRDEVICES - 1) || (dev < 0))
	return;
    if (ioctl(mixer_fd, MIXER_READ(dev), &temp) == -1) {
	CloseScreen();
	perror("MIXER_READ");
	exit(-1);
    }
    left = temp & 0x7F;
    right = (temp >> 8) & 0x7F;
    left_orig = left;
    right_orig = right;
    max = (left > right) ? left : right;
    if (max) {
	balset = (left > right) ? 50 * right / max : 100 - (50 * left / max);
    } else {
	balset = 50;
    }
    if ((1 << dev) & devmask) {
	if (dev > SOUND_MIXER_NRDEVICES - 1) {
	    dev = SOUND_MIXER_NRDEVICES - 1;
	} else {
	    dev = (dev < 0) ? 0 : dev;
	}
	max += incr / 2;
	if (balset > 49) {
	    left = max * (100 - balset) / 50;
	    right = max;
	} else {
	    right = max * balset / 50;
	    left = max;
	}
	left = (left > 100) ? 100 : left;
	right = (right > 100) ? 100 : right;
	left = (left < 0) ? 0 : left;
	right = (right < 0) ? 0 : right;
	if ((left == left_orig) && (right == right_orig)) {
	    left += incr / 2;
	    right += incr / 2;
	}
    }
    left = (left > 100) ? 100 : left;
    right = (right > 100) ? 100 : right;
    left = (left < 0) ? 0 : left;
    right = (right < 0) ? 0 : right;
    left = (setlevel > -1) ? setlevel : left;
    right = (setlevel > -1) ? setlevel : right;
    temp = (right << 8) | left;
    if (ioctl(mixer_fd, MIXER_WRITE(dev), &temp) == -1) {
	CloseScreen();
	perror("MIXER_WRITE");
	exit(-1);
    }
    EraseLevel(dev);
    /* Draw handle at new position */
    RedrawBalance(dev);
    wattrset(w_panel, COLOR_PAIR(HANDLE_COLOR) | ((has_colors())? A_BOLD : A_REVERSE));
    wmove(w_panel, YOFFSET + dev, XOFFSET + 1 + ((((temp & 0x7F) + ((temp >> 8) & 0x7F)) / 2) + 2) / 4);
    waddch(w_panel, 'O');
    wmove(w_panel, YOFFSET + dev, XOFFSET + 1 + ((((temp & 0x7F) + ((temp >> 8) & 0x7F)) / 2) + 2) / 4);
    wrefresh(w_panel);
}

void EraseLevel(int dev)
{
/* redraw level track */
    int i;
    wattrset(w_panel, COLOR_PAIR(TRACK_COLOR));
    for (i = 0; i < 26; i++) {
	wmove(w_panel, YOFFSET + dev, XOFFSET + 1 + i);
	waddch(w_panel, '+');
    }
}
void AdjustBalance(int dev, int incr, int setabs)
/*
   *  dev: device to adjust
   *  incr: signed percentage to change balance
   *  setabs: absolute balance setting, or -1 to increment only
   *  Balance settings go from 0 to 100.
   *  At setting of 0, right amplitude is 0.
 */
{
    int balset, max, left, right, temp, left_orig, right_orig;
    if (((1 << dev) & devmask) & (dev > SOUND_MIXER_NRDEVICES - 1))
	dev = SOUND_MIXER_NRDEVICES - 1;
    if (ioctl(mixer_fd, MIXER_READ(dev), &temp) == -1) {
	CloseScreen();
	perror("MIXER_READ");
	exit(-1);
    }
    left = temp & 0x7F;
    right = (temp >> 8) & 0x7F;
    left_orig = left;
    right_orig = right;
    max = (left > right) ? left : right;
    if (max) {
	balset = (left > right) ? 50 * right / max : 100 - (50 * left / max);
    } else {
	balset = 50;
    }
    balset = (setabs == -1) ? balset + incr : setabs;
    balset = (balset > 100) ? 100 : balset;
    balset = (balset < 0) ? 0 : balset;
    if (balset > 49) {
	left = max * (100 - balset) / 50;
	right = max;
    } else {
	right = max * balset / 50;
	left = max;
    }

/* hack to keep controls from getting stuck */
    if (right_orig == right && left_orig == left && setabs == -1) {
	if (incr > 0 && balset < 50) {
	    left--;
	    right++;
	    left = (left < 0) ? 0 : left;
	    right = (right > 100) ? 100 : right;
	}
	if (incr < 0 && balset > 50) {
	    left++;
	    right--;
	    left = (left > 100) ? 100 : left;
	    right = (right < 0) ? 0 : right;
	}
    }
    temp = (right << 8) | left;
    if (ioctl(mixer_fd, MIXER_WRITE(dev), &temp) == -1) {
	CloseScreen();
	perror("MIXER_WRITE");
	exit(-1);
    }
    /* draw handle at new position */
    RedrawBalance(dev);
}

void RedrawBalance(int dev)
/* redraw balance track */
{
    int left, right, max, temp, balset;
    if ((1 << dev) & stereodevs) {
	wattrset(w_panel, COLOR_PAIR(TRACK_COLOR));
	for (temp = 0; temp < 26; temp++) {
	    wmove(w_panel, YOFFSET + dev, XOFFSET + 37 + temp);
	    waddch(w_panel, '+');
	}
	if (ioctl(mixer_fd, MIXER_READ(dev), &temp) == -1) {
	    CloseScreen();
	    perror("MIXER_READ");
	    exit(-1);
	}
	left = temp & 0x7F;
	right = (temp >> 8) & 0x7F;
	max = (left > right) ? left : right;
	if (temp) {
	    balset = (left > right) ? 50 * right / max : 100 - (50 * left / max);
	} else {
	    balset = 50;
	}
	wattrset(w_panel, COLOR_PAIR(HANDLE_COLOR) | ((has_colors())? A_BOLD : A_REVERSE));
	wmove(w_panel, YOFFSET + dev, XOFFSET + 37 + balset / 4);
	waddch(w_panel, 'O');
	wmove(w_panel, YOFFSET + dev, XOFFSET + 37 + balset / 4);
	wrefresh(w_panel);
    }
}

void SwitchRP(int dev)
{
    /* change record/play and print indicators */
    if ((1 << dev) & recmask) {
	if (ioctl(mixer_fd, SOUND_MIXER_READ_RECSRC, &recsrc) == -1) {
	    perror("SOUND_MIXER_READ_RECSRC");
	    exit(-1);
	}
	if (recsrc & (1 << dev))
	    recsrc &= ~(1 << dev);	/* turn off recording */
	else
	    recsrc |= (1 << dev);	/* turn on recording */
	if (ioctl(mixer_fd, SOUND_MIXER_WRITE_RECSRC, &recsrc) == -1) {
	    CloseScreen();
	    perror("SOUND_MIXER_WRITE_RECSRC");
	    exit(-1);
	}
	if (ioctl(mixer_fd, SOUND_MIXER_READ_RECSRC, &recsrc) == -1) {
	    perror("SOUND_MIXER_READ_RECSRC");
	    exit(-1);
	}
	DrawRP(dev);
    }
}

void DrawRP(int dev)
{
    wattrset(w_panel,
	     (1 << dev) & recsrc ? COLOR_PAIR(REC_LED_COLOR) | ((has_colors())? A_NORMAL : A_REVERSE)
	     : COLOR_PAIR(PLAY_LED_COLOR)
	);
    wmove(w_panel, YOFFSET + dev, XOFFSET + 0);
    waddch(w_panel, ((1 << dev) & recsrc ? 'R' : 'P'));
    wmove(w_panel, YOFFSET + dev, XOFFSET + 0);
    wrefresh(w_panel);
}

void CloseScreen(void)
{
    if (interactive) {
	clear();
	refresh();
	endwin();
    }
}
#endif				/* end of ncurses stuff */
int MouseHandler(Gpm_Event * event, void *data)
{
    int dev, temp;
/* quit */
    if ((event->y == 3) & (event->x < 8) && (event->type & GPM_DOWN)) {
	Gpm_Close();
	close(mixer_fd);
	CloseScreen();
	exit(0);
    }
/* load from file */
    if ((event->y == 4) & (event->x < 8) && (event->type & GPM_DOWN)) {
	LoadSettings();
	return 0;
    }
/* save to file */
    if ((event->y == 5) & (event->x < 8) && (event->type & GPM_DOWN)) {
	SaveSettings();
	return 0;
    }
/* help on keystrokes */
    if ((event->y == 6) & (event->x < 8) && (event->type & GPM_DOWN)) {
	KeysBox();
	return 0;
    }
/* about */
    if ((event->y == 7) & (event->x < 8) && (event->type & GPM_DOWN)) {
	AboutBox();
	return 0;
    }
/* refresh */
    if ((event->y == 8) & (event->x < 8) && (event->type & GPM_DOWN)) {
	RefreshAllSettings();
	return 0;
    }
/* mute */
    if ((event->y == 9) & (event->x < 8) && (event->type & GPM_DOWN)) {
	ToggleMute();
	return 0;
    }
    dev = event->y - 1 - YOFFSET;
    temp = event->x - 2 - XOFFSET;
    if ((temp > 35) & (temp < 62)) {
	temp -= 36;
	temp *= 4;
	if (((event->type & GPM_DOWN) || (event->type & GPM_DRAG)) && ((1 << dev) & stereodevs)) {
	    AdjustBalance(dev, 0, temp);
	    return 0;
	}
	return 0;
    }
    if ((event->type & GPM_DOWN) && (temp == -1)) {
	SwitchRP(dev);
	return 0;
    }
    if (temp < 26) {
	temp *= 4;
	if ((event->type & GPM_DOWN) || (event->type & GPM_DRAG)) {
	    AdjustLevel(dev, 0, temp);
	}
    }
    return 0;
}

void Usage(void)
{
    printf("aumix %s usage: aumix [[ -<device option>[ ]<level>|q[uery],\n", AUMIX_VERSION);
    fputs(("\
[ -<device option>[ ]<level>|q[uery], [...]][-d] [-h] [-I{d|c}] [-L] [-q] [-S]\n\n\
device options:\n\
  v: main volume           x: mix monitor\n\
  b: bass                  W: PCM2\n\
  t: treble                r: record\n\
  s: synth                 i: input gain\n\
  w: PCM                   o: output gain\n\
  p: PC speaker            1: line1\n\
  l: line                  2: line2\n\
  m: mic                   3: line3\n\
  c: CD\n\n\
other options:\n\
   d:  dual mixers; adjust secondary\n\
   h:  this helpful message\n\
   I:  start in interactive mode after doing non-interactive functions\n\
   L:  load settings\n\
   q:  query all devices and print their settings\n\
   S:  save settings to ~/.aumixrc\n"), stdout);
    exit(1);
}
