// channelview.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include <InterViews/box.h>		/* FIX ME: put all box stuff in base class */
#include <InterViews/perspective.h>
#include "block.h"
#include "channelview.h"
#include "channelgraph.h"
#include "data.h"
#include "range.h"
#include "scale.h"
#include "viewchanger.h"
#include <X11/keysym.h>

class ChannelRangeBlock : public Block {
	Range range;
public:
	ChannelRangeBlock(const Range &r) : range(r) {}
	redefined void doIt(Interactor *i) {
		ChannelGraph *c = (ChannelGraph *) i;
		if(range.includes(c->currentChannel()))
			doThis(i);
	}
	virtual void doThis(Interactor *i) = 0;
};

//********

// some private inline methods

inline int
ChannelView::lowestChannel() {
	return channelsShown->cury;
}

inline void
ChannelView::setLowestChannel(int chan) {
	channelsShown->cury = chan;
}

inline int
ChannelView::highestChannel() {
	return channelsShown->cury + channelsShown->curheight - 1;
}

inline int
ChannelView::nchans() {
	return channelsShown->curheight;
}

inline void
ChannelView::setNChans(int chans) {
	channelsShown->curheight = chans;
}

inline int
ChannelView::maxChans() { return dataptr->channels(); }

ChannelView::ChannelView(Controller *c, ViewInfo &info) : DataView(c, info) {
	Init(info);
}

ChannelView::~ChannelView() {
	Resource::unref(channelsShown);
}

void
ChannelView::Init(ViewInfo &info) {
	BUG("ChannelView::Init()");
	SetClassName("ChannelView");
	initPerspective();
	// initialize vertical (channel) perspective
	channelsShown = new Perspective;
	channelsShown->Init(0, 0, 0, maxChans());
	setNChans(0);
	Range chanrange = (!info.channelRange.isNegative()) ?
		info.channelRange : dataptr->channelRange();
	setChannelRange(chanrange);
	if(!info.frameRange.isNegative())
		setVisibleFrameRange(info.frameRange);
}

void
ChannelView::initPerspective() {
	BUG("ChannelView::initPerspective()");
	int width = dataptr->length();
	static const int fixedHeight = 1048576;	// arbitrary large power of 2
	perspective->Init(0, 0, width, fixedHeight);
	perspective->curx = 0;
	perspective->cury = 0;
	const char *a = GetAttribute(plotWidthAttribute());
	if (a != nil)
		perspective->curwidth = min(atoi(a), perspective->width);
	else perspective->curwidth = width;
	perspective->curheight = fixedHeight;
	perspective->lx = perspective->curwidth;
	perspective->ly = 0;
	perspective->sx = perspective->lx/4;	// scroll incr. is 1/4 screen
	perspective->sy = 0;
}

void
ChannelView::checkPerspective() {
	Perspective np = *perspective;
	// the length of the data is all that could have changed
	np.width = dataptr->length();
	*perspective = constrainView(np);
}

void
ChannelView::constrainVerticalView(Perspective& np) {
	register Perspective* p = perspective;
	np.height = p->height;
	np.cury = 0;
	np.curheight = max(np.curheight, 1);	// always
}

boolean
ChannelView::keyCommand(unsigned long sym) {
	boolean interested = true;
	switch (sym) {
	case XK_plus:
		addChannel();
		break;
	case XK_minus:
		removeChannel();
		break;
	case XK_numbersign:
		setHorizScaleTo(FrameUnit);
		break;
	case XK_at:
		setHorizScaleTo(FrameTimeUnit);
		break;
	case XK_exclam:
		setHorizScaleTo(FrameSmpteUnit);
		break;
	case XK_twosuperior:
		setScaleOptions();
		break;
	default:
		interested = DataView::keyCommand(sym);
		break;
	}
	return interested;
}

void
ChannelView::shiftChannelViewUp() {
	Range chans = getChannelRange();
	chans += dataptr->frameChannels();
	setChannelRange(chans);
}

void
ChannelView::shiftChannelViewDown() {
	Range chans = getChannelRange();
	chans += -(dataptr->frameChannels());
	setChannelRange(chans);
}

int
ChannelView::graphsVisible() {
	return nchans();
}

void
ChannelView::incrementGraphsVisible() {
	DataView::incrementGraphsVisible();
	setNChans(nchans() + 1); 
}

void
ChannelView::decrementGraphsVisible() {
	DataView::decrementGraphsVisible();
	setNChans(nchans() - 1); 
}

void
ChannelView::setChannelRange(const Range& chrange) {
	Range range = chrange;
//	1 <= number of chans shown <= maxChans
	int newnchans = min(range.size(), maxChans());
	int oldnchans = graphsVisible();
	int i;
	for(i = oldnchans; i > newnchans; i--)	// remove extras if any
		removeGraph(false);
	int newmax = max(0, min(maxChans() - 1, range.intMax()));
	int newmin = max(0, min(maxChans() - newnchans, range.intMin()));
	range.set(newmin, newmax);
	struct ChanSetter : public Block {
		int chan, maxchan;
		ChanSetter(int startingChan, int endingChan)
			: chan(startingChan), maxchan(endingChan) {}
		redefined void doIt(Interactor *i) {
			ChannelGraph *c = (ChannelGraph *) i;
			if(c->currentChannel() != chan) {
				c->setChannel(chan);
				Scale* scale = c->vScale();
				scale->setLabel(c->verticalScaleLabel());
				scale->Update();	// this should be done by the graph
			}
			if(++chan > maxchan)
				done = true;
		}
	} setter(newmin, newmax);
	doForGraphs(setter);
	setLowestChannel(range.intMin());
	for(i = graphsVisible(); i < newnchans; i++)	// add new if needed
		addChannel(lowestChannel() + i);
	if(newnchans != oldnchans)
		doChange();
}

Range
ChannelView::getChannelRange() {
	return Range(lowestChannel(), highestChannel());
}

void
ChannelView::addChannel() {
	int newchan = lowestChannel() + graphsVisible();
	if(newchan < maxChans()) {
		addGraph(new ChannelGraph(controller, dataptr, newchan));
		doChange();	// public version calls this internally
	}
}

// protected internal versions of this	(these do not do the update)

void
ChannelView::addChannel(int chan) {
	addGraph(new ChannelGraph(controller, dataptr, chan));
}

void
ChannelView::addChannel(int chan, Range &range) {
	addGraph(new ChannelGraph(controller, dataptr, chan, range));
}

void
ChannelView::addGraph(Graph *graph) {
	if(graphsVisible()) {
		ChannelGraph* model = (ChannelGraph *) topGraph();
		ChannelGraph* cgraph = (ChannelGraph *) graph;
		cgraph->useLines = model->useLines;
		// FIX ME:  encapsulate in Style class
	}
	DataView::addGraph(graph);
}

// remove one channel from frame -- always the highest channel displayed

void
ChannelView::removeChannel() {
	removeGraph(true);
}

Range
ChannelView::getVisibleFrameRange() {
	return Range(shown->curx, shown->curx + shown->curwidth - 1);
}

Range
ChannelView::getHorizScaleRange() {
	return hscale->getRange();
}

void
ChannelView::setVisibleFrameRange(const Range &range) {
	BUG("ChannelView::setVisibleFrameRange()");
	Perspective np = *perspective;
	np.curx = range.intMin();
	np.curwidth = range.size();
	Adjust(np);
}

void
ChannelView::toggleGraphMode() {
    struct PlotToggle : public Block {
		redefined void doIt( Interactor *i ) {
			ChannelGraph *g = (ChannelGraph *) i;
			g->togglePlotMode();
		}
    } toggle;
    doForGraphs( toggle );
}

void
ChannelView::setInsertPoint(int point, const Range& chans, boolean scroll) {
	Range chansToSelect = chans;
	unselect();
	Range framesVisible = getVisibleFrameRange();
	Range chansVisible = getChannelRange();
	if(!framesVisible.includes(point)) {
		int margin = int(framesVisible.spread() / 2);
		framesVisible.set(point - margin, point + margin);
		setVisibleFrameRange(framesVisible);
	}		
	if(!chansVisible.includes(chansToSelect) && scroll == true) {
		Range displayChans = chansToSelect;
		displayChans.expandBy(1);
		setChannelRange(displayChans);
	}
    struct InsertSetter : public ChannelRangeBlock {
		int iPoint;
		InsertSetter(const Range &range, int pt)
			: ChannelRangeBlock(range), iPoint(pt) {}
		redefined void doThis( Interactor *i ) {
			Graph *g = (Graph *) i;
			g->setInsertPoint(iPoint);
		}
    } setter(chansToSelect, point);
    doForGraphs( setter );
}

void
ChannelView::setEditRegion(const Range& region, const Range& chans, boolean scroll) {
	Range chansToSelect = chans;
	unselect();
	Range framesVisible = getVisibleFrameRange();
	Range chansVisible = getChannelRange();
	if(!framesVisible.includes(region)) {
		int width = int(framesVisible.spread());
		int newidth = int(region.spread());
		if(newidth > width) {	// widen framesVisible to include region
			framesVisible = region;
			framesVisible.expandBy(int(newidth/2));
		}
		else {		// shift visible range to include region
			double shift = region.intMin()-framesVisible.intMin() - (width-newidth)/2;
			framesVisible += int(shift);
		}
		setVisibleFrameRange(framesVisible);
	}
	if(!chansVisible.includes(chansToSelect) && scroll == true) {
		Range displayChans = chansToSelect;
		displayChans.expandBy(1);
		setChannelRange(displayChans);
	}
    struct RegionSetter : public ChannelRangeBlock {
		Range setregion;
		RegionSetter(const Range &range, const Range &region)
			: ChannelRangeBlock(range), setregion(region) {}
		redefined void doThis( Interactor *i ) {
			Graph *g = (Graph *) i;
			g->setEditRegion(setregion);
		}
    } setter(chansToSelect, region);
    doForGraphs( setter );
}

void
ChannelView::setChannelRange() {
	ChannelChannelRange changer(this);
	if(changer.configure(controller))
		changer.apply();
}

void
ChannelView::setScaleOptions() {
	ScaleOptionsSetter setter(this);
	if(setter.configure(controller))
		setter.apply();
}

#if !defined(sgi) || defined(__GNUG__)

void
ChannelView::resetHorizontalScale() {
	RangeUnit units = horizRangeUnits();
	if(units != FrameUnit) {	// if displaying time or SMPTE
		hscale->setRange(dataptr->frameRange(units));
		hscale->Update();
	}
}

#endif
