// crossfader.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/
#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "controller.h"
#include "crossfader.h"
#include "curvegen.h"
#include "data.h"
#include "editor.h"
#include "envelope.h"
#include "localdefs.h"
#include "request.h"
#include "requester.h"

class CombineRequester : public TitledRequester {
	friend Combiner;
protected:
	CombineRequester(const char* title, Combiner *c)
		: TitledRequester(title), client(c) {}
	redefined boolean confirmValues();
protected:
	Combiner* client;
};

boolean
CombineRequester::confirmValues() {
	boolean warning = false;
	char msg[128];
	Data* target = client->target();
	const Data* source = client->source();
	// warn user about format mismatches
	if(target->channels() != source->channels()) {
		sprintf(msg, "Warning:  source channels (%d) != target channels (%d)",
			source->channels(), target->channels());
		warning = true;
	}
	else if(target->sRate() != source->sRate()) {
		sprintf(msg, "Warning:  source samp rate (%d) != target samp rate (%d)",
			source->sRate(), target->sRate());
		warning = true;
	}
	return (!warning ||
		Application::confirm(msg, "Continue anyway?", nil, Cancel) == Yes);
}

//********

Combiner::Combiner(Data* target, Data* src)
		: DataModifier(target), theSource(src) {
	theSource->ref();
}

Combiner::~Combiner() {
	Resource::unref(theSource);
}

//********

Mixer::Mixer(Data* target, Data* source) : Combiner(target, source) {
	initialize();
}

Modifier *
Mixer::create(DataEditor*) {
	return nil;
}

Mixer::~Mixer() {}

int
Mixer::doApply(Data* data) {
	data->add(source());
	return true;
}

//********

Replacer::Replacer(Data* target, Data* source) : Combiner(target, source) {
	initialize();
}

Modifier *
Replacer::create(DataEditor*) {
	return nil;
}

Replacer::~Replacer() {}

int
Replacer::doApply(Data* data) {
	return data->replaceWith(source());
}

//********

Splicer::Splicer(Data* target, Data* source) : Combiner(target, source) {
	initialize();
}

Modifier *
Splicer::create(DataEditor*) {
	return nil;
}

Splicer::~Splicer() {}

int
Splicer::doApply(Data* data) {
	data->deferUpdate(true);			// dont do update twice
	int status = (data->spliceIn(source()->frameRange())
		&& data->replaceWith(source()));
	data->deferUpdate(false);
	return status;
}

//********

class CrossfadeRequester : public CombineRequester {
	friend Crossfader;
protected:
	CrossfadeRequester(Crossfader* c)
		: CombineRequester("Crossfade Selected Region:", c) {}
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	Crossfader* fader() { return (Crossfader *) client; }
};

void
CrossfadeRequester::configureRequest(Request* request) {
	request->appendValue("Amplitude Factor:", &fader()->gain);
	request->appendChoice("Envelope for Crossfade:",
			     "|Linear|Exponential|File on Disk|",
			     &fader()->envelopeType, true);
	request->appendChoice("Envelope Direction:", "|Forward|Reverse|",
			     &fader()->envelopeDirection, true);
}

boolean
CrossfadeRequester::confirmValues() {
	boolean status = CombineRequester::confirmValues();
	if(status) {
		Controller* newController = nil;
		if(fader()->envelopeType == Crossfader::OnDisk) {
#if 0
			if((newController = c->editor()->openNewFile(
					"Select Envelope for Crossfade:")) != nil) {
				Data* source = newController->model();
				source->ref();
				envelope = new Envelope(source->length());
				envelope->ref();
				envelope->copyFrom(source);
				Resource::unref(source);
				if(envelopeDirection == Reverse)
					envelope->reverse();
			}
			else
				status = false;
		}
#else
			Application::alert("On-disk envelope not implemented.");
			status = false;
		}
#endif
	}
	return status;
}

//********

Crossfader::Crossfader(Data* target, Data* source)
	: Combiner(target, source), envelope(nil), gain(1.0),
	  envelopeType(Linear), envelopeDirection(Forward) {}

Crossfader::Crossfader(Data* target, Data* source, Envelope *evp, double amp)
		: Combiner(target, source), envelope(evp), gain(amp),
		  envelopeType(OnDisk), envelopeDirection(Forward) {
	envelope->ref();
	initialize();
}

Modifier *
Crossfader::create(DataEditor*) {
	return nil;
}

Crossfader::~Crossfader() {
	Resource::unref(envelope);
}

Requester *
Crossfader::createRequester() {
	return new CrossfadeRequester(this);
}

int
Crossfader::doApply(Data* data) {
	data->crossfade(source(), envelope);
	return true;
}

void
Crossfader::initialize() {
	Super::initialize();
	switch(envelopeType) {
		case Linear:
			envelope = new Envelope;
			envelope->ref();
			{
			LinearCurveGenerator curve(envelope, 0.0, 1.0);
			curve.apply();
			}
			break;
		case Exponential:
			envelope = new Envelope;
			envelope->ref();
			{
			ExponentialCurveGenerator curve(envelope, 0.0, 1.0, 2.0);
			curve.apply();
			}
			break;
		case OnDisk:
			break;
	}
	if(envelope && (envelopeDirection == Reverse))
		envelope->reverse();
}
