// editor.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <InterViews/button.h>
#include <InterViews/world.h>
#include <X11/keysym.h>
#include "localdefs.h"
#include <sys/time.h>
#include <assert.h>
#include "datafile.h"
#include "application.h"
#include "header.h"
#include "editor.h"
#include "comment.h"
#include "controller.h"
#include "data.h"
#include "delay.h"
#include "filename.h"
#include "filecommand.h"
#include "sound.h"
#include "lpcdata.h"
#include "formantfilter.h"
#include "pchdata.h"
#include "fftdata.h"
#include "envelope.h"
#include "datamodifier.h"
#include "analysisfun.h"
#include "optionsetter.h"
#include "phraser.h"
#include "ellipfilt.h"
#include "fftfun.h"
#include "rescaler.h"
#include "interpolater.h"
#include "lowpassfilter.h"
#include "pulsegen.h"
#include "statusaction.h"
#include "textwindow.h"
#include "pvoceditor.h"
#include "soundeditor.h"
#include "curvegen.h"
#include "crossfader.h"
#include "commenteditor.h"
#include "soundheader.h"

const char *
scat(const char *s1, const char *s2) {
	static char str[1024];
	sprintf(str, "%s%s", s1, s2);
	return str;
}

const char *
scat(const char *s1, const char *s2, const char *s3) {
	static char str[1024];
	sprintf(str, "%s%s%s", s1, s2, s3);
	return str;
}

Data *
DataEditor::internalBuffer = nil;

DataEditor::C_Constructor 
DataEditor::ctor_table1[] = {
	SoundEditor::new_DataEditor1,
	LPCEditor::new_DataEditor1,
	PCHEditor::new_DataEditor1,
	FFTEditor::new_DataEditor1,
	EnvelopeEditor::new_DataEditor1,
	PvocEditor::new_DataEditor1
};

DataEditor::C_D_Constructor 
DataEditor::ctor_table2[] = {
	SoundEditor::new_DataEditor2,
	LPCEditor::new_DataEditor2,
	PCHEditor::new_DataEditor2,
	FFTEditor::new_DataEditor2,
	EnvelopeEditor::new_DataEditor2,
	PvocEditor::new_DataEditor2
};

DataEditor::DataEditor(Controller *c, const Data *d)
		: controller(c), data(nil), selection(nil) {
	init((Data *) d);
}

inline int typeToIndex(FileType type) {
	return powerOfTwoToLinearEnum(int(type)) - 2;
}

DataEditor *
DataEditor::create(Controller *c, const char* name) {
	int type_index = typeToIndex(FileName::fileType(name));
	static const int tableEnd = sizeof(ctor_table1) / sizeof(*ctor_table1);
	assert(0 < type_index && type_index < tableEnd);
	return (*DataEditor::ctor_table1[type_index])(c);
}

DataEditor *
DataEditor::create(Controller *c, const Data *d) {
	int type_index = typeToIndex(d->fileType());
	static const int tableEnd = sizeof(ctor_table2) / sizeof(*ctor_table2);
	assert(0 < type_index && type_index < tableEnd);
	return (*DataEditor::ctor_table2[type_index])(c, d);
}

void
DataEditor::init(Data* d) {
	reset();
	if(d)
		setData(d);
}

DataEditor::~DataEditor() {
	setCopyBuffer(nil);
	setSelection(nil);
	setData(nil);
}

void
DataEditor::setData(Data* d) {
	if(d != data) {
		Resource::unref(data);
		data = d;
		if(data != nil) {
			data->ref();
			location = data->frameRange();
			channels = data->channelRange();
		}
	}
}

Data *
DataEditor::createModel(DataFile *f) {
	boolean status = false;
	Data *d = newModel();
	if(!f->readable())
		Application::error(scat("Unable to read file ", f->name()));
	else {
		Application::inform(scat("Reading ", f->name(), "..."));
		Header *header = d->createHeader(f, true);
		header->ref();
		if(!(status = d->read(f, header))) {
			if(d->length() == 0)
				/* non-read error */ ;
			else if(Application::globalResourceIsTrue("ReadRawFiles")) {
				if(header->configure(controller)) {
					f->clear();
					status = d->read(f, header);
				}
			}
		}
		Resource::unref(header);
	}
	f->failif(status != true);		// calling routine checks file state
	return d;
}

const char *
DataEditor::defaultDir() {
	const char* sfd = getenv("SFDIR");
	const char *a = 
		Application::getGlobalResource(data->defaultDirAttribute());
	return a ? a : sfd ? sfd : ".";
}

boolean
DataEditor::readFile(DataFile* file) {
	boolean status = true;
	file->ref();
	setData(createModel(file));
	status = file->good();
	Resource::unref(file);
	return status;
}

void
DataEditor::setInsertPoint(int point, int chan) {
	if(selectionMade() || chan < 0) // for first call after setting
		reset();
	setLocation(Range(point, max(point, data->length() - 1)));
	addChannel(chan);
	setSelection(nil);
}

void
DataEditor::setEditRegion(const Range &region, int chan) {
	if(!selectionMade()) {	// for first call after unsetting
		reset();
		isSelected = true;
	}
	setLocation(region);
	addChannel(chan);
	setSelection(nil);
	// when new region set in any window, it becomes default target rather
	// than any previously copied internal buffer.  Not so for insert point.
	setCopyBuffer(nil);
}

void
DataEditor::setLocation(const Range& newloc) {
	setTimeStamp();
	if(newloc != location)
		location = newloc;
}

void
DataEditor::addChannel(int chan) {
	if(chan >= 0) {
		int first = channels.intMin();
		int last = channels.intMax();
		int nfirst = (first < 0 || chan < first) ? chan : first;
		setChannels(nfirst, chan > last ? chan : last);
	}
}

Range
DataEditor::currentChannels() {
	// for now, either just one, or all channels
	// set to all if none chosen (-1 is flag for this)
	if(nchans() > 1 || channels.includes(-1))
		setChannels(model()->channelRange());
	return channels;
}

void
DataEditor::setTimeStamp() {
	struct timeval tp;
	struct timezone tzp;
	gettimeofday(&tp,&tzp);
	timestamp = tp.tv_sec;
}

void
DataEditor::setSelection(Data *s) {
	if(selection != s) {
		Resource::unref(selection);
		if((selection = s) != nil)
			selection->ref();
	}
}

Status
DataEditor::setCopyBuffer(Data *b) {
	if(internalBuffer != b) {
		Resource::unref(internalBuffer);
		if((internalBuffer = b) != nil)
			internalBuffer->ref();
	}
	return internalBuffer != nil;
}

Data *
DataEditor::currentSelection() {
	if(selection == nil)
		setSelection(data->clone(currentRegion(), currentChannels()));
	return selection;
}

Data *
DataEditor::currentSource() {
	Data *source = (copyBuffer() != nil) ?
		copyBuffer() : controller->findSelection();
	if(source == nil)
		Application::alert(
			"You must select a source to use this operation", 
			"or copy a selection into the internal buffer."
		);
	else if(*source == *data) {
		// must make deep copy if source is same as target
		Data *copyOfSource = source->copyOf();
		Resource::unref(source);
		source = copyOfSource;
		source->ref();
	}
	else
		source->ref();
	return source;
}

void
DataEditor::freeSource(Data *sel) {
	if(sel != internalBuffer)	// only unref source if it was not internal
		Resource::unref(sel);
}

Data *
DataEditor::getSelection(long &timestamp) {
	timestamp = timeStamp();
	Data *sel = nil;
	if (selectionMade())
		sel = data->clone(currentRegion(), currentChannels());
	return sel;
}

// re-select edit region (after reverting to saved, for instance)
// unless that region is no longer within the data bounds

void
DataEditor::reselect() {
	if(selectionMade() && data->frameRange().includes(currentRegion()))
			setSelection(data->clone(currentRegion(), currentChannels()));
	else {
		controller->unselectView();	// do to reset even if no selection made
		setSelection(nil);
	}
}

// static method -- used by "static constructor" methods

boolean
DataEditor::applyModifierUsing(Modifier &mod, Controller* controller) {
	boolean status = false;
	if(mod.configure(controller))
		status = mod.apply();
	return status;
}

boolean
DataEditor::applyModifier(Modifier &mod) {
	return applyModifierUsing(mod, controller);
}

boolean
DataEditor::applyModifierToNew(Modifier &mod) {
	boolean status;
	if((status = applyModifier(mod)) == true)
		displayInternalBuffer();
	setCopyBuffer(nil);
	return status;
}

void
DataEditor::markEditRegion(int size) {
	controller->showEditRegion(
		Range(currentInsert(), currentInsert() + size - 1),
		currentChannels()
	);
}

// keysym is checked first by controller, then by view, then by editor
// subclass, and lastly by editor base class.  Checking stops when matched.

boolean
DataEditor::keyCommand(unsigned long sym) {
	boolean interested = true;
	switch (sym) {
	case XK_period:
		setDefaultDir();
		break;
	case XK_n:
		newFile();
		break;
	case XK_nobreakspace:
		newLPCFile();
		break;
	case XK_exclamdown:
		newSoundFile();
		break;
	case XK_cent:
		newEnvelopeFile();
		break;
	case XK_currency:
		newPvocFile();
		break;
	case XK_o:
		openNewFile();
		break;
	case XK_s:
		saveFile();
		break;
	case XK_S:
		saveToFile();
		break;
	case XK_U:
		revertToSaved();
		break;
	case XK_quotedbl:
		changeComment();
		break;
	case XK_question:
		information();
		break;
	case XK_braceright:
		dataDump();
		break;
	case XK_I:
		displayInternalBuffer();
		break;
	case XK_c:
		copy();
		break;
	case XK_C:
		copyToNew();
		break;
	case XK_r:
		remove();
		break;
	case XK_y:
		removeToNew();
		break;
	case XK_e:
		erase();
		break;
	case XK_x:
		spliceOut();
		break;
	case XK_X:
		spliceOutToNew();
		break;
	case XK_d:
		zap();
		break;
	case XK_m:
		mix();
		break;
	case XK_R:
		replace();
		break;
	case XK_g:
		crossfade();
		break;
	case XK_v:
		spliceIn();
		break;
	case XK_P:
		scaleValues();
		break;
	case XK_b:
		reverse();
		break;
	case XK_i:
		insertSpace();
		break;
	case XK_E:
		applyEnvelope();
		break;
	case XK_O:
		vertOffset();
		break;
	case XK_bracketright:
		rescale();
		break;
	case XK_D:
		delay();
		break;
	case XK_t:
		interpolate();
		break;
	case XK_L:
		lowPassFilter();
		break;
	case XK_bracketleft:
		normalize();
		break;
	case XK_l:
		changeLength();
		break;
	case XK_0:
		findZeroCrossing();
		break;
	case XK_1:
		findSlopeChange();
		break;
	case XK_2:
		showPeakFrame();
		break;
	case XK_3:
		extractEnvelope();
		break;
	case XK_degree:
		setDataOptions();
		break;
	default:
		interested = false;
		break;
	}
	return interested;
}

Controller*
DataEditor::openNewFile(const char* windowtitle) {
	FileOpener fo(defaultDir(), windowtitle);
	applyModifier(fo);
	return fo.getNewController();
}

// the following are the menu-called functions

Status
DataEditor::openNewFile() {
	return (openNewFile("Select File to Open:") != nil);
}

Status
DataEditor::newLPCFile() {
	return (LPCEditor::new_File(controller) != nil);
}

Status
DataEditor::newSoundFile() {
	return (SoundEditor::new_File(controller) != nil);
}

Status
DataEditor::newEnvelopeFile() {
	return (EnvelopeEditor::new_File(controller) != nil);
}

Status
DataEditor::newPvocFile() {
	return (PvocEditor::new_File(controller) != nil);
}

Response
DataEditor::closeFile() {
	Response r = Yes;
	// warn if file modified and if this is the last open view of it
	if(data->modified() && data->currentViews() == 1) {
		r = Application::choice("Do you wish to save changes to",
			controller->fileName(), "before closing?", Cancel);
		switch(r) {
		case Yes:
			saveFile();
			if(data->modified())
				r = Cancel;	// assumes cancel or error during save
			else
				r = No;	// indicates warning was issued
			break;
		case No:
		case Cancel:
			break;
  		default:
			break;
		}
	}
	return r;	// defaults to Yes indicating no alert issued
}
	
Status
DataEditor::saveFile() {
	FileSaver fs(controller->fileName(), defaultDir(), model());
	return applyModifier(fs);
}

Status
DataEditor::saveToFile() {
	FileSaver fs(defaultDir(), model());
	return applyModifier(fs);
}

Status
DataEditor::revertToSaved() {
	const char* filename = controller->fileName();
	Status status = Fail;
	if(!data->modified())
		Application::alert("File has not been edited since last save.");
	else if(FileName::isTempName(filename))
		Application::alert("This is a temporary file and does not",
			"exist yet on disk.  You must save it first.");
	else if(Application::confirm(
			"Are you sure you wish to revert to the saved version?")) {
		DataFile f(filename, "r");
		if(f.readable()) {
			Application::inform("Re-reading file from disk...");
			Header *header = data->createHeader(&f, true);
			header->ref();
			status = data->read(&f, header);
			Resource::unref(header);
		}
		controller->resetScaleTimes();
		reselect();
	}
	return status;
}

Status
DataEditor::displayInternalBuffer() {
	Status status = Fail;
	if(copyBuffer() != nil) {
		if(copyBuffer()->length() > 3) {
			Controller *newCtlr = new Controller(copyBuffer());
			newCtlr->display(controller->world());
			copyBuffer()->Notify();	// to assure rescan of amps
			status = Succeed;
		}
		else
			Application::alert("Cannot display data with length < 4 frames.");
		// once displayed, not available internally to avoid confusion
		setCopyBuffer(nil);
	}
	else Application::alert("Internal buffer is currently empty.");
	return status;
}

Status
DataEditor::changeComment() {
	TextWindow* commentWindow = new TextWindow(
		scat(controller->fileName(), ": Comment"),
		new CommentEditor(new ButtonState, model())
	); 
	commentWindow->display(controller->world());
	return Succeed;
}

Status
DataEditor::dataDump() {
	DataDumper dumper(defaultDir(), currentSelection());
	return applyModifier(dumper);
}

Status
DataEditor::information() {
	data->information(controller);
	return Succeed;
}

// previous internal buffer is kept until actual copy of new buffer takes place

Status
DataEditor::copy() {
	Application::inform("Copying...");
	boolean was = Data::deferRescan(true);
	Status status = setCopyBuffer(currentSelection()->copyOf());
	Data::deferRescan(was);
	return status;
}

Status
DataEditor::copyToNew() {
	return copy() && displayInternalBuffer();
}

Status
DataEditor::remove() {
	return copy() && erase();
}

Status
DataEditor::removeToNew() {
	return remove() && displayInternalBuffer();
}

Status
DataEditor::erase() {
	currentSelection()->erase();
	return Succeed;
}

Status
DataEditor::spliceOut() {
	return copy() && zap();
}

Status
DataEditor::spliceOutToNew() {
	return spliceOut() && displayInternalBuffer();
}

Status
DataEditor::zap() {
	Status status = Fail;
	if(selectionMade()
		|| Application::confirm("Splice out everything from this point",
			"to the end of the file?")) {
		OutSplicer splicer(model(), currentRegion());
		if((status = applyModifier(splicer)))
			controller->showInsertPoint(currentInsert(), currentChannels());
	}
	return status;
}

Status
DataEditor::mix() {
	// for now this mixes at unity gain
	Data* src = nil;
	Status status = Fail;
	if((src = currentSource()) != nil) {
		Mixer mixer(currentSelection(), src);
		if((status = applyModifier(mixer)))
			markEditRegion(min(currentSelection()->length(), src->length()));
	}
	freeSource(src);
	return status;
}

Status
DataEditor::replace() {
	Data* src = nil;
	Status status = Fail;
	if((src = currentSource()) != nil) {
		Replacer replacer(currentSelection(), src);
		if((status = applyModifier(replacer)))
			markEditRegion(min(currentSelection()->length(), src->length()));
	}
	freeSource(src);
	return status;
}

Status
DataEditor::crossfade() {
	Data* src = nil;
	Status status = Fail;
	if((src = currentSource()) != nil) {
		Crossfader crossfader(currentSelection(), src);
		if((status = applyModifier(crossfader)))
			markEditRegion(min(currentSelection()->length(), src->length()));
	}
	freeSource(src);
	return status;
}

Status
DataEditor::spliceIn() {
	Data* src = nil;
	Status status = Fail;
	if((src = currentSource()) != nil) {
		Splicer splicer(currentSelection(), src);
		if((status = applyModifier(splicer))) {
			Range edit = src->frameRange();
			edit += currentInsert();		// offset for highlight display
			controller->showEditRegion(edit, data->channelRange());
		}
	}
	freeSource(src);
	return status;
}

Status
DataEditor::scaleValues() {
	Scaler s(currentSelection());
	return applyModifier(s);
}

Status
DataEditor::insertSpace() {
	SpaceInsert s(currentSelection());
	return applyModifier(s);
}

Status
DataEditor::reverse() {
	Reverser r(currentSelection());
	return applyModifier(r);
}

Status
DataEditor::vertOffset() {
	Offset o(currentSelection());
	return applyModifier(o);
}

Status
DataEditor::rescale() {
	Rescaler r(currentSelection());
	return applyModifier(r);
}

Status
DataEditor::interpolate() {
	// create new buffer which will be resized by Interpolater
	setCopyBuffer(currentSelection()->newData(1));
	Interpolater i(currentSelection(), copyBuffer());
	return applyModifierToNew(i);
}

Status
DataEditor::changeLength() {
	LengthChanger l(model());
	Status status = applyModifier(l);
	reselect();
	return status;
}

Status
DataEditor::delay() {
	Delay d(currentSelection());
	return applyModifier(d);
}

Status
DataEditor::lowPassFilter() {
	LowPassFilter l(currentSelection(), 0.5);
	return applyModifier(l);
}

Status
DataEditor::applyEnvelope() {
	Envelope *envelope = nil;
	Data *source = nil;
	Controller* ctlr = nil;
	if((source = currentSource()) != nil) {
		;
	}
	else if((ctlr = openNewFile("Select Envelope File to Apply:")) != nil) {
		source = ctlr->model();
		source->ref();
	}
	Status status = Fail;
	if(source) {
		// make copy of source so that it may be normalized, etc.
		envelope = new Envelope(source->length());
		envelope->copyFrom(source);
		envelope->ref();
		Application::inform("Scaling envelope between -1 and 1...");
		envelope->normalize();
		Phraser p(currentSelection(), envelope);
		if(p.configure(controller))
			status = p.apply();
		Resource::unref(envelope);
	}
	freeSource(source);
	return status;
}

Status
DataEditor::normalize() {
	Normalizer n(currentSelection());
	return applyModifier(n);
}

Status
DataEditor::changeSampleRate() {
	SampRateChanger s(model());
	Status status;
	if((status = applyModifier(s))) {
		controller->resetScaleTimes();
		reselect();
	}
	return status;
}

Status
DataEditor::showPeakFrame() {
	int peakchan, peakloc;
	data->maxValue(&peakchan, &peakloc);
	Range chan(peakchan, peakchan);
	controller->showInsertPoint(peakloc, chan);
	return Succeed;
}

Status
DataEditor::findZeroCrossing() {
	int offset = currentSelection()->zeroCrossing();
	Status status = Succeed;
	if(offset > 0) {
		Range chan(currentChannels().intMin(), currentChannels().intMin());
		controller->showInsertPoint(currentInsert() + offset, chan);
	}
	else {
		Application::alert("No zero crossings found.");
		status = Fail;
	}
	return status;
}

// for now, this just displays the location of the first discontinuity found, though the
// analysis data is an array of offsets to all such locations within the current selection

Status
DataEditor::findSlopeChange() {
	SlopeChangeDetecter s(currentSelection());
	Status status;
	if((status = applyModifier(s))) {
		int offset = s.getAnalysis()->get(0);
		Range chan(currentChannels().intMin(), currentChannels().intMin());
		controller->showInsertPoint(currentInsert() + offset, chan);
	}
	else {
		Application::alert("No slope changes of that magnitude found.");
		status = Fail;
	}
	return status;
}

Status
DataEditor::extractEnvelope() {
	EnvelopeExtracter e(currentSelection());
	Status status;
	if((status = applyModifier(e))) {
		Controller *newctlr = new Controller(e.getAnalysis());
		newctlr->display(controller->world());
	}
	return status;
}

//**************

Data *
LPCEditor::newModel() { return new LPCData(); }

Range
LPCEditor::currentChannels() {
	if(nchans() == 1 && !channels.includes(-1))
		;
	else if(channels.includesZero() || channels.includes(-1))  // default
		setChannels(model()->channelRange());
	else if(channels.intMin() >= 4)		// coefficient display
		setChannels(4, model()->channels() - 1);
	return channels;
}

boolean
LPCEditor::keyCommand(unsigned long sym) {
	boolean interested = true;
	switch (sym) {
	case XK_asterisk:
		stabilizeFrames();
		break;
	case XK_A:
		displayFrameAmplitudes();
		break;
	case XK_F:
		displayFrameFormants();
		break;
	case XK_k:
		changeSampleRate();
		break;
	case XK_M:
		mergePitchData();
		break;
	case XK_p:
		adjustPitchDeviation();
		break;
	default:
		interested = DataEditor::keyCommand(sym);
		break;
	}
	return interested;
}

Status
LPCEditor::newFile() {
	return (LPCEditor::new_File(controller) != nil);
}

Status
LPCEditor::saveFile() {
	LPCFileSaver lfs(controller->fileName(), defaultDir(), model());
	return applyModifier(lfs);
}

Status
LPCEditor::saveToFile() {
	LPCFileSaver lfs(defaultDir(), model());
	return applyModifier(lfs);
}

// static constructor

Controller *
LPCEditor::new_File(Controller* controller) {
	LPCFileCreator lfc;
	applyModifierUsing(lfc, controller);
	return lfc.getNewController();
}

Status
LPCEditor::stabilizeFrames() {
	FrameStabilizer fs(model());
	return applyModifier(fs);
}

Status
LPCEditor::displayFrameAmplitudes() {
	const int pulseFrameSize = 1000;
	const int pulsePerFrame = 10;
	LPCData* selected = (LPCData *) currentSelection();
	Application::inform("Creating test pattern...");
	int lpcLen = selected->length();
	int srate = selected->sRate();
	Sound* pulses = new Sound(lpcLen * pulseFrameSize, srate, 1, FloatData);
	pulses->ref();
	// add pulsePerFrame pulses per original LPC frame
	PulseGenerator pgen(pulses, pulseFrameSize/pulsePerFrame);
	pgen.apply();
	FormantFilter filter(pulses, pulses, selected, 1.0);
	filter.apply();
	Envelope* amplitudes = new Envelope(lpcLen * pulsePerFrame);
	amplitudes->setFrameRangeLabel("LPC Analysis Frames");
	amplitudes->setRangeFactor(1.0/pulsePerFrame);
	Application::inform("Extracting amplitudes...");
	pulses->getEnvelope(amplitudes, 0, AbsoluteMagnitude);
	Resource::unref(pulses);
	Controller* ampDisplay = new Controller(amplitudes, 
		scat(controller->windowName(), ":  Frame Amplitudes"));
	ampDisplay->display(controller->world());
	return Succeed;
}

Status
LPCEditor::displayFrameFormants() {
	const int pulseFrameSize = 1024;
	LPCData* selected = (LPCData *) currentSelection();
	int lpcLen = selected->length();
	int srate = selected->sRate();
	Sound* pulses = new Sound(lpcLen * pulseFrameSize, srate, 1, FloatData);
	pulses->ref();
	PulseGenerator pgen(pulses, pulseFrameSize);
	Application::inform("Creating test pattern...");
	pgen.apply();
	FormantFilter filter(pulses, pulses, selected, 1.0);
	filter.apply();
	Application::inform("Analyzing formants...");
	// fft size depends on npoles
	FFT_Function analyzer(
		pulses, selected->nPoles() > 32 ? 128 : 64, pulseFrameSize
	);
	Status status;
	if((status = analyzer.apply())) {
		Controller* fftDisplay = new Controller(
			analyzer.getAnalysis(), 
			scat(controller->windowName(), ":  Formant Frequencies"));
		fftDisplay->display(controller->world());
	}
	Resource::unref(pulses);
	return status;
}

Status
LPCEditor::mergePitchData() {
	// regardless of selection, select only channel 3 for merge
	const int pitchChannel = 3;
	setChannels(pitchChannel, pitchChannel);
	LPCData *lpc = (LPCData *) currentSelection();
	Data *pitches = nil;
	Controller* newController = nil;
	Status status = Succeed;
	if((pitches = currentSource()) != nil) {
		;
	}
	else if((newController = openNewFile("Select Pitch File for Merge:")) != nil)
		pitches = newController->model();
	if(pitches && Application::confirm("Please confirm merge operation.")) {
		Application::inform("Merging pitch data...");
		lpc->mergePitchData(pitches);	
		markEditRegion(lpc->length());
	}
	else status = Fail;
	freeSource(pitches);
	return status;
}

#if 0

Status
LPCEditor::warpFrames() {
	Data *source = nil;
	Controller* ctlr = nil;
	if((source = currentSource()) != nil) {
		copy();
		Envelope* evp = new Envelope(1);
		evp->ref();
		Interpolater i(source, evp, copyBuffer()->length());
		i.apply();		// stretch source to equal current sel's length
		FrameWarper f(evp, copyBuffer());
		if(applyModifierToNew(f)) {
			Application::alert("A new file of filter modifying values",
				"is being created, and will be needed for any",
				"subsequent warped LPC formant filtering.");
			ctlr = new Controller(evp, "new_warp.evp");
			ctlr->display(controller->world());
		}
		Resource::unref(evp);
	}
	freeSource(source);
	return Succeed;
}

#endif /* 0 */

Status
LPCEditor::setDataOptions() {
	LPCOptionSetter setter;
	return applyModifier(setter);
}

// these static functions have their addresses loaded into a ctor array
// in the DataEditor base class

DataEditor *
LPCEditor::new_DataEditor1(Controller *c) {
	return new LPCEditor(c);
}

DataEditor *
LPCEditor::new_DataEditor2(Controller *c, const Data *d) {
	return new LPCEditor(c, d);
}

//**************

Data *
PCHEditor::newModel() { return new PCHData(); }

// these static functions have their addresses loaded into a ctor array
// in the DataEditor base class

DataEditor *
PCHEditor::new_DataEditor1(Controller *c) {
	return new PCHEditor(c);
}

DataEditor *
PCHEditor::new_DataEditor2(Controller *c, const Data *d) {
	return new PCHEditor(c, d);
}

//**************

Data *
FFTEditor::newModel() { return new FFTData(); }

Status
FFTEditor::setDataOptions() { return Succeed; }

// these static functions have their addresses loaded into a ctor array
// in the DataEditor base class

DataEditor *
FFTEditor::new_DataEditor1(Controller *c) {
	return new FFTEditor(c);
}

DataEditor *
FFTEditor::new_DataEditor2(Controller *c, const Data *d) {
	return new FFTEditor(c, d);
}

//**************

Data *
EnvelopeEditor::newModel() { return new Envelope(); }

boolean
EnvelopeEditor::keyCommand(unsigned long sym) {
	boolean interested = true;
	switch (sym) {
	case XK_underscore:
		createLinearCurve();
		break;
	case XK_asterisk:
		createExponentialCurve();
		break;
	default:
		interested = DataEditor::keyCommand(sym);
		break;
	}
	return interested;
}

Status
EnvelopeEditor::newFile() {
	return (EnvelopeEditor::new_File(controller) != nil);
}

Status
EnvelopeEditor::setDataOptions() { return Succeed; }

// static constructor

Controller *
EnvelopeEditor::new_File(Controller* controller) {
	EnvelopeFileCreator efc;
	applyModifierUsing(efc, controller);
	return efc.getNewController();
}

Status
EnvelopeEditor::createLinearCurve() {
	LinearCurveGenerator l(currentSelection());
	return applyModifier(l);
}

Status
EnvelopeEditor::createExponentialCurve() {
	ExponentialCurveGenerator e(currentSelection());
	return applyModifier(e);
}

// these static functions have their addresses loaded into a ctor array
// in the DataEditor base class

DataEditor *
EnvelopeEditor::new_DataEditor1(Controller *c) {
	return new EnvelopeEditor(c);
}

DataEditor *
EnvelopeEditor::new_DataEditor2(Controller *c, const Data *d) {
	return new EnvelopeEditor(c, d);
}
