/*-
 * Copyright (c) 1995 Michael B. Durian.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Michael B. Durian.
 * 4. The name of the the Author may be used to endorse or promote 
 *    products derived from this software without specific prior written 
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
 * FreeBSD 2.2 specific driver entry points
 */
#include "midi.h"
#if NMIDI > 0

#include "sys_freebsd2.2.h"
#include "sup_export.h"

static int midiprobe __P((struct isa_device *dvp));
static int midiattach __P((struct isa_device *isdp));

struct isa_driver mididriver = {
	midiprobe, midiattach, "midi"
};

struct freebsd_midi_softc {
	struct	midi_softc s;
#ifdef DEVFS
	void	*devfs_token;
	void	*devfs_token_r;
#endif
} midi_sc[NMIDI];


static d_open_t		midiopen;
static d_close_t	midiclose;
static d_read_t		midiread;
static d_write_t	midiwrite;
static d_ioctl_t	midiioctl;
static d_select_t	midiselect;


#define CDEV_MAJOR	78
static struct cdevsw midi_cdevsw =
	{ midiopen,	midiclose,	midiread,	midiwrite,	/*78*/
	  midiioctl,	nullstop,	nullreset,	nodevtotty,	/*midi*/
	  midiselect,	nommap,		nostrat,	"midi",	NULL,	-1 };

static midi_devsw_installed = 0;

static void
midi_drvinit(void *unused)
{
	dev_t dev;

	if (!midi_devsw_installed) {
		dev = makedev(CDEV_MAJOR, 0);
		cdevsw_add(&dev, &midi_cdevsw, NULL);
		midi_devsw_installed = 1;
	}
}

SYSINIT(mididev, SI_SUB_DRIVERS, SI_ORDER_MIDDLE + CDEV_MAJOR, midi_drvinit,
    NULL)

int
midiprobe(dev)
	struct isa_device *dev;
{
	struct midi_iface *iface;
	struct midi_isa_iface loc;

	loc.io_port = dev->id_iobase;
	loc.irq = ffs(dev->id_irq) - 1;

	/* XXX does id_flags come from the config file? */
	iface = midi_dev_probe(NULL, dev->id_flags, MIDI_ISA_IFACE, &loc);
	if (iface == NULL)
		return (0);

	/*
	 * we'll do this for real in attach when we have a real softc
	 * to put it in.
	 */
	iface->free(iface);

	return (1);
}

int
midiattach(isdp)
	struct isa_device *isdp;
{
	struct freebsd_midi_softc *freebsd_softc;
	struct midi_softc *softc;
	struct midi_isa_iface loc;

	freebsd_softc = &midi_sc[isdp->id_unit];
	softc = &freebsd_softc->s;

	loc.io_port = isdp->id_iobase;
	loc.irq = ffs(isdp->id_irq) - 1;

	if (!midi_init_dev(softc, isdp->id_flags, MIDI_ISA_IFACE, &loc))
		return (0);

	printf(" %s\n", softc->iface->name(softc->iface));

	/* allocate memory for event queues */
	if ((softc->rqueue = malloc(sizeof(struct event_queue), M_DEVBUF,
	    M_NOWAIT)) == NULL) {
		log(LOG_ERR, "No memory for rqueue\n");
		softc->status |= MIDI_UNAVAILABLE;
		return (0);
	}
	if ((softc->wqueue = malloc(sizeof(struct event_queue), M_DEVBUF,
	    M_NOWAIT)) == NULL) {
		log(LOG_ERR, "No memory for wqueue\n");
		softc->status |= MIDI_UNAVAILABLE;
		free(softc->rqueue, M_DEVBUF);
		return (0);
	}
	if ((softc->thruqueue = malloc(sizeof(struct event_queue), M_DEVBUF,
	    M_NOWAIT)) == NULL) {
		log(LOG_ERR, "No memory for thruqueue\n");
		softc->status |= MIDI_UNAVAILABLE;
		free(softc->rqueue, M_DEVBUF);
		free(softc->wqueue, M_DEVBUF);
		return (0);
	}
	/* zero read/write queue to clear stynamic structures */
	bzero(softc->rqueue, sizeof(struct event_queue));
	bzero(softc->wqueue, sizeof(struct event_queue));
	bzero(softc->thruqueue, sizeof(struct event_queue));
	stynamic_init(&softc->rpartial);
	stynamic_init(&softc->wpartial);
	softc->wpartialpos = 0;

#ifdef DEVFS
	freebsd_softc->devfs_token_r = devfs_add_devswf(&midi_cdevsw,
	    isdp->id_unit | 64, DV_CHR, 0, 0, 0666, "rmidi%d", isdp->id_unit);
	freebsd_softc->devfs_token = devfs_add_devswf(&midi_cdevsw,
	    isdp->id_unit, DV_CHR, 0, 0, 0666, "midi%d", isdp->id_unit);
#endif

	return (1);
}

int
midiopen(dev, flag, mode, p)
	dev_t dev;
	int flag, mode;
	struct proc *p;
{
	int f;

	f = 0;
	if (flag & FREAD)
		f |= MIDI_OREAD;
	if (flag & FWRITE)
		f |= MIDI_OWRITE;
	return (gen_midiopen((void *)dev, f, p->p_pid, midi_sc));
}

int
midiclose(dev, flag, mode, p)
	dev_t dev;
	int flag, mode;
	struct proc *p;
{
	int f, unit;
	struct freebsd_midi_softc *sc;

	unit = minor(dev) & 0x3f;
	f = 0;
	if (flag & FREAD)
		f |= MIDI_OREAD;
	if (flag & FWRITE)
		f |= MIDI_OWRITE;
	if (minor(dev) & 0x40)
		f |= MIDI_RAW_DEVICE;

	sc = &midi_sc[unit];
	return (gen_midiclose(&sc->s, f));
}

int
midiread(dev, uio, flag)
	dev_t dev;
	struct uio *uio;
	int flag;
{
	int f, unit;
	struct freebsd_midi_softc *sc;

	f = 0;
	if (flag & IO_NDELAY)
		f |= MIDI_NDELAY;
	if (minor(dev) & 0x40)
		f |= MIDI_RAW_DEVICE;

	unit = minor(dev) & 0x3f;
	sc = &midi_sc[unit];
	return (gen_midiread(&sc->s, uio, f));
}

int
midiwrite(dev, uio, flag)
	dev_t dev;
	struct uio *uio;
	int flag;
{
	int f, unit;
	struct freebsd_midi_softc *sc;

	f = 0;
	if (flag & IO_NDELAY)
		f |= MIDI_NDELAY;
	if (minor(dev) & 0x40)
		f |= MIDI_RAW_DEVICE;

	unit = minor(dev) & 0x3f;
	sc = &midi_sc[unit];
	return (gen_midiwrite(&sc->s, uio, f));
}

void
midiintr(unit)
	int unit;
{
	struct freebsd_midi_softc *softc;

	softc = &midi_sc[unit];
	gen_midiintr(&softc->s);
}

int
midiioctl(dev, cmd, data, flag, p)
	dev_t dev;
	int cmd;
	caddr_t data;
	int flag;
	struct proc *p;
{
	int f, unit;
	struct freebsd_midi_softc *sc;

	f = 0;
	if (minor(dev) & 0x40)
		f |= MIDI_RAW_DEVICE;
	unit = minor(dev) & 0x3f;
	sc = &midi_sc[unit];
	return (gen_midiioctl(&sc->s, cmd, data, f));
}

/*
 * gen_midiselect
 * I don't have a good way of generalizing select yet, so it is done
 * on a per machine basis.
 */
int
midiselect(dev, which, p)
	dev_t dev;
	int which;
	struct proc *p;
{
	struct freebsd_midi_softc *sc;
	struct midi_softc *softc;
	int ret, s, unit;

	unit = minor(dev) & 0x3f;
	sc = &midi_sc[unit];
	softc = &sc->s;

	s = spltty();
	switch (which) {
	case FREAD:
		if (!(softc->status & MIDI_RD_BLOCK))
			ret = 1;
		else {
			ret = 0;
			softc->status |= MIDI_RSEL;
			selrecord(p, &softc->rsel);
		}
		break;
	case FWRITE:
		if (!(softc->status & MIDI_WR_BLOCK))
			ret = 1;
		else {
			ret = 0;
			softc->status |= MIDI_WSEL;
			selrecord(p, &softc->wsel);
		}
		break;
	default:
		if (softc->status & MIDI_TIME_WARP) {
			softc->status &= ~MIDI_TIME_WARP;
			ret = 1;
		} else {
			ret = 0;
			softc->status |= MIDI_ESEL;
			selrecord(p, &softc->esel);
		}
		break;
	}
	splx(s);
	return (ret);
}

struct midi_softc *
UNIT_TO_SOFTC(unit, client)
	int unit;
	void *client;
{
	struct freebsd_midi_softc *softcs = (struct freebsd_midi_softc *)client;
	struct freebsd_midi_softc *sc;
	struct midi_softc *softc;

	if (unit >= NMIDI || (sc = &softcs[unit]) == NULL) {
		softc = NULL;
		return (softc);
	}
	softc = &sc->s;
	return (softc);
}

void
SIGNAL_PG(pgid, sig)
	int pgid, sig;
{
	struct proc *p;

	if (pgid < 0)
		gsignal(-pgid, sig);
	else if ((p = pfind(pgid)) != 0)
		psignal(p, sig);
}

void
TIMEOUT(id, fn, arg, t)
	TIMEOUT_ID *id;
	TIMEOUT_FN fn;
	TIMEOUT_ARG arg;
	int t;
{

	timeout(fn, arg, t);
	*id = fn;
}

int
UWRITEC(uio)
	struct uio *uio;
{
	unsigned char b;

	if (uiomove(&b, 1, uio) != 0)
		return (-1);
	return (b);
}
#endif
