#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <x_types.h>
#include <utils.h>
#include <fileutil.h>

#include "backup.h"

#ifdef	DEBUG
#define	DB(fp, fmt, arg1, arg2, arg3)	fprintf(fp, fmt, arg1, arg2, arg3)
#else
#define	DB(fp, fmt, arg1, arg2, arg3)	/**/
#endif

#define	GETOUT	{ goto getout; }
#define CLEANUP	{ goto cleanup; }

#define	SERIAL_OPERATION	1
#define	QUEUED_OPERATION	2

#define	MAX_QUEUE_MEMORY	10485760	/* 10 MB */

typedef	struct _queueentry_ {
  UChar		*inmem;
  UChar		*outmem;
  Uns32		num_in;
  Uns32		num_out;
  Uns16		instruction;
  UChar		processed;
} QueueEntry;

#define	NOT_PROCESSED		0
#define	PROCESSED_OK		RESERVED_ERROR_CODE

UChar		comm_mode = QUEUED_OPERATION;
Uns32		queuelen = 512;
QueueEntry	*queue_entries = NULL;
UChar		*outbufmem = NULL;
UChar		*inbufmem = NULL;
Uns32		queueent_done_idx = 0;
Uns32		queueent_requested_idx = 0;
Uns32		queueent_processed_idx = 0;
Uns32		queue_insert_idx = 0;

#define	QUEUEDOP	(comm_mode == QUEUED_OPERATION)

#define	MAX_OUTARGLEN	(COMMBUFSIZ + 4)
#define	MAX_INRESULTLEN	MAX_OUTARGLEN

Uns32		bytes_transferred;
UChar		cycle_started;
Uns32		num_cycles_processing;
Uns32		transfer_per_cycle;
Uns32		transfer_per_cycle_cur;
Uns32		transferred_uninterrupted;

Uns32		bytes_transferred_prev;
Uns32		transfer_per_cycle_prev;

Uns32		queue_clock_usec;

UChar		*servername = NULL;

UChar		scbuf[COMMBUFSIZ + 4];
UChar		*cbuf = &(scbuf[1]);
Int32		cbufptr = 0;

int		commfd;
struct timeval	null_timeval;

short unsigned	portnum = DEFAULT_PORT;

Int32		cart;
Int32		filenum;
Int32		wrcart;
Int32		wrfilenum;

UChar		filenum_valid = 0;

Int32		endofrec = 0;

AarParams	params = AAR_DEFAULT_PARAMS;

UChar		bu_create = 0;
UChar		bu_extract = 0;
UChar		bu_contents = 0;
UChar		bu_verify = 0;
UChar		allfiles = 0;
UChar		c_f_verbose = 0;
UChar		o_verbose = 0;
UChar		bu_request = 0;

UChar		*clientprog = NULL;

UChar		**filelist = NULL;

char		*toextractfilename = NULL;
char		*errorlogfile = NULL;
char		*savefile = NULL;

Int32		num_errors = 0;

UChar		long_errmsgs = 0;
UChar		**gargv;

#define	MAX_NUM_ERRORS		20

#define	ESTIM_PROT_OVERHEAD	50

struct fault_msgs	fault_messages[] = FAULT_MESSAGES;

void	start_queue_timer();
void	queue_processor(int);
Int32	post_process_entry(Int32);
Int32	post_process_rest_of_queue();
Int32	append_to_queue(Uns16, UChar *, Uns32, UChar *, Uns32);

static void
errmsg(UChar * msg, ...)
{
  va_list	args;

  va_start(args, msg);

  if(long_errmsgs)
    fprintf(stderr, "%s, ", actimestr());

  vfprintf(stderr, msg, args);

  va_end(args);

  if(msg[strlen(msg) - 1] != '\n')
    fprintf(stderr, ".\n");

  fflush(stderr);
}

void
signal_handler(int s)
{
  if(long_errmsgs || s == SIGTERM || s == SIGINT)
    fprintf(stderr, "%s got signal %d, exiting.\n",
					FN_BASENAME(gargv[0]), s);

  exit(128 | s);
}

UChar
send_cmd(UChar cmd)		/* utility */
{
  if(write_split(commfd, &cmd, 1))
    return(errno ? -errno : EOF);

  return(0);
}

UChar
result()
{
  UChar		c;
  Int32		i;

  if(savefile)
    return(0);

  i = read_split(commfd, &c, 1);

  if(i){
    return(errno ? -errno : EOF);
  }

  if(! c)
    return(0);

  return(c);
}

UChar
send_buffer_to_server()
{
  Int32		i, num, r;
  UChar		*cptr;

  scbuf[0] = WRITETOTAPE;

  cptr = scbuf;
  num = COMMBUFSIZ + 1;

  if(savefile){
    cptr = cbuf;
    num = COMMBUFSIZ;
  }

  filenum_valid = 0;

  if(QUEUEDOP){
    r = append_to_queue(WRITETOTAPE, NULL, 0, cbuf, COMMBUFSIZ);
    return(r);
  }

  if( (i = write_split(commfd, cptr, num)) )
    return(i);

  r = result();

  return(r);
}

Int32
send_pending()
{
  if(cbufptr == 0)
    return(0);

  if(cbufptr < COMMBUFSIZ)
    memset(cbuf + cbufptr, 0, (COMMBUFSIZ - cbufptr) * sizeof(UChar));

  return(send_buffer_to_server());
}

Int32
send_to_server(UChar * buf, Int32 num)
{
  UChar		*cptr;
  Int32		j;

  if(num == 0)
    return(0);

  cptr = buf;

  forever{
    if(num + cbufptr <= COMMBUFSIZ){
      memcpy(cbuf + cbufptr, cptr, num * sizeof(UChar));

      cbufptr += num;

      return(0);
    }

    if(cbufptr < COMMBUFSIZ)
      memcpy(cbuf + cbufptr, cptr, (COMMBUFSIZ - cbufptr)
					* sizeof(UChar));

    j = send_buffer_to_server();
    if(j){
	return(j);
    }

    cptr += (COMMBUFSIZ - cbufptr);

    num -= (COMMBUFSIZ - cbufptr);

    cbufptr = 0;
  }

  return(0);
}

Int32
bu_output(UChar * buf, Int32 num, AarParams * params)
{
  params->vars.bytes_saved += (Real64) num;

  return(send_to_server(buf, num));
}


Int32
receive_buffer_from_server()
{
  Int32		i, j;

  filenum_valid = 0;

  if(QUEUEDOP){
    i = append_to_queue(READFROMTAPE, cbuf, COMMBUFSIZ, NULL, 0);

    j = post_process_rest_of_queue();

    return(i ? i : j);
  }

  if(!savefile){
    if( (i = send_cmd(READFROMTAPE)) )
      return(i);
  }

  if( (i = read_split(commfd, cbuf, COMMBUFSIZ)) )
    return(i);

  return(result());
}

Int32
receive(UChar * buf, Int32 num, Int32 * num_read)
{
  Int32		res, j;
  UChar		*cptr;

  if(num == 0){
    *num_read = 0;
    return(0);
  }

  cptr = buf;

  j = num;

  forever{
    if(cbufptr == 0){
	if(!endofrec)
	  res = receive_buffer_from_server();
	else
	  res = 0;

	if(res){
	  if(res < 0){
	    *num_read = num - j;
	    return(res);
	  }
	  else{
	    if(res == ENDOFFILEREACHED){
		endofrec = 1;
	    }
	    else{
		*num_read = num - j;
		return(res);
	    }
	  }
	}
    }

    if(COMMBUFSIZ - cbufptr <= j){
	if(cbufptr < COMMBUFSIZ)
	  memcpy(cptr, cbuf + cbufptr, COMMBUFSIZ - cbufptr);

	cptr += (COMMBUFSIZ - cbufptr);
	j -= (COMMBUFSIZ - cbufptr);
	cbufptr = 0;

	if(endofrec){
	  *num_read = num - j;
	  return(ENDOFFILEREACHED);
	}
	if(j == 0){
	  *num_read = num;
	  return(0);
	}
    }
    else{
	if(j > 0)
	  memcpy(cptr, cbuf + cbufptr, j);

	cbufptr += j;
	*num_read = num;
	return(0);
    }
  }

  return(0);	/* NOTREACHED */
}

Int32
bu_input(UChar * buf, Int32 num, AarParams * params)
{
  Int32		i, r;

  i = receive(buf, num, &r);

  if(i && i != ENDOFFILEREACHED){
    fprintf(params->errfp,
	"%sError: unexpected fault receiving from server: %s. Trying to continue ...\n",
		(i > 0 ? "Server " : ""),
		(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));
  }

  return(r);
}

Int32
getcartandfile(AarParams * params)
{
  static UChar	first_call = 1;

  UChar		buf[10];
  Int32		i;

  if(filenum_valid)
    return(0);

  if(QUEUEDOP){
    i = append_to_queue(QUERYPOSITION, NULL, 7, NULL, 0);

    if(first_call){
	i = post_process_rest_of_queue();

	params->vars.firstcart = cart;
	params->vars.firstfile = filenum;

	first_call = 0;
    }

    filenum_valid = 1;

    return(i);
  }

  if( (i = send_cmd(QUERYPOSITION)) )
    return(i);

  if( (i = read_split(commfd, buf, 7)) )
    return(i);

  if( (i = result()) )
    return(i);

  cart = ((Uns32) buf[0] << 16) + ((Uns32) buf[1] << 8) + buf[2];
  filenum = ((Uns32) buf[3] << 24) + ((Uns32) buf[4] << 16)
		+ ((Uns32) buf[5] << 8) + buf[6];

  filenum_valid = 1;

  if(first_call){
    params->vars.firstcart = cart;
    params->vars.firstfile = filenum;

    first_call = 0;
  }

  return(0);
}

Int32
getwrcartandfile()
{
  static UChar	first_call = 1;

  UChar		buf[10];
  Int32		i;

  if(QUEUEDOP){
    i = append_to_queue(QUERYWRPOSITION, NULL, 7, NULL, 0);

    if(first_call){
	i = post_process_rest_of_queue();
	first_call = 0;
    }

    return(i);
  }

  if( (i = send_cmd(QUERYWRPOSITION)) )
    return(i);

  if( (i = read_split(commfd, buf, 7)) )
    return(i);

  if( (i = result()) )
    return(i);

  wrcart = ((Uns32) buf[0] << 16) + ((Uns32) buf[1] << 8) + buf[2];
  wrfilenum = ((Uns32) buf[3] << 24) + ((Uns32) buf[4] << 16)
		+ ((Uns32) buf[5] << 8) + buf[6];

  return(0);
}

Int32
setcart(Int32 cartnum)
{
  UChar		buf[8];
  Int32		i;

  filenum_valid = 0;

  buf[0] = SETCARTRIDGE;
  buf[1] = (UChar) ((cartnum >> 16) & 0xff);
  buf[2] = (UChar) ((cartnum >> 8) & 0xff);
  buf[3] = (UChar) (cartnum & 0xff);

  if(QUEUEDOP){
    i = append_to_queue(buf[0], NULL, 0, buf + 1, 3);

    return(i);
  }

  if( (i = write_split(commfd, buf, 4)) )
    return(i);

  return(result());
}

Int32
setcartset(Int32 cartset)
{
  UChar		buf[8];
  Int32		i;

  filenum_valid = 0;

  buf[0] = SETCARTSET;
  buf[1] = (UChar) ((cartset >> 16) & 0xff);
  buf[2] = (UChar) ((cartset >> 8) & 0xff);
  buf[3] = (UChar) (cartset & 0xff);

  if(QUEUEDOP){
    i = append_to_queue(buf[0], NULL, 0, buf + 1, 3);

    return(i);
  }

  if( (i = write_split(commfd, buf, 4)) )
    return(i);

  return(result());
}

Int32
setfilenum(Int32 filenum)
{
  UChar		buf[8];
  Int32		i;

  filenum_valid = 0;

  buf[0] = SETFILE;
  buf[1] = (filenum >> 24) & 0x7f;
  buf[2] = (filenum >> 16) & 0xff;
  buf[3] = (filenum >> 8) & 0xff;
  buf[4] = filenum & 0xff;
  
  if(QUEUEDOP){
    i = append_to_queue(SETFILE, NULL, 0, buf + 1, 4);

    return(i);
  }

  if( (i = write_split(commfd, buf, 5)) )
    return(i);

  return(result());
}

Int32
open_write()
{
  Int32		i;

  filenum_valid = 0;

  if(QUEUEDOP)
    return(append_to_queue(OPENFORWRITE, NULL, 0, NULL, 0));

  if( (i = send_cmd(OPENFORWRITE)) )
    return(i);

  return(result());
}

Int32
open_read()
{
  Int32		i;

  filenum_valid = 0;

  if(QUEUEDOP)
    return(append_to_queue(OPENFORREAD, NULL, 0, NULL, 0));

  if( (i = send_cmd(OPENFORREAD)) )
    return(i);

  return(result());
}

Int32
closetape()
{
  Int32		i, j;

  filenum_valid = 0;

  if(QUEUEDOP){
    i = append_to_queue(CLOSETAPE, NULL, 0, NULL, 0);

    j = post_process_rest_of_queue();

    return(i ? i : j);
  }

  if( (i = send_cmd(CLOSETAPE)) )
    return(i);

  return(result());
}

UChar *
fault_string(Int32 code)
{
  Int32		i, sz;

  if(!code)
    return("");

  sz = sizeof(fault_messages) / sizeof(fault_messages[0]);

  for(i = 0; i < sz; i++){
    if(code == (Int32) fault_messages[i].code)
	return(fault_messages[i].msg);
  }

  return("unknown fault");
}


Int32
close_connection()
{
  Int32		i, j;

  filenum_valid = 0;

  if(QUEUEDOP){
    i = append_to_queue(GOODBYE, NULL, 0, NULL, 0);

    j = post_process_rest_of_queue();

    return(i ? i : j);
  }

  if( (i = send_cmd(GOODBYE)) )
    return(i);

  return(result());
}

Int32
request_client_backup(UChar * cmdname)
{
  Int32		i, n;
  UChar		*buf;

  n = strlen(cmdname);

  buf = strapp("..", cmdname);
  buf[0] = CLIENTBACKUP;
  buf[1] = n % 251;

  if(QUEUEDOP){
    i = append_to_queue(CLIENTBACKUP, NULL, 0, buf + 1, n + 1);

    free(buf);

    return(i);
  }

  i = write_split(commfd, buf, n + 2);
  free(buf);

  if(i)
    return(i);

  return(result());
}

UChar **
get_file_list(FILE * fp)
{
  UChar		buf[MAXPATHLEN + 10];
  Int32		i;
  UChar		**filelist = NULL;
  Int32		num_files = 0;

  forever{
    i = fscanwordq_forced(fp, buf);

    if(i == EOF){
	return(filelist);
    }

    if(filelist)
	filelist = RENEWP(filelist, UChar *, num_files + 2);
    else
	filelist = NEWP(UChar *, 2);

    if(!filelist){
	errmsg("Error: memory exhausted by the list of files");
	exit(-2);
    }

    filelist[num_files] = strdup(buf);
    if(!filelist[num_files]){
	for(i = 0; i < num_files; free(filelist[i++]));
	free(filelist);
	return(NULL);
    }

    num_files++;

    filelist[num_files] = NULL;
  }
}

void
pre_extended_verbose(UChar * str, AarParams * params)
{
  Int32		i;

  if( (i = getcartandfile(params)) )
    errmsg("%sError: %s", (i > 0 ? "Server " : ""),
			(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));

  params->vars.actfile = filenum;
  params->vars.actcart = cart;
}

void
extended_verbose(UChar * str, AarParams * params)
{
  FILE		*fp;

  fp = (params->mode == MODE_CONTENTS ? params->outfp : params->errfp);

  if((params->verbose & VERBOSE_CART_FILE) &&
			! (params->verbose & VERBOSE_UID))
    fprintf(fp, "%d.%d: %s", (int) params->vars.actcart,
			(int) params->vars.actfile, str);

  if((params->verbose & VERBOSE_CART_FILE) &&
			(params->verbose & VERBOSE_UID))
    fprintf(fp, "%d.%d" UIDSEP "%d: %s", (int) params->vars.actcart,
			(int) params->vars.actfile,
			(int) params->vars.uid, str);

  if(! (params->verbose & VERBOSE_CART_FILE) &&
			(params->verbose & VERBOSE_UID))
    fprintf(fp, UIDSEP "%d: %s", (int) params->vars.uid, str);
}

void
write_reportfile(UChar * filename, AarParams * params, Int32 rc)
{
  FILE		*fp;

  fp = fopen(filename, "w");
  if(!fp){
    errmsg("Error: Cannot write report file \"%s\"\n", filename);
    return;
  }

  fprintf(fp, "Backup statistics:\n");
  fprintf(fp, "==================\n\n");
  fprintf(fp, "Exit status:                 %ld\n\n", rc);
  fprintf(fp, "Start-Time:                  %s\n", params->vars.startdate);
  fprintf(fp, "End-Time:                    %s\n\n", params->vars.enddate);
  fprintf(fp, "First Cartridge used:        %ld\n", params->vars.firstcart);
  fprintf(fp, "First Tapefile accessed:     %ld\n", params->vars.firstfile);
  fprintf(fp, "Last Cartridge used:         %ld\n", params->vars.lastcart);
  fprintf(fp, "Last Tapefile accessed:      %ld\n\n", params->vars.lastfile);
  fprintf(fp, "Stored filesystem entries:   %ld\n", params->vars.num_fsentries);
  fprintf(fp, "Bytes written to media:      %s\n",
			Real642intstr(params->vars.bytes_saved));
  fprintf(fp, "Sum of filesizes:            %s\n",
			Real642intstr(params->vars.sum_filesizes));
  fprintf(fp, "Sum of compressed filesizes: %s\n",
			Real642intstr(params->vars.sum_compr_filesizes));
  if(params->vars.sum_compr_filesizes > 0)
    fprintf(fp, "Total compression factor:    %.2f\n",
		(float) params->vars.sum_filesizes /
		(float) params->vars.sum_compr_filesizes);
  fprintf(fp, "\n");

  fclose(fp);
}

void
give_help()
{
{
char *l[155];
int i;
l[0]="Description";
l[1]="===========";
l[2]="";
l[3]="This program is used to maintain archives on a backup server";
l[4]="host or in a file. Archives can be created, extracted or their";
l[5]="contents be listed. Almost one of the following flags has to";
l[6]="be supplied:";
l[7]="";
l[8]=" -c  to create an archive";
l[9]="";
l[10]=" -x  to extract from an archive";
l[11]="";
l[12]=" -t  to list the contents of an archive";
l[13]="";
l[14]=" -d  to verify (compare) the contents of an archive";
l[15]="";
l[16]=" -C  to set a certain cartridge on the backup server";
l[17]="       (makes only sense extracting or listing with -x or";
l[18]="        -t, the writing position can't be changed by clients)";
l[19]="";
l[20]=" -F  to set a certain file on the backup server's tape";
l[21]="       (the same applies as for -C)";
l[22]="";
l[23]=" -q  to printout the actual cartridge and tape file number";
l[24]="       on the backup server";
l[25]="";
l[26]=" -Q  to printout the cartridge and tape file number for the";
l[27]="       the next write access on the backup server";
l[28]="";
l[29]=" -X  followed by the full path name of a program to be started on";
l[30]="       the client. This can be used to trigger a backup remotely.";
l[31]="       If the program needs arguments, the command together with";
l[32]="       the arguments has to be enclosed by quotes";
l[33]="";
l[34]="-c, -x, -t and -X are mutual exclusive. The other options can";
l[35]="be supplied as needed. To set the cartridge and/or the tape file";
l[36]="on the backup server is only making sense when not creating";
l[37]="an archive. The serial order of writing to tape is handled by";
l[38]="the server machine independently of the client.";
l[39]="";
l[40]="";
l[41]="Filenames";
l[42]="";
l[43]="The names of the files and directories, that have to be put";
l[44]="into or extracted from an archive are by default read from the";
l[45]="standard input. If you supply filenames in the command line or";
l[46]="enter the -a flag when extracting, standard input is not read.";
l[47]="The same is valid, if filenames are read from a file with the";
l[48]="-T option. When reading the names from a file or from standard";
l[49]="input, filenames have to be seperated by whitespace. If a name";
l[50]="is containing whitespace, it has to be enclosed in double";
l[51]="quotes (\"). If a name contains double quotes or backslashes,";
l[52]="each has to be preceded by a backslash (so backslashes become";
l[53]="double backslashes).";
l[54]="";
l[55]="";
l[56]="More options in alphabetical order:";
l[57]="";
l[58]=" -A <time>    process files (save or extract) modified after";
l[59]="                the given time in seconds since 1.1.1970 00:00";
l[60]="";
l[61]=" -a           in combination with -x: extract all files and";
l[62]="                directories in the archive";
l[63]="";
l[64]=" -B <time>    process files (save or extract) modified before";
l[65]="                the given time in seconds since 1.1.1970 00:00";
l[66]="";
l[67]=" -b           don't enter buffering mode";
l[68]="";
l[69]=" -e <errlog>  Use the file <errlog> to write error messages to";
l[70]="                instead of the standard error output";
l[71]="";
l[72]=" -f <file>    write to or read from a file instead of querying";
l[73]="                the backup server";
l[74]="";
l[75]=" -g           while extracting/reading: ignore leading garbage,";
l[76]="                suppress error messages at the beginning. This";
l[77]="                is useful when extracting from tape files, that";
l[78]="                are not the first ones of a whole archive.";
l[79]="";
l[80]=" -h <host>    use the backup server with the name <host>";
l[81]="                default host is the machine with the name";
l[82]="                backuphost";
l[83]="";
l[84]=" -i           while extracting: ignore the stored ownership and";
l[85]="                do not restore it";
l[86]="";
l[87]=" -k <file>    use the contents of the given file as encryption";
l[88]="                key for authenticating to the server";
l[89]="";
l[90]=" -N <file>    while archiving: ignore files with a modification";
l[91]="                time before the one of the given file, only save";
l[92]="                newer files or such with the same age in seconds";
l[93]="";
l[94]=" -n           for each packed or unpacked filename, if sending";
l[95]="                to or receiving from a backup server in verbose";
l[96]="                   mode:";
l[97]="                printout cartridge and tape file number at the";
l[98]="                beginning of the line, e. g.: 7.15: <filename>";
l[99]="";
l[100]=" -O           for each packed file creating a backup in verbose";
l[101]="                mode: printout the user-ID of the file owner at";
l[102]="                the beginning of the line prefixed with a bar |";
l[103]="                eventually behind cartridge and file number";
l[104]="";
l[105]=" -o <uid>     archive or extract only files owned by the user";
l[106]="                with the given user-ID (an integer)";
l[107]="";
l[108]=" -p <portno>  use a different port number for communicating with";
l[109]="                the backup server. Default is TCP-Port 2988";
l[110]="";
l[111]=" -R           pack or extract directories recursively with all";
l[112]="                of their contents";
l[113]="";
l[114]=" -r           use filenames relative to the current directory,";
l[115]="                whether they start with a slash or not";
l[116]="";
l[117]=" -S <cartset> The cartridge set to use, where <cartset> is the";
l[118]="                number of a valid cartridge set on the server";
l[119]="                side. Default is 1. This option makes sense only";
l[120]="                when creating backups with -c";
l[121]="";
l[122]=" -s <filepat> do not attempt compression on files matching the";
l[123]="                given filename pattern. This parameter may";
l[124]="                appear several times";
l[125]="";
l[126]=" -T <file>    read the filenames to process from the <file>.";
l[127]="                The filenames must be seperated by whitespace.";
l[128]="                If whitespace is part of a filename, it has to";
l[129]="                be enclosed by double quotes. Double quotes or";
l[130]="                backslashes within the filename have to be";
l[131]="                preceded by a backslash";
l[132]="";
l[133]=" -u           while extracting: remove existing files with the";
l[134]="                same name as found in the archive. Otherwise";
l[135]="                no existing files are overwritten";
l[136]="";
l[137]=" -V <file>    write a report containing statistics at the end of";
l[138]="                a backup to the <file>";
l[139]="";
l[140]=" -v           verbose mode: print the filenames while creating";
l[141]="                or extracting, be a little more verbose while";
l[142]="                listing contents";
l[143]="";
l[144]=" -z <z> <uz>  use <z> as the command, that is used to compress";
l[145]="                files, <uz> for the corresponding uncompress.";
l[146]="                The command has to read from stdin and to write";
l[147]="                to stdout. If arguments have to be supplied to";
l[148]="                <z> and/or <uz>, don't forget to use quotes";
l[149]="";
l[150]=" -Z           while printing out the contents: check those files";
l[151]="                in the archive that are compressed for integrity";
l[152]="";
l[153]="";
l[154]=" -?           to printout this text";
for(i=0;i<155;i++)
fprintf(stdout,"%s\n",l[i]);
}

  exit(0);
}

void
usage(UChar * prnam)
{
  prnam = FN_BASENAME(prnam);

  errmsg("usage: %s -cxtd [ -RraunOvgiqQZb ]\n", prnam);
  errmsg("               [ -h <backup-server> ]\n");
  errmsg("               [ -z <zipcmd> <unzipcmd> ]\n");
  errmsg("               [ -T <to-extract-filename> ]\n");
  errmsg("               [ -C <cartridge-number> ]\n");
  errmsg("               [ -F <filenumber-on-tape> ]\n");
  errmsg("               [ -f <archive-filename> ]\n");
  errmsg("               [ -e <errorlog-filename> ]\n");
  errmsg("               [ -p <server-port-number> ]\n");
  errmsg("               [ -N <newer-than-filename> ]\n");
  errmsg("               [ -o <user-ID> ]\n");
  errmsg("               [ -k <encryption-key-file> ]\n");
  errmsg("               [ -s <dont-compress-filepattern> [ -s ... ] ]\n");
  errmsg("               [ -V <statistics-report-file> ]\n");
  errmsg("               [ -A <after-time-seconds> ]\n");
  errmsg("               [ -B <before-time-seconds> ]\n");
  errmsg("               [ <files> <directories> ... ]\n");
  errmsg("\n       %s -X <program>", prnam);
  errmsg("               [ -h <backup-client> ]\n");
  errmsg("\n       %s -\\?  (to get help)\n", prnam);

  exit(1);
}


main(int argc, char ** argv)
{
  Int32			i, j;
  struct sockaddr_in	addr;
  struct hostent	*hp;
  struct servent	*se;
  struct stat		statb;
  FILE			*fp;
  UChar			buf[2000];
  UChar			givehelp = 0;
  UChar			querypos = 0;
  UChar			querywrpos = 0;
  UChar			noqueued = 0;
  Int32			p = -1;
  UChar			**filelist = NULL;
  UChar			**files = NULL;
  UChar			*found_files = NULL;
  Int32			s_cart = -1;
  Int32			s_file = -1;
  Int32			s_cset = 1;
  Uns32			code;
  Int32			num_unused = 0;
  UChar			*newer_than_file = NULL;
  UChar			**unused_args = NULL;
  UChar			*cryptfile = NULL;
  UChar			*cptr = NULL;
  UChar			*reportfile = NULL;
  UChar			*older_than_sec = NULL;
  UChar			*newer_than_sec = NULL;

  gargv = (UChar **) argv;

  memset(&params.vars, 0, sizeof(params.vars));

  strcpy(params.vars.startdate, actimestr());

  if(!params.infp)
    params.infp = stdin;
  if(!params.outfp)
    params.outfp = stdout;
  if(!params.errfp)
    params.errfp = stderr;
  sigemptyset(&params.blocked_signals);
  sigprocmask(SIG_SETMASK, &params.blocked_signals, NULL);

  if(sizeof(UChar) > 4){
    fprintf(stderr, "Error: long int is wider than 4 Bytes. Inform author about machine type.\n");
    exit(1);
  }

  i = goptions(-argc, (UChar **) argv,
	"b:c;b:x;b:t;b:d;b:n;b:O;b:R;b:r;b:v;b:u;b:a;s:h;s2:z;s:T;i:p;s:e;s:f;"
	"i:C;i:F;i:S;b:g;b:i;b:?;b:q;b:Q;*;b:Z;s:X;s:N;i:o;s:k;b:b;s:s;s:V;"
	"s:B;s:A;b:E",
			&bu_create, &bu_extract, &bu_contents, &bu_verify,
			&c_f_verbose, &o_verbose,
			&params.recursive, &params.relative, &params.verbose,
			&params.unlink, &allfiles,
			&servername,
			&params.zipcmd, &params.unzipcmd,
			&toextractfilename, &p, &errorlogfile,
			&savefile, &s_cart, &s_file, &s_cset,
			&params.skip_garbage, &params.ignoreown,
			&givehelp, &querypos, &querywrpos,
			&num_unused, &unused_args, &params.check,
			&clientprog, &newer_than_file, &params.uid,
			&cryptfile, &noqueued, &cptr, &reportfile,
			&older_than_sec, &newer_than_sec,
			&long_errmsgs);

  if(i)
    usage(argv[0]);

  if(cptr){
    params.dont_compress = NEWP(UChar *, 1);
    params.dont_compress[0] = NULL;

    for(i = 1, j = 0; i < argc; i++){
      if(!strcmp(argv[i], "-s")){
	params.dont_compress = RENEWP(params.dont_compress, UChar *, j + 2);
	params.dont_compress[j] = strdup(argv[i + 1]);
	j++;
	params.dont_compress[j] = NULL;
	i++;
      }
    }
  }

  if(givehelp)
    give_help();

  if(noqueued || !bu_create)
    comm_mode = SERIAL_OPERATION;

  if(clientprog)
    bu_request = 1;

  if(bu_create + bu_extract + bu_contents + bu_request + bu_verify > 1){
    errmsg("You may supply only one of the flags: cxtX");
    usage(argv[0]);
  }
  if(bu_create + bu_extract + bu_contents + bu_request + bu_verify < 1
		&& s_cart < 0 && s_file < 0 && ! querypos && ! querywrpos){
    errmsg("You have to supply one of the flags: cxtCFqQ");
    usage(argv[0]);
  }
  if(toextractfilename && (! bu_extract && ! bu_create)){
    errmsg("The flag -T may only be given in combination wit -x or -c");
    usage(argv[0]);
  }
  if(toextractfilename && allfiles){
    errmsg("The flags -T and -a don't make sense together. Ignoring -a");
    allfiles = 0;
  }
  if(unused_args && (toextractfilename || allfiles)){
    errmsg("If you name files and/or directories in the commandline,\n -a or -T don't make any sense. Using filenames from the command line");
    allfiles = 0;
    toextractfilename = NULL;
  }
  if(allfiles && ! bu_extract){
    errmsg("Warning: the flag -a makes only sense in combination with -x");
  }
  if(params.unlink && ! bu_extract){
    errmsg("Warning: the flag -u makes only sense in combination with -x");
  }
  if(params.recursive && !(bu_create || bu_extract)){
    errmsg("Warning: the flag -R makes only sense in combination with -x or -c");
  }
  if(params.relative && ! bu_extract){
    errmsg("Warning: the flag -r makes only sense in combination with -x");
  }
  if(params.ignoreown && ! bu_extract){
    errmsg("Warning: the flag -r makes only sense in combination with -x");
  }
  if(newer_than_file && ! bu_create){
    errmsg("Warning: the flag -N only makes sense in combination with -c");
  }
  if(newer_than_file && newer_than_sec){
    errmsg("Warning: -N and -A may not be specified together, -N ignored.");
    ZFREE(newer_than_file);
  }
  if(params.zipcmd && params.unzipcmd && !bu_create){
    errmsg("Warning: to specify (de-)compression is only allowed with -c");
  }
  if(params.check && ! bu_contents){
    errmsg("Warning: the flag -Z makes only sense in combination with -t. Ignoring -Z");
    params.check = 0;
  }
  if(servername && savefile){
    errmsg("Error: You may not specify -h and -f together");
    usage(argv[0]);
  }
  if((s_cart != -1 || s_file != -1 || querypos || querywrpos) && savefile){
    errmsg("Error: with a file (-f) you cannot set/get cartridge and/or tape position");
    usage(argv[0]);
  }
  if(savefile && c_f_verbose){
    errmsg("Warning: -n makes no sense in combination with -f. -n is ignored");
    c_f_verbose = 0;
  }
  if(!bu_create && o_verbose){
    errmsg("Warning: -O is only supported in combination with -c, -O ignored");
    o_verbose = 0;
  }
  if(savefile && cryptfile){
    errmsg("Warning: -k makes no sense in combination with -f");
  }
  if(s_file >	0x7fffffff){
    errmsg("Error: requested file number is too high - must be <= %ld",
		0x7fffffffL);
    usage(argv[0]);
  }
  if(bu_request){
    if(strlen(clientprog) > 250){
      errmsg("Error: program name supplied with -X is too long (max. 250 chars)");
      exit(1);
    }
  }

  if(cryptfile){
    if(set_cryptkey(cryptfile)){
	errmsg("Warning: Cannot read enough characters from encryption key file \"%s\"",
		cryptfile);
	errmsg("		Ignoring file, using compiled-in key");
    }
  }

  if(!servername)
    servername = DEFAULT_SERVER;

  signal(SIGTERM, signal_handler);
  signal(SIGINT, signal_handler);
  signal(SIGHUP, signal_handler);
  signal(SIGSEGV, signal_handler);
  signal(SIGBUS, signal_handler);
  signal(SIGPIPE, signal_handler);

  if(params.verbose){
    params.verbose = VERBOSE_NORMAL;

    if(c_f_verbose)
	params.verbose |= VERBOSE_CART_FILE;
    if(o_verbose)
	params.verbose |= VERBOSE_UID;
  }

  if(c_f_verbose || o_verbose){
    params.verbosefunc = extended_verbose;
    params.pre_verbosefunc = pre_extended_verbose;
  }

  if(newer_than_file){
    if(stat(newer_than_file, &statb)){
      errmsg("Warning: Cannot stat file \"%s\". Option ignored",
				newer_than_file);
    }
    else{
      params.time_newer = statb.st_mtime;
    }
  }

  if(older_than_sec){
    params.time_older = strint2time(older_than_sec);
    if(params.time_older == (time_t) -1){
	errmsg("Warning: Could not read upper time limit from "
			"\"%s\". Option ignored", older_than_sec);
	params.time_older = 0;
    }
  }
  if(newer_than_sec){
    params.time_newer = strint2time(newer_than_sec);
    if(params.time_newer == (time_t) -1){
	errmsg("Warning: Could not read lower time limit from "
			"\"%s\". Option ignored", newer_than_sec);
	params.time_newer = 0;
    }
  }

  if(unused_args){
    filelist = unused_args;
  }
  else if(toextractfilename || (bu_extract && !allfiles)){
    if(toextractfilename){
	fp = fopen(toextractfilename, "r");
	if(!fp){
	  errmsg("Error: cannot open file \"%s\"", toextractfilename);
	  exit(1);
	}
    }
    else
	fp = stdin;

    filelist = get_file_list(fp);

    if(toextractfilename)
	fclose(fp);

    if(! filelist){
	errmsg("Exiting, because %s does not contain any filename",
			toextractfilename);
	exit(0);
    }
  }
  if(filelist){
    for(files = filelist, i = 0; *files; i++, files++);

    if(i > 0){
	found_files = NEWP(UChar, i);
	memset(found_files, 0, i * sizeof(UChar));

	if(!found_files){
	  errmsg("Error: cannot allocate memory");
	  exit(9);
	}
    }
  }

  if(errorlogfile){
    fp = fopen(errorlogfile, "w");
    if(!fp){
	errmsg("Error: cannot open error log file \"%s\"",
				errorlogfile);
	errmsg("	 Writing errors to stdout");
    }
    else
	params.errfp = fp;
  }

  if(!savefile){
    commfd = socket(AF_INET, SOCK_STREAM, 0);

    if(commfd < 0){
      errmsg("Error: cannot get communication socket");
      exit(-1);
    }

    addr.sin_family = AF_INET;
    addr.sin_port = htons(0);
    addr.sin_addr.s_addr = htonl(0L);
    i = sizeof(addr);

    if(bind(commfd, (struct sockaddr * ) (& addr), i) < 0){
      errmsg("Error: cannot bind socket");
      exit(-1);
    }

    hp = gethostbyname(servername);

    if(!hp){
      errmsg("Error: cannot find address of host %s", servername);
      exit(-1);
    }

    if(p > 0)
      portnum = p;
    else{
      se = getservbyname(DEFAULT_SERVICE, "tcp");

      if(!se && p < 0){
	errmsg("Error: cannot get the port number of the backup service");
	exit(-1);
      }

      portnum = ntohs(se->s_port);
    }

    memcpy(&addr.sin_addr, hp->h_addr, hp->h_length);
    addr.sin_port = htons(portnum);

    if(connect(commfd, (struct sockaddr * ) (& addr), i) < 0){
      errmsg("Error: cannot connect to host %s, port %d",
					servername, (int) portnum);
      exit(-1);
    }

    memset(buf, 0, 1000 * sizeof(UChar));

    for(i = 0; i < 1000; i++){
      j = read_split(commfd, buf + i, 1);

      if(j){
	errmsg("Error: Cannot get the greeting message from server");
	exit(-1);
      }

      if(strstr(buf, GREETING_MESSAGE))
	break;
    }
    if(i >= 1000){
      errmsg("Error: server didn't send the greeting message");
      exit(-1);
    }

    read_split(commfd, buf, 4);

    code = ((Uns32) buf[0] << 24) + ((Uns32) buf[1] << 16)
			+ ((Uns32) buf[2] << 8) + (Uns32) buf[3];

    code = encrpt(code);

    buf[0] = (code >> 24) & 0xff;
    buf[1] = (code >> 16) & 0xff;
    buf[2] = (code >> 8) & 0xff;
    buf[3] = code & 0xff;

    write_split(commfd, buf, 4);

    i = result();
    if(i){
	errmsg("Error: %s. Server closed connection", fault_string(i));
	exit(1);
    }

    if(QUEUEDOP){
      Int32	needed_entry_mem;
#ifndef	_WIN32
      struct rlimit	rlim;
#endif

      needed_entry_mem = MAX_OUTARGLEN + MAX_INRESULTLEN + sizeof(QueueEntry);

#ifndef	_WIN32
      i = getrlimit(RLIMIT_STACK, &rlim);
      if(i){
	errmsg("Error: cannot get maximum available memory");
      }
      queuelen = rlim.rlim_cur / needed_entry_mem / 4;
#else
      queuelen = 1000000 / needed_entry_mem / 4;
#endif

      if(queuelen < 1){
	comm_mode = SERIAL_OPERATION;
      }
      else{
	if(needed_entry_mem * queuelen > MAX_QUEUE_MEMORY){
	  queuelen = MAX_QUEUE_MEMORY / needed_entry_mem;
	}

	if(MAX_OUTARGLEN * queuelen < 100000){
	  comm_mode = SERIAL_OPERATION;
	}
	else{
	  outbufmem = NEWP(UChar, MAX_OUTARGLEN * queuelen);
	  inbufmem = NEWP(UChar, MAX_INRESULTLEN * queuelen);
	  queue_entries = NEWP(QueueEntry, queuelen);

	  if(!outbufmem || !inbufmem || !queue_entries){
	    errmsg("Error: Strange. Can't get memory. Disabling queued operation");
	    comm_mode = SERIAL_OPERATION;
	    ZFREE(outbufmem);
	    ZFREE(inbufmem);
	    ZFREE(queue_entries);
	  }
	  else{
	    struct sigaction	siga;

	    queue_clock_usec = 10000;

		/* assume initial max transfer rate of 1 MB / sec */
	    transfer_per_cycle = 1000000 / 1000000 * queue_clock_usec;
	    transfer_per_cycle_cur = transfer_per_cycle;

	    transfer_per_cycle_prev = transfer_per_cycle;
	    bytes_transferred_prev = transfer_per_cycle;

	    bytes_transferred = 0;
	    cycle_started = 0;
	    num_cycles_processing = 0;
	    queueent_done_idx = queue_insert_idx = queueent_requested_idx
				= queueent_processed_idx = 0;
	    transferred_uninterrupted = 0;

	    SETZERO(siga);
	    siga.sa_handler = queue_processor;
#ifdef	SA_NOMASK
	    siga.sa_flags |= SA_NOMASK;
#else
#ifdef	SA_NODEFER
	    siga.sa_flags |= SA_NODEFER;
#endif
#endif
#ifdef	SA_RESTART
	    siga.sa_flags |= SA_RESTART;
#endif
	    sigaction(SIGALRM, &siga, NULL);

	    start_queue_timer();
	  }
	}
      }
    }
  }
  else{
    comm_mode = SERIAL_OPERATION;

    if(bu_create){
      if(!strcmp(savefile, "-")){
	commfd = 1;
      }
      else{
	if( (fp = fopen(savefile, "w")) ){
	  fclose(fp);

	  commfd = open(savefile, O_WRONLY | O_CREAT | O_BINARY, 0644);
	}
	else
	  commfd = -1;
      }
    }
    else{
      if(!strcmp(savefile, "-")){
	commfd = 0;
      }
      else{
	commfd = open(savefile, O_RDONLY | O_BINARY);
      }
    }

    if(commfd < 0){
	errmsg("Error: cannot open file \"%s\"", savefile);
	exit(9);
    }
  }

  SETZERO(null_timeval);

  if(QUEUEDOP){
    sigaddset(&params.blocked_signals, SIGALRM);
    sigaddset(&params.blocked_signals, SIGPIPE);
  }

  if(s_cset != 1){
    i = setcartset(s_cset);
    if(i){
	errmsg("%sError: %s", (i > 0 ? "Server " : ""),
			(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));
	GETOUT;
    }
  }
  if(s_cart != -1){
    i = setcart(s_cart);
    if(i){
	errmsg("%sError: %s", (i > 0 ? "Server " : ""),
			(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));
	GETOUT;
    }
  }
  if(s_file != -1){
    i = setfilenum(s_file);
    if(i){
	errmsg("%sError: %s", (i > 0 ? "Server " : ""),
			(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));
	GETOUT;
    }
  }

  if(bu_request){
    if( (i = request_client_backup(clientprog)) ){
	errmsg("%sError: %s", (i > 0 ? "Server " : ""),
			(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));
    }
    if(QUEUEDOP){
	if( (i = post_process_rest_of_queue()) ){
	  errmsg("%sError: %s", (i > 0 ? "Server " : ""),
			(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));
	}
    }
  }

  if(querypos){
    if( (i = getcartandfile(&params)) ){
	errmsg("%sError: %s", (i > 0 ? "Server " : ""),
			(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));
    }
    else{
	if(QUEUEDOP){
	  if( (i = post_process_rest_of_queue()) ){
	    errmsg("%sError: %s", (i > 0 ? "Server " : ""),
			(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));
	  }
	}

	fprintf(stdout, "Actual tape access position\n");
	fprintf(stdout, "Cartridge: %d\nFile:      %d\n",
				(int) cart, (int) filenum);
    }
  }

  if(querywrpos){
    if( (i = getwrcartandfile()) ){
	errmsg("%sError: %s", (i > 0 ? "Server " : ""),
			(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));
    }
    else{
	if(QUEUEDOP){
	  if( (i = post_process_rest_of_queue()) ){
	    errmsg("%sError: %s", (i > 0 ? "Server " : ""),
			(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));
	  }
	}

	fprintf(stdout, "Next tape writing position\n");
	fprintf(stdout, "Cartridge: %d\nFile:	    %d\n",
				(int) wrcart, (int) wrfilenum);
    }
  }

  if(bu_create){
	params.mode = MODE_CREATE;

	params.outputfunc = bu_output;

	if(!savefile){
	  if( (i = open_write()) ){
	    errmsg("%sError: %s", (i > 0 ? "Server " : ""),
			(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));

	    GETOUT;
	  }
	}

	if(filelist){
	  while(*filelist){
	    i = writeout(*filelist, &params, 0);

	    if(i){
		errmsg("%sError: %s", (i > 0 ? "Server " : ""),
			(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));

		if(i > 0)
		  num_errors++;

		if(num_errors > MAX_NUM_ERRORS){
		  errmsg("Too many errors on server. Exiting");

		  break;
		}
	    }
	    else{
		num_errors = 0;
	    }

	    filelist++;
	  }
	}
	else{
	  while(EOF != fscanwordq_forced(stdin, buf)){
	    i = writeout(buf, &params, 0);

	    if(i){
		errmsg("%sError: %s", (i > 0 ? "Server " : ""),
			(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));

		if(i > 0)
		  num_errors++;

		if(num_errors > MAX_NUM_ERRORS){
		  errmsg("Too many errors on server. Exiting");

		  break;
		}
	    }
	    else{
		num_errors = 0;
	    }
	  }
	}

	writeout(NULL, &params, ENDOFARCHIVE);

	if(( i = send_pending()) ){
	  errmsg("%sError: %s", (i > 0 ? "Server " : ""),
		(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));

	  GETOUT;
	}

	if(!savefile){
	  if( (i = getcartandfile(&params)) ){
	    errmsg("%sError: %s", (i > 0 ? "Server " : ""),
		(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));

	    GETOUT;
	  }
	  params.vars.lastcart = cart;
	  params.vars.lastfile = filenum;

	  if( (i = closetape()) ){
	    errmsg("%sError: %s", (i > 0 ? "Server " : ""),
		(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));

	    GETOUT;
	  }
	}
  }
  else if(bu_contents){
	params.mode = MODE_CONTENTS;

	params.inputfunc = bu_input;

	if(!savefile){
	  if( (i = open_read()) ){
	    errmsg("%sError: %s", (i > 0 ? "Server " : ""),
			(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));

	    GETOUT;
	  }
	}

	contents(&params);

	if(!savefile){
	  if( (i = closetape()) ){
	    errmsg("%sError: %s", (i > 0 ? "Server " : ""),
			(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));

	    GETOUT;
	  }
	}
  }
  else if(bu_extract || bu_verify){
	params.mode = (bu_extract ? MODE_EXTRACT : MODE_VERIFY);

	params.inputfunc = bu_input;

	if(!savefile){
	  if( (i = open_read()) ){
	    errmsg("%sError: %s", (i > 0 ? "Server " : ""),
			(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));

	    GETOUT;
	  }
	}

	if(bu_extract)
	  readin(filelist, &params, found_files);
	else
	  verify(filelist, &params, found_files);

	if(!savefile){
	  if( (i = closetape()) ){
	    errmsg("%sError: %s", (i > 0 ? "Server " : ""),
			(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));

	    GETOUT;
	  }
	}

	if(filelist){
	  for(files = filelist, i = 0; *files; files++, i++){
	    if(!found_files[i]){
		errmsg("%s not found in archive", *files);
	    }
	  }
	}
  }

  if(QUEUEDOP){
     if( (i = post_process_rest_of_queue()) ){
       errmsg("%sError: %s", (i > 0 ? "Server " : ""),
			(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));
     }

     queue_clock_usec = 0;
     start_queue_timer();	/* in fact stop it */

     comm_mode = SERIAL_OPERATION;	/* synchronous mode for disconnect */
  }

  if(!savefile)
    close_connection();
  else{
    if(strcmp(savefile, "-"))
      close(commfd);
  }

  if(errorlogfile)
    fclose(params.errfp);

  strcpy(params.vars.enddate, actimestr());

  if(reportfile)
    write_reportfile(reportfile, &params, 0);

  exit(0);


 getout:

  if(!savefile)
    close_connection();
  else{
    if(strcmp(savefile, "-"))
      close(commfd);
  }

  if(QUEUEDOP){
     if( (i = post_process_rest_of_queue()) ){
       errmsg("%sError: %s", (i > 0 ? "Server " : ""),
			(i > 0 ? fault_string(i) : (UChar *) strerror(-i)));
     }
  }

  if(errorlogfile)
    fclose(params.errfp);

  strcpy(params.vars.enddate, actimestr());

  if(reportfile)
    write_reportfile(reportfile, &params, 1);

  exit(1);
}

/* queued operation stuff */
void
start_queue_timer()
{
  struct itimerval	queue_clock_itv;

  SETZERO(queue_clock_itv);

  queue_clock_itv.it_value.tv_usec
			= queue_clock_itv.it_interval.tv_usec
			= queue_clock_usec % 1000000;
  queue_clock_itv.it_value.tv_sec
			= queue_clock_itv.it_interval.tv_sec
			= queue_clock_usec / 1000000;

  setitimer(ITIMER_REAL, &queue_clock_itv, NULL);
}

void
reduce_transfer_rate()
{
  Uns32		diff;

  diff = transfer_per_cycle / 100;

  transfer_per_cycle -= diff;
}

void
try_increase_transfer_rate()
{
  Uns32		diff;

  diff = transfer_per_cycle / 100;
  if(diff < 1)
    diff = 1;

  transfer_per_cycle += diff;
}

Int32
process_queue_entry_response()
{
  QueueEntry	*entry;
  UChar		*inbuf;
  Int32		res, num_to_read;
  fd_set	commfdset;

  if(queueent_processed_idx == queueent_requested_idx)
    return(0);

  FD_ZERO(&commfdset);
  FD_SET(commfd, &commfdset);
  if(select(commfd + 1, &commfdset, NULL, NULL, &null_timeval) < 1)
    return(0);

  entry = queue_entries + queueent_processed_idx;

  inbuf = inbufmem + MAX_INRESULTLEN * queueent_processed_idx;
  num_to_read = entry->num_in;

  entry->processed = PROCESSED_OK;

  if(num_to_read > 0){
    res = read_split(commfd, inbuf, num_to_read);
    if(res)
	entry->processed = res;

    bytes_transferred += num_to_read + ESTIM_PROT_OVERHEAD;
  }

  if( (res = result()) )
	entry->processed = res;

  bytes_transferred += ESTIM_PROT_OVERHEAD;

  queueent_processed_idx++;
  if(queueent_processed_idx >= queuelen)
    queueent_processed_idx = 0;

  return(1);
}

Int32
process_queue_entry_request()
{
  QueueEntry	*entry;
  Uns16		instruction;
  UChar		*outbuf;
  UChar		c;
  Int32		res, num_to_write;
  fd_set	commfdset;

  if(queueent_requested_idx == queue_insert_idx)
    return(0);

  FD_ZERO(&commfdset);
  FD_SET(commfd, &commfdset);
  if(select(commfd + 1, NULL, &commfdset, NULL, &null_timeval) < 1)
    return(0);

  entry = queue_entries + queueent_requested_idx;

  outbuf = outbufmem + MAX_OUTARGLEN * queueent_requested_idx;
  num_to_write = entry->num_out;
  instruction = entry->instruction;

  if(num_to_write == 0){
    c = (UChar) instruction;
    res = write_split(commfd, &c, 1);
    if(res){
      entry->processed = res;
    }
    bytes_transferred += ESTIM_PROT_OVERHEAD;
  }

  if(num_to_write > 0){
    outbuf[1] = (UChar) instruction;

    res = write_split(commfd, outbuf + 1, num_to_write + 1);

    if(res)
	entry->processed = res;

    bytes_transferred += num_to_write + 1 + ESTIM_PROT_OVERHEAD;
  }

  bytes_transferred += ESTIM_PROT_OVERHEAD;

  queueent_requested_idx++;
  if(queueent_requested_idx >= queuelen)
    queueent_requested_idx = 0;

  return(1);
}

void
queue_processor(int sig)
{
  Uns32		new_usec_val, i;
  char		increase = 0, cycle_used_up;

#if 0	/* now using sigaction */
  signal(SIGALRM, queue_processor);
#endif

#ifndef	SA_NOMASK
#ifndef	SA_NODEFER		/* we have to do this "by hand" */
 {
  sigset_t	sigs, osigs;

  sigemptyset(&sigs);
  sigprocmask(SIG_BLOCK, &sigs, &osigs);	/* just get */
  sigdelset(&osigs, SIGALRM);
  sigprocmask(SIG_SETMASK, &osigs, NULL);
 }
#endif
#endif

  new_usec_val = queue_clock_usec;

  if(queueent_processed_idx == queue_insert_idx)	/* nothing to do */
    GETOUT;

  if(cycle_started){			/* already started */
    if(num_cycles_processing == 0)
	transferred_uninterrupted = bytes_transferred;

    num_cycles_processing++;

#if 0		/* this is not yet sufficiently worked out */
    if(num_cycles_processing < 10){	/* dont waste the rest */
					/* of the time slice */
	transfer_per_cycle_cur = bytes_transferred +
		(bytes_transferred * (10 - num_cycles_processing)
				/ num_cycles_processing);
    }
#endif

    GETOUT;
  }

  cycle_started = 1;

  while(bytes_transferred < transfer_per_cycle_cur &&
			queueent_processed_idx != queue_insert_idx){
    i = process_queue_entry_request();
    i += process_queue_entry_response();

    if(i < 1)
	break;		/* nothing can be done */
  }


  /* OPTIMIZER */
  /* Try to enhance the transfer rate */
  if(bytes_transferred < transfer_per_cycle_cur &&
				num_cycles_processing < 1L)
    cycle_used_up = 0;
  else
    cycle_used_up = 1;

#ifdef OPTTEST
  fprintf(stderr, "cycle used up: %d\n", cycle_used_up);
  fprintf(stderr, "Previously transferred: %lu, now: %lu\n",
	bytes_transferred_prev, bytes_transferred, 0);
  fprintf(stderr, "Previous rate: %lu, now: %lu\n",
	transfer_per_cycle_prev, transfer_per_cycle,0);
  fprintf(stderr, "num_cycles_processing: %d\n", num_cycles_processing);
#endif

  if(cycle_used_up){
    if(num_cycles_processing > 0){
      if(transferred_uninterrupted < transfer_per_cycle_prev)
	increase = 0;
    }
    else if(bytes_transferred_prev <= bytes_transferred){
      if(transfer_per_cycle_prev <= transfer_per_cycle)
	increase = 1;
      else
	increase = 0;
    }
    else{
      if(transfer_per_cycle_prev > transfer_per_cycle)
	increase = 1;
      else
	increase = 0;
    }
  }

  transfer_per_cycle_prev = transfer_per_cycle;

#ifdef OPTTEST
  fprintf(stderr, "increase: %d\n", increase);
  fprintf(stderr, "transf_uninterrpt: %d\n", transferred_uninterrupted);
#endif

  if(cycle_used_up){
    if(increase)
      try_increase_transfer_rate();
    else
      reduce_transfer_rate();
  }

  /* END OPTIMIZER */


  bytes_transferred_prev = bytes_transferred;

  cycle_started = 0;
  num_cycles_processing = 0;
  bytes_transferred = 0;
  transfer_per_cycle_cur = transfer_per_cycle;

 getout:

  if(queue_clock_usec != new_usec_val){
    queue_clock_usec = 0;
    start_queue_timer();
  }

  if(queue_clock_usec != new_usec_val){
    queue_clock_usec = new_usec_val;
    start_queue_timer();
  }
}

Int32
post_process_proc(Uns32 entryidx)
{
  QueueEntry	*entry;
  UChar		*inmem;
  UChar		*inbuf;
  Uns32		num_in;
  Int32		res;

  entry = queue_entries + entryidx;

  inmem = entry->inmem;
  inbuf = inbufmem + entryidx * MAX_INRESULTLEN;
  num_in = entry->num_in;

  if(inmem && num_in)
    memcpy(inmem, inbuf, num_in * sizeof(UChar));

  switch(entry->instruction){
    case QUERYPOSITION:
	cart = ((Uns32) inbuf[0] << 16)
			+ ((Uns32) inbuf[1] << 8) + inbuf[2];
	filenum = ((Uns32) inbuf[3] << 24)
		+ ((Uns32) inbuf[4] << 16)
		+ ((Uns32) inbuf[5] << 8) + inbuf[6];
	break;

    case QUERYWRPOSITION:
	wrcart = ((Uns32) inbuf[0] << 16)
			+ ((Uns32) inbuf[1] << 8) + inbuf[2];
	wrfilenum = ((Uns32) inbuf[3] << 24)
		+ ((Uns32) inbuf[4] << 16)
		+ ((Uns32) inbuf[5] << 8) + inbuf[6];
	break;
  }

  res = (entry->processed == PROCESSED_OK ? NO_ERROR
			: entry->processed);

  return(res);
}

Int32
post_process_rest_of_queue()
{
  Int32		res = NO_ERROR;

  while(queueent_done_idx != queue_insert_idx){
    if(queue_entries[queueent_done_idx].processed == NOT_PROCESSED){
	pause();

	continue;
    }

    res = post_process_proc(queueent_done_idx);
    if(res)
	return(res);

    queueent_done_idx++;
    if(queueent_done_idx >= queuelen)
	queueent_done_idx = 0;
  }

  return(res);
}

Int32
post_process_pending()
{
  Int32		res = NO_ERROR;

  while(queueent_done_idx != queueent_processed_idx){
    res = post_process_proc(queueent_done_idx);

    if(res)
	return(res);

    queueent_done_idx++;
    if(queueent_done_idx >= queuelen)
	queueent_done_idx = 0;
  }

  return(res);
}

Int32
append_to_queue(
  Uns16		instruction,
  UChar		*inmem,
  Uns32		num_in,
  UChar		*outmem,
  Uns32		num_out)
{
  Int32		newidx, res;
  UChar		*outbuf;

  if( (res = post_process_pending()) )
    return(res);

  while((newidx = (queue_insert_idx + 1) % queuelen) == queueent_processed_idx)
    pause();			/* queue full */

  outbuf = outbufmem + MAX_OUTARGLEN * queue_insert_idx;

  queue_entries[queue_insert_idx].instruction = instruction;
  queue_entries[queue_insert_idx].inmem = inmem;
  queue_entries[queue_insert_idx].num_in = num_in;
  queue_entries[queue_insert_idx].outmem = outmem;
  queue_entries[queue_insert_idx].num_out = num_out;
  queue_entries[queue_insert_idx].processed = NOT_PROCESSED;

  if(outmem && num_out)
    memcpy(outbuf + 2, outmem, num_out * sizeof(UChar));
		/* data is always put into byte 3..., so we */
		/* can put the instruction into byte 2 (and */
		/* probably 1) and send it all together in */
		/* a single packet */

  queue_insert_idx = newidx;

  return(0);
}
