/*
 *      Copyright (C) 1997 Claus-Justus Heine

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *
 *      This program contains the user level floppy tape formatting
 *      stuff for the QIC-40/80/3010/3020 floppy-tape driver "ftape"
 *      for Linux.
 */

char src[] = "$Source: /homes/cvs/ftape-stacked/contrib/ftformat/ftformat.c,v $";
char rev[] = "$Revision: 1.18.6.1 $";
char dat[] = "$Date: 1997/11/14 14:55:38 $";

#include <sys/types.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>

#include <linux/ftape.h>
#include <linux/ftape-header-segment.h>

#include "ftformat.h"
#include "ftfmt-options.h"
#include "ftfmt-bsm.h"
#include "ftfmt-tapelib.h"

int main(int argc, char *argv[])
{
	struct opt_parms opts;
	struct format_parms fmt_opts;
	int tape_fd = -1;
	void *dma_buffer = NULL;
	u_int8_t drive_status;
	u_int8_t drive_config;
	u_int8_t tape_status;
	u_int8_t hseg[FT_SEGMENT_SIZE];

	/*  These two shouldn't fail, I think.
	 */
	(void)setvbuf(stdout, NULL, _IONBF, 0);
	(void)setvbuf(stderr, NULL, _IONBF, 0);

	/* parse options will abort if there are problems
	 */
	parse_options(argc, argv, &opts, &fmt_opts);

	tape_fd = tape_open(opts.tape_dev, &opts,
						&drive_status, &drive_config, &tape_status);
	if (tape_fd == -1) {
		exit(1);
	}
	if (!(drive_status & QIC_STATUS_REFERENCED)) {
		if (!opts.write_bursts) {
			fprintf(stderr,
"Cannot omit writing of the reference bursts with an unformatted cartridge!\n"
				);
		}
		opts.read_hseg = opts.erase = 0; /* wouldn't work anyway */
	}
	if (opts.mode != FORCE) {
		if (!opts.do_format) { /* verify only */
			if (get_format_parms(tape_fd, &fmt_opts)) {
				tape_close(tape_fd, dma_buffer, opts.dma_size);
				exit(1);
			}
			/* get_parms() doesn't set QIC_WIDE as the kernel driver
			 * doesn't remember this flag as it doesn't need it
			 */
			fmt_opts.fmtparms.ft_qicstd |= tape_status & QIC_TAPE_WIDE;
		} else if (autodetect(tape_fd, drive_status, drive_config, tape_status,
							  &opts, &fmt_opts)) {
			tape_close(tape_fd, dma_buffer, opts.dma_size);
			exit(1);
		}
	}
	report_format_parms(stdout, &fmt_opts);
	if (opts.mode == PROBING) {
		tape_close(tape_fd, dma_buffer, opts.dma_size);
		exit (0);
	}
	if (opts.read_hseg) {
		if (read_header_segment(tape_fd, hseg)) {
			tape_close(tape_fd, dma_buffer, opts.dma_size);
			exit (1);
		}
	} else {
		memset(hseg, 0, FT_SEGMENT_SIZE);
		hseg[FT_FMT_ERROR] = 0xff; /* flag as invalid */
	}
	/* initialize the bsm library
	 */
	init_bsm(hseg, fmt_opts.fmtparms.ft_fmtcode);
	if (opts.erase && erase_cartridge(tape_fd, fmt_opts.fmtparms.ft_tpc)) {
		tape_close(tape_fd, dma_buffer, opts.dma_size);
		exit (1);
	}
	if (set_format_parms(tape_fd, &fmt_opts)) {
		tape_close(tape_fd, dma_buffer, opts.dma_size);
		exit (1);
	}
	if (opts.do_format) {
		/* We actually want to format the tape drive, let's map 
		 * in the dma buffers!
		 */
		if ((tape_fd = tape_reopen_rw(opts.tape_dev, tape_fd)) == -1) {
			(void)close(tape_fd);
			return -1;
		}
		if ((dma_buffer = tape_mmap(tape_fd, opts.dma_size)) == NULL) {
			(void)close(tape_fd);
			return -1;
		}
		if (opts.write_bursts) {
			/* the point of no return:
			 */
			if (write_reference_burst(tape_fd)) {
				tape_close(tape_fd, dma_buffer, opts.dma_size);
				exit (1);
			}
		}
		if (format_cartridge(tape_fd, dma_buffer, opts.dma_size, &fmt_opts)) {
			tape_close(tape_fd, dma_buffer, opts.dma_size);
			exit (1);
		}
	}
	if (verify_cartridge(tape_fd, &fmt_opts, hseg)) {
		tape_close(tape_fd, dma_buffer, opts.dma_size);
		exit (1);
	}
	if (write_header_segments(tape_fd, hseg, opts.label, &fmt_opts)) {
		tape_close(tape_fd, dma_buffer, opts.dma_size);
		exit (1);
	}
	tape_close(tape_fd, dma_buffer, opts.dma_size);
	printf("Done formatting.\n"
		   "Tracks  : %8d\n"
		   "Segments: %8d\n"
		   "Capacity: %8ldM\n",
		   fmt_opts.fmtparms.ft_tpc,
		   fmt_opts.fmtparms.ft_spt,
		   tape_capacity(&fmt_opts)>>10);
	exit (0);
}

/*
 * Local variables:
 *  version-control: t
 *  kept-new-versions: 5
 *  c-basic-offset: 4
 *  tab-width: 4
 * End:
 */
