/* parse.c - parses a trivial permissions language
   Copyright (C) 1998 Paul Sheer

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.  
 */

#include "mostincludes.h"
#include "mirrordir.h"
#include "vfs/util.h" /* for S_IFBLK with -ansi -pedantic */

/*  S_IFMT   00170000  bitmask  for  the file type bitfields */

#define S_RWX_EVERYONE (S_IRWXO | S_IRWXG | S_IRWXU)
#define S_ALL_FILES ~(S_IRWXO | S_IRWXG | S_IRWXU)

static int expression_error = 0;

unsigned long mode_t_to_unsigned_long (mode_t m)
{
    return (m & ~S_IFMT) | (010000 << ((m & S_IFMT) >> 12)) | (m & 007777);
}

/* returns the number of args processed */
static int get_mask (char **args, unsigned long *mask)
{
    char *p, *e;
    int i = 0;
    if (expression_error)
	return 0;
    for (;;) {
	if (!args[i])
	    return i;
	if (!strcmp (args[i], ")")) {
	    return i;
	} else if (!strcmp (args[i], "exec")) {
	    *mask &= S_IXUSR | S_IXGRP | S_IXOTH | S_ALL_FILES;
	} else if (!strncmp (args[i], "wri", 3)) {
	    *mask &= S_IWUSR | S_IWGRP | S_IWOTH | S_ALL_FILES;
	} else if (!strcmp (args[i], "read")) {
	    *mask &= S_IRUSR | S_IRGRP | S_IROTH | S_ALL_FILES;
	} else if (!strcmp (args[i], "user")) {
	    *mask &= S_IRWXU | S_ALL_FILES;
	} else if (!strcmp (args[i], "group")) {
	    *mask &= S_IRWXG | S_ALL_FILES;
	} else if (!strcmp (args[i], "other")) {
	    *mask &= S_IRWXO | S_ALL_FILES;
	} else if (!strncmp (args[i], "sock", 4)) {
	    *mask &= mode_t_to_unsigned_long (S_IFSOCK) | S_RWX_EVERYONE;
	} else if (!strcmp (args[i], "link")) {
	    *mask &= mode_t_to_unsigned_long (S_IFLNK) | S_RWX_EVERYONE;
	} else if (!strncmp (args[i], "reg", 3)) {
	    *mask &= mode_t_to_unsigned_long (S_IFREG) | S_RWX_EVERYONE;
	} else if (!strncmp (args[i], "blo", 3)) {
	    *mask &= mode_t_to_unsigned_long (S_IFBLK) | S_RWX_EVERYONE;
	} else if (!strncmp (args[i], "dir", 3)) {
	    *mask &= mode_t_to_unsigned_long (S_IFDIR) | S_RWX_EVERYONE;
	} else if (!strncmp (args[i], "char", 4)) {
	    *mask &= mode_t_to_unsigned_long (S_IFCHR) | S_RWX_EVERYONE;
	} else if (!strcmp (args[i], "fifo")) {
	    *mask &= mode_t_to_unsigned_long (S_IFIFO) | S_RWX_EVERYONE;
	} else if (!strncmp (args[i], "spec", 4)) {
	    *mask &= mode_t_to_unsigned_long (S_IFSOCK) | \
		mode_t_to_unsigned_long (S_IFBLK) | \
		mode_t_to_unsigned_long (S_IFCHR) | \
		mode_t_to_unsigned_long (S_IFIFO) | \
		S_RWX_EVERYONE;
	} else if (!strcmp (args[i], "and")) {
	    unsigned long t = 0xFFFFFFFFL;
	    i += get_mask (&args[i + 1], &t);
	    *mask &= t;
	} else if (!strcmp (args[i], "or")) {
	    unsigned long t = 0xFFFFFFFFL;
	    i += get_mask (&args[i + 1], &t);
	    *mask |= t;
	} else if (!strcmp (args[i], "not")) {
	    unsigned long t = 0xFFFFFFFFL;
	    i += get_mask (&args[i + 1], &t);
	    *mask &= ~t;
	} else if (!strcmp (args[i], "(")) {
	    unsigned long t = 0xFFFFFFFFL;
	    i += get_mask (&args[++i], &t);
	    if (!args[i]) {
		e = "unmatched bracket in expression";
		p = "(";
		break;
	    }
	    if (strcmp (args[i], ")")) {
		e = "parse error in expression";
		p = args[i];
		break;
	    }
	    *mask &= t;
	} else {
	    e = "unknown token in expression";
	    p = args[i];
	    break;
	}
	i++;
    }
    if (!expression_error) {
	expression_error = 1;
	progmess (e, p);
    }
    return 0;
}

static char *strlowerndup (char *p, int l)
{
    char *q, *r;
    r = q = malloc (l + 1);
    while (l--) {
	*q++ = lower_case (*p);
	p++;
    }
    *q = '\0';
    return r;
}

int parse_permissions_command (char *p, unsigned long *mask)
{
    char **arg, *q;
    int i = 0;
    if (!p)
	return 1;
    arg = malloc (1024 * sizeof (char *));	/* should be enough */
    q = p;
    for (; *p;) {
	while (this_is_whitespace (*p)) {
	    p++;
	    q++;
	}
	if (this_is_a_digit (*p)) {
	    progmess ("error, unknown token", p);
	    return 1;
	}
	switch (*p) {
	case '(':
	case ')':
	    arg[i] = strdup (" ");
	    arg[i][0] = *p;
	    i++;
	    p++;
	    q++;
	    continue;
	default:
	    while (this_is_a_char (*p))
		p++;
	    arg[i++] = strlowerndup (q, (unsigned long) p - (unsigned long) q);
	    q = p;
	}
    }
    arg[i] = 0;

    if (!get_mask (arg, mask))
	return 1;

    if (verbose > 1) {
	char s[20];
	sprintf (s, "%lo", *mask & 07777777777);
	verbmess ("permissions mask", s);
    }
    for (i = 0; arg[i]; i++)
	free (arg[i]);
    free (arg);
    return 0;
}



