/* -*- Mode:  C;-*-
 * XDELTA - RCS replacement and delta generator
 * Copyright (C) 1997  Josh MacDonald
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: xdelta.c 1.4.1.5 Mon, 13 Oct 1997 19:19:20 -0700 jmacd $
 */

#include "xdelta.h"

static void
compute_diffs (MatchQuery *query)
{
  MatchLevel *match_level;
  gint        to_index = 0;
  GSList     *to_ptr;

  match_level = (* query->find_matches) (query,
					 query->size,
					 0,
					 query->to_len);

  if (match_level == NULL)
    return;

  to_ptr = match_level->to_intervals;

  while (TRUE)
    {
      Match *match;

      if (to_index == query->to_len)
	{
	  match_level_free (match_level);
	  return;
	}

      match = to_ptr ? (Match*)to_ptr->data : (Match*)NULL;

      if (match && to_index == match->to_low)
	{
	  emit_match (query, match);
	  to_index = match->to_high;
	  to_ptr = to_ptr->next;
	}
      else
	{
	  gint next_match = to_ptr ? ((Match*)to_ptr->data)->to_low : query->to_len;
	  to_index = next_match;
	}
    }
}

static void
record_multibyte_atom (MatchQuery *query,
		       gint        begin,
		       guint16     hash,
		       guint16   **tbseg_ptr,
		       gint       *tblen_ptr,
		       gint      **map_ptr,
		       gint       *map_alloc_ptr)
{
#define RECSIZE (1<<13)
  guint16 *tbseg     = *tbseg_ptr;
  gint     tblen     = *tblen_ptr;
  gint    *map       = *map_ptr;
  gint     map_alloc = *map_alloc_ptr;

#ifdef DEBUG_CKSUM
  g_print ("Record mbck %d: %04x\n", tblen, hash);
#endif

  if (map_alloc == 0)
    {
      map = g_new (gint, RECSIZE);
      tbseg = g_new (guint16, RECSIZE);
      map_alloc = RECSIZE;
    }
  else if (tblen == map_alloc)
    {
      map_alloc *= 2;
      map = g_realloc (map, map_alloc * sizeof (gint));
      tbseg = g_realloc (tbseg, map_alloc * sizeof (guint16));
    }

  map[tblen] = begin;
  tbseg[tblen] = hash;
  tblen += 1;

  *tbseg_ptr     = tbseg;
  *tblen_ptr     = tblen;
  *map_ptr       = map;
  *map_alloc_ptr = map_alloc;
}

static void
multibyte_digest (MatchQuery     *query,
		  const guint8   *seg,
		  gint            seglen,
		  guint16       **tbseg,
		  gint           *tblen,
		  gint          **map,
		  gint           *map_alloc)
{
  int i;
  gint last_start = 0;
  gint16 h = 0, g;

  *map = NULL;
  *map_alloc = 0;
  *tbseg = NULL;
  *tblen = 0;

  for (i = 0; i < seglen; i += 1)
    {
      h = (h << 4) + seg[i];

      if ((g = h & 0xf000))
	{
	  h = h ^ (g >> 8);
	  h = h ^ g;
	}

      if ( (* query->break_segment) (query, seg, i, seglen) )
	{
	  record_multibyte_atom (query, last_start, h, tbseg, tblen, map, map_alloc);
	  h = 0;
	  last_start = i + 1;
	}
    }

  record_multibyte_atom (query, seglen, 0, tbseg, tblen, map, map_alloc);
  *tblen -= 1; /* the last atom added is past the end of the array hash, so
		* that the map contains the valid end mapping. */
}

static gint index_to_real_offset_1 (gint x, gint* map)
{
  return x;
}

static gint index_to_real_offset_2 (gint x, gint* map)
{
  return map[x];
}

/* Prep functions are responsible for filling in init_*_size_log,
 * the OB (one-byte) or TB (two-byte) pointers, the checksums,
 * the find_matches and grow functions.
 */
static void
prep_multibyte_from_seg (MatchQuery* query, FromSegment *from)
{
  guint16 *tbseg;
  gint tblen;

  if (from->ckarray)
    return;

  multibyte_digest (query,
		    from->real_seg,
		    from->real_len,
		    &tbseg,
		    &tblen,
		    &from->multibyte_map,
		    &from->multibyte_alloc);

  from->seg.tb = tbseg;
  from->len = tblen;

  from->ckarray = generate_checksums_2 (tbseg, tblen, ipow2(query->size), from);
}

static void
prep_multibyte_xdelta (MatchQuery* query)
{
  guint16 *tbseg;
  gint tblen;
  gint i;

  for (i=0; i<query->from_count; i += 1)
    prep_multibyte_from_seg (query, query->from[i]);

  multibyte_digest (query,
		    query->real_to_seg,
		    query->real_to_len,
		    &tbseg,
		    &tblen,
		    &query->multibyte_to_map,
		    &query->multibyte_to_alloc);

  query->to_seg.tb = tbseg;
  query->to_len = tblen;

  query->find_matches         = find_matches_2;
  query->grow_match           = grow_2;
  query->index_to_real_offset = index_to_real_offset_2;
}

static void
prep_singlebyte_from_seg (MatchQuery* query, FromSegment* from)
{
  if (from->ckarray)
    return;

  from->seg.ob = from->real_seg;
  from->len    = from->real_len;

  from->multibyte_map = NULL;
  from->multibyte_alloc = 0;

  from->ckarray = generate_checksums_1 (from->seg.ob, from->len, ipow2(query->size), from);
}

static void
prep_singlebyte_xdelta (MatchQuery* query)
{
  gint i;

  query->to_seg.ob   = query->real_to_seg;
  query->to_len   = query->real_to_len;

  for (i = 0; i < query->from_count; i += 1)
    prep_singlebyte_from_seg (query, query->from[i]);

  query->find_matches         = find_matches_1;
  query->grow_match           = grow_1;
  query->index_to_real_offset = index_to_real_offset_1;
}

GSList*
xdelta (MatchQuery* query)
{
  gint i;
  gint total_from_len = 0;

  for (i=0; i<query->from_count; i += 1)
    total_from_len += query->from[i]->real_len;

  if (query->real_to_len == 0)
    ; /* Nothing to output. */
  else if (total_from_len < ipow2(query->size) || query->real_to_len < ipow2(query->size))
    ; /* Lengths are too short, guaranteed to copy the whole file.
       * Output generated during emit_rest. */
  else
    {
      int i, j, total_from_ck_count = 0;

      if (query->break_segment)
	prep_multibyte_xdelta (query);
      else
	prep_singlebyte_xdelta (query);

      for (i=0; i<query->from_count; i += 1)
	{
	  total_from_ck_count += query->from[i]->ckarray->ck_count;
	  query->from[i]->segment_index = i;
	}

      query->table = c_hash_table_new(ilog2(total_from_ck_count)+1);

      for (i=0; i<query->from_count; i += 1)
	{
	  FromSegment* from = query->from[i];

	  for (j=0;j<from->ckarray->ck_count;j+=1)
	    c_hash_table_insert (query->table,
				 from->ckarray->cksums + j,
				 from->ckarray);
	}

#ifdef DEBUG_HASH
      g_print ("***** Hash stats: *****\n");
      c_hash_stats (query->table);
#endif

      compute_diffs (query);
    }

  emit_rest (query);

  return query->xdelta;
}

MatchQuery* match_query_new (gint (*break_segment) (MatchQuery* query,
						    const guint8* seg,
						    gint index,
						    gint length),
			     const guint8 *real_to_seg,
			     gint          real_to_len,
			     gint          size,
			     FromSegment  **from,
			     gint           from_count)
{
  MatchQuery* query = g_new0 (MatchQuery, 1);

  query->xdelta_chunk = g_mem_chunk_new ("xdelta", sizeof (XdeltaInstruction), 1024, G_ALLOC_ONLY);
  query->break_segment = break_segment;
  query->real_to_seg = real_to_seg;
  query->real_to_len = real_to_len;
  query->size = size;
  query->from = from;
  query->from_count = from_count;

  return query;
}

void match_query_free (MatchQuery *query)
{
  if (query->multibyte_to_map)
    {
      g_free (query->multibyte_to_map);
      g_free ((guint16*)query->to_seg.tb);
    }

  if (query->xdelta_chunk)
    g_mem_chunk_destroy (query->xdelta_chunk);

  if (query->xdelta)
    g_slist_free (query->xdelta);

  if (query->table)
    c_hash_table_free (query->table);

  g_free (query);
}

FromSegment* from_segment_new (const guint8* seg,
			       gint len)
{
  FromSegment *from = g_new0 (FromSegment, 1);

  from->real_seg = seg;
  from->real_len = len;

  return from;
}

void from_segment_free (FromSegment* from)
{
  if (from)
    {
      if (from->multibyte_map)
	{
	  g_free (from->multibyte_map);
	  g_free ((guint8*)from->seg.tb);
	}

      if (from->ckarray)
	{
	  g_free (from->ckarray->cksums);
	  g_free (from->ckarray);
	}

      g_free (from);
    }
}
