// This may look like C code, but it is really -*- C++ -*-

//<copyright>
//
// Copyright (c) 1996
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        attributes.h
//
// Purpose:     Interface to class AnyLight, PosDirLight, and SpotLight
//
// Created:     23 Feb 1996  Georg Meszaros
//
// Modified:    15 Jun 1996  Georg Meszaros     
//
//
// Description:
//
// classes AnyLight, PosDirLight and Spotlight, implemented to save
// the corresponding lights to a faceset in a LightList.
// The LightList stores AnyLights which is virtual and is dynamically
// either a SpotLight or a PosDirLight.
//
//</file>


//<class>
//
// Name: AnyLight  
//
// Purpose: The parent class for different types of light
//
//
// Public Interface:
/*  
   - AnyLight() { }
   - virtual ~AnyLight() { }
   - virtual void setLight(int index) = 0;
   - virtual void switchOff() = 0;
   - virtual void showLight() = 0;   
   - virtual void print(); 
  
*/
// Description:
//
// All of the methods need to be overridden.
// The LightList holds pointers to AnyLight(s) which are actually
// SpotLight(s) or PosDirlight(s).
//
//</class>


//<class>
//
// Name: PosDirLight  
//
// Purpose: A special type of light, child of AnyLight
//          Parallel light from point in pos_dir direction
//
// Public Interface:
/*  
   - PosDirLight(const colorRGB&,
                const point3D& point,
                int pos_dir,
                int camlg);
   - ~PosDirLight() { }

   - void setLight(int index);
     > sets the ge3d-light and turns it on     

   - void switchOff();
     > switch the light in ge3d off

   - void showLight();
     > just a help feature - draws a wired box at the light position

   - void print();
     > writes the light settings to cerr
*/
// Description:
//
// Look at the Qv-Structure or VRML for details
//
//</class>


//<class>
//
// Name: SpotLight  
//
// Purpose: A special type of light, child of AnyLight
//          Spot light from point in dir direction, droprate and the
//          cutoff angle restrict the lighting-volume
//
// Public Interface:
/*
   - SpotLight(const colorRGB& color,
              const point3D& point,
              const vector3D& dir,
              float droprate,
              float cutangle);
   - ~SpotLight() { } 

   - void setLight(int index);
     > sets the ge3d-light and turns it on     

   - void switchOff();
     > switch the light in ge3d off

   - void showLight();
     > just a help feature - draws a wired box at the light position

   - void print();
     > writes the light settings to cerr
*/
// Description:
//
// Look at the Qv-Structure or VRML for details
//
//</class>






#ifndef harmony_scene_attributes_h
#define harmony_scene_attributes_h

#include "ge3d/vectors.h"
#include "ge3d/ge3d.h"
#include "ge3d/mtl.h"



class AnyLight
{
  public:
    AnyLight() { }
    virtual ~AnyLight() { }
    virtual void setLight(int index) = 0;
    virtual void switchOff() = 0;
    virtual void showLight() = 0;   
    virtual void print(); 
};


class PosDirLight:public AnyLight
{ 
  public:
    PosDirLight(const colorRGB&,
                const point3D& point,
                int pos_dir,
                int camlg);
    ~PosDirLight() { }
    void setLight(int index);
    void switchOff();
    void showLight();
    void print();

  private:
    int index_;  // from 1 upwards
    colorRGB color_;
    point3D point_;
    int pos_dir_; 
    int camlg_;  // always false in VRML scenes  
};

   
class SpotLight:public AnyLight
{
  public:
    SpotLight(const colorRGB& color,
              const point3D& point,
              const vector3D& dir,
              float droprate,
              float cutangle);
    ~SpotLight() { } 
    void setLight(int index);
    void switchOff();
    void showLight();
    void print();

  private:
    int index_;  // from 1 upwards
    colorRGB color_;
    point3D point_;
    vector3D dir_;
    float droprate_; 
    float cutangle_;
};

#endif
