// This may look like C code, but it is really -*- C++ -*-

//<copyright>
//
// Copyright (c) 1992,93,94,95,96
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        sdfscene.h
//
// Purpose:     interface to 3D scene, SDF representation
//
// Created:     24 Apr 95   Michael Pichler (extracted from scene3d.h)
//
// Changed:     12 Jun 96   Michael Pichler
//
// $Id: sdfscene.h,v 1.9 1997/02/25 17:03:58 mpichler Exp $
//
//</file>


#ifndef harmony_scene_sdfscene_h
#define harmony_scene_sdfscene_h

#include "scenedata.h"

#include "slist.h"
struct objindexstruct;
class SDFCamera;


class SDFScene: public SceneData
{ 
  public:
    SDFScene (Scene3D* scene);
    ~SDFScene ();

    // *** input ***
    int readFile (FILE* file);
    const char* formatName () const  { return "SDF"; }
    void loadTextures ();

    // *** output ***
    int writeData (ostream& os, int format);  // write SDF file to output stream
    void printInfo (int all);           // print scene information
    int supportsOutputformat (int /*format*/)  { return 1; }

    unsigned long getNumFaces () const  // number of faces (polygons)
    { return numfaces_; }

    // *** drawing ***
    void draw (int mode);               // draw whole scene (in current mode)

    // *** picking ***
    void* pickObject (
      const point3D& A,                 //   ray start
      const vector3D& b,                //   ray direction
      float tnear,                      //   near distance
      float tfar,                       //   far distance
      GeometricObject** gobj = 0,       //   GeometricObject hit (return)
      QvNode** node = 0,                //   node hit (return)
      QvWWWAnchor** anchor = 0,         //   anchor hit (return)
      point3D* hitpoint = 0,            //   optionally calculates hit point
      vector3D* normal = 0,             //   and face normal vector (normalized)
      const StringArray** groups = 0,   //   optionally determines groups hit
      IntArray** hitpath = 0,           //   only used for VRML (ignored)
      float* hittime = 0                //   optionally returns the hit time
    );

    // *** find objects (SDF) ***
    GeometricObject* findObject (const char* objname);
    GeometricObject* findObject (int objnum);
    Material* findMaterial (const char* matname);

    // *** anchors ***
    void clearAnchors ();               // clear all anchor definitions
    void defineHyperGanchor (           // store Hyper-G anchor
      long id,                          //   anchor id
      const RString& aobj,              //   anchor object
      const char* posstr,               //   anchor position field
      const RString& title              //   anchor title (descr.)
    );
    void findTextureMaterial (          // find texture material
      const char* name,                 //   name (Hyper-G texture anchor)
      Material*& material,              //   returns SDF material
      QvTexture2*&
    )  { material = findMaterial (name); }

    // *** camera ***
    Camera* getCamera () const;         // get active camera
    void storeCamera ();                // store active camera
    void restoreCamera ();              // reset camera (to one read from file or latest stored)

  private:
    // scene reading subroutines
    int readActFile (FILE*, int&);
    int readPosFile (FILE*, int);
    int readCamFile (FILE*);
    int readMtlFile (FILE*);

    // scene writing subroutines (SDF)
    void writeActors (ostream&);
    void writePositions (ostream&);
    void writeCameras (ostream&);
    void writeMaterials (ostream&);
    void writeLights (ostream&);
    void writeObjects (ostream&);

    // VRML writing subroutines
    void writeVRML (ostream&);

    void buildMatrices ();  // compute accumulative transformation matrices
    void buildMatrix (objindexstruct*, int);

    slist gobjlist_,    // geometric objects
          camlist_,     // sdf-cameras
          lightlist_,   // lights
          allobjlist_,  // for building matrices
          materlist_;   // materials

    int num_objects_,
        num_gobj_,
        num_cam_,
        num_light_;
    unsigned long numfaces_;

    SDFCamera* activecam_;
    SDFCamera* origcam_;  // original camera (read from file)

    int matbuilt_;  // flag: trf matrices built (set on first drawing)
    int texturesloaded_;  // flag: textures loaded

    // prevent copying (declared, but not implemented)
    SDFScene (const SDFScene&);
    SDFScene& operator = (const SDFScene&);

}; // SDFScene


#endif
