//<copyright>
//
// Copyright (c) 1992,93,94,95,96,97
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        themenus.C
//
// Purpose:     implementation of menu and dialog classes for the scene viewer
//
// Created:     17 Dec 92   Michael Pichler
//
// Changed:     21 Feb 97   Alexander Nussbaumer (editing)
//
// Changed:     24 Apr 97   Michael Pichler
//
// $Id: themenus.C,v 1.50 1997/04/30 10:44:22 mpichler Exp $
//
//</file>



#include "themenus.h"

#include "icons.h"
#include "camera.h"
#include "editdlg.h"
#include "gecontext.h"
#include "hgptrhand.h"
#include "vecutil.h"
#include "stranslate.h"
#include "geomobj.h"
#include "scenewin.h"
#include "vrmlscene.h"
#include "bsptree.h"

#include <hyperg/widgets/bubblewindow.h>
#include <hyperg/widgets/coldlg.h>
#include <hyperg/widgets/cursors.h>
#include <hyperg/widgets/fchooser.h>
#include <hyperg/widgets/fieldb.h>
#include <hyperg/widgets/glyphutil.h>
#include <hyperg/widgets/menus.h>
#include <hyperg/widgets/msgbox.h>
#include <hyperg/widgets/textb.h>
#include <hyperg/widgets/xpmraster.h>

#include <ge3d/ge3d.h>

#include <hyperg/utils/verbose.h>

#include <InterViews/background.h>
#include <InterViews/helpwindowaction.h>
#include <InterViews/image.h>
#include <InterViews/label.h>
#include <InterViews/layout.h>
#include <InterViews/patch.h>
#include <InterViews/session.h>
#include <InterViews/style.h>
#include <IV-look/kit.h>
#include <IV-look/menu.h>

#include <NLS/locale.h>  /* IV locale extension (internationalization) */

#include "instrumented.h"

#include <string.h>
#include <iostream.h>
#include <fstream.h>

/* access */
#ifdef PMAX
#  include <sysent.h>
#endif
#include <unistd.h>

declareActionCallback (SceneMenus)
implementActionCallback (SceneMenus)

declareActionCallback (Scene3D)
implementActionCallback (Scene3D)

declareActionCallback (SceneWindow)
implementActionCallback (SceneWindow)

declareHelpWindowActionCallback (SceneMenus)
implementHelpWindowActionCallback (SceneMenus)




/*** MenuGroupCallback ***/


MenuGroupCallback::MenuGroupCallback (SceneMenus* sm, SceneMenuMemberFunction fu, int arg)
{
  sm_ = sm;
  fu_ = fu;
  arg_ = arg;
}


void MenuGroupCallback::execute ()
{
  (sm_->*fu_) (arg_);
}


/*** LoadDemoCallback ***/
// action for loading a demo

class LoadDemoCallback: public Action
{
  public:
    LoadDemoCallback (
      SceneMenus*,
      const char* demo  // assumed to point to static string (not copied!)
    );

    virtual void execute ();

  private:
    SceneMenus* sm_;
    const char* demo_;
};


LoadDemoCallback::LoadDemoCallback (SceneMenus* sm, const char* demo)
{
  sm_ = sm;
  demo_ = demo;
}


void LoadDemoCallback::execute ()
{
  sm_->loadDemo (demo_);
}


/*** BrowseDemoCallback ***/
// action for loading and browsing a demo (camera positioning)

class BrowseDemoCallback: public Action
{
  public:
    BrowseDemoCallback (
      SceneMenus*,
      const char* demo,  // see note of LoadDemoCallback
      float px, float py, float pz,  // camera position
      float lx, float ly, float lz   // lookat point
    );

    virtual void execute ();

  private:
    SceneMenus* sm_;
    const char* demo_;
    point3D position_, lookat_;
};


BrowseDemoCallback::BrowseDemoCallback (
  SceneMenus* sm, const char* demo,
  float px, float py, float pz,
  float lx, float ly, float lz
)
{
  sm_ = sm;
  demo_ = demo;
  init3D (position_, px, py, pz);
  init3D (lookat_, lx, ly, lz);
}


void BrowseDemoCallback::execute ()
{
  sm_->browseDemo (demo_, position_, lookat_);
}


/*** SetColourCallback ***/
// action for setting a predefined colour

class SetColourCallback: public Action
{
  public:
    SetColourCallback (
      SceneMenus*,
      colorRGB* color,
      float r, float g, float b,
      int needsrebuild = 0
    );

    virtual void execute ();

  private:
    SceneMenus* sm_;
    colorRGB* color_;
    colorRGB predef_;
    int rebuild_;
};


SetColourCallback::SetColourCallback (
  SceneMenus* sm, colorRGB* color, float r, float g, float b, int needsrebuild
)
{
  sm_ = sm;
  color_ = color;
  initRGB (predef_, r, g, b);
  rebuild_ = needsrebuild;
}


void SetColourCallback::execute ()
{
  *color_ = predef_;
  if (rebuild_)
    sm_->scene_->colorRebuild ();
  sm_->colourChanged ();
}


/*** ViewpointCallback ***/
// action for predefined viewpoints

class ViewpointCallback: public Action
{
  public:
    ViewpointCallback (
      SceneMenus* sm, const char* name, QvPerspectiveCamera* pcam, QvOrthographicCamera* ocam
    )
    { sm_ = sm;  name_ = name;  pcam_ = pcam;  ocam_ = ocam;
    }

    virtual void execute ();

  private:
    SceneMenus* sm_;
    RString name_;
    QvPerspectiveCamera* pcam_;
    QvOrthographicCamera* ocam_;
};


void ViewpointCallback::execute ()
{
  SceneWindow* scene = sm_->scene_;
  GEContext* gecontext = scene->gecontext ();

  if (gecontext)
    gecontext->pushCursor (HgCursors::instance ()->hourglass ());

  scene->activateCamera (name_, pcam_, ocam_);  // includes a rebuild

  if (gecontext)
    gecontext->popCursor ();

  sm_->inputhand_->reset ();
  scene->redraw ();
}



/*** SceneMenus ***/


SceneMenus::SceneMenus (
  Patch* menupatch, Patch* buttonpatch, SceneWindow* scene,
  GEContext* gecontext, HG3dInputHandler* inputhand,
  WidgetKit* kit, const LayoutKit* layout
)
{
  menupatch_ = menupatch;
  menubar_ = 0;
  editmapped_ = 0;
  editIdx_ = 0;
  buttonpatch_ = buttonpatch;
  scene_ = scene;
  gecontext_ = gecontext;
  inputhand_ = inputhand;
  kit_ = kit;
  layout_ = layout;
  openfc_ = 0;
//savefc_ = 0;

  // anuss
//   insertfc_ = 0;
  insertobjectdialog_ = 0; 
  texturedialog_ = 0; 
  materialdialog_ = 0; 
  wwwanchordialog_ = 0; 
  structureviewerdialog_ = 0;
  transformdialog_ = 0;
  griddialog_ = 0;
  viewpointdialog_ = 0;

  coldlg_ = 0;
  errorwindow_ = 0;
  errorbrowser_ = 0;
  navsetdlg_ = 0;
  sballwindow_ = 0;

  viewpointsPr_ = 0;
  viewptsParent_ = 0;
  viewpointsIndex_ = 0;
  selectionIndex_ = 0;
  viewproffset_ = 0;
  curviewpoint_ = 0;

  const int enabled_toggle = TelltaleState::is_enabled | TelltaleState::is_toggle;
  const int enabled_choosable = TelltaleState::is_enabled | TelltaleState::is_choosable;
#ifdef SPACEBALL
  const int enabled_toggle_choosen = enabled_toggle | TelltaleState::is_chosen;
#endif

  saveasstate_ = 0;  // new TelltaleState (0);  // disabled until scene to save (not used)
  exportSDFstate_ = new TelltaleState (0);  // disabled until scene to export
  exportVRMLstate_ = new TelltaleState (0);  // disabled until scene to export
  Resource::ref (saveasstate_);
  Resource::ref (exportSDFstate_);
  Resource::ref (exportVRMLstate_);

  // anuss
  editstate_ = new TelltaleState (TelltaleState::is_toggle);  // disabled until VRML-scene is loaded
  editmodestate_ = new TelltaleState (enabled_toggle); 
  edittoolbarstate_ = new TelltaleState (enabled_toggle); 
  constraintsstate_ = new TelltaleState (enabled_toggle);
  Resource::ref (editstate_);
  Resource::ref (editmodestate_);
  Resource::ref (edittoolbarstate_);
  Resource::ref (constraintsstate_);

  int i;
  for (i = 0;  i < edlg_num;  i++)
  { editdlgstate_ [i] = new TelltaleState (enabled_toggle);
    Resource::ref (editdlgstate_ [i]);
  }

  for (i = 0;  i < sel_num;  i++)
  { selectionstate_ [i] = new TelltaleState (0);  // disabled while no object/anchor selected
    Resource::ref (selectionstate_ [i]);          // (see selectionChanged)
  }

  anchormotionstate_ = new TelltaleState (enabled_toggle);
  velocitycontrolstate_ = new TelltaleState (enabled_toggle);
  arbitraryrotstate_ = new TelltaleState (enabled_toggle);
  colldetectionstate_ = new TelltaleState (enabled_toggle);
  showanchorstate_ = new TelltaleState (enabled_toggle);
  texmipmaponstate_ = new TelltaleState (enabled_choosable);
  texmipmapoffstate_ = new TelltaleState (enabled_choosable);
  viewlgtstate_ = new TelltaleState (enabled_toggle);
  stereostate_ = new TelltaleState (enabled_toggle);
  Resource::ref (anchormotionstate_);
  Resource::ref (velocitycontrolstate_);
  Resource::ref (arbitraryrotstate_);
  Resource::ref (colldetectionstate_);
  Resource::ref (showanchorstate_);
  Resource::ref (texmipmaponstate_);
  Resource::ref (texmipmapoffstate_);
  Resource::ref (viewlgtstate_);
  Resource::ref (stereostate_);

  for (i = 0;  i < aa_num;  i++)
  { aaliasingstate_ [i] = new TelltaleState (enabled_toggle);
    Resource::ref (aaliasingstate_ [i]);
  }

  renderzbuf_ = new TelltaleState (enabled_choosable);
  renderbsp_ = new TelltaleState (enabled_choosable);
  Resource::ref (renderzbuf_);
  Resource::ref (renderbsp_);

#ifdef SPACEBALL
  sbflagstate_ [sb_translate] = new TelltaleState (enabled_toggle_choosen);
  sbflagstate_ [sb_rotate] = new TelltaleState (enabled_toggle_choosen);
  sbflagstate_ [sb_saf] = new TelltaleState (enabled_toggle);
#endif

  for (i = 0;  i < sb_numflags;  i++)
  {
#ifdef SPACEBALL
    Resource::ref (sbflagstate_ [i]);
#else
    sbflagstate_ [i] = nil;
#endif
  }

  sbsensfieldb_ = 0;
  sbsensitivity2_ = 0;

  // need telltale states for display modes to update menu on key accelerators;
  // join to group is done by the radioMenuItems
  wireframe_ = new TelltaleState (enabled_choosable);
  hiddenline_ = new TelltaleState (enabled_choosable);
  flatshading_ = new TelltaleState (enabled_choosable);
  smoothshading_ = new TelltaleState (enabled_choosable);
  texturing_ = new TelltaleState (enabled_choosable);
  Resource::ref (wireframe_);
  Resource::ref (hiddenline_);
  Resource::ref (flatshading_);
  Resource::ref (smoothshading_);
  Resource::ref (texturing_);

  // interactive drawing modes
  intsame_ = new TelltaleState (enabled_choosable);
  intwire_ = new TelltaleState (enabled_choosable);
  inthline_ = new TelltaleState (enabled_choosable);
  intflat_ = new TelltaleState (enabled_choosable);
  intsmooth_ = new TelltaleState (enabled_choosable);
  inttexture_ = new TelltaleState (enabled_choosable);
  Resource::ref (intsame_);
  Resource::ref (intwire_);
  Resource::ref (inthline_);
  Resource::ref (intflat_);
  Resource::ref (intsmooth_);
  Resource::ref (inttexture_);

  // navigation modes
  navflipobjstate_ = new TelltaleState (enabled_choosable);
  navwalkstate_    = new TelltaleState (enabled_choosable);
  navflystate_     = new TelltaleState (enabled_choosable);
  navflytostate_   = new TelltaleState (enabled_choosable);
  navheadsupstate_ = new TelltaleState (enabled_choosable);
  Resource::ref (navflipobjstate_);
  Resource::ref (navwalkstate_);
  Resource::ref (navflystate_);
  Resource::ref (navflytostate_);
  Resource::ref (navheadsupstate_);

  kit_->style ()->find_attribute ("demoPath", demopath_);
//cerr << "loading demos from path " << demopath_ << endl;

  // initial settings (default modes)
  scene_->setNavigationMode (NavMode::flip_obj);  // navigation mode
  scene_->mode (ge3d_smooth_shading);  // display mode
  scene_->linksColor (Scene3D::l_diffcolor);  // anchor highlighting mode
} // SceneMenus


SceneMenus::~SceneMenus ()
{
  // menubar_ ref/unref'ed by menupatch_

  Resource::unref (saveasstate_);
  Resource::unref (exportSDFstate_);
  Resource::unref (exportVRMLstate_);

  // anuss
  Resource::unref (editstate_);
  Resource::unref (editmodestate_);
  Resource::unref (edittoolbarstate_);
  Resource::unref (constraintsstate_);

  int i;
  for (i = 0;  i < edlg_num;  i++)
    Resource::unref (editdlgstate_ [i]);

  for (i = 0;  i < sel_num;  i++)
    Resource::unref (selectionstate_ [i]);

  Resource::unref (anchormotionstate_);
  Resource::unref (velocitycontrolstate_);
  Resource::unref (arbitraryrotstate_);
  Resource::unref (colldetectionstate_);
  Resource::unref (showanchorstate_);
  Resource::unref (texmipmaponstate_);
  Resource::unref (texmipmapoffstate_);
  Resource::unref (viewlgtstate_);
  Resource::unref (stereostate_);
  Resource::unref (renderzbuf_);
  Resource::unref (renderbsp_);

  for (i = 0;  i < aa_num;  i++)
    Resource::unref (aaliasingstate_ [i]);

#ifdef SPACEBALL
  for (i = 0;  i < sb_numflags;  i++)
    Resource::unref (sbflagstate_ [i]);
#endif
  Resource::unref (sbsensfieldb_);

  Resource::unref (wireframe_);
  Resource::unref (hiddenline_);
  Resource::unref (flatshading_);
  Resource::unref (smoothshading_);
  Resource::unref (texturing_);

  Resource::unref (intsame_);
  Resource::unref (intwire_);
  Resource::unref (inthline_);
  Resource::unref (intflat_);
  Resource::unref (intsmooth_);
  Resource::unref (inttexture_);

  Resource::unref (navflipobjstate_);
  Resource::unref (navwalkstate_);
  Resource::unref (navflystate_);
  Resource::unref (navflytostate_);
  Resource::unref (navheadsupstate_);

  Resource::unref (openfc_);
//  Resource::unref (insertfc_);
//Resource::unref (savefc_);
  delete coldlg_;

  delete errorwindow_;  // errorbrowser_ will be destroyed as part of it
  destroyNavsetdlg ();
  destroyEditDialogs ();

  delete sballwindow_;

  Resource::unref (viewpointsPr_);

} // ~SceneMenus


// updateLanguage
// language settings; create menus, buttons, dialogs

void SceneMenus::updateLanguage ()
{
  switch (slanguage)  // locale setting
  {
    case HgLanguage::Korean:
      Locale::setlocale ("ko_KR.EUC");
    break;
    case HgLanguage::Japanese:
      Locale::setlocale ("ja_JP.SJIS");
    break;
    default:
      Locale::setlocale ("C");
    break;
  }

  createMenus ();
  createButtons ();
  updateDialogs ();
}


MenuItem* SceneMenus::editMenu (WidgetKit& kit, const LayoutKit& layout)
{
  return MenuKit::menubarItem (
    kit, STranslate::str (STranslate::FileEDIT), editPulldown (kit, layout));
}


// createMenus
// creates a menubar with all its submenus and
// puts it into the menupatch, which is redrawn

void SceneMenus::createMenus ()
{
  // right aligned help pulldown would be nice, but InterViews aligns
  // all submenus of the menubar same way (see Menu::open)
  // align () would be easy to override,
  // but place () has no access to allocation data

  WidgetKit& kit = *kit_;
  const LayoutKit& layout = *layout_;

  Menu* menubar = MenuKit::menubar (kit, layout);
  menubar_ = menubar;  // ref/unref'ed by menupatch_

  menubar->append_item (MenuKit::menubarItem (
    kit, STranslate::str (STranslate::MenuFILE), filePulldown (kit, layout)));

  editIdx_ = menubar->item_count ();  // edit menu index in menubar (1)
  editmapped_ = scene_->manipulationAllowed ();
  if (editmapped_)
    menubar->append_item (editMenu (kit, layout));

  menubar->append_item (MenuKit::menubarItem (  // selectionItems assumes viewPulldown after editPulldown
    kit, STranslate::str (STranslate::MenuVIEW), viewptsParent_ = viewPulldown (kit, layout)));
  menubar->append_item (MenuKit::menubarItem (
    kit, STranslate::str (STranslate::MenuNAVIGATE), navigatePulldown (kit, layout)));
  menubar->append_item (MenuKit::menubarItem (
    kit, STranslate::str (STranslate::MenuANCHORS), anchorsPulldown (kit, layout)));
  if (kit.style ()->value_is_on ("DocumentMenuItems"))
  { menubar->append_item (MenuKit::menubarItem (
      kit, STranslate::str (STranslate::MenuDOCUMENT), documentPulldown (kit, layout)));
  }
  menubar->append_item (MenuKit::menubarItem (
    kit, STranslate::str (STranslate::MenuDISPLAY), displayPulldown (kit, layout)));
//   menubar->append_item (MenuKit::menubarItem (
//     kit, STranslate::str (STranslate::MenuOPTIONS), optionsPulldown (kit, layout)));
  menubar->append_item (MenuKit::menubarGlue (layout));
  menubar->append_item (MenuKit::menubarItem (
    kit, STranslate::str (STranslate::MenuHELP), helpPulldown (kit, layout)));

  menupatch_->body (menubar);
  menupatch_->reallocate ();
  menupatch_->redraw ();

} // createMenus



// createButtons
// create button tool bar

void SceneMenus::createButtons ()
{
  WidgetKit& kit = *kit_;
  const LayoutKit& layout = *layout_;

  Glyph *flip, *walk, *fly, *flyto, *headsup, *objmanip, *anchors;
  int useicons = kit.style ()->value_is_on ("icons");
  int usebubbles = useicons && !kit.style ()->value_is_on ("nobubbles");

  if (useicons)
  {
    flip = new Image (XPMRaster::read ((const char**) flip_xpm));
    walk = new Image (XPMRaster::read ((const char**) walk_xpm));
    fly = new Image (XPMRaster::read ((const char**) fly_xpm));
    flyto = new Image (XPMRaster::read ((const char**) flyto_xpm));
    headsup = new Image (XPMRaster::read ((const char**) headsup_xpm));
    objmanip = new Image (XPMRaster::read ((const char**) objmanip_xpm));
    anchors = new Image (XPMRaster::read ((const char**) anchors_xpm));
  }
  else
  {
    flip = kit.label (STranslate::str (STranslate::NavFLIPOBJECT));
    walk = kit.label (STranslate::str (STranslate::NavWALK));
    fly = kit.label (STranslate::str (STranslate::NavFLY));
    flyto = kit.label (STranslate::str (STranslate::NavFLYTO));
    headsup = kit.label (STranslate::str (STranslate::NavHEADSUP));
    objmanip = kit.label (STranslate::str (STranslate::EditOBJMANIP));
    anchors = kit.label (STranslate::str (STranslate::ButtonSHOWANCHORS));
  }

  kit.begin_style ("buttons");
  if (useicons)
    kit.style ()->attribute ("minimumWidth", "0");

  // currently no flip object submode buttons
  Button* flipbutton = WWidgetKit::palette_button (flip,
    new MenuGroupCallback (this, &SceneMenus::movementMode, NavMode::flip_obj), navflipobjstate_);
  Button* walkbutton = WWidgetKit::palette_button (walk,
    new MenuGroupCallback (this, &SceneMenus::movementMode, NavMode::walk_around), navwalkstate_);
  Button* flybutton  = WWidgetKit::palette_button (fly,
    new MenuGroupCallback (this, &SceneMenus::movementMode, NavMode::fly_1), navflystate_);
  Button* fly2button = WWidgetKit::palette_button (flyto,
    new MenuGroupCallback (this, &SceneMenus::movementMode, NavMode::fly_2), navflytostate_);
  Button* hupbutton  = WWidgetKit::palette_button (headsup,
    new MenuGroupCallback (this, &SceneMenus::movementMode, NavMode::heads_up), navheadsupstate_);
  Button* editbutton = WWidgetKit::palette_button (objmanip,
    new ActionCallback(SceneMenus) (this, &SceneMenus::toggleEditMode), editmodestate_);
  Button* anchbutton = WWidgetKit::palette_button (anchors,
    new ActionCallback(SceneMenus) (this, &SceneMenus::showAnchorsChanged), showanchorstate_);

  if (usebubbles)  // bubble help windows
  {
    long helpdelay = 1;  // default: 1 s
    kit.style ()->find_attribute ("helpdelay", helpdelay);
    HelpWindowActionCallback(SceneMenus)* hwac =
      new HelpWindowActionCallback(SceneMenus) (this, &SceneMenus::bubbleHelpWindow);

    flipbutton->set_help_window_action (new HelpWindowAction (hwac, STranslate::NavFLIPOBJECT));
    flipbutton->set_help_delay (helpdelay);
    walkbutton->set_help_window_action (new HelpWindowAction (hwac, STranslate::NavWALK));
    walkbutton->set_help_delay (helpdelay);
    flybutton->set_help_window_action (new HelpWindowAction (hwac, STranslate::NavFLY));
    flybutton->set_help_delay (helpdelay);
    fly2button->set_help_window_action (new HelpWindowAction (hwac, STranslate::NavFLYTO));
    fly2button->set_help_delay (helpdelay);
    hupbutton->set_help_window_action (new HelpWindowAction (hwac, STranslate::NavHEADSUP));
    hupbutton->set_help_delay (helpdelay);
    editbutton->set_help_window_action (new HelpWindowAction (hwac, STranslate::EditOBJMANIP));
    editbutton->set_help_delay (helpdelay);
    anchbutton->set_help_window_action (new HelpWindowAction (hwac, STranslate::ButtonSHOWANCHORS));
    anchbutton->set_help_delay (helpdelay);
  }

  Glyph* toolbox = layout.hbox (
    flipbutton,
    layout.hspace (5),
    walkbutton,
    flybutton,
    fly2button,
    hupbutton
  );

  if (editstate_->test (TelltaleState::is_chosen))  // ### may add/remove this button dynamically
  {
    toolbox->append (layout.hspace (15));
    toolbox->append (editbutton);  // object manipulation
  }

  if (useicons)
  {
    toolbox->append (layout.hglue (10, fil, 0));
    toolbox->append (anchbutton);
    toolbox->append (layout.hspace (10));
    toolbox->append (new Image (XPMRaster::read ((const char**) vrwebicon_xpm)));
  }
  else
  { toolbox->append (layout.hglue (10, fil, 0));
    toolbox->append (anchbutton);
  }

  if (edittoolbarstate_->test (TelltaleState::is_chosen))
  { // edit toolbar (anuss)
    TelltaleState** eds = editdlgstate_;

    Glyph* insertbutton = WWidgetKit::palette_button (STranslate::str (STranslate::EditbuttonINSERT),
      new ActionCallback (SceneMenus)(this, &SceneMenus::toggleInsertObjectDialog), eds [edlg_insertobj]);
    Glyph* anchorbutton = WWidgetKit::palette_button (STranslate::str (STranslate::EditbuttonANCHOR),
      new ActionCallback (SceneMenus)(this, &SceneMenus::toggleWWWAnchorDialog), eds [edlg_anchor]);
    Glyph* texturebutton = WWidgetKit::palette_button (STranslate::str (STranslate::EditbuttonTEXTURE),
      new ActionCallback (SceneMenus)(this, &SceneMenus::toggleTextureDialog), eds [edlg_texture]);
    Glyph* materialbutton = WWidgetKit::palette_button (STranslate::str (STranslate::EditbuttonMATERIAL),
      new ActionCallback (SceneMenus)(this, &SceneMenus::toggleMaterialDialog), eds [edlg_material]);
    Glyph* structurebutton = WWidgetKit::palette_button (STranslate::str (STranslate::EditbuttonSTRUCTURE),
      new ActionCallback (SceneMenus)(this, &SceneMenus::toggleStructureViewerDialog), eds [edlg_structure]);
    Glyph* transformbutton = WWidgetKit::palette_button (STranslate::str (STranslate::EditbuttonTRANSFORM),
      new ActionCallback (SceneMenus)(this, &SceneMenus::toggleTransformDialog), eds [edlg_transform]);
    Glyph* viewpointbutton = WWidgetKit::palette_button (STranslate::str (STranslate::EditbuttonVIEWPOINT),
      new ActionCallback (SceneMenus)(this, &SceneMenus::toggleViewpointDialog), eds [edlg_viewpoint]);
    Glyph* gridbutton = WWidgetKit::palette_button (STranslate::str (STranslate::EditbuttonGRID),
      new ActionCallback (SceneMenus)(this, &SceneMenus::toggleGridDialog), eds [edlg_grid]);

    Glyph* editbox = layout.hbox (
      insertbutton,
      anchorbutton,
      texturebutton,
      materialbutton,
      structurebutton,
      transformbutton,
      viewpointbutton,
      gridbutton,
      layout.hglue ()
    );

    toolbox = layout.vbox (toolbox, layout.vspace (2), editbox);
  }

  kit.end_style ();  // buttons

  buttonpatch_->body (toolbox);
  buttonpatch_->reallocate ();
  buttonpatch_->redraw ();

} // createButtons


Window* SceneMenus::bubbleHelpWindow (int i)
{
  // callback for creating bubble help window
//  kit_->begin_style ("BubbleWindow");
  Glyph* label = // multiLineLabel
    kit_->label (STranslate::str ((STranslate::SStrSym) i));
//  kit_->end_style ();
  BubbleWindow* window = new BubbleWindow (label);

  // map bubble windows to the upside
  // - for OpenGL because background is garbaged otherwise on SGIs
  // - anyway (especially for Mesa) to avoid redraw of complete scene
  window->direction (BubbleWindow::RIGHT, BubbleWindow::UP);

  return window;

} // bubbleHelpWindow


void SceneMenus::updateDialogs ()
{
  updateAbout (*kit_, *layout_);

  // update file choosers
  if (openfc_)
    buildOpenFC ();
//   if (insertfc_)
//     buildInsertFC ();
// if (savefc_) ...

  if (coldlg_)
  {
    int mapped = coldlg_->ismapped ();
    buildColourDialog ();  // deletes and rebuilds colour dialog
    if (mapped)
      coldlg_->map ();
  }

} // updateDialogs


void SceneMenus::selectionChanged ()
{
  // SDF/VRML
  GeometricObject* selobj = scene_->selectedObj ();
  int selection = selobj || scene_->selectedNode ();
  int anchorsel = (selobj && selobj->getCurrentAnchor ()) || scene_->selectedAnchor ();

  // TelltaleState checks whether state really changes
  selectionstate_ [sel_sourceanchor]->set (TelltaleState::is_enabled, selection);
  selectionstate_ [sel_deleteanchor]->set (TelltaleState::is_enabled, anchorsel);
  selectionstate_ [sel_anchorinfo]->set (TelltaleState::is_enabled, anchorsel);
  selectionstate_ [sel_activateanchor]->set (TelltaleState::is_enabled, anchorsel);
  selectionstate_ [sel_parent]->set (TelltaleState::is_enabled, selection);
  selectionstate_ [sel_child]->set (TelltaleState::is_enabled, selection);
  selectionstate_ [sel_previous]->set (TelltaleState::is_enabled, selection);
  selectionstate_ [sel_next]->set (TelltaleState::is_enabled, selection);
  selectionstate_ [sel_removesel]->set (TelltaleState::is_enabled, selection);
  selectionstate_ [sel_view]->set (TelltaleState::is_enabled, selection);

} // selectionChanged


void SceneMenus::sceneChanged ()
{
//existance of a input file not known at the moment this function is called
//saveasstate_->set (TelltaleState::is_enabled, scene_->supportsOutputformat (Scene3D::write_ORIGINAL));
  exportSDFstate_->set (TelltaleState::is_enabled, scene_->supportsOutputformat (Scene3D::write_SDF));
  exportVRMLstate_->set (TelltaleState::is_enabled, scene_->supportsOutputformat (Scene3D::write_VRML));

  const char* sceneformat = scene_->formatName ();
  int sdf = sceneformat && !strcmp (sceneformat, "SDF");
  // different selection options for SDF: no parent/children/siblings, but Object/Groups option
  selectionItems (viewptsParent_, *kit_, *layout_, sdf);

  // set to viewer mode (anuss:)
  editstate_->set (TelltaleState::is_enabled, scene_->isEditable () ? 1 : 0);  // allow editing, if VRML scene

} // sceneChanged


void SceneMenus::sceneCleared ()
{
  savefilename_ = "";
}


Menu* SceneMenus::filePulldown (WidgetKit& kit, const LayoutKit& layout)
{
  Menu* file_pd = kit.pulldown ();

  // anuss
  file_pd->append_item (MenuKit::menuItem (
    kit, layout, "New", // TODO use stranslate instead
    new ActionCallback(SceneMenus) (this, &SceneMenus::newFile)));

  file_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::FileOPEN),
    new ActionCallback(SceneMenus) (this, &SceneMenus::openFile), "F3"));

  if (kit.style ()->value_is_on ("DemoMenuItems"))
  {
    file_pd->append_item (MenuKit::menuItem (
      kit, layout, STranslate::str (STranslate::FileDEMO), demoPullright (kit, layout)));
    file_pd->append_item (MenuKit::menuItem (
      kit, layout, STranslate::str (STranslate::FileUSABILITYTESTS), usabilityPullright (kit, layout)));
  } // demo items

  Menu* export_pr = kit.pullright ();  // export suboptions (TODO: should be integrated into save dialog)
  export_pr->append_item (MenuKit::menuItem (kit, layout, "VRML",
                          new ActionCallback(SceneMenus) (this, &SceneMenus::saveVRMLScene), 0, exportVRMLstate_));
  export_pr->append_item (MenuKit::menuItem (kit, layout, "SDF",
                          new ActionCallback(SceneMenus) (this, &SceneMenus::saveSDFScene), 0, exportSDFstate_));

  // File Menu - common parts
  file_pd->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::FileSAVE),  // anuss
                        new ActionCallback(SceneMenus) (this, &SceneMenus::saveCurrentScene), "F2"));
  file_pd->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::FileSAVEAS),
                        new ActionCallback(SceneMenus) (this, &SceneMenus::saveOriginalScene), "^F2", saveasstate_));
  file_pd->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::FileEXPORT),
                        export_pr));
  file_pd->append_item (MenuKit::checkMenuItem (kit, layout, STranslate::str (STranslate::FileEDIT),
    new ActionCallback(SceneMenus) (this, &SceneMenus::toggleEditScene),
    (scene_->getEditMode () != EditMode::ed_viewer), "^e", editstate_));
  file_pd->append_item (MenuKit::disable (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::FilePRINT), (Action*) nil)));
  file_pd->append_item (MenuKit::disable (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::FileMAIL), (Action*) nil)));
  file_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::FilePARSERERRORS),
    new ActionCallback(SceneMenus) (this, &SceneMenus::toggleErrorDialog)));

  if (kit.style ()->value_is_on ("SpecialMenuItems"))
  {
    file_pd->append_item (kit.menu_item_separator ());

    file_pd->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::SpecialPRINTDATA),
                          new ActionCallback(SceneMenus) (this, &SceneMenus::printData)));
    file_pd->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::SpecialPRINTALL),
                          new ActionCallback(SceneMenus) (this, &SceneMenus::printAll)));
    file_pd->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::SpecialCAMERAINFO),
                          new ActionCallback(SceneMenus) (this, &SceneMenus::cameraInfo)));
#ifdef TODO  /* saveDepth */
    // will be integrated into FileSAVEAS !!!
    file_pd->append_item (MenuKit::menuItem (kit, layout, "save depth file",
                          new ActionCallback(SceneMenus) (this, &SceneMenus::saveDepth)));
#endif
  } // special items


  file_pd->append_item (kit.menu_item_separator ());
  file_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::FileEXITVIEWER),
    new ActionCallback(SceneMenus) (this, &SceneMenus::quitScene) /*kit.quit ()*/, "^x"));

  return file_pd;

} // filePulldown


Menu* SceneMenus::demoPullright (WidgetKit& kit, const LayoutKit& layout)
{
  Menu* demo_pr = kit.pullright ();

  // old file format (SDF); will go away soon
  demo_pr->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::DemoCUBE),
                        new LoadDemoCallback (this, "/cube/cube.sdf")));
  demo_pr->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::DemoROOM),
                        new LoadDemoCallback (this, "/room/room.sdf")));
  demo_pr->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::DemoCAR_VW),
                        new LoadDemoCallback (this, "/car/car.sdf")));
  demo_pr->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::DemoCORVETTE),
                        new LoadDemoCallback (this, "/corvette/car.sdf")));
  demo_pr->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::DemoENGINE),
                        new LoadDemoCallback (this, "/engine/engine.sdf")));
  demo_pr->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::DemoCLOCKTOWER),
                        new LoadDemoCallback (this, "/uhrturm/uhrturm.sdf")));
  demo_pr->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::DemoWORLD),
                        new LoadDemoCallback (this, "/world/world.sdf")));
  demo_pr->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::DemoIICMINST),
                        new LoadDemoCallback (this, "/inst/inst.sdf")));
  demo_pr->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::DemoIICMFOYER),
                        new BrowseDemoCallback (this, "/inst/inst.sdf", 10, 2, 22, 10, 2, 21)));
  demo_pr->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::DemoKANDHLEIT),
                        new LoadDemoCallback (this, "/inst/ka_hl.sdf")));


  return demo_pr;

} // demoPullright


Menu* SceneMenus::usabilityPullright (WidgetKit& kit, const LayoutKit& layout)
{
  Menu* ut_pr = kit.pullright ();

  ut_pr->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::UTestNORTH),
                      new BrowseDemoCallback (this, "/ihci/ihci.sdf", 6, 1.8, -10, 6, 1.8, -9)));
  ut_pr->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::UTestWEST),
                      new BrowseDemoCallback (this, "/ihci/ihci.sdf", -8, 1.8, 5, -7, 1.8, 5)));
  ut_pr->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::UTestSOUTH),
                      new BrowseDemoCallback (this, "/ihci/ihci.sdf", 6, 1.8, 20, 6, 1.8, 19)));
  ut_pr->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::UTestEAST),
                      new BrowseDemoCallback (this, "/ihci/ihci.sdf", 20, 1.8, 5, 19, 1.8, 5)));

  return ut_pr;

} // usabilityPullright



Menu* SceneMenus::editPulldown (WidgetKit& kit, const LayoutKit& layout)  // anuss
{
  Menu* edit_pd = kit.pulldown ();

  editmodestate_->set (TelltaleState::is_chosen, (scene_->getEditMode () == EditMode::ed_select) ? 1 : 0);
  edit_pd->append_item (MenuKit::checkMenuItem (kit, layout, STranslate::str (STranslate::EditOBJMANIP), 
                      new ActionCallback(SceneMenus) (this, &SceneMenus::toggleEditMode),
                      0, "^o", editmodestate_));
  edit_pd->append_item (MenuKit::checkMenuItem (kit, layout, "Edit Toolbar",
                      new ActionCallback(SceneMenus) (this, &SceneMenus::toggleEditToolbar),
                      0, 0, edittoolbarstate_));
  constraintsstate_->set (TelltaleState::is_chosen, scene_->editdata ()->constraints () ? 1 : 0);
  edit_pd->append_item (MenuKit::checkMenuItem (kit, layout, "Constraints",  
                      new ActionCallback(SceneMenus) (this, &SceneMenus::toggleConstraints),
                      0, 0, constraintsstate_));

  edit_pd->append_item (kit.menu_item_separator ());

  edit_pd->append_item (MenuKit::menuItem (kit, layout, "Undo",  
    new ActionCallback (SceneMenus) (this, &SceneMenus::undoManipulation), "^z"));
  edit_pd->append_item (MenuKit::menuItem (kit, layout, "Delete",  
    new ActionCallback (SceneMenus) (this, &SceneMenus::deleteSelectedNode), 0));
  edit_pd->append_item (MenuKit::menuItem (kit, layout, "Cut",  
    new ActionCallback (SceneMenus) (this, &SceneMenus::cutSelectedNode), 0));
  edit_pd->append_item (MenuKit::menuItem (kit, layout, "Copy", 
    new ActionCallback (SceneMenus) (this, &SceneMenus::copySelectedNode), 0));
  edit_pd->append_item (MenuKit::menuItem (kit, layout, "Paste",
    new ActionCallback (SceneMenus) (this, &SceneMenus::pasteSelectedNode), 0));
//   edit_pd->append_item (MenuKit::menuItem (kit, layout, "Insert File",  
//     new ActionCallback (SceneMenus) (this, &SceneMenus::insertFile), 0));

  edit_pd->append_item (kit.menu_item_separator ());

  TelltaleState** eds = editdlgstate_;
  eds [edlg_insertobj]->set (TelltaleState::is_chosen, insertobjectdialog_ && insertobjectdialog_->isMapped ());
  eds [edlg_anchor]->set (TelltaleState::is_chosen, wwwanchordialog_ && wwwanchordialog_->isMapped ());
  eds [edlg_texture]->set (TelltaleState::is_chosen, texturedialog_ && texturedialog_->isMapped ());
  eds [edlg_material]->set (TelltaleState::is_chosen, materialdialog_ && materialdialog_->isMapped ());
  eds [edlg_structure]->set (TelltaleState::is_chosen, structureviewerdialog_ && structureviewerdialog_->isMapped ());
  eds [edlg_transform]->set (TelltaleState::is_chosen, transformdialog_ && transformdialog_->isMapped ());
  eds [edlg_viewpoint]->set (TelltaleState::is_chosen, viewpointdialog_ && viewpointdialog_->isMapped ());
  eds [edlg_grid]->set (TelltaleState::is_chosen, griddialog_ && griddialog_->isMapped ());

  RString dots = "...";

  edit_pd->append_item (MenuKit::checkMenuItem (
    kit, layout, STranslate::str (STranslate::EditdialogINSERTOBJECT) + dots,
    new ActionCallback(SceneMenus) (this, &SceneMenus::toggleInsertObjectDialog),
    0, 0, eds [edlg_insertobj]));
  edit_pd->append_item (MenuKit::checkMenuItem (
    kit, layout, STranslate::str (STranslate::EditdialogANCHOREDITOR) + dots,
    new ActionCallback(SceneMenus) (this, &SceneMenus::toggleWWWAnchorDialog),
    0, 0, eds [edlg_anchor]));
  edit_pd->append_item (MenuKit::checkMenuItem (
    kit, layout, STranslate::str (STranslate::EditdialogTEXTUREEDITOR) + dots,
    new ActionCallback(SceneMenus) (this, &SceneMenus::toggleTextureDialog),
    0, 0, eds [edlg_texture]));
  edit_pd->append_item (MenuKit::checkMenuItem (
    kit, layout, STranslate::str (STranslate::EditdialogMATERIALEDITOR) + dots,
    new ActionCallback(SceneMenus) (this, &SceneMenus::toggleMaterialDialog),
    0, 0, eds [edlg_material]));
  edit_pd->append_item (MenuKit::checkMenuItem (
    kit, layout, STranslate::str (STranslate::EditdialogSTRUCTUREVIEWER) + dots,
    new ActionCallback(SceneMenus) (this, &SceneMenus::toggleStructureViewerDialog),
    0, 0, eds [edlg_structure]));
  edit_pd->append_item (MenuKit::checkMenuItem (
    kit, layout, STranslate::str (STranslate::EditdialogTRANSFORMEDITOR) + dots,
    new ActionCallback(SceneMenus) (this, &SceneMenus::toggleTransformDialog),
    0, 0, eds [edlg_transform]));
  edit_pd->append_item (MenuKit::checkMenuItem (
    kit, layout, STranslate::str (STranslate::EditdialogVIEWPOINTSETTINGS) + dots,
    new ActionCallback(SceneMenus) (this, &SceneMenus::toggleViewpointDialog),
    0, 0, eds [edlg_viewpoint]));
  edit_pd->append_item (MenuKit::checkMenuItem (
    kit, layout, STranslate::str (STranslate::EditdialogGRIDSETTINGS) + dots,
    new ActionCallback(SceneMenus) (this, &SceneMenus::toggleGridDialog),
    0, 0, eds [edlg_grid]));

  return edit_pd;

} // editPulldown


Menu* SceneMenus::viewPulldown (WidgetKit& kit, const LayoutKit& layout)
{
  Menu* view_pd = kit.pulldown ();

  // reset view, level view, untilt view
  view_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::NavigateRESETVIEW),
    new ActionCallback(SceneMenus) (this, &SceneMenus::resetView), "r"));
  view_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::NavigateLEVELVIEW),
    new ActionCallback(SceneMenus) (this, &SceneMenus::levelView), "l"));
  view_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::NavigateUNTILTVIEW),
    new ActionCallback(SceneMenus) (this, &SceneMenus::untiltView), "u"));

  view_pd->append_item (kit.menu_item_separator ());

  // viewpoints
  emptyViewpointsPullright (kit, layout);  // build viewpointsPr_
  viewpointsIndex_ = view_pd->item_count ();  // index that viewpointsPr_ will get
  // cerr << "viewpointsPr_ gets index " << viewpointsIndex_ << " in viewptsParent_" << endl;
  view_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::MenuVIEWPOINTS), viewpointsPr_));

  // view selection, view all
  view_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::NavigateVIEWSELECTION),
    new ActionCallback (SceneMenus) (this, &SceneMenus::viewSelectedNode), "s", selectionstate_ [sel_view]));
  view_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::NavigateVIEWWORLD),
    new ActionCallback (SceneMenus) (this, &SceneMenus::viewAll), "w"));

  view_pd->append_item (kit.menu_item_separator ());

  selectionIndex_ = view_pd->item_count ();  // index from which on selection items are format dependent
  selectionItems (view_pd, kit, layout, 0);  // VRML or empty scene

  return view_pd;

} // viewPulldown


void SceneMenus::selectionItems (Menu* view_pd, WidgetKit& kit, const LayoutKit& layout, int sdf)
{
  if (!view_pd || !selectionIndex_)
    return;

  int n = view_pd->item_count ();

  if (sdf)  // SDF: 3 more items
  { if (n == selectionIndex_ + 3)
      return;
  }
  else  // VRML: 5 more items
  { if (n == selectionIndex_ + 5)
      return;
  }

  // cerr << "updating selection items" << endl;

  // clear old entries
  while (n-- > selectionIndex_)
    view_pd->remove_item (n);

  // selection of objects/groups (SDF only)
  if (sdf)
  {
    static TelltaleGroup* osgroup = 0;
    if (!osgroup)
      osgroup = new TelltaleGroup ();

    int selmode = scene_->selectionMode ();
    view_pd->append_item (MenuKit::radioMenuItem (
      kit, layout, osgroup, STranslate::str (STranslate::AnchorSELECTOBJECTS),
      new MenuGroupCallback (this, &SceneMenus::selectionMode, SceneWindow::SelectObject),
      selmode == SceneWindow::SelectObject));
    view_pd->append_item (MenuKit::radioMenuItem (
      kit, layout, osgroup, STranslate::str (STranslate::AnchorSELECTGROUPS),
      new MenuGroupCallback (this, &SceneMenus::selectionMode, SceneWindow::SelectGroup),
      selmode == SceneWindow::SelectGroup));
  }
  else  // selections of parent/child/siblings (VRML only)
  {
    view_pd->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::SelectSELECTPARENT),
      new ActionCallback (SceneMenus) (this, &SceneMenus::selectParent), "-", selectionstate_ [sel_parent]));
    view_pd->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::SelectSELECTCHILD),
      new ActionCallback (SceneMenus) (this, &SceneMenus::selectChild), "+", selectionstate_ [sel_child]));
    view_pd->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::SelectSELECTPREVIOUS),
      new MenuGroupCallback (this, &SceneMenus::selectNext, -1), "Bsp", selectionstate_ [sel_previous]));
    view_pd->append_item (MenuKit::menuItem (kit, layout, STranslate::str (STranslate::SelectSELECTNEXT),
      new MenuGroupCallback (this, &SceneMenus::selectNext, 1), "Spc", selectionstate_ [sel_next]));
  }

  // deselect Object/Group/Node (all formats)
  view_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::AnchorDESELECT),
    new ActionCallback(SceneMenus) (this, &SceneMenus::removeSelection), 0, selectionstate_ [sel_removesel]));

  // must unbind old pulldown or size won't change
  // assert: view after or at place of edit
  MenuItem* item = menubar_->item (editIdx_ + editmapped_);
  Window* pdwin = item ? item->window () : 0;
  if (pdwin)
    pdwin->unbind ();

} // selectionItems


Menu* SceneMenus::navigatePulldown (WidgetKit& kit, const LayoutKit& layout)
{
  Menu* navigate_pd = kit.pulldown ();

  // Movement modes
  static TelltaleGroup* mmgroup = 0;
  if (!mmgroup)
    mmgroup = new TelltaleGroup ();
  const int mmode = scene_->navigationMode ();

  navigate_pd->append_item (MenuKit::radioMenuItem (
    kit, layout, mmgroup, STranslate::str (STranslate::NavFLIPOBJECT),
    new MenuGroupCallback (this, &SceneMenus::movementMode, NavMode::flip_obj),
    mmode == NavMode::flip_obj, "F4", navflipobjstate_));
  navigate_pd->append_item (MenuKit::radioMenuItem (
    kit, layout, mmgroup, STranslate::str (STranslate::NavWALK),
    new MenuGroupCallback (this, &SceneMenus::movementMode, NavMode::walk_around),
    mmode == NavMode::walk_around, "F5", navwalkstate_));
  navigate_pd->append_item (MenuKit::radioMenuItem (
    kit, layout, mmgroup, STranslate::str (STranslate::NavFLY),
    new MenuGroupCallback (this, &SceneMenus::movementMode, NavMode::fly_1),
    mmode == NavMode::fly_1, "F6", navflystate_));
  navigate_pd->append_item (MenuKit::radioMenuItem (
    kit, layout, mmgroup, STranslate::str (STranslate::NavFLYTO),
    new MenuGroupCallback (this, &SceneMenus::movementMode, NavMode::fly_2),
    mmode == NavMode::fly_2, "F7", navflytostate_));
  navigate_pd->append_item (MenuKit::radioMenuItem (
    kit, layout, mmgroup, STranslate::str (STranslate::NavHEADSUP),
    new MenuGroupCallback (this, &SceneMenus::movementMode, NavMode::heads_up),
    mmode == NavMode::heads_up, "F8", navheadsupstate_));

  navigate_pd->append_item (kit.menu_item_separator ());

  // navigation options pullright
  Menu* navopt_pr = kit.pullright ();
  navopt_pr->append_item (MenuKit::checkMenuItem (
    kit, layout, STranslate::str (STranslate::OptionENABLEMOTION),
    new ActionCallback(SceneMenus) (this, &SceneMenus::anchorMotionChanged),
    scene_->anchorMotion (), 0, anchormotionstate_));
  navopt_pr->append_item (MenuKit::checkMenuItem (
    kit, layout, STranslate::str (STranslate::OptionVELOCITYCONTROL),
    new ActionCallback(SceneMenus) (this, &SceneMenus::velocityControlChanged),
    scene_->velocityControl (), 0, velocitycontrolstate_));
  navopt_pr->append_item (MenuKit::checkMenuItem (
    kit, layout, STranslate::str (STranslate::OptionARBITRARYROTATIONS),
    new ActionCallback(SceneMenus) (this, &SceneMenus::arbitraryRotationChanged),
    scene_->arbitraryRotations (), 0, arbitraryrotstate_));
  navopt_pr->append_item (MenuKit::checkMenuItem (
    kit, layout, STranslate::str (STranslate::OptionCOLLISIONDETECTION),
    new ActionCallback(SceneMenus) (this, &SceneMenus::collisionDetectionChanged),
    scene_->collisionDetection (), 0, colldetectionstate_));

  // options, settings, spaceball
  navigate_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::MenuOPTIONS), navopt_pr));
  navigate_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::NavigateSETTINGS),
    new ActionCallback(SceneMenus) (this, &SceneMenus::toggleNavigationSettingsDialog)));
  navigate_pd->append_item (
#ifndef SPACEBALL
    MenuKit::disable (
#endif
      MenuKit::menuItem (
        kit, layout, STranslate::str (STranslate::NavigateSPACEBALL),
        new ActionCallback(SceneMenus) (this, &SceneMenus::toggleSpaceballDialog))
#ifndef SPACEBALL
    )
#endif
  );

  navigate_pd->append_item (kit.menu_item_separator ());

  // common to all viewers: back, forward, history; hold
  navigate_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::NavigateBACK),
    new ActionCallback(Scene3D) (scene_, &Scene3D::back), "b"));
  navigate_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::NavigateFORWARD),
    new ActionCallback(Scene3D) (scene_, &Scene3D::forward), "f"));
  navigate_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::NavigateHISTORY),
    new ActionCallback(Scene3D) (scene_, &Scene3D::history), "h"));
  navigate_pd->append_item (kit.menu_item_separator ());
  navigate_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::NavigateHOLD),
    new ActionCallback(Scene3D) (scene_, &Scene3D::hold), "H"));

  return navigate_pd;

} // navigatePulldown


// emptyViewpointsPullright
// build empty viewpointsPr_

void SceneMenus::emptyViewpointsPullright (WidgetKit& kit, const LayoutKit& layout)
{
  Resource::unref (viewpointsPr_);
  viewpointsPr_ = kit.pullright ();
  Resource::ref (viewpointsPr_);

  viewpointsPr_->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::AnchorPREVIOUS),
    new ActionCallback(SceneMenus) (this, &SceneMenus::prevViewpoint), "PgUp"));
  viewpointsPr_->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::AnchorNEXT),
    new ActionCallback(SceneMenus) (this, &SceneMenus::nextViewpoint), "PgDn"));
  viewpointsPr_->append_item (kit.menu_item_separator ());

  curviewpoint_ = 0;  // default view
  viewproffset_ = 3;  // 3 items before actual viewpoints
}


Menu* SceneMenus::anchorsPulldown (WidgetKit& kit, const LayoutKit& layout)
{
  Menu* anchors_pd = kit.pulldown ();

  anchors_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::DocumentATTRIBUTES),
    new ActionCallback(Scene3D) (scene_, &Scene3D::anchorinfo), "a", selectionstate_ [sel_anchorinfo]));
  // next Anchor not yet implemented (no separate anchor list)
//   anchors_pd->append_item (MenuKit::menuItem (
//     kit, layout, STranslate::str (STranslate::AnchorNEXT),
//     [Action], "Tab", [state]));
  anchors_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::AnchorFOLLOW),
    new ActionCallback(SceneWindow) (scene_, &SceneWindow::activateAnchor), "Ret", selectionstate_ [sel_activateanchor]));

  anchors_pd->append_item (kit.menu_item_separator ());

  // definition of source and destination anchors
  anchors_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::AnchorDEFINESOURCE),
    new ActionCallback(Scene3D) (scene_, &Scene3D::setSourceAnchor), 0, selectionstate_ [sel_sourceanchor]));
  anchors_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::AnchorDEFINEDESTINATION),
    new ActionCallback(Scene3D) (scene_, &Scene3D::setDestinationAnchor)));
  anchors_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::AnchorDEFAULTDESTINATION),
    new ActionCallback(Scene3D) (scene_, &Scene3D::setDefDestAnchor)));

  anchors_pd->append_item (kit.menu_item_separator ());

  anchors_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::AnchorDELETE),
    new ActionCallback(Scene3D) (scene_, &Scene3D::do_deleteSourceAnchor),
    "Del", selectionstate_ [sel_deleteanchor]));

  return anchors_pd;

} // anchorsPulldown



Menu* SceneMenus::displayPulldown (WidgetKit& kit, const LayoutKit& layout)
{
  Menu* view_pd = kit.pulldown ();

  // display/drawing options (smooth shading etc.)
  static TelltaleGroup* dmgroup = 0;
  if (!dmgroup)
    dmgroup = new TelltaleGroup ();
  const int dmode = scene_->mode ();

  view_pd->append_item (MenuKit::radioMenuItem (
    kit, layout, dmgroup, STranslate::str (STranslate::DisplayWIREFRAME),
    new MenuGroupCallback (this, &SceneMenus::setDisplayMode, ge3d_wireframe),
    dmode == ge3d_wireframe, "^w", wireframe_));
  view_pd->append_item (MenuKit::radioMenuItem (
    kit, layout, dmgroup, STranslate::str (STranslate::DisplayHIDDENLINE),
    new MenuGroupCallback (this, &SceneMenus::setDisplayMode, ge3d_hidden_line),
    dmode == ge3d_hidden_line, "^h", hiddenline_));
  view_pd->append_item (MenuKit::radioMenuItem (
    kit, layout, dmgroup, STranslate::str (STranslate::DisplayFLATSHADING),
    new MenuGroupCallback (this, &SceneMenus::setDisplayMode, ge3d_flat_shading),
    dmode == ge3d_flat_shading, "^f", flatshading_));
  view_pd->append_item (MenuKit::radioMenuItem (
    kit, layout, dmgroup, STranslate::str (STranslate::DisplaySMOOTHSHADING),
    new MenuGroupCallback (this, &SceneMenus::setDisplayMode, ge3d_smooth_shading),
    dmode == ge3d_smooth_shading, "^s", smoothshading_));
  if (ge3dTexturingSupport ())  // does not require an open window
    view_pd->append_item (MenuKit::radioMenuItem (
      kit, layout, dmgroup, STranslate::str (STranslate::DisplayTEXTURING),
      new MenuGroupCallback (this, &SceneMenus::setDisplayMode, ge3d_texturing),
      dmode == ge3d_texturing, "^t", texturing_));

  // interactive display/drawing options
  static TelltaleGroup* imgroup = 0;
  if (!imgroup)
    imgroup = new TelltaleGroup ();
  const int imode = scene_->modeInteract ();

  Menu* iview_pr = kit.pullright ();

  iview_pr->append_item (MenuKit::radioMenuItem (
    kit, layout, imgroup, STranslate::str (STranslate::DisplayTHESAME),
    new MenuGroupCallback (this, &SceneMenus::setInteractiveMode, Scene3D::same_mode),
    imode == Scene3D::same_mode, "^u", intsame_));
  iview_pr->append_item (kit.menu_item_separator ());
  iview_pr->append_item (MenuKit::radioMenuItem (
    kit, layout, imgroup, STranslate::str (STranslate::DisplayWIREFRAME),
    new MenuGroupCallback (this, &SceneMenus::setInteractiveMode, ge3d_wireframe),
    imode == ge3d_wireframe, "^W", intwire_));
  iview_pr->append_item (MenuKit::radioMenuItem (
    kit, layout, imgroup, STranslate::str (STranslate::DisplayHIDDENLINE),
    new MenuGroupCallback (this, &SceneMenus::setInteractiveMode, ge3d_hidden_line),
    imode == ge3d_hidden_line, "^H", inthline_));
  iview_pr->append_item (MenuKit::radioMenuItem (
    kit, layout, imgroup, STranslate::str (STranslate::DisplayFLATSHADING),
    new MenuGroupCallback (this, &SceneMenus::setInteractiveMode, ge3d_flat_shading),
    imode == ge3d_flat_shading, "^F", intflat_));
  iview_pr->append_item (MenuKit::radioMenuItem (
    kit, layout, imgroup, STranslate::str (STranslate::DisplaySMOOTHSHADING),
    new MenuGroupCallback (this, &SceneMenus::setInteractiveMode, ge3d_smooth_shading),
    imode == ge3d_smooth_shading, "^S", intsmooth_));
  iview_pr->append_item (MenuKit::radioMenuItem (
    kit, layout, imgroup, STranslate::str (STranslate::DisplayTEXTURING),
    new MenuGroupCallback (this, &SceneMenus::setInteractiveMode, ge3d_texturing),
    imode == ge3d_texturing, 0, inttexture_));

  view_pd->append_item (kit.menu_item_separator ());
  view_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::DisplayINTERACTIVE), iview_pr));

  // rendering pullright
  Menu* rendering_pr = kit.pullright ();
  {
    static TelltaleGroup* rmgroup = 0;
    if (!rmgroup)
      rmgroup = new TelltaleGroup ();
    const int rmode = VRMLScene::renderMode ();

    rendering_pr->append_item (MenuKit::radioMenuItem (
      kit, layout, rmgroup, STranslate::str (STranslate::RenderZBUFFER),
      new MenuGroupCallback (this, &SceneMenus::renderMode, VRMLScene::rnd_zbuffer),
      rmode == VRMLScene::rnd_zbuffer, "D", renderzbuf_));  // "D" or "Z"
    rendering_pr->append_item (MenuKit::radioMenuItem (
      kit, layout, rmgroup, STranslate::str (STranslate::RenderBSPTREE),
      new MenuGroupCallback (this, &SceneMenus::renderMode, VRMLScene::rnd_bsptree),
      rmode == VRMLScene::rnd_bsptree, "B", renderbsp_));
    rendering_pr->append_item (kit.menu_item_separator ());
    rendering_pr->append_item (MenuKit::checkMenuItem (
      kit, layout, STranslate::str (STranslate::RenderSTEREO),
      new ActionCallback(SceneMenus) (this, &SceneMenus::stereoModeChanged),
      scene_->stereoMode (), "S", stereostate_));

    static TelltaleGroup* bspgroup = 0;
    if (!bspgroup)
      bspgroup = new TelltaleGroup ();
    const int bspmode = VRMLScene::bspMode ();

    rendering_pr->append_item (kit.menu_item_separator ());
    rendering_pr->append_item (MenuKit::radioMenuItem (
      kit, layout, bspgroup, STranslate::str (STranslate::RenderBSPPLAIN),
      new MenuGroupCallback (this, &SceneMenus::BSPmode, BSP_PLAIN),
      bspmode == BSP_PLAIN));
    rendering_pr->append_item (MenuKit::radioMenuItem (
      kit, layout, bspgroup, STranslate::str (STranslate::RenderBSPBACKFACECULLING),
      new MenuGroupCallback (this, &SceneMenus::BSPmode, BSP_BACK_FACE_CULLING),
      bspmode == BSP_BACK_FACE_CULLING));
    rendering_pr->append_item (MenuKit::radioMenuItem (
      kit, layout, bspgroup, STranslate::str (STranslate::RenderBSPSHADOWVOLUME),
      new MenuGroupCallback (this, &SceneMenus::BSPmode, BSP_SHADOW_VOLUME),
      bspmode == BSP_SHADOW_VOLUME));

  } // rendering pullright

  view_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::RenderRENDERMODE), rendering_pr));

  // antialiasing pullright
  Menu* antialias_pr = kit.pullright ();
  {
    int aasupport = ge3dAntialiasingSupport () || kit.style ()->value_is_on ("DemoMenuItems");
    int texmipmap = ge3dCurrentTextureMipmapping ();

    // line-antialiasing
    MenuItem* item = MenuKit::checkMenuItem (
      kit, layout, STranslate::str (STranslate::OptionAALINES),
      new ActionCallback(SceneMenus) (this, &SceneMenus::aaliasingChanged),
      antialiasing (aa_lines), "^L", aaliasingstate_ [aa_lines]);
    if (!aasupport)
      MenuKit::disable (item);
    antialias_pr->append_item (item);

    // "normal" polygon-antialiasing (bad results for adjacent polygons)
    item = MenuKit::checkMenuItem (
      kit, layout, STranslate::str (STranslate::OptionAAPOLYGONS),
      new ActionCallback(SceneMenus) (this, &SceneMenus::aaliasingChanged),
      antialiasing (aa_polygons), "^P", aaliasingstate_ [aa_polygons]);
    if (!aasupport)
      MenuKit::disable (item);
    antialias_pr->append_item (item);

    // "quality" polygon-antialiasing (requires front-to-back ordered polygons)
    RString aalabel2 (STranslate::str (STranslate::OptionAAPOLYGONS));
    aalabel2 += " (2)";
    item = MenuKit::checkMenuItem (
      kit, layout, aalabel2,
      new ActionCallback(SceneMenus) (this, &SceneMenus::aaliasingChanged),
      antialiasing (aa_polygons2), "^Q", aaliasingstate_ [aa_polygons2]);
    if (!aasupport)
      MenuKit::disable (item);
    antialias_pr->append_item (item);

    // texture anti-aliasing: filtering and mipmapping
    static TelltaleGroup* tmgroup = 0;
    if (!tmgroup)
      tmgroup = new TelltaleGroup ();
    Menu* mipmap_pr = kit.pullright ();
    mipmap_pr->append_item (MenuKit::radioMenuItem (
      kit, layout, tmgroup, STranslate::str (STranslate::TwosidedOFF),
      new MenuGroupCallback (this, &SceneMenus::textureMipmapping, 0),
      texmipmap == 0, "^N", texmipmapoffstate_));  // default for Mesa
    for (int i = 1;  i < ge3d_maxmipmap_quality;  i++)  // intermediate filter levels (experts only)
    { mipmap_pr->append_item (MenuKit::radioMenuItem (
        kit, layout, tmgroup, ".",
        new MenuGroupCallback (this, &SceneMenus::textureMipmapping, i), 0));
    }
    mipmap_pr->append_item (MenuKit::radioMenuItem (
      kit, layout, tmgroup, STranslate::str (STranslate::TwosidedON),
      new MenuGroupCallback (this, &SceneMenus::textureMipmapping, ge3d_maxmipmap_quality),
      texmipmap == ge3d_maxmipmap_quality, "^M", texmipmaponstate_));  // default for OpenGL

    item = MenuKit::menuItem (
      kit, layout, STranslate::str (STranslate::OptionAATEXTURES),
      mipmap_pr);
    // also mesa does texture filtering now
    antialias_pr->append_item (item);

    // disable menu items if no anti-aliasing support
  }

  // twosided polygons pullright
  Menu* twoside_pr = kit.pullright ();
  {
    static TelltaleGroup* tsgroup = 0;
    if (!tsgroup)
      tsgroup = new TelltaleGroup ();
    const int tpolys = scene_->twosidedpolys ();

    twoside_pr->append_item (MenuKit::radioMenuItem (
      kit, layout, tsgroup, STranslate::str (STranslate::TwosidedON),
      new MenuGroupCallback (this, &SceneMenus::twosidedPolys, Scene3D::twosided_always),
      tpolys == Scene3D::twosided_always));
    twoside_pr->append_item (MenuKit::radioMenuItem (
      kit, layout, tsgroup, STranslate::str (STranslate::TwosidedAUTO),
      new MenuGroupCallback (this, &SceneMenus::twosidedPolys, Scene3D::twosided_auto),
      tpolys == Scene3D::twosided_auto));
    twoside_pr->append_item (MenuKit::radioMenuItem (
      kit, layout, tsgroup, STranslate::str (STranslate::TwosidedOFF),
      new MenuGroupCallback (this, &SceneMenus::twosidedPolys, Scene3D::twosided_never),
      tpolys == Scene3D::twosided_never));
  }

  // transparency pullright
  Menu* transp_pr = kit.pullright ();
  {
    static TelltaleGroup* trgroup = 0;
    if (!trgroup)
      trgroup = new TelltaleGroup ();
    const int transparency = scene_->transparency ();

    transp_pr->append_item (MenuKit::radioMenuItem (
      kit, layout, trgroup, STranslate::str (STranslate::TwosidedOFF),
      new MenuGroupCallback (this, &SceneMenus::transparencyMode, transp_off),
      transparency == transp_off));
    transp_pr->append_item (MenuKit::radioMenuItem (
      kit, layout, trgroup, STranslate::str (STranslate::TransparSTIPPLE),
      new MenuGroupCallback (this, &SceneMenus::transparencyMode, transp_stipple),
      transparency == transp_stipple));
    transp_pr->append_item (MenuKit::radioMenuItem (
      kit, layout, trgroup, STranslate::str (STranslate::TransparBLENDING),
      new MenuGroupCallback (this, &SceneMenus::transparencyMode, transp_blend),
      transparency == transp_blend));

    transp_pr->append_item (kit.menu_item_separator ());
    transp_pr->append_item (MenuKit::checkMenuItem (
      kit, layout, STranslate::str (STranslate::OptionTEXTURELIGHTING),  // "on texturing"
      new ActionCallback(SceneMenus) (this, &toggleTextureTransparency),
      scene_->textureTransparency ()));
  }

  // lighting calculations pullright
  Menu* lightcalc_pr = kit.pullright ();
  {
    static TelltaleGroup* lcgroup = 0;
    if (!lcgroup)
      lcgroup = new TelltaleGroup ();
    const int lighting = scene_->dolighting ();

    lightcalc_pr->append_item (MenuKit::radioMenuItem (
      kit, layout, lcgroup, STranslate::str (STranslate::TwosidedON),
      new MenuGroupCallback (this, &SceneMenus::doLighting, Scene3D::lighting_always),
      lighting == Scene3D::lighting_always));
    lightcalc_pr->append_item (MenuKit::radioMenuItem (
      kit, layout, lcgroup, STranslate::str (STranslate::TwosidedAUTO),
      new MenuGroupCallback (this, &SceneMenus::doLighting, Scene3D::lighting_auto),
      lighting == Scene3D::lighting_auto));
    lightcalc_pr->append_item (MenuKit::radioMenuItem (
      kit, layout, lcgroup, STranslate::str (STranslate::TwosidedOFF),
      new MenuGroupCallback (this, &SceneMenus::doLighting, Scene3D::lighting_never),
      lighting == Scene3D::lighting_never));

    lightcalc_pr->append_item (kit.menu_item_separator ());
    lightcalc_pr->append_item (MenuKit::checkMenuItem (
      kit, layout, STranslate::str (STranslate::OptionTEXTURELIGHTING),
      new ActionCallback(SceneMenus) (this, &SceneMenus::toggleTextureLighting),
      scene_->textureLighting ()));

    lightcalc_pr->append_item (kit.menu_item_separator ());
    lightcalc_pr->append_item (MenuKit::checkMenuItem (
      kit, layout, STranslate::str (STranslate::OptionNOMATERIALS),
      new ActionCallback(SceneMenus) (this, &SceneMenus::toggleNoMaterials),
      scene_->noMaterials ()));
  }

  view_pd->append_item (kit.menu_item_separator ());

  view_pd->append_item (MenuKit::checkMenuItem (
    kit, layout, STranslate::str (STranslate::OptionSHOWFRAMERATE),
    new ActionCallback(SceneWindow) (scene_, &SceneWindow::toggleShowFramerate),
    scene_->showFramerate ()));
  view_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::OptionSHOWTITLE),
    new ActionCallback(SceneWindow) (scene_, &SceneWindow::showTitle)));
  view_pd->append_item (MenuKit::menuItem (
     kit, layout, STranslate::str (STranslate::OptionAALIASING), antialias_pr));
  view_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::OptionTWOSIDEDPOLYS), twoside_pr));
  view_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::OptionDOLIGHTING), lightcalc_pr));
  view_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::OptionTRANSPARENCY), transp_pr));

  view_pd->append_item (MenuKit::checkMenuItem (
    kit, layout, STranslate::str (STranslate::ColorVIEWINGLIGHT),
    new ActionCallback(SceneMenus) (this, &SceneMenus::viewingLightChanged),
    scene_->viewingLight (), 0, viewlgtstate_));

  view_pd->append_item (kit.menu_item_separator ());

  view_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::MenuANCHORS), anchorsPullright (kit, layout)));

  view_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::MenuCOLOR), colorPullright (kit, layout)));

  return view_pd;

} // displayPulldown


Menu* SceneMenus::anchorsPullright (WidgetKit& kit, const LayoutKit& layout)
{
  Menu* anchors_pr = kit.pullright ();  // view/anchor pullright menu

  // anchor highlighting mode
  static TelltaleGroup* ahgroup = 0;
  if (!ahgroup)
    ahgroup = new TelltaleGroup ();
  const int ahmode = scene_->linksColor ();

  MenuItem* showanchors = MenuKit::checkMenuItem (
    kit, layout, STranslate::str (STranslate::AnchorDISPLAY),
    new ActionCallback(SceneMenus) (this, &SceneMenus::showAnchorsChanged),
    scene_->linksActive (), "F9", showanchorstate_);
  anchors_pr->append_item (showanchors);

  anchors_pr->append_item (kit.menu_item_separator ());

  anchors_pr->append_item (MenuKit::radioMenuItem (
    kit, layout, ahgroup, STranslate::str (STranslate::AnchBRIGHTNESS),
    new MenuGroupCallback (this, &SceneMenus::highlightingMode, Scene3D::l_brightness), ahmode == Scene3D::l_brightness));
//   anchors_pr->append_item (MenuKit::radioMenuItem (
//     kit, layout, ahgroup, STranslate::str (STranslate::AnchBOUNDINGCUBE),
//     new MenuGroupCallback (this, &SceneMenus::highlightingMode, Scene3D::l_boundingbox),
//     ahmode == Scene3D::l_boundingbox));
  anchors_pr->append_item (MenuKit::radioMenuItem (
    kit, layout, ahgroup, STranslate::str (STranslate::AnchCOLORCODE),
    new MenuGroupCallback (this, &SceneMenus::highlightingMode, Scene3D::l_diffcolor), ahmode == Scene3D::l_diffcolor));
  anchors_pr->append_item (MenuKit::radioMenuItem (
    kit, layout, ahgroup, STranslate::str (STranslate::AnchCOLOREDGES),
    new MenuGroupCallback (this, &SceneMenus::highlightingMode, Scene3D::l_coloredges), ahmode == Scene3D::l_coloredges));

  return anchors_pr;

} // anchorsPullright


Menu* SceneMenus::colorPullright (WidgetKit& kit, const LayoutKit& layout)
{
  // colour submenus (set predefined colour or select other colour)

  // background colour submenu
  Menu* backgcolor_pr = kit.pullright ();
  {
    colorRGB* backg = &scene_->col_background;
    backgcolor_pr->append_item (MenuKit::menuItem (
      kit, layout, STranslate::str (STranslate::ColBLACK),
      new SetColourCallback (this, backg, 0.0, 0.0, 0.0)));  // black
    backgcolor_pr->append_item (MenuKit::menuItem (
      kit, layout, STranslate::str (STranslate::ColGRAY),
      new SetColourCallback (this, backg, 0.5, 0.5, 0.5)));  // grey50
    backgcolor_pr->append_item (MenuKit::menuItem (
      kit, layout, STranslate::str (STranslate::ColLIGHTGRAY),
      new SetColourCallback (this, backg, 0.75, 0.75, 0.75)));  // grey75
    backgcolor_pr->append_item (MenuKit::menuItem (
      kit, layout, STranslate::str (STranslate::ColWHITE),
      new SetColourCallback (this, backg, 1.0, 1.0, 1.0)));  // white
  }

  // heads-up display colour submenu
  Menu* hudisplaycolor_pr = kit.pullright ();
  {
    colorRGB* hudisp = &scene_->col_hudisplay;
    hudisplaycolor_pr->append_item (MenuKit::menuItem (
      kit, layout, STranslate::str (STranslate::ColPURPLE),
      new SetColourCallback (this, hudisp, 0.5, 0, 1.0)));  // purple
    hudisplaycolor_pr->append_item (MenuKit::menuItem (
      kit, layout, STranslate::str (STranslate::ColORANGE),
      new SetColourCallback (this, hudisp, 1.0, 0.5, 0.0)));  // orange
    hudisplaycolor_pr->append_item (MenuKit::menuItem (
      kit, layout, STranslate::str (STranslate::ColCYAN),
        new SetColourCallback (this, hudisp, 0.0, 1.0, 1.0)));  // cyan
  }

  // anchor faces colour submenu
  Menu* anfacecolor_pr = kit.pullright ();
  {
    colorRGB* anface = &scene_->col_anchorface;
    anfacecolor_pr->append_item (MenuKit::menuItem (
      kit, layout, STranslate::str (STranslate::ColREDBROWN),
        new SetColourCallback (this, anface, 1.0, 0.25, 0.0, 1/*rebuild*/)));  // red-brown
    anfacecolor_pr->append_item (MenuKit::menuItem (
      kit, layout, STranslate::str (STranslate::ColGREEN),
        new SetColourCallback (this, anface, 0.0, 1.0, 0.0, 1)));  // green
  }

  // anchor edge colour submenu
  Menu* anedgecolor_pr = kit.pullright ();
  {
    colorRGB* anedge = &scene_->col_anchoredge;
    anedgecolor_pr->append_item (MenuKit::menuItem (
      kit, layout, STranslate::str (STranslate::ColYELLOW),
        new SetColourCallback (this, anedge, 1.0, 1.0, 0.25)));  // yellow
    anedgecolor_pr->append_item (MenuKit::menuItem (
      kit, layout, STranslate::str (STranslate::ColMAGENTA),
        new SetColourCallback (this, anedge, 1.0, 0.5, 1.0)));  // magenta
  }

  Menu* color_pr = kit.pullright ();  // colour pullright menu

  color_pr->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::ColorOTHER),
    new ActionCallback(SceneMenus) (this, &SceneMenus::toggleColourDialog)));
  color_pr->append_item (kit.menu_item_separator ());
  color_pr->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::ColorBACKGROUND), backgcolor_pr));
  color_pr->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::ColorHEADSUP), hudisplaycolor_pr));
  color_pr->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::ColorANCHORFACES), anfacecolor_pr));
  color_pr->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::ColorANCHOREDGES), anedgecolor_pr));

  return color_pr;

} // colorPullright


Menu* SceneMenus::documentPulldown (WidgetKit& kit, const LayoutKit& layout)
{
  Menu* docinfo_pd = kit.pulldown ();

  docinfo_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::DocumentREFERENCES),
    new MenuGroupCallback (this, &SceneMenus::documentInfo, Scene3D::info_references)));
  docinfo_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::DocumentANNOTATIONS),
    new MenuGroupCallback (this, &SceneMenus::documentInfo, Scene3D::info_annotations)));
  docinfo_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::DocumentPARENTS),
    new MenuGroupCallback (this, &SceneMenus::documentInfo, Scene3D::info_parents)));
  docinfo_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::DocumentATTRIBUTES),
    new MenuGroupCallback (this, &SceneMenus::documentInfo, Scene3D::info_attributes)));
  docinfo_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::DocumentTEXTURES),
    new MenuGroupCallback (this, &SceneMenus::documentInfo, Scene3D::info_textures)));

  return docinfo_pd;

} // documentPulldown


// Menu* SceneMenus::optionsPulldown (WidgetKit& kit, const LayoutKit& layout)
// {
// enable motion and velocity control --> option submenu under navigate pulldown
// show framerate and twosided polygons --> display (view) pulldown
// }


Menu* SceneMenus::helpPulldown (WidgetKit& kit, const LayoutKit& layout)
{
  Menu* help_pd = kit.pulldown ();

  help_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::HelpOVERVIEW),
    new ActionCallback(Scene3D) (scene_, &Scene3D::doShowHelp), "F1"));
  help_pd->append_item (MenuKit::disable (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::HelpINDEX), (Action*) nil)));
//  help_pd->append_item (kit.menu_item_separator ());
  help_pd->append_item (MenuKit::menuItem (
    kit, layout, STranslate::str (STranslate::HelpABOUTPRODUCT),
    new ActionCallback(SceneMenus) (this, &SceneMenus::toggleAbout)));

  return help_pd;

} // helpPulldown



// updateAbout moved to dialogs.C



/*** settings done with radio menus ***/


void SceneMenus::movementMode (int mode)
{ // menu callback, group updated by InterViews
#ifdef INSTRUMENTED
  RString rstr (n_move);
  switch (mode)
  {
    case NavMode::flip_obj:     rstr += "Flip";     break;
    case NavMode::walk_around:  rstr += "Walk";     break;
    case NavMode::fly_1:        rstr += "Fly";      break;
    case NavMode::fly_2:        rstr += "FlyTo";    break;
    case NavMode::heads_up:     rstr += "HeadsUp";  break;
  }
  Instrumentation::instance()->write_to_log(rstr); //--- by iegger
#endif

  scene_->setNavigationMode (mode);
  scene_->giveNavigationHint ();
}


void SceneMenus::navigationMode (int mode)
{ // extern callable, update group

  TelltaleState* ts = 0;

  switch (mode)
  {
    case NavMode::flip_obj:
      ts = navflipobjstate_;
    break;
    case NavMode::walk_around:
      ts = navwalkstate_;
    break;
    case NavMode::fly_1:
      ts = navflystate_;
    break;
    case NavMode::fly_2:
      ts = navflytostate_;
    break;
    case NavMode::heads_up:
      ts = navheadsupstate_;
    break;
  }

  if (ts)  // update current item of radio menu group
  {
    ts->set (TelltaleState::is_chosen, true);

    movementMode (mode);
  }

} // navigationMode


void SceneMenus::selectionMode (int selmode)
{
  scene_->selectionMode (selmode);
}


void SceneMenus::renderMode (int mode)
{ // menu callback
  VRMLScene::renderMode (mode);
  scene_->showTitle ();  // get rid of old BSP info
  scene_->redraw ();
}


void SceneMenus::setRenderMode (int mode)
{ // extern callable, sets telltale state

  TelltaleState* ts = (mode == VRMLScene::rnd_bsptree) ? renderbsp_ : renderzbuf_;

  ts->set (TelltaleState::is_chosen, true);
  renderMode (mode);
}

void SceneMenus::BSPmode (int mode)
{
  VRMLScene::bspMode (mode);
  // gmes-TODO: switch to BSP rendering and update radio buttons
  // VRMLScene::renderMode (VRMLScene::rnd_bsptree);
  if (VRMLScene::renderMode () == VRMLScene::rnd_bsptree)
    scene_->redraw ();
}


void SceneMenus::textureMipmapping (int quality)
{
  ge3dTextureMipmapping (quality);
  if (!quality)
    texmipmapoffstate_->set (TelltaleState::is_chosen, true);
  else if (quality >= ge3d_maxmipmap_quality)
    texmipmaponstate_->set (TelltaleState::is_chosen, true);
  if (scene_->mode () == ge3d_texturing)
    scene_->redraw ();
}


void SceneMenus::twosidedPolys (int mode)
{
  scene_->twosidedpolys (mode);
  scene_->redraw ();
}


void SceneMenus::doLighting (int mode)
{
  scene_->dolighting (mode);
  scene_->redraw ();
}


void SceneMenus::transparencyMode (int mode)
{
  scene_->transparency (mode);
  ge3dHint (hint_transparency, mode);
  scene_->redraw ();
}


void SceneMenus::highlightingMode (int mode)
{
  scene_->linksColor (mode);

  if (scene_->linksActive ())  // when anchors are highlighted
    scene_->redraw ();         // redraw whole window
}


void SceneMenus::setDisplayMode (int mode)
{ // menu callback, group updated by InterViews
#ifdef INSTRUMENTED
  RString rstr (dis_drawmode);
  switch (mode)
  {
    case ge3d_wireframe:       rstr += "Wireframe";      break;
    case ge3d_hidden_line:     rstr += "HiddenLine";     break;
    case ge3d_flat_shading:    rstr += "FlatShading";    break;
    case ge3d_smooth_shading:  rstr += "SmoothShading";  break;
    case ge3d_texturing:       rstr += "Texturing";      break;
  }
  Instrumentation::instance()->write_to_log(rstr); //--- by iegger
#endif

  scene_->mode (mode);
  scene_->redraw ();  // redraw whole window
}


void SceneMenus::displayMode (int mode, int interactive)
{ // extern callable, update group

  TelltaleState* ts = 0;

  switch (mode)
  {
    case Scene3D::same_mode:
      ts = interactive ? intsame_ : 0;
    break;
    case ge3d_wireframe:
      ts = interactive ? intwire_ : wireframe_;
    break;
    case ge3d_hidden_line:
      ts = interactive ? inthline_ : hiddenline_;
    break;
    case ge3d_flat_shading:
      ts = interactive ? intflat_ : flatshading_;
    break;
    case ge3d_smooth_shading:
      ts = interactive ? intsmooth_ : smoothshading_;
    break;
    case ge3d_texturing:
      ts = interactive ? inttexture_ : texturing_;
    break;
  }

  if (ts)  // update current item of radio menu group
  {
    ts->set (TelltaleState::is_chosen, true);

    if (interactive)
      setInteractiveMode (mode);  // no redraw
    else
      setDisplayMode (mode);  // and redraw
  }

} // DisplayMode


void SceneMenus::setInteractiveMode (int mode)
{
#ifdef INSTRUMENTED
  RString rstr (dis_drawmodeinter);
  switch (mode)
  {
    case ge3d_wireframe:       rstr += "Wireframe";      break;
    case ge3d_hidden_line:     rstr += "HiddenLine";     break;
    case ge3d_flat_shading:    rstr += "FlatShading";    break;
    case ge3d_smooth_shading:  rstr += "SmoothShading";  break;
    case ge3d_texturing:       rstr += "Texturing";      break;
  }
  Instrumentation::instance()->write_to_log(rstr); //--- by iegger
#endif

  scene_->modeInteract (mode);
  // no need to redraw when called from menu
}


/*** callback actions ***/


void SceneMenus::loadDemo (const char* demo)
{
  char filename [512];

  strcpy (filename, demopath_);
  strcat (filename, demo);
//cerr << "loading demo " << filename << endl;

  scene_->readSceneFile (filename);
  scene_->setTitle (filename);
  scene_->reset ();  // and redraw
}


void SceneMenus::browseDemo (const char* demo, const point3D& position, const point3D& lookat)
{
  char filename [512];

  strcpy (filename, demopath_);
  strcat (filename, demo);

  scene_->readSceneFile (filename);
  scene_->setTitle (filename);
  scene_->setCamera (position, lookat);
  scene_->reset ();  // and redraw
}


void SceneMenus::buildOpenFC ()
{
  DEBUGNL ("SceneMenus::buildOpenFC");

  Resource::unref (openfc_);  // destroy old dialog

  WidgetKit& kit = *kit_;
  kit.begin_style ("Open");
  Style* style = kit.style ();

  style->attribute ("name", STranslate::str (STranslate::DialogTitleOPENFILE));
  style->attribute ("open", STranslate::str (STranslate::DialogButtonOPEN));
  style->attribute ("cancel", WTranslate::str (WTranslate::CANCEL, slanguage));
  style->attribute ("subcaption", STranslate::str (STranslate::DialogStringFILENAME));

  openfc_ = new WFileChooser ("", &kit, style, 0, "*.wrl*");

  Resource::ref (openfc_);  // prevent immediate destruction

  kit.end_style ();  // "Open"

} // buildOpenFC


void SceneMenus::openFile ()
{
#ifdef INSTRUMENTED
  Instrumentation::instance()->write_to_log(f_open); //--- by iegger
#endif

  if (!openfc_)
    buildOpenFC ();
  // otherwise reuse existent file chooser

  if (openfc_->post_for_aligned (scene_->appwin (), 0.5, 0.5))  // "OK"
  {
    const char* filename = openfc_->selected ();

    int res = scene_->readSceneFile (filename);

    if (res)  // error occured
    {
      RString errormsg (STranslate::str (STranslate::MessageERRORONREADING));  // ready to append filename
      errormsg += filename;

      scene_->reset ();  // scene may already have been cleared
      // TODO: "ready" title in status line in case scene has been cleared

      MessageBox::message (
        slanguage, scene_->appwin (), errormsg,
        STranslate::str (STranslate::MsgboxtitleERROR), MessageBox::Ok
      );

    }  // invalid file
    else  // no error on reading
    {
      scene_->setTitle (filename);
      scene_->reset ();  // and redraw
    }
  }  // OK was pressed

} // openFile


void SceneMenus::saveVRMLScene ()
{
  // file format should be an option in save dialog (TODO)
  saveScene (Scene3D::write_VRML, "*.wrl");  // save as VRML (export)
}


void SceneMenus::saveSDFScene ()
{
  saveScene (Scene3D::write_SDF, "*.sdf");  // save as SDF (export)
}


void SceneMenus::saveOriginalScene ()
{
  saveScene (Scene3D::write_ORIGINAL, "");  // save as ...
  // does write_VRML on edited scenes (see writeScene)
}


void SceneMenus::saveCurrentScene ()  // anuss/mpi
{
  // on editing: save using filename specified last time (if done)
  // does write_VRML on edited scenes (see writeScene)
  if (savefilename_.length ())  // save ...
  {
    // ofstream fout("out.wrl");
    // scene_->saveVRMLSceneFile (fout);  // resets manipulation flag after writeScene (os, write_VRML);
    writeScene (Scene3D::write_ORIGINAL, savefilename_.string ());
  }
  else
    saveScene (Scene3D::write_ORIGINAL, "");  // save as
}


// "Save as ..." dialog
// asks for confirmation before overwriting existing files

void SceneMenus::saveScene (int format, const char* filemask)
{
#ifdef INSTRUMENTED
  Instrumentation::instance()->write_to_log(f_save_as); //--- by iegger
#endif

  // under write_ORIGINAL format, save current data on editing, original input otherwise
  // (filechooser provides no way to choose it)

  // see writeScene; allow saving a local scene copy in harscened
  int editing = scene_->data () && scene_->data ()->startedEditing ();
  if (editing)
    format = Scene3D::write_VRML;

  if (!scene_->supportsOutputformat (format))
  {
    MessageBox::message (
      slanguage, scene_->appwin (), STranslate::str (STranslate::MessageERRORNOINPUTFILE),
      STranslate::str (STranslate::MsgboxtitleERROR),
      MessageBox::Ok
    );
    return;
  }

  WidgetKit& kit = *kit_;
  kit.begin_style ("Save");
  Style* style = kit.style ();

  style->attribute ("name", STranslate::str (STranslate::DialogTitleSAVEFILE));
  style->attribute ("open", STranslate::str (STranslate::DialogButtonSAVE));
  style->attribute ("cancel", WTranslate::str (WTranslate::CANCEL, slanguage));
  style->attribute ("subcaption", STranslate::str (STranslate::DialogStringFILENAME));

  // should set path here

  WFileChooser* fc = new WFileChooser ("", &kit, style, 0, filemask);
  Resource::ref (fc);  // prevent immediate destruction

  if (fc->post_for_aligned (scene_->appwin (), 0.5, 0.5))  // "OK"
  {
    const char* filename = fc->selected ();
    RString messagestring (filename);
    messagestring += RString ("\n") + STranslate::str (STranslate::DialogStringOVERWRITEEXISTENTFILE);

    // can write if file not yet exists or user wants to overwrite it
    int write = access (filename, F_OK) ||
    MessageBox::message (
      slanguage, scene_->appwin (), messagestring,
      STranslate::str (STranslate::DialogTitleSAVEFILE), MessageBox::Ok | MessageBox::Cancel
    ) == MessageBox::Ok;

    if (write)  // do it
      writeScene (format, filename);

  }  // OK was pressed

  Resource::unref (fc);  // destroy file chooser

  kit.end_style ();  // "Save"
} // saveScene


// do the work: write the scene
// pops up a message in case of errors

void SceneMenus::writeScene (int format, const char* filename)
{
  ofstream ofs (filename);
  int ok;

  // when output format were contained in "Save as" dialog, would change the default above
  // instead of changing it here
  int editing = scene_->data () && scene_->data ()->startedEditing ();
  if (editing)
    format = Scene3D::write_VRML;

  DEBUGNL ("save in " << (editing ? "edit" : "normal") << " mode");
  DEBUGNL ("data will be written to file " << filename);

  if (gecontext_)
    gecontext_->setCursor (HgCursors::instance ()->hourglass ());

  ok = ofs && !scene_->writeScene (ofs, format);

  if (gecontext_)
    gecontext_->resetCursor ();

  if (editing && ok)  // use this filename on next "save"
    savefilename_ = filename;

  if (!ok)
  {
    RString msgstring = STranslate::str (
      /* ok ? STranslate::MessageSUCCESSFULLYWRITTEN : */ STranslate::MessageERRORONWRITING
    );
    msgstring += filename;  // append file name

    MessageBox::message (
      slanguage, scene_->appwin (), msgstring,
      STranslate::str (/*ok ? STranslate::DialogTitleSAVEFILE : */ STranslate::MsgboxtitleERROR),
      MessageBox::Ok
    );
  }

} // writeScene


void SceneMenus::saveDepth ()  // will be merged into saveScene
{
// must query contents of Z-buffer for reasonable performance!
#ifdef TODO  /* saveDepth */
  ofstream ofs ("tmpDepthFile.pgm");
  scene_->writeDepth (ofs, 320, 200, Scene3D::depth_PGMbin);
#endif
} // saveScene


/*********** editing ****************/


// undo last manipulation (selection and transformation)

void SceneMenus::undoManipulation ()
{
  if (scene_->manipulationAllowed ())
    if (scene_->editdata ()->undoManipulation ())
      scene_->redraw ();

} // undoManipulation


void SceneMenus::viewSelectedNode ()
{
  // if (scene_->manipulationAllowed ())
  if (scene_->editdata () && scene_->editdata ()->viewSelectedNode ())
    scene_->redraw ();

} // viewSelectedNode


void SceneMenus::viewAll ()
{
  if (!scene_->editdata ())
    return;

  scene_->editdata ()->viewAll ();
  scene_->redraw ();
} // viewAll


void SceneMenus::selectParent ()
{
  if (scene_->editdata () && scene_->editdata ()->selectParent ())
    scene_->redraw ();
}


void SceneMenus::selectChild ()
{
  if (scene_->editdata () && scene_->editdata ()->selectChild ())
    scene_->redraw ();
}


void SceneMenus::selectNext (int step)
{
  if (scene_->editdata () && scene_->editdata ()->selectNext (step))
    scene_->redraw ();
} // selectNext


void SceneMenus::deleteSelectedNode ()
{
  if (scene_->manipulationAllowed ())
    if (scene_->editdata ()->deleteSelectedNode ())
      scene_->redraw ();

} // deleteSelectedNode


void SceneMenus::cutSelectedNode ()
{
  if (scene_->manipulationAllowed ())
    if (scene_->editdata ()->cutSelectedNode ())
      scene_->redraw ();

} // cutSelectedNode


void SceneMenus::copySelectedNode ()
{
  if (scene_->manipulationAllowed ())
    if (scene_->editdata ()->copySelectedNode ())
      scene_->redraw ();

} // copySelectedNode


void SceneMenus::pasteSelectedNode ()
{
  if (scene_->manipulationAllowed ())
    if (scene_->editdata ()->pasteSelectedNode ())
      scene_->redraw ();

} // pasteNode


// makes a new vrml scene

void SceneMenus::newFile ()
{
//   if (!scene_->manipulationAllowed ())
//   { cerr << "'newFile' only allowed in edit mode" << endl;
//     return;
//   }

  // Dialogs must be unmapped, because the old VRMLScene knows the dialog,
  // but the new one does not; if a dialog is mapped, it registers itself in the VRMLScene
  destroyEditDialogs (); 

  const char* filename = "noname.wrl";

  scene_->newVRMLScene (filename);
  scene_->setTitle (filename);

  if (scene_->getEditMode () == EditMode::ed_viewer)  // turn on editing
    toggleEditScene ();

  scene_->reset ();  // and redraw

} // newFile


// void SceneMenus::buildInsertFC ()
// {
//   //  DEBUGNL ("SceneMenus::buildOpenFC");

//   Resource::unref (insertfc_);  // destroy old dialog

//   WidgetKit& kit = *kit_;
//   kit.begin_style ("Insert"); 
//   Style* style = kit.style ();

//   style->attribute ("name", "Insert File");
//   style->attribute ("open", "Insert");
//   style->attribute ("cancel", WTranslate::str (WTranslate::CANCEL, slanguage));
//   style->attribute ("subcaption", STranslate::str (STranslate::DialogStringFILENAME));

//   insertfc_ = new WFileChooser ("", &kit, style, 0, "*.wrl*");

//   Resource::ref (insertfc_);  // prevent immediate destruction

//   kit.end_style ();  // "Insert"

// } // buildInsertFC


// void SceneMenus::insertFile ()
// {
//   if (scene_->getEditMode () == EditMode::ed_viewer)
//     return;

//   if (!insertfc_)
//     buildInsertFC ();
//   // otherwise reuse existent file chooser

//   if (insertfc_->post_for_aligned (scene_->appwin (), 0.5, 0.5))  // "OK"
//   {
//     const char* filename = insertfc_->selected ();
        
//     scene_->setInsertFlag (1);  // set flag, 
//     if (!scene_->readSceneFile (filename))  // no errorcode
//       scene_->redraw ();
//     scene_->setInsertFlag (0);  // set flag, 
//   }

// }  // insertFile


// void SceneMenus::insertThisFile (const char* filename)
// {
//   scene_->setInsertFlag (1);  // set flag, 
//   if (!scene_->readSceneFile (filename))  // no errorcode
//     scene_->redraw ();
//   scene_->setInsertFlag (0);  // set flag, 
// }



void SceneMenus::quitScene ()
{
  if (scene_->getSceneManipulated () && !continueOnSaveState ())
    return;

  Session::instance ()->quit ();
}


// anuss
void SceneMenus::toggleEditToolbar ()
{
  if (scene_->getEditMode () == EditMode::ed_viewer)
    return;

  createButtons ();  // new toolbar
  scene_->toolbarLayoutChanged ();              // redraw buttons in toolbar

} // toggleEditToolbar



// toggle between viewer and manipulation mode

void SceneMenus::toggleEditScene ()
{
  if (!scene_->isEditable () || !menubar_)
    return;

  int menuchanged = 0;

  if (scene_->getEditMode () == EditMode::ed_viewer)  // turn on editing
  {
    editMode (EditMode::ed_camera);  // mpi: ed_select likely to cause unwanted object manipulation
    scene_->editdata ()->checkVRMLScene (); // check vrml scene graph and prepare it for editing
    // TODO: sufficient on first time switch

    if (!editmapped_)
    {
      menubar_->insert_item (editIdx_, editMenu (*kit_, *layout_));
      editmapped_ = 1;
      menuchanged = 1;
    }
  }
  else  // return to viewing mode
  {
    editMode (EditMode::ed_viewer);
    if (editmapped_)
    {
      menubar_->remove_item (editIdx_);
      editmapped_ = 0;
      menuchanged = 1;
    }
  }

  if (menuchanged)
  {
    menupatch_->reallocate ();
    menupatch_->redraw ();
  }

  // ### may need to change this too (like editmenu_)
  createButtons ();                             // update edit buttons
  scene_->toolbarLayoutChanged ();              // redraw buttons in toolbar

} // toggleEditScene 


// toggle between selection and camera mode

void SceneMenus::toggleEditMode ()
{
  if (scene_->getEditMode () == EditMode::ed_viewer)
    return;

  if (scene_->getEditMode () == EditMode::ed_select)
    editMode (EditMode::ed_camera);
  else
    editMode (EditMode::ed_select);

  // set/unset editmode button if shortcut ^o is pressed
  editmodestate_->set (TelltaleState::is_chosen, scene_->getEditMode () == EditMode::ed_select); 

} // toggleEditMode


// set scene editmode (EditMode::ed_*)

void SceneMenus::editMode (int mode)
{
  if (mode == scene_->getEditMode ())
    return;  // nothing to do

  if (scene_->getEditMode () == EditMode::ed_select)
    scene_->editdata ()->unselectPropertyNodes ();

  scene_->setEditMode (mode);

  if (scene_->selectedNode () && scene_->manipulationAllowed ())
  { scene_->editdata ()->updateEditContext ();          // selectTransform checks the conditions
    scene_->redraw ();  // force redraw for matrix calculation of the selected object
  }

  if (scene_->modeDrawsUI (scene_->navigationMode ()))
  {                                                 // only for navigation-modes, which draw UI
    inputhand_->reset ();                           // reset Cursor
    scene_->redraw ();                              // draw or remove UI
  }
  else if (mode == EditMode::ed_select)
    inputhand_->reset ();

  if (mode == EditMode::ed_viewer)
  { 
    destroyEditDialogs ();
    scene_->editdata ()->unselectPropertyNodes ();
    edittoolbarstate_->set (TelltaleState::is_chosen, 0); 
  }

  editstate_->set (TelltaleState::is_chosen, scene_->manipulationAllowed () ? 1 : 0);

  // enable and disable navigationstates (is_choosable need not be changed)
  const int enabled_choosable = TelltaleState::is_enabled | TelltaleState::is_choosable;
  const int editmode = (mode != EditMode::ed_select);

  navflipobjstate_->set (enabled_choosable, editmode);
  navwalkstate_->set    (enabled_choosable, editmode);
  navflystate_->set     (enabled_choosable, editmode);
  navflytostate_->set   (enabled_choosable, editmode);
  navheadsupstate_->set (enabled_choosable, editmode);
} // editMode


void SceneMenus::toggleConstraints ()
{
  scene_->editdata ()->constraints (!scene_->editdata ()->constraints ());
  // was toolbar button some time ago
} // toggleConstraints


void SceneMenus::documentInfo (int info)
{
  scene_->docinfo (info);  // Scene3D::info_...
}


// dialog creating functions moved to dialogs.C:
// buildErrorDialog
// buildSpaceballDialog
// buildNavigationSettingsDialog
// buildColourDialog
// toggleColourDialog


void SceneMenus::applyColours (ColourDialog* coldlg)
{
  DEBUGNL ("SceneMenus::applyColours");

  // be sure to match order with buildColourDialog (see dialogs.C)

  float r, g, b;
  colorRGB* col;

  if (coldlg->modified (0))
  { col = &scene_->col_background;
    coldlg->getRGB (r, g, b, 0);
    initRGB (*col, r, g, b);
  }
  if (coldlg->modified (1))
  { col = &scene_->col_hudisplay;
    coldlg->getRGB (r, g, b, 1);
    initRGB (*col, r, g, b);
  }
  if (coldlg->modified (2))
  { col = &scene_->col_anchorface;
    coldlg->getRGB (r, g, b, 2);
    initRGB (*col, r, g, b);
    scene_->colorRebuild ();
  }
  if (coldlg->modified (3))
  { col = &scene_->col_anchoredge;
    coldlg->getRGB (r, g, b, 3);
    initRGB (*col, r, g, b);
  }
  if (coldlg->modified (4))
  { col = &scene_->col_viewinglight;
    coldlg->getRGB (r, g, b, 4);
    initRGB (*col, r, g, b);
  }
  if (coldlg->modified (5))
  { col = &scene_->col_ambient;
    coldlg->getRGB (r, g, b, 5);
    initRGB (*col, r, g, b);
    // set new ambient color
    ge3dGlobalAmbient (col);
  }

  colourChanged ();  // redraw scene

} // applyColours


void SceneMenus::colourChanged ()
{
  // set new background (may have changed)
  ge3dBackgroundColor (&scene_->col_background);

  scene_->redraw ();  // redraw with new colour
}


void SceneMenus::anchorMotionChanged ()
{
#ifdef INSTRUMENTED
  Instrumentation::instance()->write_to_log (RString (n_options) + "AnchorMotion");
#endif

  scene_->anchorMotion ((anchormotionstate_->flags () & TelltaleState::is_chosen) ? 1 : 0);

  if (scene_->linksActive ())  // when anchors are highlighted
    scene_->redraw ();     // redraw whole window
}


void SceneMenus::velocityControlChanged ()
{
#ifdef INSTRUMENTED
  Instrumentation::instance()->write_to_log (RString (n_options) + "VelocityControl");
#endif

  scene_->velocityControl ((velocitycontrolstate_->flags () & TelltaleState::is_chosen) ? 1 : 0);
}


void SceneMenus::arbitraryRotationChanged ()
{
#ifdef INSTRUMENTED
  Instrumentation::instance()->write_to_log (RString (n_options) + "ArbitraryRotations");
#endif

  scene_->arbitraryRotations ((arbitraryrotstate_->flags () & TelltaleState::is_chosen) ? 1 : 0);
}


void SceneMenus::setCollisionDetection (int flag)
{
  scene_->collisionDetection (flag);
  colldetectionstate_->set (TelltaleState::is_chosen, flag);
}


void SceneMenus::collisionDetectionChanged ()
{
#ifdef INSTRUMENTED
  Instrumentation::instance()->write_to_log (RString (n_options) + "CollisionDetection");
#endif

  scene_->collisionDetection ((colldetectionstate_->flags () & TelltaleState::is_chosen) ? 1 : 0);
}


void SceneMenus::toggleShowAnchors ()
{
  int flag = (showanchorstate_->flags () & TelltaleState::is_chosen) ? 1 : 0;
  showanchorstate_->set (TelltaleState::is_chosen, !flag);
  showAnchorsChanged ();
}


void SceneMenus::showAnchorsChanged ()
{
#ifdef INSTRUMENTED
  Instrumentation::instance()->write_to_log (dis_anchors);
#endif

  scene_->activateLinks ((showanchorstate_->flags () & TelltaleState::is_chosen) ? 1 : 0);
  scene_->redraw ();
  scene_->statusMessage (STranslate::str (STranslate::AnchorHintACTIVATION));
}


void SceneMenus::aaliasingChanged ()
{
#ifdef INSTRUMENTED
  Instrumentation::instance()->write_to_log (dis_lineantialiasing);
#endif

  int aa = 0;
  if (antialiasing (aa_lines))
    aa = ge3d_aa_lines;
  if (antialiasing (aa_polygons))
    aa += ge3d_aa_polygons;
  if (antialiasing (aa_polygons2))
    aa += ge3d_aa_polygons_front2back;

  ge3dAntialiasing (aa);
  ge3dBackgroundColor (&scene_->col_background);
  scene_->front2backDrawing (antialiasing (aa_polygons2));

  scene_->redraw ();
}


void SceneMenus::toggleAntialiasing (int which)
{
  TelltaleState* ts = aaliasingstate_ [which];
  ts->set (TelltaleState::is_chosen, !ts->test (TelltaleState::is_chosen));
  aaliasingChanged ();
}


void SceneMenus::viewingLightChanged ()
{
#ifdef INSTRUMENTED
  Instrumentation::instance()->write_to_log (dis_viewinglight);
#endif

  scene_->viewingLight ((viewlgtstate_->flags () & TelltaleState::is_chosen) ? 1 : 0);
  scene_->redraw ();
}

void SceneMenus::stereoModeChanged ()
{ // menu callback
  int stereomode = (stereostate_->flags () & TelltaleState::is_chosen)
  ? Scene3D::stereo_redgreen : Scene3D::stereo_off;
  scene_->stereoMode (stereomode);
  if (!stereomode)
    ge3dBackgroundColor (&scene_->col_background);
  scene_->redraw ();
}

void SceneMenus::toggleStereoMode ()
{ // extern callable
  int flag = (stereostate_->flags () & TelltaleState::is_chosen) ? 1 : 0;
  stereostate_->set (TelltaleState::is_chosen, !flag);
  stereoModeChanged ();
}

void SceneMenus::toggleTextureLighting ()
{
#ifdef INSTRUMENTED
  Instrumentation::instance()->write_to_log (dis_texturelighting);
#endif

  scene_->textureLighting (!scene_->textureLighting ());
  scene_->redraw ();
}

void SceneMenus::toggleTextureTransparency ()
{
#ifdef INSTRUMENTED
  Instrumentation::instance()->write_to_log (dis_texturetransparency);
#endif

  scene_->textureTransparency (!scene_->textureTransparency ());
  if (scene_->mode () == ge3d_texturing)
    scene_->redraw ();
}

void SceneMenus::toggleNoMaterials ()
{
  scene_->noMaterials (!scene_->noMaterials ());
  scene_->redraw ();
}

void SceneMenus::resetView ()
{
#ifdef INSTRUMENTED
  Instrumentation::instance()->write_to_log(n_resetview); //--- by iegger
#endif
  scene_->restoreCamera ();
  inputhand_->reset ();
  scene_->redraw ();
}


void SceneMenus::levelView ()
{
#ifdef INSTRUMENTED
  Instrumentation::instance()->write_to_log(n_levelview); //--- by iegger
#endif
  Camera* cam = scene_->getCamera ();
  if (cam)
  { cam->makeHoricontal ();
    inputhand_->reset ();
    scene_->redraw ();
  }
  // center of rotation is camera position
  // for flip object center of scene should be used, but
  // makehoricontal is not really necessary for flip object
}


void SceneMenus::untiltView ()
{
#ifdef INSTRUMENTED
  Instrumentation::instance()->write_to_log(n_untiltview); //--- by iegger
#endif
  Camera* cam = scene_->getCamera ();
  if (cam)
  { cam->untilt ();
    inputhand_->reset ();
    scene_->redraw ();
  }
}


void SceneMenus::addViewpoint (
  const char* name, QvPerspectiveCamera* pcam, QvOrthographicCamera* ocam
)
{
  // cerr << "adding camera '" << name << "' to viewpoint menu." << endl;
  if (!viewpointsPr_)
    return;

  // preset items (prev/next, Separator), let numbers start with 1
  int num = (int) viewpointsPr_->item_count () - viewproffset_ + 1;
  char defname [32];
  char number [16];
  strcpy (defname, STranslate::str (STranslate::NavigateVIEWPOINT));
  sprintf (number, " %d", num);
  strcat (defname, number);  // "Camera NN"
  if (!name || !*name)
    name = defname;

  viewpointsPr_->append_item (MenuKit::menuItem (
    *kit_, *layout_, name,
    new ViewpointCallback (this, name, pcam, ocam), (num < 10) ? number+1 : (const char*) 0));

} // addViewpoint


void SceneMenus::clearViewpoints ()
{
  if (!viewpointsPr_ || !viewptsParent_)
    return;

  int i = (int) viewpointsPr_->item_count ();
  while (i-- > viewproffset_)  // remove old viewpoints
    viewpointsPr_->remove_item (i);

  // must unbind old pullright or its size won't change
  MenuItem* item = viewptsParent_->item (viewpointsIndex_);
  Window* prwin = item ? item->window () : 0;
  if (prwin)
    prwin->unbind ();

} // clearViewpoints


void SceneMenus::selectViewpoint (int i)
{
  // i runs from 0 (for 1st camera)
  if (!viewpointsPr_)
    return;

  int index = i + viewproffset_;  // preset items (prev/next, Separator)
  int num = (int) viewpointsPr_->item_count ();
  if (i >= 0 && index < num)
  {
    MenuItem* item = viewpointsPr_->item (index);
    item->action ()->execute ();
    curviewpoint_ = i;
    // cerr << "current viewpoint set to " << i << endl;
  }
}


void SceneMenus::prevViewpoint ()
{
  int i = curviewpoint_ - 1;
  int num = (int) viewpointsPr_->item_count () - viewproffset_;

  if (i < 0)
    selectViewpoint (num - 1);
  else
    selectViewpoint (i);
}


void SceneMenus::nextViewpoint ()
{
  int i = curviewpoint_ + 1;
  int num = (int) viewpointsPr_->item_count () - viewproffset_;

  if (i >= num)
    selectViewpoint (0);
  else
    selectViewpoint (i);
}


void SceneMenus::removeSelection ()
{
  if (scene_->selectObj (0) || scene_->selectNode (0))
  { 
    if (scene_->editdata () && scene_->manipulationAllowed ())
      scene_->editdata ()->updateEditContext (); // update edit dialogs
    scene_->redraw ();
  }
}


void SceneMenus::printData ()
{
  scene_->printInfo (0);
}


void SceneMenus::printAll ()
{
  scene_->printInfo (1);
}


void SceneMenus::cameraInfo ()
{
  Camera* cam = scene_->getCamera ();

  if (!cam)
  { cerr << "no camera!";
    return;
  }

  point3D p;

  cerr << "current camera:" << endl;
  cam->print ();

  cam->getposition (p);
  cerr << "camera position: " << p << endl;
  cam->getlookat (p);
  cerr << "lookat position: " << p << endl;
  cam->getupvector (p);
  cerr << "up direction   : " << p << endl
       << endl;
}


// dialog opening/closing functions moved to dialogs.C:
// toggleAbout
// toggleErrorDialog
// closeErrorDialog
// toggleSpaceballDialog
// closeSpaceballDialog


void SceneMenus::editdialogClosed (int i)
{
  // assert: 0 <= i < edlg_num
  TelltaleState* t = editdlgstate_ [i];
  t->set (TelltaleState::is_chosen, 0);
}


int SceneMenus::getSballFlag (int i)
{
#ifdef SPACEBALL
  TelltaleState* t = sbflagstate_ [i];
  return (int) t->test (TelltaleState::is_chosen);
#else
  return i;
#endif
}

int SceneMenus::toggleSballFlag (int i)
{
#ifdef SPACEBALL
  TelltaleState* t = sbflagstate_ [i];

  int newflag = !t->test (TelltaleState::is_chosen);
  t->set (TelltaleState::is_chosen, newflag);

  sb_checkbuttons (i);

  return newflag;
#else
  return i;
#endif
}


void SceneMenus::toggleSballMode ()
{
#ifdef SPACEBALL
  // spaceball metaphors: flip (manipulate object) and walk (manipulate view)
  if (scene_->navigationMode () == NavMode::flip_obj)
    navigationMode (NavMode::walk_around);
  else
    navigationMode (NavMode::flip_obj);
#endif
}


void SceneMenus::sb_checkbuttons (int i)
{
#ifdef SPACEBALL
  // ensure at least one of translation/rotation is turned on
  // button i is the most recently toggled one
  TelltaleState* t = sbflagstate_ [i];
  if (t->test (TelltaleState::is_chosen))
    return;

  if (i == sb_translate)
    sbflagstate_ [sb_rotate]->set (TelltaleState::is_chosen, true);
  else if (i == sb_rotate)
    sbflagstate_ [sb_translate]->set (TelltaleState::is_chosen, true);
#endif
}


float SceneMenus::sballSensitivity () const
{
  // internally stored as power of 2
  if (sbsensitivity2_ < 0)
    return 1.0 / (1 << -sbsensitivity2_);
  return (float) (1 << sbsensitivity2_);
}


void SceneMenus::updateSensitivity ()
{
#ifdef SPACEBALL
  if (!sbsensfieldb_)
    return;

  char buf [64];
  if (sbsensitivity2_ < 0)
    sprintf (buf, "1/%d", 1 << -sbsensitivity2_);
  else
    sprintf (buf, "%d", 1 << sbsensitivity2_);

  sbsensfieldb_->field (buf);
#endif
}


void SceneMenus::decreaseSballSensitivity ()
{
  if (sbsensitivity2_ > -10)
  { sbsensitivity2_--;
    updateSensitivity ();
  }
}


void SceneMenus::increaseSballSensitivity ()
{
  if (sbsensitivity2_ < 10)
  { sbsensitivity2_++;
    updateSensitivity ();
  }
}


void SceneMenus::resetSballSensitivity ()
{
  if (sbsensitivity2_ != 0)
  { sbsensitivity2_ = 0;
    updateSensitivity ();
  }
}
