#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <unistd.h>

#include "main.h"
#include "feature.h"
#include "debug.h"
#include "mem.h"
#include "options.h"

#ifdef USE_IMLIB
#  include <X11/imlib.h>
#endif

#ifdef PIXMAP_SUPPORT
/* Specifying a single extension is irrelevant with Imlib. -vendu */
#  ifdef USE_IMLIB
#    define PIXMAP_EXT NULL
#  endif

extern XWindowAttributes attr;
extern char * rs_path;
extern const char * rs_pixmapScale;

/* the originally loaded pixmap and its scaling */
Pixmap set_bgPixmap (const char * /* file */);

static struct {
  short w, h, x, y;
  Pixmap pixmap;
} bgPixmap = { 0, 0, 50, 50, None };

#  ifdef IMLIB_SCALING
static struct {
  ImlibData *id;
  Image *im;
  int last_w,last_h;
} imlib = { NULL, NULL, 0, 0 };
#  endif

#  ifdef USE_IMLIB
#    ifdef IMLIB_SCALING

void
ReadPixmapViaImlib(Display *d, char *filename)
{
#      ifdef DEBUG_IMLIB
  fprintf(stderr, "ReadPixmapViaImlib(%s)\n", filename);
#      endif
  if (imlib.id == NULL) {
#      ifdef DEBUG_IMLIB
	fprintf(stderr, "ImlibInit()\n");
#      endif
	imlib.id = ImlibInit(d);
  }
#      ifdef DEBUG_IMLIB
  fprintf(stderr, "ImlibLoadImage(%s)\n", filename);
#      endif
  imlib.im = ImlibLoadImage(imlib.id, filename, NULL);
}

#    else /* IMLIB_SCALING */
Pixmap
ReadFileToPixmapViaImlib(Display *d, char *filename, int *w, int *h)
{
  ImlibData  *id;
  Image      *im;
  Pixmap      pmap;
  
#      ifdef DEBUG_IMLIB
  fprintf(stderr, "ReadFileToPixmapViaImlib(%s)\n", filename);
#      endif
  id = ImlibInit(d);
  im = ImlibLoadImage(id,filename,NULL);
  
  if (!im) return 0;

  *w=im->rgb_width;
  *h=im->rgb_height;
  
  ImlibRender(id,im,*w,*h);
  pmap = ImlibCopyImageToPixmap(id,im);
  ImlibDestroyImage(id,im);
  
  return pmap;
}
#    endif /* IMLIB_SCALING */
#  endif /* USE_IMLIB */

/*
 * These GEOM strings indicate absolute size/position:
 * @ `WxH+X+Y'
 * @ `WxH+X'	-> Y = X
 * @ `WxH'	-> Y = X = 50
 * @ `W+X+Y'	-> H = W
 * @ `W+X'	-> H = W, Y = X
 * @ `W'	-> H = W, X = Y = 50
 * @ `0xH'	-> H *= H/100, X = Y = 50 (W unchanged)
 * @ `Wx0'	-> W *= W/100, X = Y = 50 (H unchanged)
 * @ `=+X+Y'	-> (H, W unchanged)
 * @ `=+X'	-> Y = X (H, W unchanged)
 *
 * These GEOM strings adjust position relative to current position:
 * @ `+X+Y'
 * @ `+X'	-> Y = X
 *
 * And this GEOM string is for querying current scale/position:
 * @ `?'
 */


  /*   '[', 2*4 + 2*3 digits + 3 delimiters, ']'. -vendu */
#define GEOM_LEN 19    /* #undef'd immediately after scale_pixmap(). - vendu */
int
scale_pixmap (const char * geom)
{
  /*   static char str [] = "[1000x1000+100+100]";   /* should be big enough */
  static char str[GEOM_LEN+1] = {'\0'};
  int w = 0, h = 0, x = 0, y = 0;
  int flags;
  int changed = 0;
  char * p, tmp;
  int n;
  
  if (geom == NULL) return 0;
  
  dprintf(("scale_pixmap(\"%s\")\n", geom));
  if (!strcmp (geom, "?"))
	{
	  sprintf (str,
			   "[%dx%d+%d+%d]",
			   bgPixmap.w,
			   bgPixmap.h,
			   bgPixmap.x,
			   bgPixmap.y);
	  xterm_seq (XTerm_title, str);
	  return 0;
	}
  
  {
	
	if ((p = strchr (geom, ';')) == NULL)
	  p = strchr (geom, '\0');
	n = (p - geom);
	/*	if (n >= sizeof(str) - 1) return 0;    NO! -vendu */
	if (n > GEOM_LEN - 1) return 0;
	strncpy (str, geom, n);
	str [n] = '\0';
  }
  
  flags = XParseGeometry (str, &x, &y, &w, &h);
  
  if (!flags) {
	flags |= WidthValue;    /* default is tile */
	w = 0; 
  }
  
  if (flags & WidthValue)
	{
	  if (!(flags & XValue)) { x = 50; }
	  if (!(flags & HeightValue)) h = w;
	  
	  if (w && !h)
		{
		  w = bgPixmap.w * ((float)w/100);
		  h = bgPixmap.h;
		}
	  else if (h && !w)
		{
		  w = bgPixmap.w;
		  h = bgPixmap.h * ((float)h/100);
		}
	  
	  /* FIXME: Should be fixed to use automagic limits. -vendu */
	  
	  if (w > 1000) w = 1000;
	  if (h > 1000) h = 1000;
	  
	  if (bgPixmap.w != w)
		{
		  bgPixmap.w = w;
		  changed++;
		}
	  if (bgPixmap.h != h)
		{
		  bgPixmap.h = h;
		  changed++;
		}
	}
  
  if (!(flags & YValue))
	{
	  if (flags & XNegative) flags |= YNegative;
	  y = x;
	}
  if (!(flags & WidthValue) && geom [0] != '=')
	{
	  x += bgPixmap.x;
	  y += bgPixmap.y;
	}
  else
	{
	  if (flags & XNegative) x += 100;
	  if (flags & YNegative) y += 100;
	}
  
  x = (x <= 0 ? 0 : (x >= 100 ? 100 : x));
  y = (y <= 0 ? 0 : (y >= 100 ? 100 : y));;
  if (bgPixmap.x != x)
	{
	  bgPixmap.x = x;
	  changed++;
	}
  if (bgPixmap.y != y)
	{
	  bgPixmap.y = y;
	  changed++;
	}
  return changed;
}
#undef GEOM_LEN

void
resize_pixmap (void)
{
  XGCValues gcvalue;
  GC gc;
  unsigned int width = TermWin_TotalWidth ();
  unsigned int height = TermWin_TotalHeight ();
  float p, incr;
  Pixmap tmp;
  int xsize, ysize;
#if 0
  Window W_dummy;
  int dummy;
  unsigned int u_dummy;
#endif
  
  /* This *might* be useful later... -vendu */
#if 0
  /* Ignore Xroot, x, y, border and depth. */
  XGetGeometry(Xdisplay, TermWin.vt, &W_dummy, &dummy, &dummy, &width, &height, &u_dummy, &u_dummy);
#endif
  
  if (!imlib.im) return;

#  ifdef IMLIB_SCALING
#    ifdef DEBUG_IMLIB
  fprintf(stderr, "resize_pixmap() block 1\n");
#    endif
	xsize = imlib.im->rgb_width;
	ysize = imlib.im->rgb_height;
#  else
  xsize = attr.width;
  ysize = attr.height;
#  endif
  
  if (TermWin.pixmap != None) {
#  ifdef DEBUG_IMLIB
	fprintf(stderr, "ImlibFreePixmap(TermWin.pixmap)\n");
#  endif
	if (imlib.id) ImlibFreePixmap(imlib.id, TermWin.pixmap);
	TermWin.pixmap = None;
  }
  
#  ifdef DEBUG_PIXMAP
  fprintf(stderr, "XCreatePixmap(TermWin.pixmap)\n");
#  endif
  TermWin.pixmap = XCreatePixmap (Xdisplay, TermWin.vt,
								  width, height, Xdepth);
  
  gcvalue.foreground = PixColors [bgColor];
  gc = XCreateGC (Xdisplay, TermWin.vt, GCForeground, &gcvalue);
  
#  ifdef IMLIB_SCALING
#    ifdef DEBUG_IMLIB
  fprintf(stderr, "resize_pixmap() block 2\n");
#    endif
  if (imlib.im)
#  else
	if (bgPixmap.pixmap)
#  endif
	  {
		int w = bgPixmap.w;
		int h = bgPixmap.h;
		int x = bgPixmap.x;
		int y = bgPixmap.y;
		
		/*
		 * don't zoom pixmap too much nor expand really small pixmaps
		 */
		
#  ifdef DEBUG_IMLIB
		fprintf(stderr, "resize_pixmap() block 3\n");
#  endif
		if (w > 1000 || h > 1000) {
		  w = 1;
		} else if (width > (10 * xsize) || height > (10 * ysize))
		  w = 0;		/* tile */
		
		if (w)
		  {
#  if defined(USE_IMLIB) && !defined(IMLIB_SCALING)
#    ifdef DEBUG_PIXMAP
			fprintf(stderr, "tmp = XCreatePixmap()\n");
#    endif
			tmp = XCreatePixmap (Xdisplay, TermWin.vt,
								 width, ysize,
								 Xdepth);
			
			XFillRectangle (Xdisplay, tmp, gc, 0, 0, width, ysize);
#  endif
			/*
			 * horizontal scaling
			 */
#  ifdef DEBUG_IMLIB
			fprintf(stderr, "resize_pixmap() block 4\n");
#  endif
			incr = (float)xsize;
			
			p = 0;
			
			if (w == 1)
			  {
				/* display image directly - no scaling at all */
				incr = width;
#  ifdef DEBUG_IMLIB
				fprintf(stderr, "resize_pixmap() block 5\n");
#  endif
				if (xsize <= width)
				  {
#  ifdef DEBUG_IMLIB
					fprintf(stderr, "resize_pixmap() block 6\n");
#  endif
					w = xsize;
					x = (width - w) / 2;
					w += x;
				  }
				else
				  {
					x = 0;
					w = width;
				  }
			  }
			else if (w < 10)
			  {
				incr *= w;	/* fit W images across screen */
				x = 0;
				w = width;
			  }
			else
			  {
				incr *= 100.0 / w;
				if (w < 100)	/* contract */
				  {
					w = (w * width) / 100;
					if (x >= 0)	/* position */
					  {
						float pos;
						pos = (float)x / 100 * width - (w/2);
						
						x = (width - w);
						if (pos <= 0) x = 0; else if (pos < x) x = pos;
					  }
					else
					  {
						x = (width - w) / 2;
					  }
					w += x;
				  }
				else if (w >= 100)		/* expand */
				  {
					if (x > 0)	/* position */
					  {
						float pos;
#  ifdef DEBUG_IMLIB
						fprintf(stderr, "resize_pixmap() block 7\n");
#  endif
						pos = (float)x / 100 * xsize - (incr/2);
						p = xsize - (incr);
						if (pos <= 0) p = 0; else if (pos < p) p = pos;
					  }
					x = 0;
					w = width;
				  }
			  }
			incr /= width;
			
#  ifndef IMLIB_SCALING
			for (/*nil*/; x < w; x++, p += incr)
			  {
#    ifdef USE_IMLIB
				if (p >= xsize) p = 0;
				XCopyArea (Xdisplay, bgPixmap.pixmap, tmp, gc,
						   (int)p, 0, 1, ysize, x, 0);
#    endif
			  }
#  endif /* !IMLIB_SCALING */
			
			/*
			 * vertical scaling
			 */
#  ifdef DEBUG_IMLIB
			fprintf(stderr, "resize_pixmap() block 8\n");
#  endif
			incr = (float)ysize;
			p = 0;
			
			if (h == 1)
			  {
				/* display image directly - no scaling at all */
				incr = height;
#  ifdef DEBUG_IMLIB
				fprintf(stderr, "resize_pixmap() block 9\n");
#  endif
				if (ysize <= height)
				  {
#  ifdef DEBUG_IMLIB
					fprintf(stderr, "resize_pixmap() block 10\n");
#  endif
					h = ysize;
					y = (height - h) / 2;
					h += y;
				  }
				else
				  {
					y = 0;
					h = height;
				  }
			  }
			else if (h < 10)
			  {
				incr *= h;	/* fit H images across screen */
				y = 0;
				h = height;
			  }
			else
			  {
				incr *= 100.0 / h;
				if (h < 100)		/* contract */
				  {
					h = (h * height) / 100;
					if (y >= 0)	/* position */
					  {
						float pos;
						pos = (float)y / 100 * height - (h/2);
						
						y = (height - h);
						if (pos < 0.0f) y = 0; else if (pos < y) y = pos;
					  }
					else
					  {
						y = (height - h) / 2;
					  }
					h += y;
				  }
				else if (h >= 100)		/* expand */
				  {
					if (y > 0)		/* position */
					  {
						float pos;
#  ifdef DEBUG_IMLIB
						fprintf(stderr, "resize_pixmap() block 11\n");
#  endif
						pos = (float)y / 100 * ysize - (incr/2);
						p = ysize - (incr);
						if (pos < 0) p = 0; else if (pos < p) p = pos;
					  }
					y = 0;
					h = height;
				  }
			  }
			incr /= height;
			
#  ifdef IMLIB_SCALING
#    ifdef DEBUG_IMLIB
			fprintf(stderr, "resize_pixmap() block 12\n");
#    endif
			imlib.last_w = w; imlib.last_h = h;
#    ifdef DEBUG_IMLIB
			fprintf(stderr, "ImlibRender(%dx%d)\n", w, h);
#    endif
			ImlibRender(imlib.id, imlib.im, w, h);
#    ifdef DEBUG_PIXMAP
			fprintf(stderr, "bgPixmap.pixmap = ImlibCopyImageToPixmap()\n");
#    endif
			bgPixmap.pixmap = ImlibCopyImageToPixmap(imlib.id, imlib.im);
#    ifdef DEBUG_IMLIB
			fprintf(stderr, "ImlibDestroyImage()\n");
#    endif
			ImlibDestroyImage(imlib.id, imlib.im);
#    ifdef DEBUG_PIXMAP
			fprintf(stderr, "TermWin.pixmap = bgPixmap.pixmap\n");
#    endif
			TermWin.pixmap = bgPixmap.pixmap;
#  else
			for (/*nil*/; y < h; y++, p += incr)
			  {
#    ifdef USE_IMLIB
				if (p >= ysize) p = 0;
#    endif
				XCopyArea (Xdisplay, tmp, TermWin.pixmap, gc,
						   0, (int)p, width, 1, 0, y);
			  }
#    ifdef DEBUG_IMLIB
			fprintf(stderr, "ImlibFreePixmap(tmp)\n");
#    endif
			ImlibFreePixmap(imlib.id, tmp);
#  endif /* IMLIB_SCALING */
		  }
		
		else /* if (w), light years above. -vendu*/
		  {
			/* tiled */
			
#  ifdef IMLIB_SCALING
#    ifdef DEBUG_IMLIB
			fprintf(stderr, "resize_pixmap() block 13\n");
			fprintf(stderr, "ImlibRender(%dx%d)\n", xsize, ysize);
#    endif
			ImlibRender(imlib.id, imlib.im, xsize, ysize);
#    ifdef DEBUG_IMLIB
			fprintf(stderr, "bgPixmap.pixmap = ImlibCopyImageToPixmap()\n");
#    endif
			bgPixmap.pixmap=ImlibCopyImageToPixmap(imlib.id, imlib.im);
#    ifdef DEBUG_IMLIB
			fprintf(stderr, "ImlibDestroyImage()\n");
#    endif
			ImlibDestroyImage(imlib.id, imlib.im);
#  endif
			
#  ifdef DEBUG_IMLIB
			fprintf(stderr, "resize_pixmap() block 14\n");
#  endif
			for (y = 0; y < height; y += ysize)
			  {
				unsigned int h = (height - y);
#  ifdef USE_IMLIB
#    ifdef DEBUG_IMLIB
				fprintf(stderr, "resize_pixmap() block 15\n");
#    endif
				if (h > ysize) h = ysize;
				for (x = 0; x < width; x += xsize)
#  endif
				  {
					unsigned int w = (width - x);
#  ifdef DEBUG_IMLIB
					fprintf(stderr, "resize_pixmap() block 16\n");
#  endif
					if (w > xsize) w = xsize;
					XCopyArea (Xdisplay,
							   bgPixmap.pixmap,
							   TermWin.pixmap,
							   gc,
							   0, 0, w, h, x, y);
				  }
			  }
		  }
	  }
	else
	  XFillRectangle (Xdisplay, TermWin.pixmap, gc,
					  0, 0, width, height);
  
#  ifdef PIXMAP_BUFFERING
  if (TermWin.buf_pixmap) XFreePixmap (Xdisplay, TermWin.buf_pixmap);
  TermWin.buf_pixmap = XCreatePixmap (Xdisplay, TermWin.vt,
									  width, height, Xdepth);
  XCopyArea (Xdisplay,
			 TermWin.pixmap,
			 TermWin.buf_pixmap,
			 gc,
			 0, 0,
			 width, height,
			 0, 0);
  XSetWindowBackgroundPixmap (Xdisplay, TermWin.vt, TermWin.buf_pixmap);
#  else	/* PIXMAP_BUFFERING */
  XSetWindowBackgroundPixmap (Xdisplay, TermWin.vt, TermWin.pixmap);
#  endif	/* PIXMAP_BUFFERING */
  XSync (Xdisplay, 0);
  XFreeGC (Xdisplay, gc);
  
  XClearWindow (Xdisplay, TermWin.vt);
  XFlush (Xdisplay);
  
  XSync (Xdisplay, 0);
  
}
#else	/* PIXMAP_SUPPORT */
#  define scale_pixmap(str)	((void)0)
#  define resize_pixmap()	((void)0)
#endif	/* PIXMAP_SUPPORT */

#if defined(PIXMAP_SUPPORT) || (MENUBAR_MAX)
/*
 * search for FILE in the current working directory, and within the
 * colon-delimited PATHLIST, adding the file extension EXT if required.
 *
 * FILE is either @ or zero terminated
 */
const char *
search_path(const char *pathlist, const char *file, const char *ext) {
  
  /* FIXME: the 256 below should be changed to some #define in <dirent.h> */
  static char name [256];
  char * p;
  const char * path; /* Was char * before. -vendu */
  int maxpath, len;
  struct stat fst;
  
#  ifdef DEBUG_OPTIONS
  fprintf(stderr, "search_path(\"%s\", \"%s\", \"%s\") called.\n",
		  pathlist, file, ext);
  fprintf(stderr, "search_path():  Checking for file \"%s\"\n", file);
#  endif
  if (!access(file, R_OK)) {
    if (stat(file, &fst)) {
      dprintf(("Unable to stat %s -- %s\n", file, strerror(errno)));
    } else {
      dprintf(("Stat returned mode 0x%08o, S_ISDIR() == %d\n", fst.st_mode, S_ISDIR(fst.st_mode)));
    }
    if (!S_ISDIR(fst.st_mode)) return file;
  }
  
  /* Changed to use '@' as the delimiter. -vendu */
  if ((p = strchr(file, '@')) == NULL) p = strchr(file, '\0');
  len = (p - file);
  
  /* check about adding a trailing extension */
  if (ext != NULL) {
	
    char *dot;
	
    dot  = strrchr(p, '.');
    path = strrchr(p, '/');
    if (dot != NULL || (path != NULL && dot <= path)) ext = NULL;
  }
  
  /* leave room for an extra '/' and trailing '\0' */
  maxpath = sizeof(name) - (len + (ext ? strlen(ext) : 0) + 2);
  if (maxpath <= 0) return NULL;
  
  /* check if we can find it now */
  strncpy(name, file, len);
  name[len] = '\0';
  
#  ifdef DEBUG_OPTIONS
  fprintf(stderr, "search_path():  Checking for file \"%s\"\n", name);
#  endif
  if (!access(name, R_OK)) {
    stat(name, &fst);
    if (!S_ISDIR(fst.st_mode)) return name;
  }
  if (ext) {
    strcat(name, ext);
#  ifdef DEBUG_OPTIONS
    fprintf(stderr, "search_path():  Checking for file \"%s\"\n", name);
#  endif
    if (!access(name, R_OK)) {
      stat(name, &fst);
      if (!S_ISDIR(fst.st_mode)) return name;
    }
  }
  
  for (path = pathlist; path != NULL && *path != '\0'; path = p) {
	
    int n;
	
    /* colon delimited */
    if ((p = strchr(path, ':')) == NULL) p = strchr(path, '\0');
	
    n = (p - path);
    if (*p != '\0') p++;
	
    if (n > 0 && n <= maxpath) {
	  
      strncpy(name, path, n);
      if (name[n-1] != '/') name[n++] = '/';
      name[n] = '\0';
      strncat(name, file, len);
	  
#  ifdef DEBUG_OPTIONS
      fprintf(stderr, "search_path():  Checking for file \"%s\"\n", name);
#  endif
      if (!access(name, R_OK)) {
		stat(name, &fst);
		if (!S_ISDIR(fst.st_mode)) return name;
      }
      if (ext) {
		strcat(name, ext);
#  ifdef DEBUG_OPTIONS
		fprintf(stderr, "search_path():  Checking for file \"%s\"\n", name);
#  endif
		if (!access(name, R_OK)) {
		  stat(name, &fst);
		  if (!S_ISDIR(fst.st_mode)) return name;
		}
      }
    }
  }
  return NULL;
}
#endif	/* PIXMAP_SUPPORT || (MENUBAR_MAX) */

#ifdef PIXMAP_SUPPORT
/* I think it might be cool to make Eterm load the pixmaps in background.
 * You'd be able to start typing commands without waiting for the bg
 * pixmap processing to end. Thoughts right now: fork(), pthreads. -vendu
 */

Pixmap
set_bgPixmap (const char *file) {
  
  const char *f; /* Was char * before. -vendu */
  
  /* FIXME: assert() looks like a bad thing. Calls abort(). IMHO, stupid. */
  assert(file != NULL);
  
#  ifdef DEBUG_IMLIB
  fprintf(stderr, "[debug] set_bgPixmap(%s) called.\n", file);
#  endif
  
  /* Turn on scaling */
  if ((Options & Opt_pixmapScale) || (rs_pixmapScale)) {
    bgPixmap.h = 100; bgPixmap.w = 100;
  }
  
  XSetWindowBackground(Xdisplay, TermWin.vt, PixColors[bgColor]);
  
  if (*file != '\0') {
    XGetWindowAttributes(Xdisplay, TermWin.vt, &attr);
	
    /* search environment variables here too */
#  ifdef USE_IMLIB
    if ((f = search_path(rs_path, file, PIXMAP_EXT)) == NULL)
#  endif
#  ifdef PATH_ENV
#    ifdef USE_IMLIB
      if ((f = search_path(getenv (PATH_ENV), file, PIXMAP_EXT)) == NULL)
#    endif
#  endif /* PATH_ENV */
#  ifdef USE_IMLIB
		f = search_path(getenv("PATH"), file, PIXMAP_EXT);
#  endif
	
#  ifdef USE_IMLIB
#    ifdef IMLIB_SCALING
    if (f != NULL) {
      if (imlib.im != NULL) {
#      ifdef DEBUG_IMLIB
		fprintf(stderr, "ImlibDestroyImage()\n");
#      endif
		ImlibDestroyImage(imlib.id, imlib.im);
	  }
#      ifdef DEBUG_IMLIB
	  fprintf(stderr, "ReadPixmapViaImlib(%s)\n", (char*)f);
#      endif
      ReadPixmapViaImlib(Xdisplay, (char *)f);
    }
    if (imlib.im == NULL) {
#    else
#      ifdef DEBUG_IMLIB
	  fprintf(stderr, "ReadFileToPixmapViaImlib(%s)\n", (char*)f);
#      endif
	  if (f == NULL || !(bgPixmap.pixmap = 
						 ReadFileToPixmapViaImlib(Xdisplay, (char *)f,
												  &attr.width,
												  &attr.height))) {
#    endif
#  endif
		char * p;
		
		if ((p = strchr(file, ';')) == NULL) p = strchr(file, '\0');
		print_error("couldn't load image file \"%.*s\"", (p - file), file);
		resize_pixmap();
	  } else if (bgPixmap.pixmap != None) resize_pixmap();
	} else {
#  ifdef DEBUG_PIXMAP
	  fprintf(stderr, "Nuking pixmap pointers");
	  bgPixmap.pixmap = None;
#  endif
	  rs_pixmaps[pixmap_bg] = "";
#  ifdef USE_IMLIB
	  imlib.im = 0;
#  endif
	}
	
	/* FIXME: The call to XClearWindow could probably be removed. -vendu */
	XClearWindow(Xdisplay, TermWin.vt);
	scr_touch();
	XFlush(Xdisplay);
	return bgPixmap.pixmap;
  }
#  ifdef USE_IMLIB
#    undef PIXMAP_EXT
#  endif
#endif	/* PIXMAP_SUPPORT */
