/*
 * IceWM
 *
 * Copyright (C) 1997 Marko Macek
 */

#include "icewm.h"

void YApplication::popup(YWindow *forWindow, YPopupWindow *popup) {
    assert(popup != 0);
    if (fPopup == 0) {
        Cursor changePointer = (popup->popupFlags() & YPopupWindow::pfNoPointerChange) ? None : rightPointer;
        
        if (!grabEvents(forWindow ? forWindow : popup, changePointer,
                        ButtonPressMask | ButtonReleaseMask | ButtonMotionMask))
        {
            popup->cancelPopup();
            return ;
        }
    }
    popup->setPrevPopup(fPopup);
    fPopup = popup;
}

void YApplication::popdown(YPopupWindow *popdown) {
    if (popdown != fPopup) {
	MSG(("popdown: 0x%lX  fPopup: 0x%lX", popdown, fPopup));
        return ;
    }
    assert(fPopup != 0);
    fPopup = fPopup->prevPopup();
    
    if (fPopup == 0)
        releaseEvents();
}

YPopupWindow::YPopupWindow(YWindow *aParent): YWindow(aParent) {
    fForWindow = 0;
    fPrevPopup = 0;
    fCommandListener = 0;
    fFlags = 0;
    setStyle(wsSaveUnder | wsOverrideRedirect);
}

YPopupWindow::~YPopupWindow() {
}

void YPopupWindow::updatePopup() {
}

void YPopupWindow::sizePopup() {
}

void YPopupWindow::activatePopup() {
}

void YPopupWindow::deactivatePopup() {
}

void YPopupWindow::popup(YWindow *forWindow, CommandListener *commandListener, unsigned int flags) {
    fPrevPopup = 0;
    fFlags = flags;
    fForWindow = forWindow;
    fCommandListener = commandListener;
    if (fCommandListener)
        fCommandListener->setPopupActive(this);

    raise();
    show();

    app->popup(forWindow, this);
    activatePopup();
}

void YPopupWindow::popup(YWindow *forWindow, CommandListener *commandListener, int x, int y, int x_delta, int y_delta, unsigned int flags) {

    if ((flags & pfPopupMenu) &&
        (wmLook == lookWarp3 || wmLook == lookWarp4))
        flags |= pfFlipVertical;
    
    fFlags = flags;

    updatePopup();

    sizePopup();

    /* !!! FIX this to maximize visible area */
    if ((x + width() > app->root()->width()) || (fFlags & pfFlipHorizontal))
        if (fFlags & (pfCanFlipHorizontal | pfFlipHorizontal)) {
            x -= width() + x_delta;
            fFlags |= pfFlipHorizontal;
        } else
            x = app->root()->width() - width();
    if ((y + height() > app->root()->height()) || (fFlags & pfFlipVertical))
        if (fFlags & (pfCanFlipVertical | pfFlipVertical)) {
            y -= height() + y_delta;
            fFlags |= pfFlipVertical;
        } else
            y = app->root()->height() - height();
    if (x < 0)
        if (fFlags & pfCanFlipHorizontal)
            x += width() + x_delta;
        else
            x = 0;
    if (y < 0)
        if (fFlags & pfCanFlipVertical)
            y += height() + y_delta;
        else
            y = 0;

    setPosition(x, y);

    popup(forWindow, commandListener, fFlags);
}

void YPopupWindow::popdown() {
    if (fCommandListener)
        fCommandListener->setPopupActive(0);
    
    deactivatePopup();
    app->popdown(this);
    hide();

    fFlags = 0;
    fForWindow = 0;
    fCommandListener = 0;
}

void YPopupWindow::cancelPopup() {
    if (fForWindow) {
        fForWindow->donePopup(this);
    } else {
        popdown();
    }
}

void YPopupWindow::finishPopup(WMCommand command, void *context, unsigned int modifiers) {
    CommandListener *cmd = fCommandListener;
    
    while (app->popup())
        app->popup()->cancelPopup();

    if (cmd)
        cmd->handleCommand(command, context, modifiers);
    else
        app->handleCommand(command, context, modifiers);
}

bool YPopupWindow::handleKey(const XKeyEvent &/*key*/) {
    return true;
}

void YPopupWindow::handleButton(const XButtonEvent &button) {
    if (button.x_root >= x() &&
        button.y_root >= y() &&
        button.x_root < int (x() + width()) &&
        button.y_root < int (y() + height()) &&
        button.window == handle())
    {
        YWindow::handleButton(button);
    } else {
        if (fForWindow) {
            XEvent xev;

            xev.xbutton = button;

            app->handleGrabEvent(fForWindow, xev);
        } else {
            popdown();
        }       
    }
}

void YPopupWindow::handleMotion(const XMotionEvent &motion) {
    if (motion.x_root >= x() &&
        motion.y_root >= y() &&
        motion.x_root < int (x() + width()) &&
        motion.y_root < int (y() + height()) &&
       motion.window == handle())
    {
        YWindow::handleMotion(motion);
    } else {
        if (fForWindow) {
            XEvent xev;

            xev.xmotion = motion;
            
            app->handleGrabEvent(fForWindow, xev);
        }
    }
}

YMenu::YMenu(YWindow *parent): YPopupWindow(parent) {
    fItems = 0;
    fItemCount = 0;
    paintedItem = selectedItem = -1;
    fPopup = 0;
    fPopupActive = 0;
}

YMenu::~YMenu() {
    if (fPopup) {
        fPopup->popdown();
        fPopup = 0;
    }

    for (int i = 0; i < fItemCount; i++)
        delete fItems[i];
    free(fItems);
}

void YMenu::sizePopup() {
    int width = 2;
    int height = 2 + 3;
    int fontHeight = menuFont->height() + 1;
    int maxWidth = 0;
    int maxName = 0;
    int maxParam = 0;
    unsigned int maxh = 16;

    for (int i = 0; i < itemCount(); i++) {
        if (item(i)->command() == cmdSeparator) {
            height += 4;
        } else {
            int tw = 0;
            
            char *name = item(i)->name();
            char *param = item(i)->param();
            
            if (name) {
                tw = XTextWidth(menuFont->getFontStruct(), name, strlen(name));
                if (tw >= maxName)
                    maxName = tw;
            }

            if (param) {
                tw = XTextWidth(menuFont->getFontStruct(), param, strlen(param));
                if (tw >= maxParam)
                    maxParam = tw;
            } else if (item(i)->submenu()) {
                tw = fontHeight + 2;
                if (tw >= maxParam)
                    maxParam = tw;
            }

            unsigned int h = fontHeight;

            if (item(i)->getPixmap() && item(i)->getPixmap()->height() > h)
                h = item(i)->getPixmap()->height();

            if (h > maxh)
                maxh = h;

            height += h + 3;
        }
    }

    maxWidth = maxName + maxParam + (maxParam ? 8 : 0);

    width += 3 + maxh + 2 + 2 + maxWidth + 4 + 4;
    setSize(width, height);
    paramPos = maxName;
}

void YMenu::activatePopup() {
    if (popupFlags() & pfButtonDown)
        selectedItem = -1;
    else
        focusItem(findActiveItem(itemCount() - 1, 1), 0, 0);
}

void YMenu::deactivatePopup() {
    if (fPopup) {
        fPopup->popdown();
        fPopup = 0;
    }
}

void YMenu::donePopup(YPopupWindow *popup) {
    if (fPopup)
        assert(popup == fPopup);
    if (fPopup) {
        fPopup->popdown();
        fPopup = 0;
        if (selectedItem != -1)
            if (item(selectedItem)->submenu() == popup)
                paintItems();
    }
}

void YMenu::paintItems() {
    Graphics &g = getGraphics();
    int yy = 2;
    for (int i = 0; i < itemCount(); i++)
        paintItem(g, i, yy, (i == selectedItem || i == paintedItem) ? 1 : 0);
    paintedItem = selectedItem;
}

int YMenu::findItemPos(int itemNo, int &x, int &y) {
    x = -1;
    y = -1;
    
    if (itemNo < 0 || itemNo > itemCount())
        return -1;

    int fontHeight = menuFont->height() + 1;
    y = 3;
    x = 2;
    
    for (int i = 0; i < itemNo; i++) {
        if (item(i)->command() == cmdSeparator)
            y += 4;
        else {
            unsigned int h = fontHeight;

            if (item(i)->getPixmap() && item(i)->getPixmap()->height() > h)
                h = item(i)->getPixmap()->height();

            y += h + 3;
        }
    }
    return 0;
}

int YMenu::findItem(int x, int y) {
    if (x < 2 || x > int (width() - 3))
        return -1;

    int fontHeight = menuFont->height() + 1;
    int yPos = 2;
    
    for (int i = 0; i < itemCount(); i++) {
        if (item(i)->command() == cmdSeparator) {
            if (y >= yPos && y < yPos + 4)
                return i;
            
            yPos += 4;
        } else {
            unsigned int h = fontHeight;

            if (item(i)->getPixmap() && item(i)->getPixmap()->height() > h)
                h = item(i)->getPixmap()->height();

            if (y >= yPos && y < int(yPos + h + 3))
                return i;

            yPos += h + 3;
        }
    }
    return -1;
}

int YMenu::onCascadeButton(int selItem, int x, int /*y*/) {
    if (selItem != -1 &&
        item(selItem)->command() != cmdSubmenu &&
        item(selItem)->submenu())
    {
        int fontHeight = menuFont->height() + 1;

        unsigned int h = fontHeight;

        if (item(selItem)->getPixmap() && item(selItem)->getPixmap()->height() > h)
            h = item(selItem)->getPixmap()->height();

        if (x <= int(width() - h - 4))
            return 1;
    }
    return 0;
}

void YMenu::focusItem(int itemNo, int submenu, int byMouse) {
    selectedItem = itemNo;

    YMenu *sub = 0;
    if (selectedItem != -1)
        sub = item(selectedItem)->submenu();

    if (sub != fPopup) {
        int repaint = 0;
        
        if (fPopup) {
            fPopup->popdown();
            fPopup = 0;
            repaint = 1;
        }

        if (submenu && sub && item(selectedItem)->isEnabled()) {
            int xp, yp;
            
            findItemPos(selectedItem, xp, yp);
            sub->popup(this, commandListener(),
                       x() + width() - 2, y() + yp - 3,
                       width() - 4, -1,
                       YPopupWindow::pfCanFlipHorizontal |
                       (popupFlags() & YPopupWindow::pfFlipHorizontal) |
                       (byMouse ? (unsigned int)YPopupWindow::pfButtonDown : 0U));
            fPopup = sub;
            repaint = 1;
        }
        if (repaint && selectedItem != -1 && paintedItem == selectedItem &&
            item(selectedItem)->command() != cmdSubmenu)
            paintItems();
    }
    if (paintedItem != selectedItem)
        paintItems();
}

int YMenu::findActiveItem(int cur, int direction) {
    assert(direction == -1 || direction == 1);
    
    if (cur == -1 || itemCount() == 0)
        return -1;

    assert(cur >= 0 && cur < itemCount());

    int c = cur;
    do {
        c += direction;
        if (c < 0) c = itemCount() - 1;
        if (c >= itemCount()) c = 0;
    } while (c != cur && item(c)->command() == cmdSeparator);
    return c;
}

int YMenu::activateItem(int no, int byMouse, unsigned int modifiers) {
    assert(selectedItem == no && selectedItem != -1);
    
    if (item(selectedItem)->isEnabled()) {
        if (item(selectedItem)->command() == cmdSubmenu) {
            focusItem(selectedItem, 1, byMouse);
        } else {
            finishPopup(WMCommand(item(selectedItem)->command()),
                        item(selectedItem)->context(), modifiers);
        }
    } else {
        //XBell(app->display(), 50);
        return -1;
    }
    return 0;
}

bool YMenu::handleKey(const XKeyEvent &key) {
    if (key.type == KeyPress) {
        int k = XKeycodeToKeysym(app->display(), key.keycode, 0);
        int m = KEY_MODMASK(key.state);

        if ((m & ~ShiftMask) == 0) {
            if (itemCount() > 0) {
                if (k == XK_Up)
                    focusItem(findActiveItem(selectedItem, -1), 0, 0);
                else if (k == XK_Down)
                    focusItem(findActiveItem(selectedItem, 1), 0, 0);
                else if (k == XK_Home)
                    focusItem(findActiveItem(itemCount() - 1, 1), 0, 0);
                else if (k == XK_End)
                    focusItem(findActiveItem(0, -1), 0, 0);
                else if (k == XK_Right)
                    focusItem(selectedItem, 1, 0);
                else if (k == XK_Left) {
                    if (prevPopup())
                        cancelPopup();
                } else if (k == XK_Return || k == XK_KP_Enter) {
                    if (selectedItem != -1 && item(selectedItem)->command() != cmdSeparator) {
                        activateItem(selectedItem, 0, key.state);
                        return true;
                    }
                } else if (k == XK_Escape) {
                    cancelPopup();
                } else if ((k < 256) && ((m & ~ShiftMask) == 0)) {
                    for (int i = 0; i < itemCount(); i++) {
                        int hot = item(i)->hotChar();
                        if (hot != -1 &&
                            TOUPPER(char(hot)) == TOUPPER(k))
                        {
                            focusItem(i, 0, 0);
                            if (!(m & ShiftMask))
                                activateItem(i, 0, key.state);
                            return true;
                        }
                    }
                }
            }
        }
    }
    return YPopupWindow::handleKey(key);
}

void YMenu::handleButton(const XButtonEvent &button) {
    if (button.button != 0) {
        int selItem = findItem(button.x_root - x(), button.y_root - y());
        int submenu = onCascadeButton(selItem,
                                      button.x_root - x(),
                                      button.y_root - y()) ? 0 : 1;
        if (button.type == ButtonRelease && fPopupActive == fPopup && fPopup != 0 && submenu) {
            fPopup->popdown();
            fPopupActive = fPopup = 0;
            focusItem(selItem, 0, 1);
            if (submenu)
                paintItems();
            return ;
        } else if (button.type == ButtonPress) {
            fPopupActive = fPopup;
        }
        focusItem(selItem, submenu, 1);
        if (selectedItem != -1 &&
            button.type == ButtonRelease && 
            item(selectedItem)->command() != cmdSeparator &&
            (item(selectedItem)->command() == cmdSubmenu ||
             !item(selectedItem)->submenu() || !submenu)
           )
        {
            activateItem(selectedItem, 1, button.state);
            return ;
        }
        if (button.type == ButtonRelease &&
            (selectedItem == -1 || item(selectedItem)->command() == cmdSeparator))
            focusItem(findActiveItem(itemCount() - 1, 1), 0, 0);
    }
    YPopupWindow::handleButton(button);
}

void YMenu::handleMotion(const XMotionEvent &motion) {
    if ((motion.state & (Button1Mask | Button2Mask | Button3Mask | Button4Mask | Button5Mask))) {
        int selItem = findItem(motion.x_root - x(), motion.y_root - y());
        if (selItem != -1 || app->popup() == this) {
            int submenu = onCascadeButton(selItem,
                                          motion.x_root - x(),
                                          motion.y_root - y()) ? 0 : 1;
            focusItem(selItem, submenu, 1);
        }
    }

    YPopupWindow::handleMotion(motion);
}

YMenu::YMenuItem::YMenuItem(const char *name, int aHotCharPos, const char *param, WMCommand command, YMenu *submenu, void *context) {
    if (name)
        fName = new char[strlen(name) + 1];
    else
        fName = 0;
    if (fName) strcpy(fName, name);
    if (param)
        fParam = new char[strlen(param) + 1];
    else
        fParam = 0;
    if (fParam) strcpy(fParam, param);
    fCommand = command;
    fContext = context;
    fSubmenu = submenu;
    fEnabled = 1;
    fHotCharPos = aHotCharPos;
    fPixmap = 0;
    fChecked = 0;
    if (!fName || fHotCharPos >= int(strlen(name)) || fHotCharPos < -1)
        fHotCharPos = -1;
}

YMenu::YMenuItem::~YMenuItem() {
    delete fName;
    delete fParam;
    delete fSubmenu;
}

void YMenu::YMenuItem::setChecked(bool c) {
    fChecked = c;
}

void YMenu::YMenuItem::setPixmap(YMaskPixmap *pixmap) {
    fPixmap = pixmap;
}

YMenu::YMenuItem *YMenu::addItem(const char *name, int hotCharPos, const char *param, WMCommand command, void *context) {
    return add(new YMenuItem(name, hotCharPos, param, command, 0, context));
}

YMenu::YMenuItem * YMenu::addSubmenu(const char *name, int hotCharPos, YMenu *submenu) {
    return add(new YMenuItem(name, hotCharPos, 0, cmdSubmenu, submenu, 0));
}

YMenu::YMenuItem * YMenu::addSeparator() {
    return add(new YMenuItem());
}

void YMenu::removeAll() {
    if (fPopup) {
        fPopup->popdown();
        fPopup = 0;
    }
    for (int i = 0; i < itemCount(); i++)
        delete fItems[i];
    free(fItems);
    fItems = 0;
    fItemCount = 0;
    paintedItem = selectedItem = -1;
    fPopup = 0;
}

YMenu::YMenuItem * YMenu::add(YMenuItem *item) {
    if (item) {
        YMenuItem **newItems = (YMenuItem **)realloc(fItems, sizeof(YMenuItem *) * ((fItemCount) + 1));
        if (newItems) {
            fItems = newItems;
            fItems[fItemCount++] = item;
            return item;
        } else {
            delete item;
            return 0;
        }
    }
    return item;
}

YMenu::YMenuItem *YMenu::findItem(WMCommand cmd) {
    for (int i = 0; i < itemCount(); i++)
        if (cmd == item(i)->command())
            return item(i);
    return 0;
}
YMenu::YMenuItem *YMenu::findSubmenu(YMenu *sub) {
    for (int i = 0; i < itemCount(); i++)
        if (sub == item(i)->submenu())
            return item(i);
    return 0;
}

void YMenu::enableCommand(WMCommand cmd) {
    for (int i = 0; i < itemCount(); i++)
        if (cmd == cmdNone || cmd == item(i)->command())
            item(i)->setEnabled(true);
}

void YMenu::disableCommand(WMCommand cmd) {
    for (int i = 0; i < itemCount(); i++)
        if (cmd == cmdNone || cmd == item(i)->command())
            item(i)->setEnabled(false);
}

void YMenu::paint(Graphics &g, int /*_x*/, int /*_y*/, unsigned int /*_width*/, unsigned int /*_height*/) {
    g.setColor(menuBg);
    g.drawBorderW(0, 0, width() - 1, height() - 1, true);
    g.drawLine(1, 1, width() - 3, 1);
    g.drawLine(1, 2, width() - 3, 2);
    g.drawLine(1, 1, 1, height() - 3);
    g.drawLine(2, 1, 2, height() - 3);
    g.drawLine(1, height() - 3, width() - 3, height() - 3);
    g.drawLine(width() - 3, 1, width() - 3, height() - 3);

    int y = 2;
    for (int i = 0; i < itemCount(); i++)
        paintItem(g, i, y, 1);
    paintedItem = selectedItem;
}

void YMenu::paintItem(Graphics &g, int i, int &y, int paint) {
    int fontHeight = menuFont->height() + 1;
    int fontBaseLine = menuFont->ascent();
    YMenuItem *mitem = item(i);
    char *name = mitem->name();
    char *param = mitem->param();

    //if (wmLook == lookMotif) {
    //   topGC = brightMenuGC;
    //    bottomGC = darkMenuGC;
    //} else {
    //    bottomGC = brightMenuGC;
    //    topGC = darkMenuGC;
    //}
    g.setColor(menuBg);
    if (mitem->command() == cmdSeparator) {
        if (paint) {
            g.drawLine(1, y, width() - 3, y);
            g.setColor(menuBg->darker());
            g.drawLine(1, y + 1, width() - 3, y + 1);
            g.setColor(menuBg->brighter());
            g.drawLine(1, y + 2, width() - 3, y + 2);
            g.setColor(menuBg);
            g.drawLine(1, y + 3, width() - 3, y + 3);
        }
        y += 4;
    } else {
        unsigned int hh = fontHeight;
        
        if (item(i)->getPixmap() && item(i)->getPixmap()->height() > hh)
            hh = item(i)->getPixmap()->height();

        hh++;
        if (i == selectedItem)
            g.setColor(activeMenuItemBg);
        if (paint) {
            if (wmLook == lookWin95 || wmLook == lookWarp4) {
                g.fillRect(2, y, width() - 5, hh + 2);
            } else {
                g.fillRect(2, y, width() - 5, hh + 2);
                
                if (i == selectedItem) {
//#define NEWMOTIF
#ifdef NEWMOTIF
                    bool raised = false;
#else
                    bool raised = (wmLook != lookMotif) ? false : true;
#endif
                    g.setColor(menuBg);
                    g.draw3DRect(2, y, width() - 6, hh + 1, raised);

                    if (wmLook == lookMotif) {
                        g.draw3DRect(3, y + 1, width() - 6 - 2, hh - 1, raised);
                        /*XDrawLine(app->display(), handle(), topGC,
                                  4, y + 1, width() - 5, y + 1);
                        XDrawLine(app->display(), handle(), bottomGC,
                                  4, 0 + y + hh - 2, width() - 5, 0 + y + hh - 2);
                        XDrawLine(app->display(), handle(), topGC,
                                  4, y + 1, 4, 0 + y + hh - 2);
                        XDrawLine(app->display(), handle(), bottomGC,
                                  width() - 5, y + 1, width() - 5, 0 + y + hh - 2);*/
                    }
                }
            }

            YColor *fg;
            if (!mitem->isEnabled())
                fg = disabledMenuItemFg;
            else if (i == selectedItem)
                fg = activeMenuItemFg;
            else
                fg = menuItemFg;
            g.setColor(fg);
            g.setFont(menuFont);
            
            int delta = (i == selectedItem) ? 1 : 0;
            if (wmLook == lookMotif || wmLook == lookWarp4 || wmLook == lookWin95)
                delta = 0;

            if (mitem->isChecked()) {
                XPoint points[4];

                points[0].x = delta + (fontHeight - 10) / 2 + 3;
                points[1].x = 5;
                points[2].x = 5;
                points[3].x = -5;
                points[0].y = y + delta + (fontHeight - 10) / 2 + 4;
                points[1].y = -5;
                points[2].y = 5;
                points[3].y = 5;

                g.fillPolygon(points, 4, Convex, CoordModePrevious);
            } else {
                if (mitem->getPixmap())
                    g.drawMaskPixmap(mitem->getPixmap(),
                                     2 + delta + 1 + (wmLook == lookMotif ? 1 : 0),
                                     y + delta + 1 + (hh - mitem->getPixmap()->height()) / 2);
            }

            if (name) {
                if (!mitem->isEnabled()) {
                    g.setColor(white);
                    g.drawChars((char *)name, 0, strlen((char *)name),
                                1 + 3 + hh + 4 + delta,
                                1 + 1 + y + (hh - fontHeight) / 2 + fontBaseLine + delta);
                    
                    if (mitem->hotCharPos() != -1) {
                        g.drawCharUnderline(1 + 2 + hh + 4 + delta,
                                            1 + 1 + y + (hh - fontHeight) / 2 + fontBaseLine + delta,
                                            (char *)name, mitem->hotCharPos());
                    }
                }
                g.setColor(fg);
                g.drawChars((char *)name, 0, strlen((char *)name),
                            2 + hh + 4 + delta,
                            1 + y + (hh - fontHeight) / 2 + fontBaseLine + delta);

                if (mitem->hotCharPos() != -1) {
                    g.drawCharUnderline(3 + hh + 4 + delta,
                                        1 + y + (hh - fontHeight) / 2 + fontBaseLine + delta,
                                        (char *)name, mitem->hotCharPos());
                }
            }
            
            if (param) {
                if (!mitem->isEnabled()) {
                    g.setColor(white);
                    g.drawChars((char *)param, 0, strlen((char *)param),
                                1 + 2 + hh + 4 + 8 + paramPos + delta,
                                1 + 1 + y + (hh - fontHeight) / 2 + fontBaseLine + delta);
                }
                g.setColor(fg);
                g.drawChars((char *)param, 0, strlen((char *)param),
                            2 + hh + 4 + 8 + paramPos + delta,
                            1 + y + (hh - fontHeight) / 2 + fontBaseLine + delta);
            }
            else if (mitem->submenu() || mitem->command() == cmdSubmenu) {
                if (mitem->command() != cmdSubmenu) {
                    int active = (fPopup == mitem->submenu()) ? 1 : 0;
                    g.setColor(menuBg);
                    g.fillRect(width() - 3 - hh - 3 + 2,
                               1 + y,
                               hh - 1,
                               hh - 1);
                    g.drawBorderW(width() - 3 - hh - 3 + 1, 1 + y,
                                  hh - 1, hh - 1,
                                  active ? false : true);
                    delta = active ? 1 : 0;
                }

                XPoint points[3];

                points[0].x = width() - 3 - hh + 2 + delta + 2;
                points[2].x = - (hh / 2 - 2);
                points[1].x = hh / 2 - 2;
                points[0].y = 1 + y + delta + 1;
                points[1].y = hh / 2 - 2;
                points[2].y = hh / 2 - 2;

                g.setColor(fg);
                g.fillPolygon(points, 3, Convex, CoordModePrevious);

            }
        }
        y += hh + 2;
    }
}
