/*
 * IceWM
 *
 * Copyright (C) 1997 Marko Macek
 *
 * Window list
 */

#include "icewm.h"
YListItem::YListItem(YFrameWindow *frame) {
    fPrevItem = fNextItem = 0;
    fFrame = frame;
    fSelected = 0;
}

YListItem::~YListItem() {
    if (fFrame) {
        fFrame->setWinListItem(0);
        fFrame = 0;
    }
}

YListItem *YListItem::getNext() {
    return fNextItem;
}

YListItem *YListItem::getPrev() {
    return fPrevItem;
}

void YListItem::setNext(YListItem *next) {
    fNextItem = next;
}

void YListItem::setPrev(YListItem *prev) {
    fPrevItem = prev;
}

YFrameWindow *YListItem::frame() {
    return fFrame;
}

int YListItem::getSelected() {
    return fSelected;
}

void YListItem::setSelected(int aSelected) {
    fSelected = aSelected;
}

YListBox::YListBox(YScrollBar *vert, YWindow *aParent, Window win): YWindow(aParent, win) {
    fVerticalScroll = vert;
    if (vert)
        vert->setListener(this);
    fOffsetY = 0;
    fFirst = fLast = 0;
    fFocusedItem = 0;
    fSelectStart = fSelectEnd = -1;
    fSelecting = -1;
}

YListBox::~YListBox() {
    fFirst = fLast = 0;
}

bool YListBox::isFocusTraversable() {
    return true;
}

int YListBox::addItem(YListItem *item) {
    YFrameWindow *w = item->frame();

    if (w->owner() &&
        w->owner()->winListItem())
    {
        YListItem *i = w->owner()->winListItem();
        assert(w->owner() != w);

        item->setNext(i->getNext());
        if (item->getNext())
            item->getNext()->setPrev(item);
        else
            fLast = item;
        
        item->setPrev(i);
        i->setNext(item);
    } else {
        item->setNext(0);
        item->setPrev(fLast);
        if (fLast)
            fLast->setNext(item);
        else
            fFirst = item;
        fLast = item;
    }
    repaint();
    return 1;
}

void YListBox::removeItem(YListItem *item) {
    if (item->getPrev())
        item->getPrev()->setNext(item->getNext());
    else
        fFirst = item->getNext();

    if (item->getNext())
        item->getNext()->setPrev(item->getPrev());
    else
        fLast = item->getPrev();
    item->setPrev(0);
    item->setNext(0);
    repaint();
}

YListItem *YListBox::findItem(int mouseY, int &no) {
    int y = 0;
    int lh, fh = windowListFont->height();
    YListItem *a = fFirst;
    
    no = 0;
    if (ICON_SMALL > fh)
        lh = ICON_SMALL;
    else
        lh = fh;
    lh += 2;
    while (a) {
        if (mouseY >= y - fOffsetY && mouseY < y - fOffsetY + lh)
            return a;
        y += lh;
        a = a->getNext();
        no++;
    }
    no = -1;
    return 0;
}

void YListBox::selectItem(YListItem *i) {
    if (i == 0) {
        fFocusedItem = 0;
        fSelectStart = 0;
        fSelectEnd = 0;
    } else {
        int n = 0;
        YListItem *f = fFirst;
        for(; f ; f = f->getNext(), n++)
            if (i == f) {
                fFocusedItem = n;
                fSelectStart = fSelectEnd = n;
            }
    }
    setSelection();
    repaint();
}

void YListBox::updateSelection(int apply) {
    if (fSelectStart != -1) {
        assert(fSelectEnd != -1);
        
        int beg = (fSelectStart < fSelectEnd) ? fSelectStart : fSelectEnd;
        int end = (fSelectStart < fSelectEnd) ? fSelectEnd : fSelectStart;

        YListItem *i;
        int n;

        for (i = fFirst, n = 0; i; i = i->getNext(), n++) {
            int s = i->getSelected();
            int t = (n >= beg) && (n <= end);

            if (fSelecting) {
                s &= 1;
                if (t)
                    s |= apply ? 1 : 2;
            } else {
                s &= 1;
                if (t) {
                    if (s & 1)
                        if (apply)
                            s = 0;
                        else
                            s |= 2;
                }
            }
            if (apply)
                s &= 1;
            i->setSelected(s);
        }
    }
}

void YListBox::setSelection() {
    if (fSelectStart != -1) {
        assert(fSelectEnd != -1);
        
        int beg = (fSelectStart < fSelectEnd) ? fSelectStart : fSelectEnd;
        int end = (fSelectStart < fSelectEnd) ? fSelectEnd : fSelectStart;

        YListItem *i;
        int n;

        for (i = fFirst, n = 0; i; i = i->getNext(), n++) {
            int t = ((n >= beg) && (n <= end)) ? 1 : 0;
            i->setSelected(t);
        }
    }
}

int YListBox::itemCount() {
    YListItem *i;
    int n;
    
    for (i = fFirst, n = 0; i; i = i->getNext(), n++)
        ;
    return n;
}

YListItem *YListBox::item(int no) {
    YListItem *a = fFirst;
    int n;
    
    for (n = 0; a; a = a->getNext(), n++)
        if (n == no)
            return a;
    return 0;
}

int YListBox::getLineHeight() {
    int lh, fh = windowListFont->height();
    if (ICON_SMALL > fh)
        lh = ICON_SMALL;
    else
        lh = fh;
    lh += 2;
    return lh;
}

void YListBox::ensureVisibility(int item) {
    if (item >= 0) {
        int lh = getLineHeight();
        int fy = item * lh;
        
        if (fy + lh >= fOffsetY + (int)height())
            fOffsetY = fy + lh - height();
        if (fy <= fOffsetY)
            fOffsetY = fy;
    }
}

void YListBox::focusVisible() {
    int ty = fOffsetY;
    int by = ty + height();

    if (fFocusedItem >= 0) {
        int lh = getLineHeight();

        while (ty > fFocusedItem * lh)
            fFocusedItem++;
        while (by - lh < fFocusedItem * lh)
            fFocusedItem--;
    }
}

bool YListBox::handleKey(const XKeyEvent &key) {
    if (key.type == KeyPress) {
        int k = XKeycodeToKeysym(app->display(), key.keycode, 0);
        int m = KEY_MODMASK(key.state);

        int SelPos, OldPos = fFocusedItem, count = itemCount();

        if (m & ShiftMask) {
            SelPos = fFocusedItem;
        } else {
            SelPos = -1;
        }
        
        switch (k) {
        case XK_Escape:
            windowList->frame()->wmHide();
            return true;
        case XK_Return:
            {
                YListItem *i = item(fFocusedItem);
                YFrameWindow *f = 0;
                if (i)
                    f = i->frame();
                if (f) {
                    f->activate();
                    windowList->frame()->wmHide();
                }
            }
            break;
        case ' ':
            {
                YListItem *i = item(fFocusedItem);
                if (i) {
                    i->setSelected(i->getSelected() ? 0 : 1);
                    repaint();
                }
            }
            break;
        case XK_Up:
            if (fFocusedItem > 0)
                fFocusedItem--;
            break;
        case XK_Down:
            if (fFocusedItem < count - 1)
                fFocusedItem++;
            break;
        case XK_Prior:
            fVerticalScroll->setValue(fVerticalScroll->getValue() -
                                      fVerticalScroll->getBlockIncrement());
            fOffsetY = fVerticalScroll->getValue();
            fFocusedItem -= height() / getLineHeight();
            if (fFocusedItem < 0)
                if (count > 0)
                    fFocusedItem = 0;
                else
                    fFocusedItem = -1;
            repaint();
            break;
        case XK_Next:
            fVerticalScroll->setValue(fVerticalScroll->getValue() +
                                      fVerticalScroll->getBlockIncrement());
            fOffsetY = fVerticalScroll->getValue();
            fFocusedItem += height() / getLineHeight();
            if (fFocusedItem > count - 1)
                fFocusedItem = count - 1;
            repaint();
            break;
        case XK_Home:
            fFocusedItem = 0;
            break;
        case XK_End:
            fFocusedItem = count - 1;
            break;
        case XK_F10:
        case XK_Menu:
            if (k != XK_F10 || m == ShiftMask) {
                windowListPopup->popup(0, this,
                                       key.x_root, key.y_root, -1, -1,
                                       YPopupWindow::pfCanFlipVertical |
                                       YPopupWindow::pfCanFlipHorizontal |
                                       YPopupWindow::pfPopupMenu);
            }
            break;
        case XK_Delete:
            {
                handleCommand(cmdClose, 0, key.state);
            }
            break;
        default:
            if (k < 256) {
                unsigned char c = TOUPPER(k);
                int count = itemCount();
                int i = fFocusedItem;
                YListItem *it = 0;
                unsigned char *title;

                for (int n = 0; n < count; n++) {
                    i = (i + 1) % count;
                    it = item(i);
                    title = it->frame()->client()->windowTitle();
                    if (title && TOUPPER(title[0]) == c) {
                        fFocusedItem = i;
                        break;
                    }
                }
            } else
                return YWindow::handleKey(key);
        }
        if (fFocusedItem != OldPos) {
            if (SelPos == -1) {
                fSelectStart = fFocusedItem;
                fSelectEnd = fFocusedItem;
                fSelecting = 1;
            } else {
                if (fSelectStart == OldPos)
                    fSelectStart = fFocusedItem;
                else if (fSelectEnd == OldPos)
                    fSelectEnd = fFocusedItem;
                else
                    fSelectStart = fSelectEnd = fFocusedItem;
                if (fSelectStart == -1)
                    fSelectStart = fSelectEnd;
                if (fSelectEnd == -1)
                    fSelectEnd = fSelectStart;

                fSelecting = 1;
            }
            setSelection();
            ensureVisibility(fFocusedItem);
            repaint();
        }
        return true;
    }
    return YWindow::handleKey(key);
}

void YListBox::handleButton(const XButtonEvent &button) {
    if (button.button == 1) {
        int no;
        YListItem *i = findItem(button.y, no);
        fFocusedItem = no;
        
        if (button.type == ButtonPress) {
            if (!(button.state & ControlMask))
                for (YListItem *j = fFirst; j; j = j->getNext())
                    j->setSelected(0);
            if (i) {
                fSelectStart = no;
                fSelectEnd = no;
                fFocusedItem = no;
                fSelecting = i->getSelected() ? 0 : 1;
                updateSelection(0);
                ensureVisibility(fFocusedItem);
                repaint();
            } else {
                fSelectStart = -1;
                fSelectEnd = -1;
                fSelecting = 1;
                repaint();
            }
        } else if (button.type == ButtonRelease) {
            if (no != -1 && no != fSelectEnd) {
                fSelectEnd = no;
                updateSelection(0);
            }
            updateSelection(1);
            fSelecting = -1;
            ensureVisibility(fFocusedItem);
            repaint();
        }
    }
    YWindow::handleButton(button);
}

void YListBox::handleClick(const XButtonEvent &/*down*/, const XButtonEvent &up, int count) {
    if (up.button == 3 && count == 1) {
        int no;
        YListItem *i = findItem(up.y, no);

        if (i) {
            if (i->getSelected() != 1) {
                fSelectStart = no;
                fSelectEnd = no;
                fSelecting = -1;
                fFocusedItem = no;
                setSelection();
            } else {
                fFocusedItem = -1;
            }
            ensureVisibility(fFocusedItem);
            repaint();
            windowListPopup->popup(0, this,
                                   up.x_root, up.y_root, -1, -1,
                                   YPopupWindow::pfCanFlipVertical |
                                   YPopupWindow::pfCanFlipHorizontal |
                                   YPopupWindow::pfPopupMenu);
        }
    } else if (up.button == 1 && count == 2) {
        int no;
        YListItem *i = findItem(up.y, no);
        YFrameWindow *f = 0;
        if (i)
            f = i->frame();
        if (f) {
            f->activate();
            windowList->frame()->wmHide();
        }
    }
}

void YListBox::handleMotion(const XMotionEvent &motion) {
    if (motion.state & Button1Mask) {
        int no;

        findItem(motion.y, no);

        fFocusedItem = no;
        if (no != -1 && fSelectEnd != no) {
            if (fSelectStart == -1)
                fSelectStart = no;
            fSelectEnd = no;
            updateSelection(0);
            ensureVisibility(fFocusedItem);
            repaint();
        }
    }
    YWindow::handleMotion(motion);
}

void YListBox::handleDrag(const XButtonEvent &down, const XMotionEvent &motion) {
    if (down.button == 2) {
        int dx = motion.y - down.y;

        fVerticalScroll->setValue(fVerticalScroll->getValue() - dx);
        fOffsetY = fVerticalScroll->getValue();
        repaint();
    }
}

void YListBox::scroll(YScrollBar *scroll, int delta) {
    if (scroll == fVerticalScroll) {
        fOffsetY += delta;
        focusVisible();
        repaint();
    }
}

void YListBox::move(YScrollBar *scroll, int pos) {
    if (scroll == fVerticalScroll) {
        fOffsetY = pos;
        focusVisible();
        repaint();
    }
}

void YListBox::paint(Graphics &g, int /*x*/, int /*y*/, unsigned int /*width*/, unsigned int /*height*/) {
    int y = 0, x = 3;
    int lh = getLineHeight();
    int fh = windowListFont->height();
    YListItem *a = fFirst;
    int n = 0;

    g.setColor(listBoxBg);
    g.fillRect(0, 0, width(), height());

    while (a) {
        if (a->frame()) {
            int xpos = 0;
            unsigned char *title = a->frame()->client()->windowTitle();
            int yPos = y + lh - (lh - fh) / 2 - windowListFont->descent();
            
            YFrameWindow *w = a->frame();
            
            while (w->owner()) {
                xpos += 20;
                w = w->owner();
            }
            
            int s = a->getSelected();
            if (fSelecting == -1)
                s = (s == 1) ? 1 : 0;
            if (fSelecting == 0 && (s & 2))
                s = 0;

            if (s)
                g.setColor(listBoxSelBg);
            else
                g.setColor(listBoxBg);
            g.fillRect(0, y - fOffsetY, width(), lh);
            if (fFocusedItem == n) {
                g.setColor(black);
                g.drawRect(0, y - fOffsetY, width() - 1, lh - 1);
            }
            if (a->frame()->clientIcon() && a->frame()->clientIcon()->small())
                g.drawMaskPixmap(a->frame()->clientIcon()->small(),
                                 xpos + x, y - fOffsetY + 1);
            if (s)
                g.setColor(listBoxSelFg);
            else
                g.setColor(listBoxFg);
            g.setFont(windowListFont);
            if (title) {
                g.drawChars((char *)title, 0, strlen((char *)title),
                            xpos + x + 20, yPos - fOffsetY);
            }
        }
        y += lh;
        a = a->getNext();
        n++;
    }
    fVerticalScroll->setValues(fOffsetY, height(), 0, y);
    fVerticalScroll->setBlockIncrement(height());
    fVerticalScroll->setUnitIncrement(fh);
}

void YListBox::setPopupActive(YPopupWindow */*popup*/) {
}

void YListBox::handleCommand(WMCommand command, void *context, unsigned int modifiers) {
    YListItem *i;

    switch (command) {
    case cmdCascade:
    case cmdTileVertical:
    case cmdTileHorizontal:
        break;
    default:
        if (fSelectStart != -1 && fSelectEnd != -1) {
            for (i = fFirst; i; i = i->getNext()) {
                if (i->getSelected()) {
                    if (command == cmdHide)
                        if (i->frame()->hidden())
                            continue;
                    i->frame()->handleCommand(command, context, modifiers);
                }
            }
        }
    }
}

WindowList::WindowList(YWindow *aParent, Window win): YFrameClient(aParent, 0, win) {
    scrollVert = new YScrollBar(YScrollBar::Vertical, this);
    scrollVert->show();
    list = new YListBox(scrollVert, this);
    list->show();

    int w = app->root()->width();
    int h = app->root()->height();
    
    setGeometry(3 * w / 8, 2 * h / 6, w / 4, h / 3);

    windowList = this;
    setWindowTitle((unsigned char *)"Window list");
    setIconTitle((unsigned char *)"Window list");
}

WindowList::~WindowList() {
    delete list; list = 0;
    windowList = 0;
}

void WindowList::handleFocus(const XFocusChangeEvent &focus) {
    if (focus.type == FocusIn) {
        list->setFocus();
    } else if (focus.type == FocusOut) {
    }
}

void WindowList::relayout() {
    list->repaint();
}

YListItem *WindowList::addWindowListApp(YFrameWindow *frame) {
    if (frame->client() == windowList)
        return 0;
    YListItem *item = new YListItem(frame);
    if (item)
        list->addItem(item);
    return item;
}

void WindowList::removeWindowListApp(YListItem *item) {
    if (item) {
        list->removeItem(item);
        delete item;
    }
}

void WindowList::configure(int x, int y, unsigned int width, unsigned int height) {
    YFrameClient::configure(x, y, width, height);

    scrollVert->setGeometry(width - SCROLLBAR_SIZE, 0,
                            SCROLLBAR_SIZE, height);
    list->setGeometry(0, 0, width - SCROLLBAR_SIZE, height);
}

void WindowList::handleClose() {
    frame()->wmHide();
}

void WindowList::showFocused() {
    YFrameWindow *f = app->focus();

    if (f != frame()) {
        if (f)
            list->selectItem(f->winListItem());
        else
            list->selectItem(0);
    }
    setAllWorkspaces(1);
    setWorkspaces(ALLWORKSPACES);
    show();
    raise();
}
