/*****************************************************************************/
/*                                                                           */
/*  wmmail - WindowMaker Mail                                                */
/*                                                                           */
/*  Copyright (c) 1996, 1997, 1998  Per Liden                                */
/*  Copyright (c) 1997, 1998  Bryan Chan                                     */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                */
/*                                                                           */
/*    mail.c: functions to handle UNIX-style mailboxes                       */
/*                                                                           */
/*****************************************************************************/

#include "wmmail.h"
#include <time.h>

/* 
 * This module contains routines to check an UNIX-style mailbox in various
 * ways during the periodic update of wmmail_status. We can classify a
 * given mailbox as being in one of the states listed below.
 *
 * The states and their corresponding effects on wmmail_status are:
 *
 *   - no mailbox or empty mailbox              NO_MAIL
 *   - no new mails                             OLD_MAIL
 *   - same number of new mails or fewer        no change
 *   - more new mails than last time            NEW_MAIL
 *
 * or, if TimeStampMode is used:
 *
 *   - no mailbox or empty mailbox              NO_MAIL
 *   - smaller than last time but non-zero      OLD_MAIL
 *   - read after most recent write             OLD_MAIL
 *   - no read after most recent write and
 *     - same size as last time                 no change
 *     - bigger than last time                  NEW_MAIL
 *
 * This module assumes that overriding automatically changes wmmail_status
 * to OLD_MAIL.
 *
 */

/* internal functions */
static void check_time_stamp(Mailbox *, int *, int *, int *);
static void get_new_count(Mailbox *, int *, int *, int *);
static int  count_mail(Mailbox *);


void check_unix_mbox(mbox, should_beep, should_redraw, should_run)
  Mailbox *mbox;
  int     *should_beep,
          *should_redraw,
          *should_run;
{
  if (use_time_stamp)
  {
#ifdef DEBUG
    struct stat __debug;
#endif

    check_time_stamp(mbox, should_beep, should_redraw, should_run);

#ifdef DEBUG
    if (!stat(mbox->path, &__debug))
      printf("%s\nsize:  %d\nmtime: %d\natime: %d\n\n",
             mbox->path, mbox->size, __debug.st_mtime, __debug.st_atime);
    else
      printf("cannot stat %s\n\n", mbox->path);
#endif
  }
  else
  {
    get_new_count(mbox, should_beep, should_redraw, should_run);

#ifdef DEBUG
    printf("%s\ntotal:  %d\nnew:    %d\nstatus: %d\n\n", 
           mbox->path, mbox->mail_count, mbox->new_mail_count, mbox->status);
#endif
  }
}


static void check_time_stamp(mbox, should_beep, should_redraw, should_run)
  Mailbox  *mbox;
  int      *should_beep,
           *should_redraw,
           *should_run;
{
  struct stat t;
  int    prev_status;

  prev_status = mbox->status;

  if (!stat(mbox->path, &t))
  {
    if (t.st_size == 0)
    {
      mbox->status = NO_MAIL;
    }
    else if (t.st_size < mbox->size)
    {
      /* mailbox smaller in size; some mails have been deleted */
      mbox->status = OLD_MAIL;
    }
    else if (t.st_atime > t.st_mtime)
    { 
      /* mailbox read after most recent write */
      mbox->status = OLD_MAIL;
    }
    else if (t.st_size > mbox->size)
    {
      /* mailbox modified after most recent read, and larger in size */
      /* this implies the arrival of some new mails */

      if (mbox->status == NEW_MAIL && always_new_mail_exec)
        *should_run |= True;
      else if (mbox->status == NEW_MAIL)
        *should_run |= False;
      else
        *should_run |= True;

      *should_beep = True;

      mbox->status = NEW_MAIL;
    }

    /* else no change */

    mbox->size = t.st_size;                     /* record size of mbox */
  }
  else
    mbox->status = NO_MAIL;                         /* no such mailbox */

  *should_redraw |= (prev_status != mbox->status);
}


static void get_new_count(mbox, should_beep, should_redraw, should_run)
  Mailbox *mbox;
  int     *should_beep,
          *should_redraw,
          *should_run;
{
  int prev_status,
      prev_new_mail_count;
  
  prev_status = mbox->status;
  prev_new_mail_count = mbox->new_mail_count;
        
  if (count_mail(mbox))
  {
    if (mbox->mail_count == 0)
    {                               /* there is no mail in the mailbox */
      mbox->status = NO_MAIL;
    }
    else if (mbox->new_mail_count == 0)
    {                                        /* there are no new mails */
      mbox->status = OLD_MAIL;
    }
    else if (mbox->new_mail_count > prev_new_mail_count)
    {                                       /* new mails have arrived! */
      if (mbox->status == NEW_MAIL && always_new_mail_exec)
        *should_run |= True;
      else if (mbox->status == NEW_MAIL)
        *should_run |= False;
      else
        *should_run |= True;

      *should_beep = True;

      mbox->status = NEW_MAIL;
    }

    /* else no change */
  }
  else
    mbox->status = NO_MAIL;                         /* no such mailbox */

  *should_redraw |= (prev_status != mbox->status);
}


static int count_mail(Mailbox *mbox)
{
  FILE        *file = NULL;
  struct stat  t;
  char         buf[13];
  int          end_of_header = False,
               is_old        = False;
  int          m = 0,                     /* mail_count */
               n = 0;                     /* new_mail_count */

  /* only count the mails in the mailbox if it was modified after last check */
  if (!stat(mbox->path, &t) && t.st_ctime == mbox->last_check)
    return True;

  mbox->last_check = t.st_ctime;

  if ((file = fopen(mbox->path, "r")) == NULL)
    return False;

  while (fgets(buf, 12, file) != NULL)
  {
    if (!strncmp(buf, "From ", 5))
    {
      end_of_header = False;
      m++; 

      /* do not increment n if the _previous_ mail was already read */
      /* always assume the first mail has not been read */
      if (is_old)
        is_old = False;
      else
        n++;
    }
    else if (!strcmp(buf, "\n"))
    {
      /* the first blank line marks the end of headers */
      end_of_header = True;
    }
    else if (!strncmp(buf, "Status: RO", 10) || 
             !strncmp(buf, "Status: OR", 10))
    {
      /* set is_old to True only if we are still reading the headers */
      is_old |= !end_of_header;
    }

    if (buf[strlen(buf) - 1] != '\n')
    {
      /* skip everything until the next newline */
      int i;

      for (i = '\0'; i != '\n' && i != EOF; i = fgetc(file));
    }
  }

  /* must correct the obiwan error by decrementing n if the last mail is old */
  if (is_old)
    n--;

  fclose(file);

  mbox->mail_count = m;
  mbox->new_mail_count = n;

  return True;
}
