/*****************************************************************************/
/*                                                                           */
/*  wmmail - WindowMaker Mail                                                */
/*                                                                           */
/*  Copyright (c) 1996, 1997, 1998  Per Liden                                */
/*  Copyright (c) 1997, 1998  Bryan Chan                                     */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                */
/*                                                                           */
/*    wmmail.c: essential routines                                           */
/*                                                                           */
/*****************************************************************************/

#include "wmmail.h"
#include "mail.h"
#include "mh.h"
#include "std_icons.h"


extern FILE *yyin;


/* function prototypes */
int   main(int, char **);
void  help(void);
void  initialize(int, char **);
void  create_appicon(void);
void  redraw_appicon(void);
void  animate(XtPointer, XtIntervalId *);
Pixel get_pixel_by_color(char *);
void  update_status(XtPointer, XtIntervalId *);
void  handle_expose(Widget, XtPointer, XEvent *);
void  handle_button(Widget, XtPointer, XEvent *);
  
int main(int argc, char **argv)
{
  /* handle command line options, open the configuration file for reading */
  initialize(argc, argv);

  create_appicon();

  update_status(NULL, NULL);
  animate(NULL, NULL);

  XtAppMainLoop(wmmail_context);
}


void help(void)
{
  fprintf(stderr, 

    "\n"
    "%s version %s, by Bryan Chan (bryan.chan@utoronto.ca)\n"
    "based on asmail version 0.50, by Per Liden (per@rsn.hk-r.se)\n"
    "\n"
    "usage: \t%s [options ... ] [<filename>]\n"
    "\n"
    "<filename>:\tconfiguration file to use (%s by default)\n"
    "\n"
    "options:\t-help, -h           display this message\n"
    "        \t-swallowed          show window as well as appicon;\n"
    "        \t                    good for swallowing with AfterStep\n"
    "\n",
     
    app_name, VERSION, app_name, DEFAULT_CONFIG_FILE);

  exit(1);
}


void initialize(int argc, char **argv)
{
  /* determine how this program was invoked */
  if (rindex(argv[0], '/') != NULL)
    app_name = (char *) rindex(argv[0], '/') + 1;
  else
    app_name = argv[0];

  wmmail_widget = XtAppInitialize(&wmmail_context, WMMAIL_CLASS, 
                                  (XrmOptionDescRec *) NULL, 0, &argc, argv,
                                  (String *) NULL, (ArgList) NULL, 0);

  disp = XtDisplay(wmmail_widget);

  root = RootWindow(disp, DefaultScreen(disp));

  wmmail_font = XLoadFont(disp, DEFAULT_FONT);

  if (argc > 1)
  {
    int i;

    for (i = 1; i < argc; i++)
      if (!strcmp(argv[i], "-help") || !strcmp(argv[i], "-h"))
      {
        help();
        exit(OK);
      }
      else if (!strcmp(argv[i], "-swallowed"))
      {
        use_appicon_only = False;
      }
      else if (argv[i][0] == '-')
      {
        croak("no such option '%s'", argv[i]);
        exit(ERR_CALL);
      }
      else
      {
        config_file_path = strdup(argv[i]);
      }
  }

  if (config_file_path == NULL)
  {
    config_file_path = safe_malloc(strlen(getenv("HOME")) + 
                                   strlen(DEFAULT_CONFIG_FILE) + 2);

    strcpy(config_file_path, getenv("HOME"));
    strcat(config_file_path, "/");
    strcat(config_file_path, DEFAULT_CONFIG_FILE);
  }  

  /* open configuration file */
  if ((yyin = fopen(config_file_path, "r")) == (FILE *) NULL)
  {
    croak("cannot open file %s", config_file_path);
    exit(ERR_IO);
  }

  /* parse configuration file, then either proceed or quit */
  if (yyparse())
  {
    croak("failed to parse configuration file");
    exit(ERR_SYNTAX);
  }

  /* if no mailbox is specified in the configuration file, use $MAIL */
  if (mbox_list == NULL)  
  {
    if (getenv("MAIL") != NULL)
    {
      struct stat t;

      mbox_list = (Mailbox *) safe_malloc(sizeof(Mailbox));

      mbox_list->path           = getenv("MAIL");
      mbox_list->last_check     = 0;
      mbox_list->mail_count     = 0;
      mbox_list->new_mail_count = 0;
      mbox_list->status         = NO_MAIL;
      mbox_list->next           = NULL;

      if (!stat(mbox_list->path, &t))
      {
        if (S_ISDIR(t.st_mode))
          mbox_list->mbox_type  = TYPE_MH;
        else
          mbox_list->mbox_type  = TYPE_UNIX;

        mbox_list->size         = t.st_size;
      }
      else
      {
        croak("cannot stat default mailbox %s", mbox_list->path);
        exit(ERR_IO);
      }
    }
    else
    {
      croak("no mailbox specified");
      exit(ERR_CALL);
    }
  }
}


void create_appicon(void)
{
  Arg            args[4];
  XGCValues      gc_values;
  unsigned long  gc_mask;

  load_std_icons();

  wmmail_appicon = XtCreateWidget("appicon", topLevelShellWidgetClass,
                                  wmmail_widget, NULL, 0);

  XtAddEventHandler(wmmail_appicon, ExposureMask, False, 
                    (XtEventHandler) handle_expose, NULL);

  XtAddEventHandler(wmmail_appicon, ButtonPressMask, False,
                    (XtEventHandler) handle_button, NULL);

  XtResizeWidget(wmmail_appicon, animations[wmmail_status]->attributes.width, 
                 animations[wmmail_status]->attributes.height, 1);

  XtRealizeWidget(wmmail_appicon);

  XtSetArg(args[0], XtNiconName, "wmmail");
  XtSetArg(args[1], XtNiconWindow, XtWindow(wmmail_appicon));

  if (!use_appicon_only)
    XtSetArg(args[2], XtNinitialState, NormalState);
  else
    XtSetArg(args[2], XtNinitialState, WithdrawnState);
  
  XtSetValues(wmmail_widget, args, 3); 

  if (!use_appicon_only)
  {
    XtAddEventHandler(wmmail_widget, ExposureMask, False, 
                      (XtEventHandler) handle_expose, NULL);

    XtAddEventHandler(wmmail_widget, ButtonPressMask, False,
                      (XtEventHandler) handle_button, NULL);
  }

  XtResizeWidget(wmmail_widget, animations[wmmail_status]->attributes.width, 
                 animations[wmmail_status]->attributes.height, 1);

  XtRealizeWidget(wmmail_widget);

  XtMapWidget(wmmail_widget);

  shape_mask = XCreatePixmap(disp, XtWindow(wmmail_widget),
                               animations[wmmail_status]->attributes.width,
                               animations[wmmail_status]->attributes.height, 1);

  shape_mask_gc = XCreateGC(disp, shape_mask, 0, &gc_values);
  XSetGraphicsExposures(disp, shape_mask_gc, False);
  XSetBackground(disp, shape_mask_gc, 0);
  XSetForeground(disp, shape_mask_gc, 1);

  wmmail_gc = XCreateGC(disp, root, 0, &gc_values);
  XSetGraphicsExposures(disp, wmmail_gc, False);
  XSetBackground(disp, wmmail_gc, get_pixel_by_color("black"));
  XSetForeground(disp, wmmail_gc, get_pixel_by_color(wmmail_color));

  redraw_appicon();
}


void redraw_appicon(void)
{
  char          buf[30];
  XTextItem     num_of_msg;
  XGCValues     gc_values;

  /* create shape mask from the mask of the XPM animation frame */
  XFillRectangle(disp, shape_mask, shape_mask_gc, 0, 0,
                 animations[wmmail_status]->attributes.width,
                 animations[wmmail_status]->attributes.height);
                 
  XCopyArea(disp, animations[wmmail_status]->mask, shape_mask, 
            shape_mask_gc, 0, 0, 
            animations[wmmail_status]->attributes.width,
            animations[wmmail_status]->attributes.height,
            0, 0);

  /* generate NumOfMsg if NumOfMsgMode is used */
  if (num_of_msg_mode != SHOW_NONE)
  {
    switch(num_of_msg_mode)
    {
      case SHOW_NEW_OVER_TOTAL:
        sprintf(buf, "%d/%d", mbox_list->new_mail_count, mbox_list->mail_count);
        break;
      case SHOW_NEW_ONLY:
        sprintf(buf, "%d", mbox_list->new_mail_count);
        break;
      case SHOW_TOTAL_ONLY:
        sprintf(buf, "%d", mbox_list->mail_count);
        break;
    }

    num_of_msg.chars  = buf;
    num_of_msg.nchars = strlen(buf);            
    num_of_msg.delta  = 1;       
    num_of_msg.font   = wmmail_font;

    /* add the text to the shape mask */
    XDrawText(disp, shape_mask, shape_mask_gc,
              num_of_msg_x, num_of_msg_y, &num_of_msg, 1);

    XDrawText(disp, shape_mask, shape_mask_gc,
              num_of_msg_x, num_of_msg_y, &num_of_msg, 1);
  }
   
  /* apply the shape mask */
  XShapeCombineMask(disp, XtWindow(wmmail_appicon), ShapeBounding, 0, 0,
                    shape_mask, ShapeSet);

  if (!use_appicon_only)
    XShapeCombineMask(XtDisplay(wmmail_widget), XtWindow(wmmail_widget), ShapeBounding, 0, 0,
                      shape_mask, ShapeSet);

  /* draw XPM animation frame */
  XCopyArea(disp, animations[wmmail_status]->pixmap, XtWindow(wmmail_appicon),
            wmmail_gc, 0, 0, animations[wmmail_status]->attributes.width,
            animations[wmmail_status]->attributes.height, 0, 0);            

  if (!use_appicon_only)
    XCopyArea(XtDisplay(wmmail_widget), animations[wmmail_status]->pixmap, XtWindow(wmmail_widget),
              wmmail_gc, 0, 0, animations[wmmail_status]->attributes.width,
              animations[wmmail_status]->attributes.height, 0, 0);            

  /* draw NumOfMsg */
  if (num_of_msg_mode != SHOW_NONE)
  {
    XDrawText(disp, XtWindow(wmmail_appicon), wmmail_gc, 
              num_of_msg_x, num_of_msg_y, &num_of_msg, 1);

    if (!use_appicon_only)
      XDrawText(disp, XtWindow(wmmail_widget), wmmail_gc,
                num_of_msg_x, num_of_msg_y, &num_of_msg, 1);
  }

  XFlush(disp);
}


void animate(XtPointer XtP, XtIntervalId *XtI)
{
  animations[wmmail_status]  = animations[wmmail_status]->next;

  redraw_appicon();

  XtAppAddTimeOut(wmmail_context, anim_speed[wmmail_status], animate, NULL);
}


Pixel get_pixel_by_color(char *color_name)
{
  XColor            color;
  XWindowAttributes attributes;

  XGetWindowAttributes(disp, root, &attributes);
  color.pixel = 0;

  if (!XParseColor(disp, attributes.colormap, color_name, &color))
    croak("cannot parse color '%s', using black", color_name);
  else if (!XAllocColor(disp, attributes.colormap, &color))
    croak("cannot allocate color '%s', using black", color_name);

  return color.pixel;
}


void update_status(XtPointer XtP, XtIntervalId * XtI)
{
  Mailbox    *i;
  static int update_in_progress = False;
  int        new_status         = NO_MAIL;
  int        should_beep        = False,
             should_redraw      = False,
             should_run         = False;

  if (exec_on_update_path != NULL)
    system(exec_on_update_path);

  if (!update_in_progress)
  {
    update_in_progress = True;

    for (i = mbox_list; i != NULL; i = i->next)
    {
      if (IS_UNIX(i))
        check_unix_mbox(i, &should_beep, &should_redraw, &should_run);
      else if (IS_MH(i))
	check_mh_folder(i, &should_beep, &should_redraw, &should_run);
      else if (IS_POP3(i))
        check_pop3(i, &should_beep, &should_redraw, &should_run);
      else
        /* ignore anything that we cannot handle */
        continue;
  
      if (i->status > new_status)
        new_status = i->status;
    }

    wmmail_status = new_status;

    update_in_progress = False;
  }

  if (should_run && exec_on_new_path != NULL)
    system(exec_on_new_path);

  if (should_beep && use_beep)
    XBell(disp, 0);

  if (should_redraw)
    redraw_appicon();

  XtAppAddTimeOut(wmmail_context, update_interval * 1000, update_status, NULL);
}


void handle_expose(Widget w, XtPointer p, XEvent *e)
{
  if (e->xexpose.count == 0)
    redraw_appicon();
}


void handle_button(Widget w, XtPointer p, XEvent *e)
{
  static Time last_left_mouse_click   = -1;
  static Time last_middle_mouse_click = -1;

  if (e->xbutton.button == LEFT_BUTTON && exec_on_click_path != NULL)
  {
    if (last_left_mouse_click > 0 &&
        e->xbutton.time - last_left_mouse_click <= DOUBLE_CLICK_TIME)

    {
      last_left_mouse_click = -1;
      system(exec_on_click_path);
    }

    last_left_mouse_click = e->xbutton.time;
  }
  else if (e->xbutton.button == MIDDLE_BUTTON)
  {
    if (last_middle_mouse_click > 0 && 
        e->xbutton.time - last_middle_mouse_click <= DOUBLE_CLICK_TIME)
    {
      /* double-clicking with the middle button overrides mailbox status */

      Mailbox *i;
      int     new_status = NO_MAIL;

      for (i = mbox_list; i != NULL; i = i->next)
      {
        if (i->status == NEW_MAIL)
          i->status = OLD_MAIL;

        if (i->status > new_status)
          new_status = i->status;
      }

      if (wmmail_status != new_status)
      {
        wmmail_status = new_status;
        redraw_appicon();
      }
      else
        wmmail_status = new_status;
    }

    last_middle_mouse_click = e->xbutton.time;
  }
}
