/* option utilities library
   Copyright 1998 Gene McCulley <mcculley@cuspy.com>, Cuspy Solutions, Inc.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include "config.h"
#include "optutil.h"

XtResource *OU_GetResources(const ResourceOption *ros, unsigned num_ros)
{
  unsigned k;

  XtResource *resources = (XtResource *)XtMalloc(sizeof(XtResource) * num_ros);
  for (k = 0; k < num_ros; k++)
    resources[k] = ros[k].resource;

  return resources;
}

XrmOptionDescRec *OU_GetOptions(const ResourceOption *ros, unsigned num_ros)
{
  unsigned k;

  XrmOptionDescRec *options =
    (XrmOptionDescRec *)XtMalloc(sizeof(XrmOptionDescRec) * num_ros);
  for (k = 0; k < num_ros; k++)
    options[k] = ros[k].option;

  return options;
}

void OU_DumpOptions(const ResourceOption *ros, unsigned num_ros, FILE *fp)
{
  unsigned j;
  unsigned biggest_option = 0;
  unsigned biggest_description = 0;
  const unsigned spacing = 2; /* Spacing between categories in help output. */
  const char format_template[] = "%%-%us%%-%us";
  char format[sizeof(format_template)]; /* Here we are assuming that the result
                                           is never bigger than the template. */

  for (j = 0; j < num_ros; j++) {
    unsigned option_length = strlen(ros[j].option.option) +
      (ros[j].parameters ? strlen(ros[j].parameters) + 1 : 0);
                               /* The 1 here is a space added between the
                                  option and its parameter description, if it
                                  has one. */

    if (option_length > biggest_option)
      biggest_option = option_length;
    if (strlen(ros[j].help) > biggest_description)
      biggest_description = strlen(ros[j].help);
  }

  assert(sprintf(format, format_template, biggest_option + spacing,
                 biggest_description + spacing) < (int)(sizeof(format)));
  fprintf(fp, format, "Option", "Description");
  fprintf(fp, "Default\n");
  for (j = 0; j < num_ros; j++) {
    char *whole_option = (char *)alloca(biggest_option + 1);

    sprintf(whole_option, "%s %s", ros[j].option.option,
            ros[j].parameters ? ros[j].parameters : "");
    fprintf(fp, format, whole_option, ros[j].help);
    if (ros[j].resource.default_type == XtRString) {
      fprintf(fp, "%s\n", (char *)ros[j].resource.default_addr);
    }
    else {
      fprintf(fp, "\nerror: unknown type %s for %s at %s:%d\n",
              ros[j].resource.default_type, ros[j].option.option, __FILE__,
              __LINE__);
      return;
    }
  }
}

void OU_GetEnvironment(const char *varname, char **argv[], int *argc)
{
  char *env_args;

  if (!(env_args = getenv(varname)))
    return;
  else {
    unsigned num_args = 0;
    char **env_argv = NULL;
    char *arg;

    while ((arg = strsep(&env_args, " \t\n\r\f\v"))) {
      if (strlen(arg)) {
        num_args++;
        env_argv = (char **)realloc(env_argv, sizeof(char *) * num_args);
        env_argv[num_args - 1] = arg;
      }
    }

    if (num_args) {
      char **new_argv = (char **)malloc(sizeof(char *) * (num_args + *argc) +
                                        1);
      unsigned i;

      for (i = 0; i < (unsigned)*argc; i++)
        new_argv[i] = (*argv)[i];

      for (i = *argc; i < num_args + *argc; i++)
        new_argv[i] = env_argv[i - *argc];

      new_argv[i] = NULL;

      *argv = new_argv;
      *argc = num_args + *argc;
    }
  }
}
