/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <sys/types.h>

#include "version.h"
#include "loaderdefs.h"

/*--------------------------------------------------------------------------*/

#define LILO_CONFIGFILE			"/etc/vmelilo.conf"
#define LILO_LOADERFILE			"/boot/boot.loader"
#define LILO_BACKUPFILEPREFIX	"/boot/backup"

/*--------------------------------------------------------------------------*/
/* Configuration File Parsing
 */

typedef struct {
	const char			*boot_device_name;
	const char			*default_password;
	const char			*default_kernel;
	const char			*default_cmdline;
	const char			*default_append;
	const char			*default_root;
	const char			*default_console;
	const unsigned long	*default_read_only;
	const unsigned long	*default_restricted;
 	BOOTOPTIONS			options;
	BOOTRECORD			*records;
	FILEDEF				*files;

} CONFIG;

/*--------------------------------------------------------------------------*/

typedef struct {
	int			maxsize;			/* space allocated for tag data			*/
	int			cursize;			/* amount of tag data so far			*/
	char		*data;				/* pointer to collected tag data		*/

} MEMFILE;

/*--------------------------------------------------------------------------*/

typedef struct {
 const char		*monitor_name;				/* name of debug monitor		*/
 unsigned char	*loader_code;				/* boot loader code				*/
 int			loader_code_size;			/* size of boot loader code		*/
 int			boot_block_size;			/* size of boot block in bytes	*/
 void			(*init_boot_block)(void *);	/* initialise a boot block		*/
 int			(*valid_boot_block)(void *);/* validate a boot block		*/

} SUBARCH;

/*--------------------------------------------------------------------------*/

#ifndef DECLARE_GLOBALS
#define GLOBAL	extern
#define _I_(v)
#else
#define GLOBAL
#define _I_(v)	v
#endif

/*--------------------------------------------------------------------------*/

GLOBAL const char	LiloVersion[]			_I_( = VERSION );

/*--------------------------------------------------------------------------*/
/* Command Line Options
 */
GLOBAL const char		*program_name;
GLOBAL int				f_verbose;
GLOBAL int				f_install;
GLOBAL int				f_backup;
GLOBAL int				f_uninstall;
GLOBAL int				f_force;
GLOBAL int				f_test;
GLOBAL const char		*save_boot_block;
GLOBAL const char		*restore_boot_block;
GLOBAL const char		*root_path;
GLOBAL int				m68k_model _I_( = -1 );
#define MODEL_BVME		0			/* BVME4000/6000 using BVMBug monitor	*/
#define MODEL_MVME		1			/* Motorola SBC using Motorola monitor	*/

/*--------------------------------------------------------------------------*/
/*	Lilo Data
 */
GLOBAL void				*boot_block_buffer;
GLOBAL const FILEMAP	*loader_map;
GLOBAL MEMFILE			map_data;
GLOBAL dev_t			boot_device;
GLOBAL CONFIG			config;

/*--------------------------------------------------------------------------*/
/* Files
 */
GLOBAL const char		*config_file;
GLOBAL const char		*loader_file;
GLOBAL const char		*backup_file;

/*--------------------------------------------------------------------------*/
/* Drivers
 */

extern SUBARCH bvme_subarch;
extern SUBARCH mvme_subarch;

GLOBAL SUBARCH *subarch[]
#ifdef DECLARE_GLOBALS
  =	{	&bvme_subarch,
		&mvme_subarch
	}
#endif
;

#define monitor_name		( subarch[m68k_model]->monitor_name    )
#define loader_code			( subarch[m68k_model]->loader_code     )
#define loader_code_size	( subarch[m68k_model]->loader_code_size)
#define boot_block_size		( subarch[m68k_model]->boot_block_size )
#define init_boot_block		(*subarch[m68k_model]->init_boot_block )
#define valid_boot_block	(*subarch[m68k_model]->valid_boot_block)

/*--------------------------------------------------------------------------*/
/* Function Prototypes
 */

char * skip_white(char *);
int equal_strings (const char *s1, const char *s2);
int case_equal_strings (const char *s1, const char *s2);
void die (const char *fmt, ...) __attribute__ ((noreturn, format (printf, 1, 2)));
void error_nomemory (void);
void error_open (const char *name);
void error_opendir (const char *name);
void error_read (const char *name);
void error_write (const char *name);
void error_seek (const char *name);
void error_stat (const char *name);
void error_ioctl (const char *name,const char *ioctl);
void read_config_file (void);
int is_scsi_partition (unsigned rdev);
const FILEMAP * create_file_map (const char *name);
const char * create_backup_filename (void);

/*-----------------------------< end of file >------------------------------*/
