/*
 * Defines for the dmalloc-leap routines.
 *
 * Copyright 1999 by Gray Watson
 *
 * This file is part of the dmalloc package.
 *
 * Permission to use, copy, modify, and distribute this software for
 * any purpose and without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies, and that the name of Gray Watson not be used in advertising
 * or publicity pertaining to distribution of the document or software
 * without specific, written prior permission.
 *
 * Gray Watson makes no representations about the suitability of the
 * software described herein for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * The author may be contacted via http://www.dmalloc.com/
 *
 * $Id: dmalloc_lp.h,v 1.38 1999/10/13 22:39:52 gray Exp $
 */

#ifndef __DMALLOC_LP_H__
#define __DMALLOC_LP_H__

#define DEBUG_PRE_NONE		(-1)	/* no pre debug call */

#include <stdarg.h>				/* for ... */

/*
 * The following function types are only needed if the user has
 * defined USE_DMALLOC_LEAP to 1 in the settings.h file.
 */
#if USE_DMALLOC_LEAP

/* function type for the shutdown function */
typedef void		(*dmalloc_shutdown_func_t)(void);

/* function type for the malloc function */
typedef DMALLOC_PNT	(*dmalloc_malloc_func_t)(const char *file,
						 const int line,
						 const DMALLOC_SIZE size,
						 const int func_id,
						 const DMALLOC_SIZE alignment);

/* function type for the realloc function */
typedef DMALLOC_PNT	(*dmalloc_realloc_func_t)(const char *file,
						  const int line,
						  DMALLOC_PNT old_pnt,
						  DMALLOC_SIZE new_size,
						  const int func_id);

/* function type for the free function */
typedef int		(*dmalloc_free_func_t)(const char *file,
					       const int line,
					       DMALLOC_PNT pnt);

/* function type for log_heap_map function */
typedef void		(*dmalloc_log_heap_map_func_t)(const char *file,
						       const int line);

/* function type for log_stats function */
typedef void		(*dmalloc_log_stats_func_t)(const char *file,
						    const int line);

/* function type for log_unfreed function */
typedef void		(*dmalloc_log_unfreed_func_t)(const char *file,
						      const int line);

/* function type for verify function */
typedef int		(*dmalloc_verify_func_t)(const DMALLOC_PNT pnt);

/* function type for debug function */
typedef void		(*dmalloc_debug_func_t)(const int flags);

/* function type for debug_current function */
typedef int		(*dmalloc_debug_current_func_t)(void);

/* function type for examine function */
typedef int		(*dmalloc_examine_func_t)(const char *file,
						  const int line,
						  const DMALLOC_PNT pnt,
						  DMALLOC_SIZE *size_p,
						  char **file_p,
						  unsigned int *line_p,
						  DMALLOC_PNT *ret_attr_p);

/* function type for message function */
typedef void		(*dmalloc_vmessage_func_t)(const char *format,
						   va_list args);

/* function type for tracking function */
typedef void	(*dmalloc_track_func_t)(const dmalloc_track_t track_func);

/* function type for strerror function */
typedef const char	*(*dmalloc_strerror_func_t)(const int error_num);

#endif /* if USE_DMALLOC_LEAP */

/*<<<<<<<<<<  The below prototypes are auto-generated by fillproto */

/* internal dmalloc error number for reference purposes only */
extern
int	dmalloc_errno;

/* pre-set dmalloc_debug() value before the library is setup */
extern
int	_dmalloc_debug_preset;

#if USE_DMALLOC_LEAP
/* pointers to shutdown function to allow calls without linked routine */
extern
dmalloc_shutdown_func_t		_dmalloc_shutdown_func;
#endif /* if USE_DMALLOC_LEAP */

#if USE_DMALLOC_LEAP
/* pointers to the standard malloc function */
extern
dmalloc_malloc_func_t		_dmalloc_malloc_func;
#endif /* if USE_DMALLOC_LEAP */

#if USE_DMALLOC_LEAP
/* pointers to the standard realloc function */
extern
dmalloc_realloc_func_t		_dmalloc_realloc_func;
#endif /* if USE_DMALLOC_LEAP */

#if USE_DMALLOC_LEAP
/* pointers to the standard free function */
extern
dmalloc_free_func_t		_dmalloc_free_func;
#endif /* if USE_DMALLOC_LEAP */

#if USE_DMALLOC_LEAP
/* pointers to log_heap_map function to allow calls without linked routine */
extern
dmalloc_log_heap_map_func_t	_dmalloc_log_heap_map_func;
#endif /* if USE_DMALLOC_LEAP */

#if USE_DMALLOC_LEAP
/* pointers to log_stats function to allow calls without linked routine */
extern
dmalloc_log_stats_func_t	_dmalloc_log_stats_func;
#endif /* if USE_DMALLOC_LEAP */

#if USE_DMALLOC_LEAP
/* pointers to log_unfreed function to allow calls without linked routine */
extern
dmalloc_log_unfreed_func_t	_dmalloc_log_unfreed_func;
#endif /* if USE_DMALLOC_LEAP */

#if USE_DMALLOC_LEAP
/* pointers to verify function to allow calls without linked routine */
extern
dmalloc_verify_func_t		_dmalloc_verify_func;
#endif /* if USE_DMALLOC_LEAP */

#if USE_DMALLOC_LEAP
/* pointers to debug function to allow calls without linked routine */
extern
dmalloc_debug_func_t		_dmalloc_debug_func;
#endif /* if USE_DMALLOC_LEAP */

#if USE_DMALLOC_LEAP
/* pointers to debug_current function to allow calls without linked routine */
extern
dmalloc_debug_current_func_t	_dmalloc_debug_current_func;
#endif /* if USE_DMALLOC_LEAP */

#if USE_DMALLOC_LEAP
/* pointers to examine function to allow calls without linked routine */
extern
dmalloc_examine_func_t		_dmalloc_examine_func;
#endif /* if USE_DMALLOC_LEAP */

#if USE_DMALLOC_LEAP
/* pointers to message function to allow calls without linked routine */
extern
dmalloc_vmessage_func_t		_dmalloc_vmessage_func;
#endif /* if USE_DMALLOC_LEAP */

#if USE_DMALLOC_LEAP
/* pointers to track function to allow calls without linked routine */
extern
dmalloc_track_func_t		_dmalloc_track_func;
#endif /* if USE_DMALLOC_LEAP */

#if USE_DMALLOC_LEAP
/* pointers to strerror function to allow calls without linked routine */
extern
dmalloc_strerror_func_t		_dmalloc_strerror_func;
#endif /* if USE_DMALLOC_LEAP */

/*
 * leap routine to malloc
 */
extern
DMALLOC_PNT	_malloc_leap(const char *file, const int line,
			     DMALLOC_SIZE size);

/*
 * leap routine to calloc
 */
extern
DMALLOC_PNT	_calloc_leap(const char *file, const int line,
			     DMALLOC_SIZE ele_n, DMALLOC_SIZE size);

/*
 * leap routine to realloc
 */
extern
DMALLOC_PNT	_realloc_leap(const char *file, const int line,
			      DMALLOC_PNT old_p, DMALLOC_SIZE new_size);

/*
 * leap routine to recalloc
 */
extern
DMALLOC_PNT	_recalloc_leap(const char *file, const int line,
			       DMALLOC_PNT old_p, DMALLOC_SIZE new_size);

/*
 * leap routine to memalign
 */
extern
DMALLOC_PNT	_memalign_leap(const char *file, const int line,
			       DMALLOC_SIZE alignment, DMALLOC_SIZE size);

/*
 * leap routine to valloc
 */
extern
DMALLOC_PNT	_valloc_leap(const char *file, const int line,
			     DMALLOC_SIZE size);

/*
 * leap routine to strdup
 */
extern
char	*_strdup_leap(const char *file, const int line, const char *str);

/*
 * leap routine to free
 */
extern
DMALLOC_FREE_RET	_free_leap(const char *file, const int line,
				   DMALLOC_PNT pnt);

/*
 * leap routine to malloc with error checking
 */
extern
DMALLOC_PNT	_xmalloc_leap(const char *file, const int line,
			      DMALLOC_SIZE size);

/*
 * leap routine to calloc with error checking
 */
extern
DMALLOC_PNT	_xcalloc_leap(const char *file, const int line,
			      DMALLOC_SIZE ele_n, DMALLOC_SIZE size);

/*
 * leap routine to realloc with error checking
 */
extern
DMALLOC_PNT	_xrealloc_leap(const char *file, const int line,
			       DMALLOC_PNT old_p, DMALLOC_SIZE new_size);

/*
 * leap routine to recalloc with error checking
 */
extern
DMALLOC_PNT	_xrecalloc_leap(const char *file, const int line,
				DMALLOC_PNT old_p, DMALLOC_SIZE new_size);

/*
 * leap routine to memalign with error checking
 */
extern
DMALLOC_PNT	_xmemalign_leap(const char *file, const int line,
				DMALLOC_SIZE alignment, DMALLOC_SIZE size);

/*
 * leap routine to valloc with error checking
 */
extern
DMALLOC_PNT	_xvalloc_leap(const char *file, const int line,
			      DMALLOC_SIZE size);

/*
 * leap routine for strdup with error checking
 */
extern
char 	*_xstrdup_leap(const char *file, const int line,
		       const char *str);

/*
 * leap routine to free
 */
extern
DMALLOC_FREE_RET	_xfree_leap(const char *file, const int line,
				    DMALLOC_PNT pnt);

/*
 * routine to call dmalloc_shutdown when linked in
 */
extern
void	dmalloc_shutdown(void);

/*
 * routine to call dmalloc_log_heap_map when linked in
 */
extern
void	dmalloc_log_heap_map(void);

/*
 * routine to call dmalloc_log_stats when linked in
 */
extern
void	dmalloc_log_stats(void);

/*
 * routine to call dmalloc_log_unfreed when linked in
 */
extern
void	dmalloc_log_unfreed(void);

/*
 * routine to call dmalloc_verify when linked in
 */
extern
int	dmalloc_verify(const DMALLOC_PNT pnt);

/*
 * routine to call malloc_verify when linked in
 */
extern
int	malloc_verify(const DMALLOC_PNT pnt);

/*
 * routine to call dmalloc_debug when linked in
 */
extern
void	dmalloc_debug(const int flags);

/*
 * routine to call dmalloc_debug_current when linked in
 */
extern
int	dmalloc_debug_current(void);

/*
 * routine to call dmalloc_examine when linked in
 */
extern
int	dmalloc_examine(const DMALLOC_PNT pnt, DMALLOC_SIZE *size_p,
			char **file_p, unsigned int *line_p,
			DMALLOC_PNT *ret_attr_p);

#if HAVE_STDARG_H
/*
 * routine to call dmalloc_vmessage when linked in
 */
extern
void	dmalloc_message(const char *format, ...);
#endif /* if HAVE_STDARG_H */

#if HAVE_STDARG_H
/*
 * routine to call dmalloc_vmessage when linked in
 */
extern
void	dmalloc_vmessage(const char *format, va_list args);
#endif /* if HAVE_STDARG_H */

/*
 * routine to call dmalloc_track when linked in
 */
extern
void	dmalloc_track(const dmalloc_track_t track_func);

/*
 * routine to call dmalloc_strerror when linked in
 */
extern
const char	*dmalloc_strerror(const int error_num);

/*<<<<<<<<<<   This is end of the auto-generated output from fillproto. */

#endif /* ! __DMALLOC_LP_H__ */
