/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998, 1999  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_mutex.c,v 1.2 1999/11/23 19:01:41 hercules Exp $";
#endif

#ifdef linux
/* Look to see if glibc is available, and if so, what version */
#include <features.h>

#if (__GLIBC__ == 2) && (__GLIBC_MINOR__ == 0)
#warning Working around a bug in glibc 2.0 pthreads
#undef SDL_USE_PTHREADS
/* The bug is actually a problem where threads are suspended, but don't
   wake up when the thread manager sends them a signal.  This is a problem
   with thread creation too, but it happens less often. :-/
   We avoid this by using System V IPC for mutexes.
 */
#endif /* glibc 2.0 */
#endif /* linux */

#ifdef SDL_USE_PTHREADS

#include <stdlib.h>
#include <stdio.h>
#include <pthread.h>

#include "SDL_error.h"
#include "SDL_mutex.h"


struct SDL_mutex {
	pthread_mutex_t id;
};

/* Create a blockable semaphore */
SDL_mutex * SDL_CreateMutex (void)
{
	SDL_mutex *mutex;

	/* Allocate the structure */
	mutex = (SDL_mutex *)malloc(sizeof(*mutex));
	if ( mutex == NULL ) {
		SDL_OutOfMemory();
		return(NULL);
	}

	if ( pthread_mutex_init(&mutex->id, NULL) != 0 ) {
		SDL_SetError("Couldn't create mutex");
		free(mutex);
		return(NULL);
	}
	return(mutex);
}

/* Lock the semaphore */
int SDL_mutexP (SDL_mutex *mutex)
{
	if ( ! mutex ) {
		SDL_SetError("Passed a NULL mutex");
		return(-1);
	}
	if ( pthread_mutex_lock(&mutex->id) != 0 ) {
		SDL_SetError("Couldn't lock mutex");
		return(-1);
	}
	return(0);
}

/* Unlock the semaphore */
int SDL_mutexV (SDL_mutex *mutex)
{
	if ( ! mutex ) {
		SDL_SetError("Passed a NULL mutex");
		return(-1);
	}
	if ( pthread_mutex_unlock(&mutex->id) != 0 ) {
		SDL_SetError("Couldn't unlock mutex");
		return(-1);
	}
	return(0);
}

/* Free the semaphore */
void SDL_DestroyMutex (SDL_mutex *mutex)
{
	if ( mutex ) {
		pthread_mutex_destroy(&mutex->id);
		free(mutex);
	}
}

#else /* System V IPC based implementation */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <errno.h>

#include "SDL_error.h"
#include "SDL_mutex.h"


struct SDL_mutex {
	int id;
};

/* Not defined by Solaris or later versions of Linux */
#if defined(__SVR4) || defined(_SEM_SEMUN_UNDEFINED)
union semun {
	int val;
	struct semid_ds *buf;
	ushort *array;
};
#endif

static struct sembuf op_lock[2] = {
	{ 0, 0, 0 },			/* Wait for semaphore to reach 0 */
	{ 0, 1, SEM_UNDO },		/* Increment semaphore */
};
static struct sembuf op_unlock[1] = {
	{ 0, -1, (IPC_NOWAIT|SEM_UNDO) }	/* Decrement semaphore */
};

/* Create a blockable semaphore */
SDL_mutex * SDL_CreateMutex (void)
{
	SDL_mutex *mutex;
	union semun init;

	mutex = (SDL_mutex *)malloc(sizeof(*mutex));
	if ( mutex == NULL ) {
		SDL_OutOfMemory();
		return(NULL);
	}
	mutex->id = semget(IPC_PRIVATE, 1, (0600|IPC_CREAT));
	if ( mutex->id < 0 ) {
		SDL_SetError("Couldn't create semaphore");
		free(mutex);
		return(NULL);
	}
	init.val = 0;		/* Initialize semaphore */
	semctl(mutex->id, 0, SETVAL, init);
	return(mutex);
}

/* Lock the semaphore */
int SDL_mutexP (SDL_mutex *mutex)
{
	if ( ! mutex ) {
		SDL_SetError("Passed a NULL mutex");
		return(-1);
	}
tryagain:
	if ( semop(mutex->id, op_lock, 2) < 0 ) {
		if ( errno == EINTR ) {
			goto tryagain;
		}
		SDL_SetError("Semaphore operation error");
		return(-1);
	}
	return(0);
}

/* Unlock the semaphore */
int SDL_mutexV (SDL_mutex *mutex)
{
	if ( ! mutex ) {
		SDL_SetError("Passed a NULL mutex");
		return(-1);
	}
	if ( semop(mutex->id, op_unlock, 1) < 0 ) {
		SDL_SetError("Semaphore operation error");
		return(-1);
	}
	return(0);
}

/* Free the semaphore */
void SDL_DestroyMutex (SDL_mutex *mutex)
{
	if ( mutex ) {
#ifdef _SGI_SOURCE
		semctl(mutex->id, 0, IPC_RMID);
#else
		semctl(mutex->id, 0, IPC_RMID, (union semun)0);
#endif
		free(mutex);
	}
}

#endif /* SDL_USE_PTHREADS */
