/*								-*- C++ -*-
 * $Id: WIN_slider.cpp,v 1.2 1998/03/31 22:16:07 wg Exp $
 *
 * Purpose: slider panel item
 *
 * Authors: Markus Holzem and Julian Smart
 *
 * Copyright: (C) 1995, AIAI, University of Edinburgh (Julian)
 * Copyright: (C) 1995, GNU (Markus)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Additionally everyone using this library has to announce it with:
 *
 *   This software uses the wxWindows-Xt GUI library
 *   (C) Markus Holzem, available via
 *       ftp://ftp.aiai.ed.ac.uk/pub/packages/wxwin/ports/xt
 */

#ifdef __GNUG__
#pragma implementation "WIN_slider.h"
#endif

#define  Uses_XtIntrinsic
#define  Uses_wxSlider
#include "wx.h"
#define  Uses_EnforcerWidget
#define  Uses_ScaleWidget
#include <widgets.h>

//-----------------------------------------------------------------------------
// create and destroy button
//-----------------------------------------------------------------------------

IMPLEMENT_DYNAMIC_CLASS(wxSlider, wxItem)

wxSlider::wxSlider(void) : wxItem()
{
    __type = wxTYPE_SLIDER;

    minimum = maximum = value = 0;
}

wxSlider::wxSlider(wxPanel *panel, wxFunction func, Const char *label,
		   int _value, int min_value, int max_value, int width,
		   int x, int y, long style, Constdata char *name)
{
   __type = wxTYPE_SLIDER;

    minimum = maximum = value = 0;

    Create(panel, func, label, _value, min_value, max_value, width, 
	   x, y, style, name);
}

Bool wxSlider::Create(wxPanel *panel, wxFunction func, Const char *label,
		      int init_value, int min_value, int max_value, int length,
		      int x, int y, long style, Constdata char *name)
{
    ChainToPanel(panel, style, name);

    minimum = min_value;
    maximum = max_value;

    // compute sane length if not supplied
    if(length < 0) {
	length = (max_value - min_value) + 10;
	if(length < 100)
	    length = 100;
	else if(length > 500)
	    length = 500;
    }

    // create frame
    FWidget() = XtVaCreateManagedWidget
	(name, xfwfEnforcerWidgetClass, GetParentWidget(parent),
	 XtNlabel,       label,
	 XtNalignment,   (((label_position==wxVERTICAL) || (style&wxVERTICAL)) ?
			  XfwfTop : XfwfLeft),
	 XtNbackground,  bg.GetPixel(&cmap),
	 XtNforeground,  label_fg.GetPixel(&cmap),
	 XtNfont,        label_font.GetInternalFont(),
	 XtNframeType,   XfwfSunken,
	 XtNframeWidth,  2,
	 XtNshrinkToFit, TRUE,
	 XtNtraversalKeys, XfwfTraverseKeyAll & (~XfwfTraverseKeyHome)
			   & ~((style & wxVERTICAL) ?
			       XfwfTraverseKeyUp   | XfwfTraverseKeyDown :
			       XfwfTraverseKeyLeft | XfwfTraverseKeyRight),
	 NULL);
    // compute sizes of the slider widget
    float swidth, sheight; char tempstring[80];
    sprintf(tempstring, "-%d-", wxMax(wxAbs(max_value), wxAbs(min_value)));
    GetTextExtent(tempstring, &swidth, &sheight);
    swidth += 8; sheight += 8; // shadows and margin
    // create the slider widget
    HWidget() = XtVaCreateManagedWidget
	(name, xfwfScaleWidgetClass, FWidget(),
	 XtNbackground,    bg.GetPixel(&cmap),
	 XtNforeground,    fg.GetPixel(&cmap),
	 XtNthumbColor,    bg.GetPixel(&cmap),
	 XtNfont,          font.GetInternalFont(),
	 XtNframeType,     XfwfRaised,
	 XtNframeWidth,    0,
	 XtNhighlightThickness, 0,
	 XtNtraversalOn,   FALSE, 
	 (style & wxVERTICAL) ? XtNwidth : XtNheight,
	                   Dimension((style & wxVERTICAL) ? swidth : sheight),
	 NULL);
    if (style & wxVERTICAL) {
	XfwfResizeThumb(HWidget(), 1.0, wxMax(0.1,wxMin(0.9,sheight/length)));
    } else {
	XfwfResizeThumb(HWidget(), wxMax(0.1,wxMin(0.9, swidth/length)), 1.0);
    }
    SetValue(init_value);
    // propagate key events from frame to button widget
    XtVaSetValues(FWidget(), XtNpropagateTarget, HWidget(), NULL);
    // set data declared in wxItem
    callback = func;
    XtAddCallback(HWidget(), XtNscrollCallback, wxSlider::EventCallback,
		  (XtPointer)this);
    // panel positioning
    panel->PositionItem(this, x, y, (style & wxVERTICAL) ? -1 : length, (style & wxVERTICAL) ? length : -1);
    AddEventHandlers();

    return TRUE;
}

//-----------------------------------------------------------------------------
// OnChar handles cursor keys
//-----------------------------------------------------------------------------

void wxSlider::OnChar(wxKeyEvent& event)
{
    Boolean handled = FALSE;
    Boolean vert = (style & wxVERTICAL);

    int pg = (maximum - minimum) / 10;

    switch (event.KeyCode()) {
    case WXK_HOME:  SetValue(minimum);			handled = TRUE;   break;
    case WXK_END:   SetValue(maximum);			handled = TRUE;   break;
    case WXK_PRIOR: SetValue(wxMax(minimum, value-pg)); handled = TRUE;   break;
    case WXK_NEXT:  SetValue(wxMin(maximum, value+pg)); handled = TRUE;   break;
    case WXK_UP:    if ( vert)   { SetValue(value - 1); handled = TRUE; } break;
    case WXK_DOWN:  if ( vert)   { SetValue(value + 1); handled = TRUE; } break;
    case WXK_LEFT:  if (!vert)   { SetValue(value - 1); handled = TRUE; } break;
    case WXK_RIGHT: if (!vert)   { SetValue(value + 1); handled = TRUE; } break;
    }
    if (handled) {
	// process event
	wxCommandEvent command_event(wxEVENT_TYPE_SLIDER_COMMAND);
	command_event.eventObject = this;
	command_event.commandInt  = value;
	ProcessCommand(command_event);
    } else {
	wxItem::OnChar(event);
    }
}

//-----------------------------------------------------------------------------
// methods to access internal data
//-----------------------------------------------------------------------------

void wxSlider::SetButtonColour(wxColour &col)
{
    if (&col && HWidget())
	XtVaSetValues(HWidget(), XtNthumbColor, col.GetPixel(&cmap), NULL);
    wxItem::SetButtonColour(col);
}

void wxSlider::SetRange(int smin, int smax)
{
    minimum = wxMin(smin, smax);
    maximum = wxMax(smin, smax);

    // resize only possible for horizontal sliders
    if ( !(style & wxVERTICAL) ) {
	float swidth, sheight; char tempstring[80];
	sprintf(tempstring, "-%d", wxMax(wxAbs(maximum), wxAbs(minimum)));
	GetTextExtent(tempstring, &swidth, &sheight);
	swidth += 8;
	int width, length;
	GetSize(&length, &width);
	XfwfResizeThumb(HWidget(), wxMax(0.1,wxMin(0.9, swidth/length)), 1.0);
    }

    // adjust value
    value = wxMax(value, minimum);
    value = wxMin(value, maximum);
    SetValue(value);
}

void wxSlider::SetValue(int new_value)
{
    if (minimum <= new_value && new_value <= maximum) {
	value = new_value;
	char tempstring[80];
	sprintf(tempstring, "%d", value);
	//XtVaSetValues(HWidget(), XtNlabel, tempstring, NULL);
	XfwfSetLabel(HWidget(), tempstring);
	if (style & wxVERTICAL)
	    XfwfMoveThumb(HWidget(),
			  0.0, float(value-minimum)/float(maximum-minimum));
	else
	    XfwfMoveThumb(HWidget(),
			  float(value-minimum)/float(maximum-minimum), 0.0);
    }
}

//-----------------------------------------------------------------------------
// callbacks for xfwfGroupWidgetClass
//-----------------------------------------------------------------------------

void wxSlider::EventCallback(Widget WXUNUSED(w),
			     XtPointer dclient, XtPointer dcall)
{
    wxSlider       *slider = (wxSlider*)dclient;
    XfwfScrollInfo *info   = (XfwfScrollInfo*)dcall;
    Bool           process = FALSE;
    int            new_value = 0;

    if ((slider->style & wxVERTICAL) && (info->flags & XFWF_VPOS)) {
	if (info->reason == XfwfSPageUp || info->reason == XfwfSPageDown) {
	    XfwfMoveThumb(slider->HWidget(), 0.0, info->vpos);
	}
	new_value = int(slider->minimum 
			+ info->vpos * (slider->maximum-slider->minimum));
	process = TRUE;
    } else if (!(slider->style & wxVERTICAL) && (info->flags & XFWF_HPOS)) {
	if (info->reason == XfwfSPageLeft || info->reason == XfwfSPageRight) {
	    XfwfMoveThumb(slider->HWidget(), info->hpos, 0.0);
	}
	new_value = int(slider->minimum
			+ info->hpos * (slider->maximum-slider->minimum));
	process = TRUE;
    }
    if (process && new_value != slider->value) {
	// set and display new value
	slider->value = new_value;
	char tempstring[80];
	sprintf(tempstring, "%d", new_value);
	//XtVaSetValues(slider->HWidget(), XtNlabel, tempstring, NULL);
	XfwfSetLabel(slider->HWidget(), tempstring);
	// process event
	wxCommandEvent event(wxEVENT_TYPE_SLIDER_COMMAND);
	event.eventObject = slider;
	event.commandInt = new_value;
	slider->ProcessCommand(event);
    }
}
