/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#if !str_h
#define str_h 1

/* Lengthed strings:

   In several areas in the program it is necessary to avoid
   null-terminated strings because NULs may be present in the strings
   (specifically, in the data input and output code and in som and
   relations).  There are three flavors of non-null-terminated strings,
   defined in "str.h".  Functions that deal with lengthed strings often
   have the types of their arguments encoded in their names (in C++ the
   same thing would be expressed by function overloading).

   The type letters are as follows:

   a: a_string

   b: b_string

   c: c_string

   d: string passed as pointers to beginning and one char after the
   end

   i: a_string passed as individual arguments rather than
   encapsulated in a structure

   s: ordinary null-terminated strings. */

/* Headers and miscellaneous. */

#include <stdarg.h>

#if STDC_HEADERS
#include <string.h>
#else
#ifndef HAVE_STRCHR
#define strchr index
#define strrchr rindex
#endif
char *strchr (), *strrchr ();
#endif

#if !HAVE_STRTOK_R
char *strtok_r (char *, const char *, char **);
#endif

#if !HAVE_STPCPY && !__linux__
char *stpcpy (char *dest, const char *src);
#endif

#if !HAVE_STRCASECMP
int strcasecmp (const char *s1, const char *s2);
#endif

#if !HAVE_STRNCASECMP
int strncasecmp (const char *s1, const char *s2, size_t n);
#endif

#if !HAVE_MEMMEM
void *memmem (const void *haystack, size_t haystack_len,
	      const void *needle, size_t needle_len);
#endif

#define streq(A, B) 				\
	(!strcmp ((A), (B)))

/* spprintf() calls sprintf() and returns the address of the null
   terminator in the resulting string.  It should be portable the way
   it's been implemented. */
#if __GNUC__
#if HAVE_GOOD_SPRINTF
#define spprintf(buf, format, args...)			\
	((buf) + sprintf((buf), (format) , ## args))
#else /* not good sprintf */
#define spprintf(buf, format, args...)					\
	({ sprintf((buf), (format) , ## args); strchr((buf), 0); })
#endif /* not good sprintf */
#else /* not gcc */
char *spprintf (char *buf, const char *format,...);
#endif /* not gcc */

/* nsprintf() calls sprintf() and returns the strlen() of the
   resulting string.  It should be portable the way it's been
   implemented. */
#if __GNUC__
#if HAVE_GOOD_SPRINTF
#define nsprintf(BUF, FORMAT, ARGS...) 		\
	(sprintf ((BUF), (FORMAT) , ## ARGS))
#define nvsprintf(BUF, FORMAT, ARGS) 		\
	(vsprintf ((BUF), (FORMAT), (ARGS)))
#else /* not good sprintf */
#define nsprintf(BUF, FORMAT, ARGS...)		\
	({					\
	  char *_buf = BUF;			\
	  sprintf ((_buf), (FORMAT) , ## ARGS);	\
	  strlen (_buf);			\
	})
#define nvsprintf(BUF, FORMAT, ARGS)		\
	({					\
	  char *_buf = BUF;			\
	  vsprintf ((_buf), (FORMAT), (ARGS));	\
	  strlen (_buf);			\
	})
#endif /* not good sprintf */
#else /* not gcc */
#if HAVE_GOOD_SPRINTF
#define nsprintf sprintf
#define nvsprintf vsprintf
#else /* not good sprintf */
int nsprintf (char *buf, const char *format,...);
int nvsprintf (char *buf, const char *format, va_list args);
#endif /* not good sprintf */
#endif /* not gcc */

/* Weird PSPP string types. */

/* These are the three named varieties of strings, besides the
   ordinary null-terminated flavor.  Each flavor is marked with the
   symptoms of its most-common usage. */
/* Passed to functions. */
typedef struct a_string
  {
    int l;
    char *s;
  }
a_string;

/* Stored in memory. */
typedef struct b_string
  {
    int l;
    char s[1];
  }
b_string;

/* Excessive memory-efficiency.  See pref.h for the methods used for
   reading and writing to L. */
typedef struct c_string
  {
    char l1, l2;		/* Not an array, to avert mistakes. */
    char s[1];
  }
c_string;

/* A `d' string is passed to a function as two char * arguments (it
   isn't a structure).  Arg 1 points to the first character, arg 2
   points to the character after the last one. */

/* An `s' string is an ordinary null-terminated string. */

/* String functions. */

char *spacing (int);
void strbarepadlencpy (char *dest, const char *src, size_t n,
		       int pad, size_t len);
void strbarepadcpy (char *dest, const char *src, size_t n, int pad);
void strmaxcpy (char *, const char *, size_t);
void strpadcpy (char *, const char *, size_t, int);
void blpstrset (char *, int, int);

void memrev (void *, size_t);
char *memrmem (const char *, size_t, const char *, size_t);
int cmp_str (const char *, int, const char *, int);

a_string *aa_strcpy (a_string *, const a_string *);
a_string *ab_strcpy (a_string *, const b_string *);
a_string *ac_strcpy (a_string *, const c_string *);
b_string *ba_strcpy (b_string *, const a_string *);
b_string *bb_strcpy (b_string *, const b_string *);
c_string *ca_strcpy (c_string *, const a_string *);

a_string *aa_strdup (const a_string *);
char *sa_strdup (const a_string *);
a_string *as_strdup (const a_string *, const char *);
a_string *as_strdupcpy (a_string *, const char *);
a_string *aa_strdupcpy (a_string *, const a_string *);
b_string *ba_strdup (const a_string *);
b_string *ba_strdupcpy (b_string *, const a_string *);

int aa_printf (a_string *, a_string *,...);
int ab_printf (a_string *, b_string *,...);
int as_printf (a_string *, char *,...);

#define as_streq(A, B)							\
	((int) strlen (B)==(A)->l && !memcmp ((A)->s, (B), (A)->l))
#define bs_streq(A, B) 							\
	((int) strlen (B)==(A)->l && !memcmp ((A)->s, (B), (A)->l))
#define cs_streq(A, B)					\
	((int) strlen (B)==LOAD_2(&A->l1)		\
	 && !memcmp ((A)->s, (B), LOAD_2(&A->l1)))
#define sa_streq(B, A) 							\
	((int) strlen (B)==(A)->l && !memcmp ((A)->s, (B), (A)->l))
#define sb_streq(B, A) 							\
	((int) strlen (B)==(A)->l && !memcmp ((A)->s, (B), (A)->l))

#endif /* str_h */
