/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTSPPrefs.h

	Contains:	Object store for RTSP server preferences.
	
	$Log: RTSPPrefs.h,v $
	Revision 1.2  1999/02/19 23:08:30  ds
	Created
	
	
*/

#ifndef __RTSPPREFS_H__
#define __RTSPPREFS_H__


#include "StrPtrLen.h"
#include "OSMutex.h"
#include "PrefsSource.h"

class RTSPPrefs
{
	public:

		//Arguments:	inPrefsSource: an object to fetch preferences from
		//				inLogTimestamps:	Specify "log timestamps" setting.
		//				inAppendRandomOffsets: Specify "append random offsets" setting
		RTSPPrefs(PrefsSource* inPrefsSource, bool inLogTimestamps, bool inAppendRandomOffsets,
					bool inKeyFramesOnly, bool inNoBFrames);
		~RTSPPrefs() {}
		
		//This is callable at any time, and is thread safe wrt to the accessors
		void RereadPreferences();

		//if you want to do something that must be preemtive safe
		OSMutex*	GetMutex() 			{ return &fMutex; }
		
		//Prefs source: if you want raw access to the source of the preferences
		PrefsSource* GetPrefsSource()	{ return fPrefsSource; }
		
		//Individual accessor methods for preferences.
		
		//This function returns a newly allocated buffer with the movie folder
		//path in it. You may opt for some padding in the buffer if you'd like,
		//and the length of the path is returned to you.
		char*	GetMovieFolderPath(UInt32 inPadding, UInt32* outLength);
		char*	GetMovieFolderPath();	//csl- added this for the rest of us...
		//This function returns a newly allocated buffer with the movie folder
		//path for users' home directory. You may opt for some padding in
		//the buffer if you'd like, and the length of the path is returned
		//to you.
		char*	GetUsersMovieFolderPath(UInt32 inPadding, UInt32* outLength);

		//this may return INADDR_ANY
		UInt32	GetRTSPIPAddress() 						{ return fIPAddress; }
		
		//Amount of idle time after which respective protocol sessions are timed out
		//(stored in seconds)
		
		//This is the value we advertise to clients (lower than the real one)
		UInt32	GetRTSPTimeoutInSecs()	{ return fRTSPTimeoutInSecs; }
		UInt32	GetRTPTimeoutInSecs()	{ return fRTPTimeoutInSecs; }
		
		//This is the real timeout
		UInt32	GetRealRTSPTimeoutInSecs(){ return fRealRTSPTimeoutInSecs; }
		
		StrPtrLen*	GetRTSPTimeoutAsString() { return &fRTSPTimeoutString; }
		
		UInt32	GetSecondsToBuffer()	{ return fBufferSeconds; }
		//-1 means unlimited
		SInt32	GetMaxConnections()			{ return fMaximumConnections; }
		SInt32	GetMaxKBitsBandwidth()		{ return fMaxBandwidthInKBits; }
		
		//Access log
		bool	IsTransferLogEnabled() 	{ return fTransferLogEnabled; }

		//these allocate a new buffer
		char*	GetTransferLogDir(){ return GetStringPreference(fTransferLogDir.Ptr); }
		char*	GetTransferLogName(){ return GetStringPreference(fTransferLogName.Ptr); }

		SInt32	GetMaxTransferLogBytes() 		{ return fTransferLogBytes; }
		SInt32	GetTransferRollIntervalInDays() { return fTransferRollIntervalInDays; }
		
		//Error log
		bool	IsErrorLogEnabled() 	{ return fErrorLogEnabled; }
		bool	IsScreenLoggingEnabled(){ return fScreenLoggingEnabled; }
		
		//these allocate a new buffer
		char*	GetErrorLogDir()	{ return GetStringPreference(fErrorLogDir.Ptr); }
		char*	GetErrorLogName()	{ return GetStringPreference(fErrorLogName.Ptr); }
		
		SInt32	GetMaxErrorLogBytes() 			{ return fErrorLogBytes; }
		SInt32	GetErrorRollIntervalInDays() 	{ return fErrorRollIntervalInDays; }
		UInt32	GetErrorLogVerbosity() 			{ return fErrorLogVerbosity; }

		//for debugging, mainly
		bool		ShouldServerBreakOnAssert()			{ return fBreakOnAssert; }
		bool		IsAutoRestartEnabled()				{ return fAutoRestart; }
		bool		DisableTimeoutOnPlay()				{ return fDisableTimeout; }
		UInt32		GetTotalBytesUpdateTimeInSecs()		{ return fTBUpdateTimeInSecs; }
		
		SInt64		GetReflectorDelayTimeInMilSecs()		{ return fDelayTimeInMilSecs; }
		UInt32		GetReflectorBucketSize()				{ return fBucketSize; }

		SInt32		GetHistoryUpdateIntervaInSecs(){ return fHistoryIntervalInSecs; }
		
		UInt32		GetAvgBandwidthUpdateTimeInSecs()		{ return fABUpdateTimeInSecs; }
		UInt32		GetSafePlayDurationInSecs()				{ return fSafePlayDurationInSecs; }
		UInt32		GetMaxAdvanceSendTimeInMilSecs()		{ return fMaxAdvanceTimeInMilSecs; }
		
		bool		GetLogTimestamps()						{ return fLogTimestamps; }
		bool		GetAppendRandomOffsets()				{ return fAppendRandomOffsets; }
		bool		GetKeyFramesOnly()						{ return fKeyFramesOnly; }
		bool		GetNoBFrames()							{ return fNoBFrames; }
		bool		IsFlowControlLogEnabled()				{ return fFlowControlLogEnabled; }
		
		//used for generating dynamic SDP
		StrPtrLen*	GetSDPURL()			{ return &fSDPURL; }
		StrPtrLen*	GetAdminEmail()		{ return &fAdminEmail; }
		StrPtrLen*	GetWebStatsURL()		{ return &fWebStatsURL; }

		//remote status server
		UInt32		GetRemoteStatsAddr()	{ return fRemoteStatsAddr; }
		UInt32		GetRemoteStatsPort()	{ return fRemoteStatsPort; }
		
		//thinning settings
		UInt32		GetPercentLossThinTolerance()	{ return fLossThinTolerance; }
		UInt32		GetNumLossesToThin()			{ return fNumLossesToThin; }
		UInt32		GetPercentLossThickTolerance()	{ return fLossThickTolerance; }
		UInt32		GetNumLossesToThick()			{ return fLossesToThick; }
		UInt32		GetNumWorsesToThin()			{ return fWorsesToThin; }
		
		static StrPtrLen&	GetServerName() 	{ return sServerNameStr; }
		static StrPtrLen&	GetServerVersion() 	{ return sServerVersionStr; }
		static StrPtrLen&	GetServerBuildDate() { return sServerBuildDateStr; }
		
	private:

		static StrPtrLen	sServerNameStr;
		static StrPtrLen	sServerVersionStr;
		static StrPtrLen	sServerBuildDateStr;

		enum
		{
			//All the preferences read by this object
			kRTSPPortPref = 0,
			kRTSPTimeout = 1,
			kMaximumConnections = 2,
			kMovieFolder = 3,
			kTransferLog = 4,
			kTransferLogDir = 5,
			kMaxTransferLogSize = 6,
			kTransferRollInterval = 7,
			kErrorLog = 8,
			kErrorLogDir = 9,
			kMaxErrorLogSize = 10,
			kErrorRollInterval = 11,
			kErrorLogVerbosity = 12,
			kScreenLogging = 13,
			kBreakOnAssert = 14,
			kBindToThisIPAddr = 15,
			kHistoryUpdateInterval = 16,
			kMaxBandwidth = 17,
			kBufferSeconds = 18,
			kAutoRestart = 19,
			kRTPTimeout = 20,
			kTransferLogName = 21,
			kErrorLogName = 22,
			kTotalBytesUpdateTime = 23,
			kReflectorDelayTime = 24,
			kReflectorBucketSize = 25,
			kABUpdateTime = 26,
			kSafePlayDuration = 27,
			kSDPURL = 28,
			kAdminEmail = 29,
			kDisableTimeout = 30,
			kMaxAdvanceTime = 31,
			kWebStatsURL = 32,
			kRemoteStatsAddr = 33,
			kRemoteStatsPort = 34,
			kLossThinTolerance = 35,
			kNumLossesToThin = 36,
			kLossThickTolerance = 37,
			kNumLossesToThick = 38,
			kNumWorsesToThin = 39,
			kRealRTSPTimeout = 40,
			kUsersMovieFolder = 41,
			kFlowControlLogEnabled = 42,
			
			kNumPreferences = 43
		};
		
		//for keeping this stuff pre-emptive safe
		OSMutex	fMutex;
		PrefsSource* fPrefsSource;

		SInt32 	fRTSPTimeoutInSecs;
		SInt32 	fRTPTimeoutInSecs;
		SInt32 	fMaximumConnections;
		SInt32	fMaxBandwidthInKBits;
		StrPtrLen fMoviesFolder;
		StrPtrLen fUsersMovieFolder;
		bool 	fTransferLogEnabled;
		StrPtrLen	fTransferLogName;
		StrPtrLen	fTransferLogDir;
		SInt32	fTransferLogBytes;
		SInt32	fTransferRollIntervalInDays;
		bool	fErrorLogEnabled;
		StrPtrLen	fErrorLogDir;
		StrPtrLen	fErrorLogName;
		SInt32	fErrorLogBytes;
		SInt32	fErrorRollIntervalInDays;
		UInt32	fErrorLogVerbosity;
		bool	fScreenLoggingEnabled;
		bool	fBreakOnAssert;
		UInt32	fIPAddress;
		SInt32	fHistoryIntervalInSecs;
		UInt32	fBufferSeconds;
		bool	fAutoRestart;
		UInt32	fTBUpdateTimeInSecs;
		SInt64	fDelayTimeInMilSecs;
		UInt32	fBucketSize;
		UInt32	fABUpdateTimeInSecs;
		UInt32	fSafePlayDurationInSecs;
		StrPtrLen fSDPURL;
		StrPtrLen fAdminEmail;
		UInt16 	fPortOverride;
		bool	fDisableTimeout;
		UInt32	fMaxAdvanceTimeInMilSecs;
		StrPtrLen fWebStatsURL;
		UInt32	fRemoteStatsAddr;
		UInt16	fRemoteStatsPort;
		bool	fLogTimestamps;
		bool	fAppendRandomOffsets;
		bool	fKeyFramesOnly;
		char		fRTSPTimeoutBuf[20];
		StrPtrLen 	fRTSPTimeoutString;
		UInt32	fThinProbeIntSecs;
		bool	fNoBFrames;
		UInt32 fLossThinTolerance;
		UInt32 fNumLossesToThin;
		UInt32 fLossThickTolerance;
		UInt32 fLossesToThick;
		UInt32 fWorsesToThin;
		UInt32 fRealRTSPTimeoutInSecs;
		bool	fFlowControlLogEnabled;
				
		//Helper functions for RereadPreferences()
		void ProcessStringPref(char* inBuffer, StrPtrLen* inPref, char* inDefault);
		void ProcessBooleanPref(char* inBuffer, bool* inPref, bool inDefault);
		void ProcessUIntPref(char* inBuffer, UInt32* inPref, UInt32 inDefault);
		void ProcessSIntPref(char* inBuffer, SInt32* inPref, SInt32 inDefault);

		char*	GetStringPreference(char* inPref);
		
		//There needs to be separate storage of defaults from standard prefs, because
		//if user removes some pref from the file and asks server to reread, we need
		//to go back to a default.
		
		static SInt32 	sDefaultRTSPTimeout;
		static SInt32 	sDefaultRTPTimeout;
		static SInt32 	sDefaultMaximumConnections;
		static char*  	sDefaultMoviesFolder;
		static bool 	sDefaultTransferLogEnabled;
		static char*	sDefaultTransferLogDir;
		static char*	sDefaultTransferLogName;
		static SInt32	sDefaultTransferLogSize;
		static SInt32	sDefaultTransferRollInterval;
		static bool		sDefaultErrorLogEnabled;
		static char*	sDefaultErrorLogDir;
		static char*	sDefaultErrorLogName;
		static SInt32	sDefaultErrorLogSize;
		static SInt32	sDefaultErrorRollInterval;
		static UInt32	sDefaultErrorLogVerbosity;
		static bool		sDefaultScreenLoggingEnabled;
		static bool		sDefaultBreakOnAssert;
		static UInt32	sDefaultIPAddress;
		static SInt32	sDefaultHistoryInterval;
		static SInt32	sDefaultMaxBandwidth;
		static UInt32	sDefaultBufferSeconds;
		static bool		sDefaultAutoRestart;
		static UInt32	sDefaultTBUpdateTime;
		static SInt64	sDefaultDelayTime;
		static UInt32	sDefaultBucketSize;
		static UInt32	sDefaultABUpdateTime;
		static UInt32	sDefaultSafePlayDuration;
		static char*	sDefaultSDPURL;
		static char*	sDefaultAdminEmail;
		static bool 	sDefaultDisableTimeout;
		static UInt32 	sDefaultMaxAdvanceTimeInMilSecs;
		static char* 	sDefaultWebStatsURL;
		static UInt32	sDefaultLossThinTolerance;
		static UInt32	sDefaultNumLossesToThin;
		static UInt32	sDefaultLossThickTolerance;
		static UInt32	sDefaultLossesToThick;
		static UInt32	sDefaultWorsesToThin;
		static UInt32	sDefaultRealRTSPTimeoutInSecs;
		static char*	sDefaultUsersMovieFolder;
		static bool		sDefaultFlowControlLogEnabled;
		
		static char* 	sEnabledString;
		static char* 	sPrefsKeyStrings[];
};
#endif //__RTSPPREFS_H__
