/*
** Copyright (c) Massachusetts Institute of Technology 1994, 1995, 1996.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/

/*
** ltoken.c
*/

# include "lclintMacros.nf"
# include "llbasic.h"
# include "llgrammar.h"

/* 
** Place to store the \keyword (\forall ...) tokens.  These tokens will	    
** have to be modified when the extensionChar ("\") changes. 		    
** set in LCLScanLineInit of lclscanline.c or in scanline.c 
*/

ltoken ltoken_forall;
ltoken ltoken_exists;
ltoken ltoken_true;
ltoken ltoken_false;
ltoken ltoken_not;
ltoken ltoken_and;
ltoken ltoken_or;
ltoken ltoken_implies;
ltoken ltoken_eq;
ltoken ltoken_neq;
ltoken ltoken_equals;
ltoken ltoken_eqsep;
ltoken ltoken_select;
ltoken ltoken_open;
ltoken ltoken_sep;
ltoken ltoken_close;
ltoken ltoken_id;
ltoken ltoken_arrow;
ltoken ltoken_marker;
ltoken ltoken_pre;
ltoken ltoken_post;
ltoken ltoken_comment;
ltoken ltoken_any;
ltoken ltoken_compose;
ltoken ltoken_if;
ltoken ltoken_result;
ltoken ltoken_typename;
ltoken ltoken_bool;
ltoken ltoken_farrow;
ltoken ltoken_lbracked;
ltoken ltoken_rbracket;

static /*@notnull@*/ ltoken ltoken_new (void)
{
  ltoken tok = (ltoken) dmalloc (sizeof (*tok));

  tok->code = NOTTOKEN;
  tok->col = 0;
  tok->line = 0;
  tok->text = lsymbol_undefined;
  tok->fname = lsymbol_undefined;
  tok->rawText = lsymbol_undefined;
  tok->defined = FALSE;
  tok->hasSyn = FALSE;

  return tok;
}

ltoken ltoken_create (unsigned int code, lsymbol text)
{
  ltoken tok = ltoken_new ();

  tok->code = code;
  tok->text = text;

  DPRINTF (("create: %s", ltoken_unparse (tok)));
  return tok;
}

ltoken ltoken_createRaw (unsigned int code, lsymbol text)
{
  ltoken tok = ltoken_new ();

  tok->code = code;
  tok->rawText = text;

  DPRINTF (("create: %s", ltoken_unparse (tok)));
  return tok;
}

ltoken ltoken_createType (unsigned int code, SimpleIdCode idtype, lsymbol text)
{
  ltoken tok = ltoken_new ();

  /* no...what's the real llassert (code == simpleId); */

  tok->code = code;
  tok->idtype = idtype;
  tok->text = text;

  DPRINTF (("create type: %s", ltoken_unparse (tok)));
  return tok;
}

ltoken ltoken_createFull (unsigned int code, lsymbol text, 
			      cstring file, unsigned int line, 
			      unsigned int col)
{
  ltoken tok = (ltoken) dmalloc (sizeof (*tok));

  tok->code = code;
  tok->text = text;
  tok->fname = lsymbol_fromString (file);
  tok->line = line;
  tok->col = col;
  tok->rawText = lsymbol_undefined;
  tok->defined = FALSE;
  tok->hasSyn = FALSE;

  DPRINTF (("full token: %s", ltoken_unparse (tok)));
  return tok;
}

static /*@only@*/ cstring LCLTokenCode_unparseCodeName (unsigned int t)
{
  switch (t)
    {
    case NOTTOKEN:         return cstring_makeLiteral ("*** NOTTOKEN ***");      
    case quantifierSym:    return cstring_makeLiteral ("QUANTIFIERSYM");      
    case logicalOp:        return cstring_makeLiteral ("LOGICALOP");
    case selectSym:        return cstring_makeLiteral ("SELECTSYM");    
    case openSym:          return cstring_makeLiteral ("OPENSYM");
    case preSym:           return cstring_makeLiteral ("\\pre");
    case postSym:          return cstring_makeLiteral ("\\post");
    case anySym:           return cstring_makeLiteral ("\\any");
    case sepSym:           return cstring_makeLiteral ("SEPSYM");
    case closeSym:         return cstring_makeLiteral ("CLOSESYM");
    case simpleId:         return cstring_makeLiteral ("simpleId");
    case TYPEDEF_NAME:     return cstring_makeLiteral ("TYPEDEF_NAME");
    case mapSym:           return cstring_makeLiteral ("MAPSYM");
    case markerSym:        return cstring_makeLiteral ("MARKERSYM");
    case commentSym:       return cstring_makeLiteral ("COMMENTSYM");
    case simpleOp:         return cstring_makeLiteral ("SIMPLEOP");
    case COLON:            return cstring_makeLiteral ("COLON");
    case COMMA:            return cstring_makeLiteral ("COMMA");
    case LLT_EQUALS:           return cstring_makeLiteral ("LLT_EQUALS");
    case LBRACE:           return cstring_makeLiteral ("LBRACE");
    case LBRACKET:         return cstring_makeLiteral ("LBRACKET");
    case LPAR:             return cstring_makeLiteral ("LPAR");
    case QUOTE:            return cstring_makeLiteral ("QUOTE");
    case RBRACE:           return cstring_makeLiteral ("RBRACE");
    case RBRACKET:         return cstring_makeLiteral ("RBRACKET");
    case RPAR:             return cstring_makeLiteral ("RPAR");
    case SEMI:             return cstring_makeLiteral ("SEMI");
    case VERTICALBAR:      return cstring_makeLiteral ("VERTICALBAR");
    case eqOp:             return cstring_makeLiteral ("EQOP");
    case MULOP:            return cstring_makeLiteral ("MULOP");
    case WHITESPACE:       return cstring_makeLiteral ("WHITESPACE,");
    case LEOFTOKEN:        return cstring_makeLiteral ("EOFTOKEN");
    case LLT_EOL:         return cstring_makeLiteral ("LLT_EOL");
    case CCHAR:            return cstring_makeLiteral ("CCHAR");
    case CFLOAT:           return cstring_makeLiteral ("CFLOAT");
    case CINTEGER:         return cstring_makeLiteral ("CINTEGER");
    case LCSTRING:         return cstring_makeLiteral ("CSTRING");
    case ALL:              return cstring_makeLiteral ("allTOKEN");
    case ANYTHING:         return cstring_makeLiteral ("anythingTOKEN");
    case BE:               return cstring_makeLiteral ("beTOKEN");
    case CONSTANT:         return cstring_makeLiteral ("constantTOKEN");
    case LLT_ELSE:         return cstring_makeLiteral ("elseTOKEN");
    case LLT_ENSURES:      return cstring_makeLiteral ("ensuresTOKEN");
    case IF:               return cstring_makeLiteral ("ifTOKEN");
    case IMMUTABLE:        return cstring_makeLiteral ("immutableTOKEN");
    case OBJ:              return cstring_makeLiteral ("objTOKEN");
    case IMPORTS:          return cstring_makeLiteral ("importsTOKEN");
    case CONSTRAINT:       return cstring_makeLiteral ("constraintTOKEN");
    case LET:              return cstring_makeLiteral ("letTOKEN");
    case MODIFIES:         return cstring_makeLiteral ("modifiesTOKEN");
    case CLAIMS:           return cstring_makeLiteral ("claimsTOKEN");
    case MUTABLE:          return cstring_makeLiteral ("mutableTOKEN");
    case FRESH:            return cstring_makeLiteral ("freshTOKEN");
    case NOTHING:          return cstring_makeLiteral ("nothingTOKEN");
    case PRIVATE:          return cstring_makeLiteral ("privateTOKEN");
    case SPEC:             return cstring_makeLiteral ("specTOKEN");
    case REQUIRES:         return cstring_makeLiteral ("requiresTOKEN");
    case BODY:             return cstring_makeLiteral ("bodyTOKEN");
    case RESULT:           return cstring_makeLiteral ("resultTOKEN");
    case SIZEOF:           return cstring_makeLiteral ("sizeofTOKEN");
    case THEN:             return cstring_makeLiteral ("thenTOKEN");
    case TYPE:             return cstring_makeLiteral ("typeTOKEN");
    case TYPEDEF:          return cstring_makeLiteral ("typedefTOKEN");
    case UNCHANGED:        return cstring_makeLiteral ("unchangedTOKEN");
    case USES:             return cstring_makeLiteral ("usesTOKEN");
    case CHAR:             return cstring_makeLiteral ("charTOKEN");
    case CONST:            return cstring_makeLiteral ("constTOKEN");
    case DOUBLE:           return cstring_makeLiteral ("doubleTOKEN");
    case LLT_ENUM:         return cstring_makeLiteral ("enumTOKEN");
    case FLOAT:            return cstring_makeLiteral ("floatTOKEN");
    case INT:              return cstring_makeLiteral ("intTOKEN");
    case LONG:             return cstring_makeLiteral ("longTOKEN");
    case SHORT:            return cstring_makeLiteral ("shortTOKEN");
    case STRUCT:           return cstring_makeLiteral ("structTOKEN");
    case LLT_SIGNED:       return cstring_makeLiteral ("signedTOKEN");
    case UNION:            return cstring_makeLiteral ("unionTOKEN");
    case UNKNOWN:          return cstring_makeLiteral ("unknownTOKEN");
    case UNSIGNED:         return cstring_makeLiteral ("unsignedTOKEN");
    case VOID:             return cstring_makeLiteral ("voidTOKEN");
    case VOLATILE:         return cstring_makeLiteral ("volatileTOKEN");
    case TELIPSIS:         return cstring_makeLiteral ("elipsisTOKEN");
    case ITER:             return cstring_makeLiteral ("iterTOKEN");
    case YIELD:            return cstring_makeLiteral ("yieldTOKEN");
    default:               return cstring_makeLiteral ("*** invalid token code ***"); 
    } /* end switch */
}

cstring ltoken_unparseCodeName (ltoken tok)
{
  return LCLTokenCode_unparseCodeName (ltoken_getCode (tok));
}

/*@observer@*/ cstring ltoken_unparse (ltoken s)
{
  if (ltoken_isValid (s))
    {
      return (lsymbol_toString (s->text));
    }
  else
    {
      return cstring_undefined;
    }
}

ltoken ltoken_copy (ltoken tok)
{
  if (ltoken_isValid (tok))
    {
      ltoken ret = (ltoken) dmalloc (sizeof (*ret));
      
      ret->code = tok->code;
      ret->text = tok->text;
      ret->fname = tok->fname;
      ret->line = tok->line;
      ret->col = tok->col;
      ret->rawText = tok->rawText;
      ret->defined = tok->defined;
      ret->hasSyn = tok->hasSyn;
      ret->idtype = tok->idtype;
      ret->intfield = tok->intfield;
      
      return ret;
    }
  else
    {
      return ltoken_undefined;
    }
}

lsymbol ltoken_getRawText (ltoken tok)
{
  if (ltoken_isValid (tok))
    {
      lsymbol ret = tok->rawText;

      if (lsymbol_isUndefined (ret))
	{
	  ret = tok->text;
	}
      
      return ret;
    }
  else
    {
      return lsymbol_undefined;
    }
}

/*@only@*/ cstring ltoken_unparseLoc (ltoken t)
{
  if (ltoken_getCode (t) != NOTTOKEN)
    {
      if (context_getFlag (FLG_SHOWCOL))
	{
	  return (message ("%s:%u,%u", ltoken_fileName (t), 
			   ltoken_getLine (t), ltoken_getCol (t)));
	}
      else
	{
	  return (message ("%s:%u", ltoken_fileName (t), ltoken_getLine (t)));
	}
    }
  else
    {
      llcontbug (cstring_makeLiteral ("ltoken_unparseLoc: not token"));
      return cstring_undefined; 
    }
}

void ltoken_markOwned (/*@owned@*/ ltoken tok)
{
  DPRINTF (("ltokenOwned: %s", ltoken_unparse (tok)));
  sfreeEventually (tok);
}

void ltoken_free (/*@only@*/ ltoken tok)
{
  sfree (tok); 
}
