#include "showimg.h"
#include <qpainter.h>
#include <qapp.h>


static const char *errorText = "Couldn't load image.";

/*
  This function loads an image from a file and resizes the widget to
  exactly fit the image size. If the file was not found or the image
  format was unknown it will resize the widget to fit the errorText
  message (see above) displayed in the current font.
  Returns TRUE if the image was successfully loaded.
*/

bool CuteWidget::loadImage( const char *fileName )
{
    bool success = FALSE;
    QApplication::setOverrideCursor( waitCursor ); // this might take time
    if ( pm.load( fileName ) ) {		// try to load the image
	setCaption( fileName );			// set window caption
	pmScaled = pm;				// no need to scale because
	resize( pm.size() );			// we resize to image size
	success = TRUE;				// load successful
    } else {
	pm.resize(0,0);				// couldn't load image
	QFontMetrics fm = fontMetrics();	// resize to fit errorText
	resize( fm.width(errorText) + 10, fm.height() + 10 );// 10 pixel border
    }
    repaint();					// show image in widget
    QApplication::restoreOverrideCursor();	// restore original cursor
    return success;				// TRUE if loaded OK
}

/*
  This functions scales the pixmap in the member variable "pm" to fit the
  widget size and  puts the resulting pixmap in the member variable "pmScaled".
*/

void CuteWidget::scale()
{
    QApplication::setOverrideCursor( waitCursor ); // this might take time
    if ( size() == pm.size() ) {		// no need to scale if widget
	pmScaled = pm;				// size equals pixmap size
    } else {
	QWMatrix m;				// transformation matrix
	m.scale(((double)width())/pm.width(),	// define scale factors
		((double)height())/pm.height());
	pmScaled = pm.xForm( m );		// create scaled pixmap
    }
    QApplication::restoreOverrideCursor();	// restore original cursor
}

/*
  The resize event handler, if a valid pixmap was loaded it will call
  scale() to fit the pixmap to the new widget size.
*/

void CuteWidget::resizeEvent( QResizeEvent *e )
{
    if ( pm.size() == QSize( 0, 0 ) )		// we couldn't load the image
	return;
    if ( e->size() != pmScaled.size() )		// if new size
	scale();				// scale pmScaled to window
}

/*
  Draws the portion of the scaled pixmap that needs to be updated or prints
  an error message if no legal pixmap has been loaded.
  Note that this example never needs to use a QPainter, this is only
  true for trivial widgets, and rarely happens in a real-life widget.
*/

void CuteWidget::paintEvent( QPaintEvent *e )
{
    if ( pm.size() != QSize( 0, 0 ) ) {		// is an image loaded?
	bitBlt( this, e->rect().topLeft(), &pmScaled, e->rect() );
    } else {					// no image, display errorText
	QFontMetrics fm = fontMetrics();	// metrics of widget font
	int xpos = (width() - fm.width(errorText))/2;	  // center x start pos
	int ypos = (height() - fm.height())/2;	// center y start pos
	drawText( xpos, ypos + fm.ascent(), errorText );  // baseline pos
    }
}
