------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             E X P _ P A K D                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.40 $                             --
--                                                                          --
--   Copyright (C) 1992,1993,1994,1995,1996 Free Software Foundation, Inc.  --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Exp_Ch4;  use Exp_Ch4;
with Exp_Util; use Exp_Util;
with Freeze;   use Freeze;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Rtsfind;  use Rtsfind;
with Sem;      use Sem;
with Sem_Eval; use Sem_Eval;
with Sem_Res;  use Sem_Res;
with Sem_Util; use Sem_Util;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Stand;    use Stand;
with Tbuild;   use Tbuild;
with Ttypes;   use Ttypes;

package body Exp_Pakd is

   ---------------------------
   -- Endian Considerations --
   ---------------------------

   --  As described in the specification, bit numbering in a packed array
   --  is consistent with bit numbering in a record representation clause,
   --  and hence dependent on the endianness of the machine:

   --    For little-endian machines, element zero is at the right hand end
   --    (low order end) of a bit field.

   --    For big-endian machines, element zero is at the left hand end
   --    (high order end) of a bit field.

   --  The shifts that are used to right justify a field therefore differ
   --  in the two cases. For the little-endian case, we can simply use the
   --  bit number (i.e. the element number * element size) as the count for
   --  a right shift. For the big-endian case, we have to subtract the shift
   --  count from an appropriate constant to use in the right shift. We use
   --  rotates instead of shifts (which is necessary in the store case to
   --  preserve other fields), and we expect that the backend will be able
   --  to change the right rotate into a left rotate, avoiding the subtract,
   --  if the architecture provides such an instruction.

   -----------------------
   -- Local Subprograms --
   -----------------------

   procedure Convert_To_PAT_Type (Aexp : Node_Id);
   --  Given an expression of a packed array type, builds a corresponding
   --  expression whose type is the implementation type used to represent
   --  the packed array. Aexp is analyzed on entry, and on return Aexp
   --  is rewritten (using Rewrite_Substitute_Tree) by this expression.

   procedure Setup_Packed_Array_Reference
     (N      : Node_Id;
      Cmask  : out Uint;
      Obj    : out Node_Id;
      Shift  : out Node_Id);
   --  This procedure performs common processing on the N_Indexed_Component
   --  parameter given as N, whose prefix is a reference to a packed array.
   --  On return:

   --    Obj is the object containing the desired bit field. It is of type
   --    Unsigned or Long_Long_Unsigned, and is either the entire value,
   --    for the small static case, or the proper selected bute from the
   --    array in the large or dynamic case. This node is analyzed and
   --    resolved on return.
   --
   --    Shift is a node representing the shift count to be used in the
   --    rotate right instruction that positions the field for access.
   --    This node is analyzed and resolved on return.
   --
   --    Cmask is a mask corresponding to the width of the component field.
   --    It is 2#1#, 2#11#, or 2#1111# for a component size of 1,2, or 4.
   --
   --  Note: in some cases the call to this routine may generate actions
   --  (for handling multi-use references and the generation of the packed
   --  array type on the fly). Such actions are inserted into the tree
   --  directly using Insert_Action.

   -------------------------
   -- Convert_To_PAT_Type --
   -------------------------

   --  The PAT is always obtained from the actual subtype

   procedure Convert_To_PAT_Type (Aexp : Entity_Id) is
      Act_ST : Entity_Id;
      PAT    : Entity_Id;
      Decl   : Node_Id;
      Esiz   : Uint;

   begin
      Convert_To_Actual_Subtype (Aexp);
      Act_ST := Etype (Aexp);
      Expand_Packed_Array_Type (Act_ST, PAT, Decl, Esiz);

      if Present (Decl) then
         Insert_Action (Aexp, Decl);
         Set_Esize (PAT, Esiz);
         Freeze_Itype (PAT, Aexp);
      end if;

      --  Replace the reference with an unchecked conversion from the
      --  actual subtype to the appropriate packed array type.

      Rewrite_Substitute_Tree (Aexp,
        Unchecked_Convert_To (PAT, Relocate_Node (Aexp)));

      Analyze_And_Resolve (Aexp, PAT);
   end Convert_To_PAT_Type;

   ------------------------------
   -- Expand_Packed_Array_Type --
   ------------------------------

   procedure Expand_Packed_Array_Type
     (Typ  : Entity_Id;
      PAT  : out Entity_Id;

      Decl : out Node_Id;
      Esiz : out Uint)
   is
      Loc      : constant Source_Ptr := Sloc (Typ);
      Base     : constant Entity_Id  := Base_Type (Typ);
      Frst     : constant Entity_Id  := First_Subtype (Base);
      Ctyp     : constant Entity_Id  := Component_Type (Typ);
      Styp     : constant Entity_Id  := Etype (First_Index (Typ));
      Lo_Bound : constant Node_Id    := Type_Low_Bound (Styp);
      Hi_Bound : constant Node_Id    := Type_High_Bound (Styp);
      Csize    : constant Uint       := Component_Size (Typ);
      Len_Dim  : Node_Id;
      Len_Expr : Node_Id;
      Len_Bits : Uint;
      Bits_U1  : Node_Id;
      PAT_High : Node_Id;
      Btyp     : Entity_Id;
      Lit      : Node_Id;

   begin
      --  If we already have a packed array type, just return it

      if Present (Packed_Array_Type (Typ)) then
         Decl := Empty;
         PAT := Packed_Array_Type (Typ);
         return;

      --  For the case where the first subtype is constrained, we build
      --  a modular type if the bounds are appropriate (static and small
      --  enough). The code for this is below. It is unusual but possible
      --  to have subtypes of such a type (happens for example in the
      --  case of aggregates). We simply use the same representation for
      --  such subtypes.

      elsif Present (Packed_Array_Type (Frst))
        and then Is_Integer_Type (Packed_Array_Type (Frst))
      then
         PAT := Packed_Array_Type (Frst);
         Set_Packed_Array_Type (Typ, PAT);
         Decl := Empty;
         return;

      --  For unconstrained array types, we simply use Packed_Bytes, and we
      --  do not need to construct a special packed array type.

      elsif not Is_Constrained (Typ) then
         PAT := RTE (RE_Packed_Bytes);
         Set_Packed_Array_Type (Typ, PAT);
         Decl := Empty;
         return;
      end if;

      --  Preset result type size to zero (unknown), unless a specific
      --  size was given for the original array type. This size really
      --  belongs to the packed array type.

      if Esize (Typ) /= Uint_0 then
         Esiz := Esize (Typ);
         Set_Esize (Typ, Uint_0);
      else
         Esiz := Uint_0;
      end if;

      --  The name of the packed array subtype is

      --    tttPx

      --  where x is 1,2 or 4 for the component size in bits and ttt is
      --  the name of the parent packed type.

      PAT :=
        Make_Defining_Identifier (Loc,
          Chars => New_External_Name (Chars (Typ), 'P', UI_To_Int (Csize)));

      Set_Packed_Array_Type (Typ, PAT);

      --  Build an expression for the length of the array in bits.
      --  This is the product of the length of each of the dimensions

      for J in 1 .. Number_Dimensions (Typ) loop
         Len_Dim :=
           Make_Attribute_Reference (Loc,
             Attribute_Name => Name_Length,
             Prefix         => New_Occurrence_Of (Typ, Loc),
             Expressions    => New_List (
               Make_Integer_Literal (Loc, UI_From_Int (J))));

         if J = 1 then
            Len_Expr := Len_Dim;

         else
            Len_Expr :=
              Make_Op_Multiply (Loc,
                Left_Opnd  => Len_Expr,
                Right_Opnd => Len_Dim);
         end if;
      end loop;

      --  Temporarily attach the length expression to the tree and analyze
      --  and resolve it, so that we can test its value. We assume that the
      --  total length fits in type Integer.

      Set_Parent (Len_Expr, Typ);
      Analyze_And_Resolve (Len_Expr, Standard_Integer);

      --  Use a modular type if possible. We can do this if we are a
      --  constrained first subtype, with static bounds, and the length
      --  in bits is in the range 1 .. Word Size.

      if Typ = Frst
        and then Compile_Time_Known_Value (Len_Expr)
      then
         Len_Bits := UI_Max (Uint_1, (Expr_Value (Len_Expr) * Csize));

         if Len_Bits <= System_Word_Size then

            --  We can use the modular type, it has the form:

            --    subtype tttPn is btyp
            --      range 0 .. 2 ** (Esize (Typ) * Csize) - 1;

            --  Here Siz is 1, 2 or 4, as computed above, and btyp is either
            --  Unsigned or Long_Long_Unsigned depending on the length.

            if Len_Bits <= Standard_Integer_Size then
               Btyp := RTE (RE_Unsigned);
            else
               Btyp := RTE (RE_Long_Long_Unsigned);
            end if;

            Lit := Make_Integer_Literal (Loc, Intval => 2 ** Len_Bits - 1);
            Set_Print_In_Hex (Lit);

            Decl :=
              Make_Subtype_Declaration (Loc,
                Defining_Identifier => PAT,
                  Subtype_Indication =>
                    Make_Subtype_Indication (Loc,
                      Subtype_Mark => New_Occurrence_Of (Btyp, Loc),

                      Constraint =>
                        Make_Range_Constraint (Loc,
                          Range_Expression =>
                            Make_Range (Loc,
                              Low_Bound =>
                                Make_Integer_Literal (Loc, Intval => Uint_0),
                              High_Bound => Lit))));

            if Esiz = Uint_0 then
               Esiz := Len_Bits;
            end if;

            return;
         end if;
      end if;

      --  Could not use a modular type, for all other cases, we build
      --  a packed array subtype:

      --    subtype tttPn is
      --      System.Packed_Bytes (0 .. (Bits + 7) / 8 - 1);

      --  Bits is the length of the array in bits.

      Bits_U1 :=
        Make_Op_Add (Loc,
          Left_Opnd =>
            Make_Op_Multiply (Loc,
              Left_Opnd  =>
                Make_Integer_Literal (Loc, Csize),
              Right_Opnd => Len_Expr),

          Right_Opnd =>
            Make_Integer_Literal (Loc, Uint_7));

      Set_Paren_Count (Bits_U1, 1);

      PAT_High :=
        Make_Op_Subtract (Loc,
          Left_Opnd =>
            Make_Op_Divide (Loc,
              Left_Opnd => Bits_U1,
              Right_Opnd => Make_Integer_Literal (Loc, Uint_8)),
          Right_Opnd => Make_Integer_Literal (Loc, Uint_1));

      Decl :=
        Make_Subtype_Declaration (Loc,
          Defining_Identifier => PAT,
            Subtype_Indication =>
              Make_Subtype_Indication (Loc,
                Subtype_Mark => New_Occurrence_Of (RTE (RE_Packed_Bytes), Loc),
                Constraint =>

                  Make_Index_Or_Discriminant_Constraint (Loc,
                    Constraints => New_List (
                      Make_Range (Loc,
                        Low_Bound =>
                          Make_Integer_Literal (Loc,
                            Intval => Uint_0),
                        High_Bound => PAT_High)))));

   end Expand_Packed_Array_Type;

   ------------------------------------
   -- Expand_Packed_Boolean_Operator --
   ------------------------------------

   --  This routine expands "a op b" for the packed cases

   procedure Expand_Packed_Boolean_Operator (N : Node_Id) is
      Loc : constant Source_Ptr := Sloc (N);
      Typ : constant Entity_Id  := Etype (N);
      L   : constant Node_Id    := Relocate_Node (Left_Opnd  (N));
      R   : constant Node_Id    := Relocate_Node (Right_Opnd (N));

      Rtyp : Entity_Id;
      PAT  : Entity_Id;

   begin
      Convert_To_Actual_Subtype (L);
      Convert_To_Actual_Subtype (R);
      Rtyp := Etype (L);

      Convert_To_PAT_Type (L);
      Convert_To_PAT_Type (R);
      PAT := Etype (L);

      --  For the modular case, we expand a op b into

      --    rtyp!(pat!(a) op pat!(b))

      --  where rtyp is the Etype of the left operand. Note that we do not
      --  convert to the base type, since this would be unconstrained, and
      --  hence not have a corresponding packed array type set.

      if Is_Modular_Integer_Type (PAT) then
         declare
            P : Node_Id;

         begin
            if Nkind (N) = N_Op_And then
               P := Make_Op_And (Loc, L, R);

            elsif Nkind (N) = N_Op_Or then
               P := Make_Op_Or  (Loc, L, R);

            else -- Nkind (N) = N_Op_Xor
               P := Make_Op_Xor (Loc, L, R);
            end if;

            Rewrite_Substitute_Tree (N, Unchecked_Convert_To (Rtyp, P));
         end;

      --  For the non-modular case, we use Exp_Ch4.Make_Boolean_Array to build
      --  a function that does the necessary loop of operations on the array,
      --  and then replace the operation with a call to this function, doing
      --  the necessary unchecked conversions. What we are relying on here is
      --  that the operation for a byte containing a single boolean bit is the
      --  same as the operation for a byte containing 8 boolean bits.

      --    rtyp!(func (pat!(a), pat!(b)))

      --  where rtyp is the actual subtype of the left operand

      else
         declare
            Func_Body : constant Node_Id   := Make_Boolean_Array_Op (PAT, N);
            Func_Name : constant Entity_Id := Defining_Unit_Name
                                                (Specification (Func_Body));

         begin
            Insert_Action (N, Func_Body);

            Rewrite_Substitute_Tree (N,
              Unchecked_Convert_To (Rtyp,
                Make_Function_Call (Loc,
                  Name => New_Reference_To (Func_Name, Loc),
                  Parameter_Associations => New_List (L, R))));
         end;
      end if;

      Analyze_And_Resolve (N, Typ, Suppress => All_Checks);
   end Expand_Packed_Boolean_Operator;

   -------------------------------
   -- Expand_Packed_Element_Get --
   -------------------------------

   procedure Expand_Packed_Element_Get (N : Node_Id) is
      Loc   : constant Source_Ptr := Sloc (N);
      Ctyp  : constant Entity_Id  := Component_Type (Etype (Prefix (N)));
      Obj   : Node_Id;
      Shift : Node_Id;
      Cmask : Uint;
      Lit   : Node_Id;

   begin
      Setup_Packed_Array_Reference (N, Cmask, Obj, Shift);
      Lit   := Make_Integer_Literal (Loc, Cmask);
      Set_Print_In_Hex (Lit);

      --  We generate a shift right to position the field, followed by a
      --  masking operation to extract the bit field, and we finally do an
      --  unchecked conversion to convert the result to the required target.

      Rewrite_Substitute_Tree (N,
        Unchecked_Convert_To (Ctyp,
          Make_Op_And (Loc,
            Left_Opnd =>
              Make_Op_Shift_Right (Loc,
                Left_Opnd => Obj,
                Right_Opnd => Shift),
            Right_Opnd => Lit)));

      Analyze_And_Resolve (N, Ctyp, Suppress => All_Checks);
   end Expand_Packed_Element_Get;

   -------------------------------
   -- Expand_Packed_Element_Set --
   -------------------------------

   procedure Expand_Packed_Element_Set (N : Node_Id) is
      Loc    : constant Source_Ptr := Sloc (N);
      Lhs    : constant Node_Id    := Name (N);
      Rhs    : constant Node_Id    := Expression (N);
      Ctyp   : constant Entity_Id  := Etype (Rhs);
      Obj    : Node_Id;
      Shift  : Node_Id;
      Cmask  : Uint;
      Or_Rhs : Node_Id;
      Lit1   : Node_Id;
      Lit2   : Node_Id;

   begin
      Setup_Packed_Array_Reference (Lhs, Cmask, Obj, Shift);
      Lit1  := Make_Integer_Literal (Loc, Cmask);
      Lit2  := Make_Integer_Literal (Loc, Cmask);
      Set_Print_In_Hex (Lit1);
      Set_Print_In_Hex (Lit2);

      --  The statement to be generated is:

      --    Obj := ((((Obj >> Shift) or Cmask) xor Cmask) or Or_Rhs) << Shift)

      --  where >> and << are rotate right and left respectively, and Cmask is
      --  the mask returned by the call to Setup_Packed_Array_Reference.

      --  The right hand side, Or_Rhs must be of Etype (Obj). A special
      --  case arises if what we have now is a Val attribute reference whose
      --  expression type is Etype (Obj). This happens for assignments of
      --  fields from the same array. In this case we get the required right
      --  hand side by simply removing the inner attribute reference.

      if Nkind (Rhs) = N_Attribute_Reference
        and then Attribute_Name (Rhs) = Name_Val
        and then Etype (First (Expressions (Rhs))) = Etype (Obj)
      then
         Or_Rhs := Relocate_Node (First (Expressions (Rhs)));

      --  Otherwise we get the expression to the right type by using
      --  an unchecked conversion to Etype (Obj).

      else
         Or_Rhs := Unchecked_Convert_To (Etype (Obj), Rhs);
      end if;

      --  Now do the rewrite

      Rewrite_Substitute_Tree (N,
        Make_Assignment_Statement (Loc,
          Name => Duplicate_Subexpr (Obj, True),
          Expression =>
            Make_Op_Rotate_Left (Loc,

              Left_Opnd =>
                Make_Op_Or (Loc,

                  Left_Opnd =>
                    Make_Op_Xor (Loc,

                      Left_Opnd =>
                        Make_Op_Or (Loc,
                          Left_Opnd =>
                            Make_Op_Rotate_Right (Loc,
                              Left_Opnd  => Duplicate_Subexpr (Obj, True),
                              Right_Opnd => Duplicate_Subexpr (Shift)),
                          Right_Opnd => Lit1),

                      Right_Opnd => Lit2),

                  Right_Opnd => Or_Rhs),

              Right_Opnd => Duplicate_Subexpr (Shift))));

      Analyze (N, Suppress => All_Checks);

   end Expand_Packed_Element_Set;

   -----------------------
   -- Expand_Packed_Not --
   -----------------------

   --  Handles expansion of "not" on packed array types

   procedure Expand_Packed_Not (N : Node_Id) is
      Loc  : constant Source_Ptr := Sloc (N);
      Typ  : constant Entity_Id  := Etype (N);
      Opnd : constant Node_Id    := Relocate_Node (Right_Opnd (N));

      Rtyp : Entity_Id;
      PAT  : Entity_Id;
      Lit  : Node_Id;

   begin
      Convert_To_Actual_Subtype (Opnd);
      Rtyp := Etype (Opnd);
      Convert_To_PAT_Type (Opnd);
      PAT := Etype (Opnd);

      --  For the case where the packed array type is a modular type,
      --  not A expands simply into:

      --     rtyp!(PAT!(A) xor mask)

      --  where PAT is the packed array type, and mask is a mask of all
      --  one bits of length equal to the size of this packed type and
      --  rtyp is the actual subtype of the operand

      Lit := Make_Integer_Literal (Loc, 2 ** Esize (PAT) - 1);
      Set_Print_In_Hex (Lit);

      if not Is_Array_Type (PAT) then
         Rewrite_Substitute_Tree (N,
           Unchecked_Convert_To (Rtyp,
             Make_Op_Xor (Loc,
               Left_Opnd  => Opnd,
               Right_Opnd => Lit)));

      --  For the array case, we can't use the same approach as we used for
      --  logical operations (see Expand_Packed_Boolean_Operator), because
      --  the last byte must be handled specially (it cannot be blindly
      --  inverted, since the unused bits must be left alone). Instead we
      --  call a library function:

      --    System.Bitops.Bit_Not (Opnd, Len);

      --  where Opnd is the operand converted to Packed_Bytes, and the second
      --  argument is the actual operand length in bits.

      else
         Rewrite_Substitute_Tree (N,
           Unchecked_Convert_To (Rtyp,
             Make_Function_Call (Loc,
               Name => New_Occurrence_Of (RTE (RE_Bit_Not), Loc),
               Parameter_Associations => New_List (
                 Opnd,
                 Make_Attribute_Reference (Loc,
                   Prefix =>
                     New_Occurrence_Of (Etype (First_Index (Typ)), Loc),
                   Attribute_Name => Name_Range_Length)))));
      end if;

      Analyze_And_Resolve (N, Typ, Suppress => All_Checks);

   end Expand_Packed_Not;

   ---------------------------------------
   -- Needs_Packed_Array_Initialization --
   ---------------------------------------

   function Needs_Packed_Array_Initialization (T : Entity_Id) return Boolean is
   begin
      return
        Is_Array_Type (T)
          and then Present (Packed_Array_Type (T));
   end Needs_Packed_Array_Initialization;

   ---------------------------
   -- Packed_Array_Init_Val --
   ---------------------------

   function Packed_Array_Init_Val
     (Typ  : Entity_Id;
      Loc  : Source_Ptr;
      Nod  : Node_Id)
      return Node_Id
   is
      PAT  : constant Entity_Id  := Packed_Array_Type (Typ);
      Expr : Node_Id;

   begin
      --  For the modular case, the initial value is just zero

      if Is_Modular_Integer_Type (PAT) then
         Expr := Make_Integer_Literal (Loc, Uint_0);

      --  For the Packed_Bytes case, we build an aggregate

      else
         declare
            Indx_Typ : constant Entity_Id := Etype (First_Index (PAT));
            High_Bd  : constant Node_Id   := Type_High_Bound (Indx_Typ);
            Exp_List : List_Id := No_List;
            Cmp_List : List_Id := No_List;

         begin
            --  For small static case, build an explicit aggregate of zeroes
            --  Note that we know that the low bound is zero.

            if Compile_Time_Known_Value (High_Bd)
              and then Expr_Value (High_Bd) <= 15
            then
               Exp_List := New_List;

               for J in 0 .. UI_To_Int (Expr_Value (High_Bd)) loop
                  Append_To (Exp_List,
                    Make_Integer_Literal (Loc, Uint_0));
               end loop;

            --  For other cases, use an aggregate of the form (others => 0)

            else
               Cmp_List := New_List (
                 Make_Component_Association (Loc,
                   Choices    => New_List (
                     Make_Others_Choice (Loc)),
                   Expression =>
                     Make_Integer_Literal (Loc, Uint_0)));
            end if;

            Expr :=
              Make_Qualified_Expression (Loc,
                Subtype_Mark => New_Occurrence_Of (PAT, Loc),
                Expression   =>
                  Make_Aggregate (Loc,
                    Expressions            => Exp_List,
                    Component_Associations => Cmp_List));
         end;
      end if;

      Set_Parent (Expr, Nod);
      Analyze_And_Resolve (Expr, PAT);
      return Unchecked_Convert_To (Typ, Expr);
   end Packed_Array_Init_Val;

   ----------------------------------
   -- Setup_Packed_Array_Reference --
   ----------------------------------

   procedure Setup_Packed_Array_Reference
     (N      : Node_Id;
      Cmask  : out Uint;
      Obj    : out Node_Id;
      Shift  : out Node_Id)
   is
      Loc    : constant Source_Ptr := Sloc (N);
      Atyp   : Entity_Id;
      Ctyp   : Entity_Id;
      Oldsub : Node_Id;
      Newsub : Node_Id;
      Indx   : Node_Id;
      PAT    : Entity_Id;
      Styp   : Entity_Id;
      Otyp   : Entity_Id;
      Csiz   : Uint;
      Osiz   : Uint;

   begin
      Obj := Relocate_Node (Prefix (N));
      Convert_To_Actual_Subtype (Obj);
      Atyp := Etype (Obj);
      Ctyp := Component_Type (Atyp);

      Convert_To_PAT_Type (Obj);
      PAT  := Etype (Obj);

      Csiz := Component_Size (Atyp);

      Cmask := 2 ** Csiz - 1;

      if Is_Array_Type (PAT) then
         Otyp := Component_Type (PAT);
      else
         Otyp := PAT;
      end if;

      Osiz := Esize (Otyp);

      --  Get expression for the shift count

      Shift  := Empty;
      Indx   := First_Index (Atyp);
      Styp   := Etype (Indx);
      Oldsub := First (Expressions (N));

      --  Loop through dimensions

      while Present (Indx) loop
         Styp := Etype (Indx);
         Newsub := Relocate_Node (Oldsub);

         --  Get expression for the subscript value. First, if Do_Range_Check
         --  is set on a subscript, then we must do a range check against the
         --  original bounds (not the bounds of the packed array type). We do
         --  this by introducing a subtype conversion.

         if Do_Range_Check (Newsub)
           and then Etype (Newsub) /= Styp
         then
            Newsub := Convert_To (Styp, Newsub);
         end if;

         --  Now evolve the expression for the subscript. First convert
         --  the subscript to be zero based and of an integer type.

         --  If it is of an integer type now, we just subtract:

         --      Integer (subscript) - Integer (Styp'First)

         if Is_Integer_Type (Styp) then
            Newsub :=
              Make_Op_Subtract (Loc,
                Left_Opnd =>
                  Convert_To (Standard_Integer, Newsub),
                Right_Opnd =>
                  Convert_To (Standard_Integer,
                    Make_Attribute_Reference (Loc,
                      Prefix => New_Occurrence_Of (Styp, Loc),
                      Attribute_Name => Name_First)));

         --  For the enumeration case, we have to use 'Pos to get the value
         --  to work with before subtracting the lower bound.

         --    Integer (Styp'Pos (subscr)) - Integer (Styp'Pos (Styp'First));

         else
            pragma Assert (Is_Enumeration_Type (Styp));

            Newsub :=
              Make_Op_Subtract (Loc,
                Left_Opnd => Convert_To (Standard_Integer,
                  Make_Attribute_Reference (Loc,
                    Prefix => New_Occurrence_Of (Styp, Loc),
                    Attribute_Name => Name_Pos,
                    Expressions => New_List (Newsub))),

                Right_Opnd =>
                  Convert_To (Standard_Integer,
                    Make_Attribute_Reference (Loc,
                      Prefix => New_Occurrence_Of (Styp, Loc),
                      Attribute_Name => Name_Pos,
                      Expressions => New_List (
                        Make_Attribute_Reference (Loc,
                        Prefix => New_Occurrence_Of (Styp, Loc),
                        Attribute_Name => Name_First)))));
         end if;

         --  For the first subscript, we just copy that subscript value

         if No (Shift) then
            Shift := Newsub;

         --  Otherwise, we must multiply what we already have by the current
         --  stride and then add in the new value to the evolving subscript.

         else
            Set_Paren_Count (Shift, 1);

            Shift :=
              Make_Op_Add (Loc,
                Left_Opnd =>
                  Make_Op_Multiply (Loc,
                    Left_Opnd  => Shift,
                    Right_Opnd =>
                      Make_Attribute_Reference (Loc,
                        Attribute_Name => Name_Range_Length,
                        Prefix         => New_Occurrence_Of (Styp, Loc))),
                Right_Opnd => Newsub);
         end if;

         --  Move to next subscript

         Indx   := Next_Index (Indx);
         Oldsub := Next (Oldsub);
      end loop;

      --  If the component size is 2 or 4, then the subscript must be
      --  multiplied by the component size to get the shift count.

      if Csiz /= 1 then
         Shift :=
           Make_Op_Multiply (Loc,
             Left_Opnd => Make_Integer_Literal (Loc, Csiz),
             Right_Opnd => Shift);
      end if;

      --  If we have the array case, then this shift count must be broken
      --  down into a byte subscript, and a shift within the byte.

      if Is_Array_Type (PAT) then

         declare
            New_Shift : Node_Id;

         begin
            --  We must analyze shift, since we will duplicate it

            Set_Parent (Shift, N);
            Analyze_And_Resolve
              (Shift, Standard_Integer, Suppress => All_Checks);

            --  The shift count within the word is
            --    shift mod Osiz

            New_Shift :=
              Make_Op_Mod (Loc,
                Left_Opnd  => Duplicate_Subexpr (Shift),
                Right_Opnd => Make_Integer_Literal (Loc, Osiz));

            --  The subscript to be used on the PAT array is
            --    shift / Osiz

            Obj :=
              Make_Indexed_Component (Loc,
                Prefix => Obj,
                Expressions => New_List (
                  Make_Op_Divide (Loc,
                    Left_Opnd => Duplicate_Subexpr (Shift),
                    Right_Opnd => Make_Integer_Literal (Loc, Osiz))));

            Shift := New_Shift;
         end;

      --  For the non-array case, the byte shift count is already
      --  set, and all we need is the unchecked conversion of the
      --  array to the PAT type.

      else
         Obj := Unchecked_Convert_To (PAT, Obj);
      end if;

      --  The one remaining step is to modify the shift count for the
      --  big-endian case. Consider the following example in a byte:

      --     xxxxxxxx  bits of byte
      --     vvvvvvvv  bits of value
      --     33221100  little-endian numbering
      --     00112233  big-endian numbering

      --  Here we have the case of 2-bit fields

      --  For the little-endian case, we already have the proper rotate
      --  count set, e.g. for element 2, the shift count is 2*2 = 4.

      --  For the big endian case, we have to adjust the shift count,
      --  computing it as (N - F) - shift, where N is the number of bits
      --  in an element of the array used to implement the packed array,
      --  F is the number of bits in a source level array element, and
      --  shift is the count so far computed.

      if Bytes_Big_Endian then
         Shift :=
           Make_Op_Subtract (Loc,
             Left_Opnd  => Make_Integer_Literal (Loc, Osiz - Csiz),
             Right_Opnd => Shift);
      end if;

      Set_Parent (Shift, N);
      Set_Parent (Obj, N);
      Analyze_And_Resolve (Obj,   Otyp,             Suppress => All_Checks);
      Analyze_And_Resolve (Shift, Standard_Integer, Suppress => All_Checks);

   end Setup_Packed_Array_Reference;

end Exp_Pakd;
