------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             E X P _ P A K D                              --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.10 $                             --
--                                                                          --
--   Copyright (C) 1992,1993,1994,1995,1996 Free Software Foundation, Inc.  --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). --
--                                                                          --
------------------------------------------------------------------------------

--  Expand routines for manipulation of packed arrays

with Types; use Types;
with Uintp; use Uintp;

package Exp_Pakd is

   -------------------------------------
   -- Implementation of Packed Arrays --
   -------------------------------------

   --  When a packed array (sub)type is frozen, we create a corresponding
   --  type that will be used to hold the bits of the packed value, and
   --  store the entity for this type in the Packed_Array_Type field of the
   --  E_Array_Type or E_Array_Subtype entity for the packed array.

   --  This packed array type has the name xxxPn, where xxx is the name
   --  of the packed type, and n is 1, 2, or 4 giving the number of bits
   --  per component (note that we use 4 bits for a size of 3, as permitted
   --  in the RM). The created declaration declares a type derived from one
   --  of the types in package System.Unsigned_Types as follows:

   --    For an unconstrained array, we declare:

   --      type xxxPn is new Packed_Bytes;

   --    For a constrained array with a static index type where the number
   --    of bits does not exceed the size of Unsigned, we use:

   --       type xxxPn is new Unsigned range 0 .. 2 ** nbits - 1;

   --    For a constrained array with a static index type where the number
   --    of bits is greater than the size of Unsigned, but does not exceed
   --    the size of Long_Long_Unsigned, we use

   --       type xxxPn is new Long_Long_Unsigned range 0 .. 2 ** nbits - 1;

   --    For all other constrained arrays, we use

   --       type xxxPn is new Packed_Bytes (0 .. m);

   --    where m is calculated (from the length of the original packed array)
   --    to hold the required number of bits.

   --  When a variable of packed array type is allocated, gigi will allocate
   --  the amount of space indicated by the corresponding packed array type.
   --  However, we do NOT attempt to rewrite the types of any references or
   --  to retype the variable itself, since this would cause all kinds of
   --  semantic problems in the front end (remember that expansion proceeds
   --  at the same time as analysis).

   --  For an indexed reference to a packed array, we simply convert the
   --  reference to the appropriate equivalent reference to the object
   --  of the packed array type (using unchecked conversion).

   --  In some cases (for internally generated types, and for the subtypes
   --  for record fields that depend on a discriminant), the corresponding
   --  packed type cannot be easily generated in advance. In these cases,
   --  we generate the required subtype on the fly at the reference point.

   ---------------------------
   -- Endian Considerations --
   ---------------------------

   --  The standard does not specify the way in which bits are numbered in
   --  a packed array. There are two reasonable rules for deciding this:

   --    Store the first bit at right end (low order) word. This means
   --    that the scaled subscript can be used directly as a right shift
   --    count (if we put bit 0 at the left end, then we need an extra
   --    subtract to compute the shift count.

   --    Layout the bits so that if the packed boolean array is overlaid on
   --    a record, using unchecked conversion, then bit 0 of the array is
   --    the same as the bit numbered bit 0 in a record representation
   --    clause applying to the record. For example:

   --       type Rec is record
   --          C : Bits4;
   --          D : Bits7;
   --          E : Bits5;
   --       end record;

   --       for Rec use record
   --          C at 0 range  0  .. 3;
   --          D at 0 range  4 .. 10;
   --          E at 0 range 11 .. 15;
   --       end record;

   --       type P16 is array (0 .. 15) of Boolean;
   --       pragma Pack (P16);

   --    Now if we use unchecked conversion to convert a value of the record
   --    type to the packed array type, according to this second criterion,
   --    we would expect field D to occupy bits 4..10 of the Boolean array.

   --  Although not required, this correspondence seems a highly desirable
   --  property, and is one that GNAT decides to guarantee. For a little
   --  endian machine, we can also meet the first requirement, but for a
   --  big endian machine, it will be necessary to store the first bit of
   --  a Boolean array in the left end (most significant) bit of the word.
   --  This may cost an extra instruction on some machines, but we consider
   --  that a worthwhile price to pay for the consistency.

   -----------------
   -- Subprograms --
   -----------------

   procedure Expand_Packed_Array_Type
     (Typ  : Entity_Id;
      PAT  : out Entity_Id;
      Decl : out Node_Id;
      Esiz : out Uint);
   --  Typ is a array type or subtype to which pragma Pack applies. This
   --  procedure creates a declaration for the corresponding representation
   --  type or subtype and returns the entity for the type that is declared
   --  in PAT, and the corresponding declaration in Decl. The caller is
   --  responsible for installing this declaration in the tree, analyzing
   --  it, and making sure that the type gets frozen. In some cases, the
   --  PAT may be available without needing to be declared. In such cases
   --  Decl is set to Empty on return. If Decl is non-Empty, then Esiz is
   --  set to the appropriate setting of the Esize field for PAT after its
   --  declaration has been analyzed. In all cases, the Packed_Array_Type
   --  field of Typ is set to PAT on exit.

   procedure Expand_Packed_Boolean_Operator (N : Node_Id);
   --  N is an N_Op_And, N_Op_Or or N_Op_Xor node whose operand type is a
   --  packed boolean array. This routine expands the appropriate operations
   --  to carry out the logical operation on the packed arrays. It handles
   --  both the modular and array representation cases.

   procedure Expand_Packed_Element_Get (N : Node_Id);
   --  N is an N_Indexed_Component node that appears as an expression
   --  whose value is needed (i.e. it is not the left hand side of an
   --  assignment). This procedure rewrites the node, replacing it with
   --  an appropriate reference to the corresponding packed array type.

   procedure Expand_Packed_Element_Set (N : Node_Id);
   --  N is an N_Assignment_Statement node whose name is an indexed
   --  component of a packed array. This procedure rewrites the entire
   --  assignment statement with appropriate code to set the referenced
   --  bits of the packed array type object.

   procedure Expand_Packed_Not (N : Node_Id);
   --  N is an N_Op_Not node where the operand is packed array of Boolean
   --  in standard representation (i.e. component size is one bit). This
   --  procedure expands the corresponding not operation. Note that the
   --  non-standard representation case is handled by using a loop through
   --  elements generated by the normal non-packed circuitry.

   function Needs_Packed_Array_Initialization (T : Entity_Id) return Boolean;
   --  Returns true if T is a packed array type for which allocated variables
   --  must be initialized. The need for initialization arises because these
   --  values can have extra bits that are not part of the logical value, and
   --  these extra bits must be set to zero, since equality checks check all
   --  the bits of the representation value.

   function Packed_Array_Init_Val
     (Typ  : Entity_Id;
      Loc  : Source_Ptr;
      Nod  : Node_Id)
      return Node_Id;
   --  Given a packed array typ Typ which Needs_Packed_Array_Initialization
   --  (see above), this function returns an appropriate initial value, using
   --  Loc as the location for the constructed value. Node is the node in the
   --  tree to which this value will be attached (used to provide a possible
   --  tree attachment point for analyze actions).

end Exp_Pakd;
