------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              I T Y P E S                                 --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.23 $                             --
--                                                                          --
--   Copyright (C) 1992,1993,1994,1995,1996 Free Software Foundation, Inc.  --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). --
--                                                                          --
------------------------------------------------------------------------------

with Alloc;    use Alloc;
with Atree;    use Atree;
with Debug;    use Debug;
with Einfo;    use Einfo;
with Exp_Util; use Exp_Util;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Sem_Util; use Sem_Util;
with Sinfo;    use Sinfo;
with Stand;    use Stand;
with Table;

package body Itypes is

   -----------------------
   -- Local subprograms --
   -----------------------

   procedure Append_Itype (N : Node_Id; E : Entity_Id);
   --  Add the Itype E at the end of implicit types list attached to N. If
   --  E is the head of an implicit type list, the full list is appended.

   ------------------
   -- Append_Itype --
   ------------------

   procedure Append_Itype (N : Node_Id; E : Entity_Id) is
      Ityp : Entity_Id;

   begin
      pragma Assert (Nkind (N) not in N_Entity and Nkind (E) in N_Entity);

      --  Static Itypes and anonymous access types don't need to be hooked
      --  to the tree because there is no elaboration for those types

      if Ekind (E) = E_Anonymous_Access_Type
        or else False --  ??? should be "Is_Static_Itype (E)" not impl. yet
      then
         return;
      end if;

      --  The new model for dynamic itypes is to attach them to an
      --  N_Implicit_Types node that is inserted with the Insert_Action
      --  mechanism, but this doesn't work in all cases:

      --  First exception, when the related node is a N_Implicit_Types
      --  itself, the new itype must go at the end of the itype list and
      --  not on another node inserted before

      if Nkind (N) = N_Implicit_Types

      --  Second exception, when the itype is inside a record declaration,
      --  we cannot insert N_Implicit_Types nodes in the the component list.

        or else Is_Record_Type (Current_Scope)
        or else Is_Protected_Type (Current_Scope)

      then

         --  If the itype is generated by a component declaration whose type
         --  is a constrained access type, do not attach itype at all. It is
         --  created only for constraint-checking purposes, and is not used by
         --  the back-end.

         if Nkind (N) = N_Component_Declaration
          and then Nkind (Subtype_Indication (N)) = N_Subtype_Indication
          and then
            Is_Access_Type (Entity (Subtype_Mark (Subtype_Indication (N))))
         then
            null;

         elsif No (First_Itype (N)) then
            Set_First_Itype (N, E);

         else
            Ityp := First_Itype (N);
            while Present (Next_Itype (Ityp)) loop
               Ityp := Next_Itype (Ityp);
            end loop;

            Set_Next_Itype (Ityp, E);
         end if;

      else
         declare
            Inode : constant Node_Id := Make_Implicit_Types (Sloc (N));

         begin
            Set_First_Itype (Inode, E);
            Insert_Action (N, Inode);
         end;
      end if;
   end Append_Itype;

   ------------------
   -- Create_Itype --
   ------------------

   function Create_Itype
     (Ekind        : Entity_Kind;
      Loc          : Source_Ptr;
      Related_Id   : Entity_Id := Empty;
      Suffix       : Character := ' ';
      Suffix_Index : Nat       := 0;
      Scope_Id     : Entity_Id := Current_Scope)
      return         Entity_Id
   is
      Typ : Entity_Id;

   begin
      if Related_Id = Empty then
         Typ := New_Internal_Entity (Ekind, Scope_Id, Loc, 'T');

         Set_Public_Status (Typ);

      else
         Typ := New_External_Entity
           (Ekind, Scope_Id, Loc, Related_Id, Suffix, Suffix_Index, 'T');
      end if;

      Set_Etype (Typ, Any_Type);
      Set_Is_Itype (Typ);
      return Typ;
   end Create_Itype;

   -----------------------------
   -- Create_Itype_And_Insert --
   -----------------------------

   function Create_Itype_And_Insert
     (Ekind        : Entity_Kind;
      Ins_Node     : Node_Id;
      Related_Id   : Entity_Id   := Empty;
      Suffix       : Character   := ' ';
      Suffix_Index : Nat         := 0;
      Scope_Id     : Entity_Id   := Current_Scope)
      return         Entity_Id
   is
      Typ : constant Entity_Id :=
              Create_Itype
                (Ekind        => Ekind,
                 Loc          => Sloc (Ins_Node),
                 Related_Id   => Related_Id,
                 Suffix       => Suffix,
                 Suffix_Index => Suffix_Index,
                 Scope_Id     => Scope_Id);

   begin
      Insert_Itype (Ins_Node, Typ);
      return Typ;
   end Create_Itype_And_Insert;

   ------------------
   -- Insert_Itype --
   ------------------

   procedure Insert_Itype (Ins_Node : Node_Id; E : Entity_Id) is
   begin
      pragma Assert (Next_Itype (E) = Empty);

      --  Make sure Is_Itype is set. This really should not be necessary,
      --  but some people are doing their own Itype manufacturering ???

      Set_Is_Itype (E);
      Append_Itype (Ins_Node, E);
   end Insert_Itype;

   ---------------------
   -- Transfer_Itypes --
   ---------------------

   procedure Transfer_Itypes (From : Node_Id; To : Node_Id) is
   begin
      pragma Assert (Nkind (From) not in N_Entity
                       and then Nkind (To) not in N_Entity);

      if From /= To
        and then Nkind (From) in N_Has_Itypes
        and then Present (First_Itype (From))
      then
         Append_Itype (To, First_Itype (From));
         Set_First_Itype (From, Empty);

         if Has_Dynamic_Itype (From) then
            Set_Has_Dynamic_Itype (To,   True);
            Set_Has_Dynamic_Itype (From, False);
         end if;
      end if;
   end Transfer_Itypes;

end Itypes;
