/**
 *
 * $Id: BulletinBoard.c,v 1.34 1996/04/22 22:54:20 miers Exp $
 *
 * Copyright (C) 1995 Free Software Foundation, Inc.
 *
 * This file is part of the GNU LessTif Library.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **/

static char rcsid[] = "$Id";

#include <LTconfig.h>
#include <Xm/XmP.h>
#include <Xm/BaseClassP.h>
#include <Xm/BulletinBP.h>
#include <Xm/DebugUtil.h>
#include <Xm/TransltnsP.h>
#include <Xm/DialogS.h>
#include <Xm/ArrowB.h>
#include <Xm/ArrowBG.h>
#include <Xm/LabelP.h>
#include <Xm/LabelGP.h>
#include <Xm/PushB.h>
#include <Xm/PushBG.h>
#include <Xm/XmI.h>
#include <stdio.h>

/* Forward Declarations */

static void class_initialize();
static void class_part_initialize(WidgetClass class);
static void initialize(Widget request, Widget new, ArgList args, Cardinal *num_args);
static void destroy(Widget w);
static XtGeometryResult query_geometry(Widget w, XtWidgetGeometry *proposed, XtWidgetGeometry *answer);
static Boolean set_values(Widget current, Widget request, Widget new, ArgList args, Cardinal *num_args);
static void expose(Widget w, XEvent *event, Region region);
static XtGeometryResult geometry_manager(Widget w, XtWidgetGeometry *request, XtWidgetGeometry *reply);
static void change_managed(Widget w);
static void ConstraintInitialize(Widget request, Widget new, Arg *args, Cardinal *nargs);
static Boolean ConstraintSetValues(Widget old, Widget request, Widget new, Arg *args, Cardinal *nargs);
static void DoResize(Widget w);
static void realize(Widget w, XtValueMask *value_mask, XSetWindowAttributes *attributes);
static void InsertChild(Widget w);
static void resize(Widget w);
static Boolean _XmBBParentProcess(Widget widget, XmParentProcessData data);

/*
 * Resources for the Bulletin board class
 */
#define Offset(field) XtOffsetOf(XmBulletinBoardRec, bulletin_board.field)
#define MGR_Offset(field) XtOffsetOf(XmBulletinBoardRec, manager.field)
static XtResource resources[] = {
    {
	XmNshadowType, XmCShadowType, XmRShadowType,
	sizeof(unsigned char), Offset(shadow_type),
	XmRImmediate, (XtPointer)XmSHADOW_OUT
    },
    {
	XmNshadowThickness, XmCShadowThickness, XmRHorizontalDimension,
	sizeof(Dimension), MGR_Offset(shadow_thickness),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNmarginWidth, XmCMarginWidth, XmRHorizontalDimension,
	sizeof(Dimension), Offset(margin_width),
	XmRImmediate, (XtPointer)10
    },    
    {
	XmNmarginHeight, XmCMarginHeight, XmRVerticalDimension,
	sizeof(Dimension), Offset(margin_height),
	XmRImmediate, (XtPointer)10
    },
    {
	XmNdefaultButton, XmCWidget, XmRWidget,
	sizeof(Widget), Offset(default_button),
	XmRImmediate, (XtPointer)NULL
    },
    {
	XmNcancelButton, XmCWidget, XmRWidget,
	sizeof(Widget), Offset(cancel_button),
	XmRImmediate, (XtPointer)NULL
    },
    {
	XmNfocusCallback, XmCCallback, XmRCallback,
	sizeof(XtCallbackList), Offset(focus_callback_list),
	XmRImmediate, (XtPointer)NULL
    },
    {
	XmNmapCallback, XmCCallback, XmRCallback,
	sizeof(XtCallbackList), Offset(map_callback_list),
	XmRImmediate, (XtPointer)NULL
    },
    {
	XmNunmapCallback, XmCCallback, XmRCallback,
	sizeof(XtCallbackList), Offset(unmap_callback_list),
	XmRImmediate, (XtPointer)NULL
    },
    {
	XmNbuttonFontList, XmCButtonFontList, XmRFontList,
	sizeof(XmFontList), Offset(button_font_list),
	XmRFontList, (XtPointer)NULL
    },
    {
	XmNlabelFontList, XmCLabelFontList, XmRFontList,
	sizeof(XmFontList), Offset(label_font_list),
	XmRFontList, (XtPointer)NULL
    },
    {
	XmNtextFontList, XmCTextFontList, XmRFontList,
	sizeof(XmFontList), Offset(text_font_list),
	XmRFontList, (XtPointer)NULL
    }, 
    {
	XmNtextTranslations, XmCTranslations, XmRTranslationTable,
	sizeof(XtTranslations), Offset(text_translations),
	XmRImmediate, (XtPointer)NULL
    },
    {
	XmNallowOverlap, XmCAllowOverlap, XmRBoolean,
	sizeof(Boolean), Offset(allow_overlap),
	XmRImmediate, (XtPointer)True
    },
    {
	XmNautoUnmanage, XmCAutoUnmanage, XmRBoolean,
	sizeof(Boolean), Offset(auto_unmanage),
	XmRImmediate, (XtPointer)True
    },
    {
	XmNdefaultPosition, XmCDefaultPosition, XmRBoolean,
	sizeof(Boolean), Offset(default_position),
	XmRImmediate, (XtPointer)True
    },
    {
	XmNresizePolicy, XmCResizePolicy, XmRResizePolicy,
	sizeof(unsigned char), Offset(resize_policy),
	XmRImmediate, (XtPointer)XmRESIZE_ANY
    },
    {
	XmNnoResize, XmCNoResize, XmRBoolean,
	sizeof(Boolean), Offset(no_resize),
	XmRImmediate, (XtPointer)False
    },
    {
	XmNdialogStyle, XmCDialogStyle, XmRDialogStyle,
	sizeof(unsigned char), Offset(dialog_style),
	XmRCallProc, (XtPointer)_XmBulletinBoardDialogStyleDefault
    },
    {
	XmNdialogTitle, XmCDialogTitle, XmRXmString,
	sizeof(XmString), Offset(dialog_title),
	XmRString, (XtPointer)NULL
    },
};

static XmSyntheticResource syn_resources[] = {
    {
	XmNdialogTitle,
	sizeof(XmString), Offset(dialog_title),
	NULL /* FIXME */, NULL
    },
    {
	XmNmarginWidth,
	sizeof(Dimension), Offset(margin_width),
	_XmFromHorizontalPixels, _XmToHorizontalPixels
    },
    {
	XmNmarginHeight,
	sizeof(Dimension), Offset(margin_height),
	_XmFromVerticalPixels, _XmToVerticalPixels
    }
};

char _XmBulletinB_defaultTranslations[] = 
    "<BtnMotion>:               ManagerGadgetButtonMotion()\n\
     <Btn1Down>:                ManagerGadgetArm()\n\
     <Btn1Down>,<Btn1Up>:       ManagerGadgetActivate()\n\
     <Btn1Up>:                  ManagerGadgetActivate()\n\
     <Btn1Down>(2+):            ManagerGadgetMultiArm()\n\
     <Btn1Up>(2+):              ManagerGadgetMultiActivate()\n\
     <Btn2Down>:                ManagerGadgetDrag()\n\
     <Key>osfHelp:              ManagerGadgetHelp()\n\
     <Key>osfActivate:          ManagerParentActivate()\n\
     <Key>osfCancel:            ManagerParentCancel()\n\
     <Key>osfSelect:            ManagerGadgetSelect()\n\
     <Key>space:                ManagerGadgetSelect()\n\
     <Key>Return:               ManagerParentActivate()\n\
     <Key>:                     ManagerGadgetKeyInput()";

static XmBaseClassExtRec _XmBulletinBCoreClassExtRec = {
    /* next_extension            */ NULL,
    /* record_type               */ NULLQUARK,                             
    /* version                   */ XmBaseClassExtVersion,
    /* size                      */ sizeof(XmBaseClassExtRec),
    /* initialize_prehook        */ NULL, /* FIXME */
    /* set_values_prehook        */ NULL, /* FIXME */
    /* initialize_posthook       */ NULL, /* FIXME */
    /* set_values_posthook       */ NULL, /* FIXME */
    /* secondary_object_class    */ NULL, /* FIXME */
    /* secondary_object_create   */ NULL, /* FIXME */
    /* get_secondary_resources   */ NULL, /* FIXME */
    /* fast_subclass             */ { 0 }, /* FIXME */
    /* get_values_prehook        */ NULL, /* FIXME */
    /* get_values_posthook       */ NULL, /* FIXME */
    /* class_part_init_prehook   */ NULL, /* FIXME */
    /* class_part_init_posthook  */ NULL, /* FIXME */
    /* ext_resources             */ NULL, /* FIXME */
    /* compiled_ext_resources    */ NULL, /* FIXME */
    /* num_ext_resources         */ 0, /* FIXME */
    /* use_sub_resources         */ FALSE, /* FIXME */
    /* widget_navigable          */ NULL, /* FIXME */
    /* focus_change              */ NULL, /* FIXME */
    /* wrapper_data              */ NULL
};

static CompositeClassExtensionRec bbCompositeExt = 
{
    /* next_extension */  NULL,
    /* record_type    */  NULLQUARK,
    /* version        */  XtCompositeExtensionVersion,
    /* record_size    */  sizeof(CompositeClassExtensionRec),
    /* accepts_objects */ True,
#if XtSpecificationRelease >= 6
    /* allows_change_managed_set */ True
#endif
};

static XmManagerClassExtRec _XmBulletinBMClassExtRec = {
    /* next_extension            */ NULL,
    /* record_type               */ NULLQUARK,
    /* version                   */ XmManagerClassExtVersion,
    /* record_size               */ sizeof(XmManagerClassExtRec),
    /* traversal_children        */ NULL /* FIXME */
};

XmBulletinBoardClassRec xmBulletinBoardClassRec = {
    /* Core class part */
    {
	/* superclass            */ (WidgetClass) &xmManagerClassRec,
        /* class_name            */ "XmBulletinBoard",
	/* widget_size           */ sizeof(XmBulletinBoardRec),
	/* class_initialize      */ class_initialize,
	/* class_part_initialize */ class_part_initialize,
	/* class_inited          */ FALSE,
	/* initialize            */ initialize,
	/* initialize_hook       */ NULL,
	/* realize               */ realize,
	/* actions               */ NULL,
	/* num_actions           */ 0,
	/* resources             */ resources,
	/* num_resources         */ XtNumber(resources),
	/* xrm_class             */ NULLQUARK,
	/* compress_motion       */ TRUE,
	/* compress_exposure     */ XtExposeCompressMultiple,
	/* compress_enterleave   */ TRUE,
	/* visible_interest      */ FALSE,
	/* destroy               */ destroy,
	/* resize                */ resize,
	/* expose                */ expose,
	/* set_values            */ set_values,
	/* set_values_hook       */ NULL,
	/* set_values_almost     */ XtInheritSetValuesAlmost,
	/* get_values_hook       */ NULL,
	/* accept_focus          */ NULL,
	/* version               */ XtVersion,
	/* callback offsets      */ NULL,
	/* tm_table              */ _XmBulletinB_defaultTranslations,
	/* query_geometry        */ query_geometry,
	/* display_accelerator   */ NULL,
	/* extension             */ (XtPointer)&_XmBulletinBCoreClassExtRec
    },
    /* Composite class part */
    {
	/* geometry manager */ geometry_manager, 
        /* change_managed   */ change_managed, 
        /* insert_child     */ InsertChild,
        /* delete_child     */ XtInheritDeleteChild,
        /* extension        */ (XtPointer) &bbCompositeExt,
    },
    /* Constraint class part */
    {
	/* subresources      */ NULL,  /* FIX ME */
        /* subresource_count */ 0,     /* FIX ME */
        /* constraint_size   */ 0,     /* FIX ME */
        /* initialize        */ ConstraintInitialize,
        /* destroy           */ NULL,  /* FIX ME */
        /* set_values        */ ConstraintSetValues,
        /* extension         */ NULL,   /* FIX ME */
    },
    /* XmManager class part */
    {
	/* translations                 */ XmInheritTranslations,
        /* syn_resources                */ syn_resources,
        /* num_syn_resources            */ XtNumber(syn_resources),
        /* syn_constraint_resources     */ NULL,
        /* num_syn_constraint_resources */ 0,
        /* parent_process               */ _XmBBParentProcess,
	/* extension                    */ (XtPointer)&_XmBulletinBMClassExtRec
    },
    /* XmBulletinBoard class part */
    {
	/* always_install_accelerators	*/ False,
	/* geo_matrix_create		*/ NULL,
	/* focus_moved_proc		*/ NULL,
	/* extension			*/ NULL,
    },
};

WidgetClass xmBulletinBoardWidgetClass = (WidgetClass)&xmBulletinBoardClassRec;

static void 
class_initialize()
{
    _XmBulletinBCoreClassExtRec.record_type = XmQmotif;
}

static void
class_part_initialize(WidgetClass widget_class)
{
    XmBulletinBoardWidgetClass bbclass = (XmBulletinBoardWidgetClass) widget_class;
    XmBulletinBoardWidgetClass sclass =
	(XmBulletinBoardWidgetClass) (widget_class->core_class.superclass);
    CompositeClassExtension ext, *extptr;

    extptr = (CompositeClassExtension*)_XmGetClassExtensionPtr(
		(XmGenericClassExt*)&(bbclass->composite_class.extension),
		NULLQUARK);

    if (extptr == NULL || *extptr == NULL)
    {
	ext = (CompositeClassExtension) XtNew(CompositeClassExtensionRec);
	if (ext != NULL)
	{
	    ext->next_extension = bbclass->composite_class.extension;
	    ext->record_type = NULLQUARK;
	    ext->version = XtCompositeExtensionVersion;
	    ext->record_size = sizeof(CompositeClassExtensionRec);
	    ext->accepts_objects = True;
#if XtSpecificationRelease >= 6
	    ext->allows_change_managed_set = True;
#endif
	    bbclass->composite_class.extension = (XtPointer) ext;
	}
    }    
    /* MLM: This seems to be necessary in XFree86/R6 on linux */
    else if (!(*extptr)->accepts_objects)
	(*extptr)->accepts_objects = True;

    if (bbclass->bulletin_board_class.geo_matrix_create ==
	XmInheritGeoMatrixCreate && widget_class != xmBulletinBoardWidgetClass)
	bbclass->bulletin_board_class.geo_matrix_create = 
		sclass->bulletin_board_class.geo_matrix_create;

    if (bbclass->bulletin_board_class.focus_moved_proc ==
	XmInheritFocusMovedProc)
	if (sclass->bulletin_board_class.focus_moved_proc)
	    bbclass->bulletin_board_class.focus_moved_proc =
		sclass->bulletin_board_class.focus_moved_proc;
	else
	    bbclass->bulletin_board_class.focus_moved_proc =
		_XmBulletinBoardFocusMoved;

    _XmFastSubclassInit(widget_class, XmBULLETIN_BOARD_BIT);
}

static void
initialize(Widget request,
	   Widget new,
	   ArgList args,
	   Cardinal *num_args)
{
    XmBulletinBoardWidget bb = (XmBulletinBoardWidget)new;
    XmBulletinBoardWidgetClass bbclass;

    if (!BB_LabelFontList(bb))
	BB_LabelFontList(bb) = _XmGetDefaultFontList(new, XmLABEL_FONTLIST);

    if (!BB_ButtonFontList(bb))
	BB_ButtonFontList(bb) = _XmGetDefaultFontList(new, XmBUTTON_FONTLIST);

    if (!BB_TextFontList(bb))
	BB_TextFontList(bb) = _XmGetDefaultFontList(new, XmTEXT_FONTLIST);

    if (XtIsSubclass(XtParent(new), xmDialogShellWidgetClass)) {
	/* Shouldn't this be in synthetic resource handlers ? FIX ME */
	if (bb->bulletin_board.dialog_title) {
	    char *p;

	    if (XmStringGetLtoR(bb->bulletin_board.dialog_title,
				XmFONTLIST_DEFAULT_TAG, &p)) {
		XtVaSetValues(XtParent(new), XtNtitle, p, NULL);
		XtFree(p);
	    }
	}
    }

    if (XtIsSubclass(XtParent(new), xmDialogShellWidgetClass) ||
	XtIsSubclass(XtParent(new), vendorShellWidgetClass)) {
	if (MGR_ShadowThickness(new) == 0)
	    MGR_ShadowThickness(new) = 1;
    }

    BB_DynamicDefaultButton(new) = NULL;
    BB_DynamicCancelButton(new) = NULL;

    bbclass = (XmBulletinBoardWidgetClass)XtClass(new);
    if (bbclass->bulletin_board_class.focus_moved_proc) {
	XtAddCallback(new, XmNfocusCallback,
		      bbclass->bulletin_board_class.focus_moved_proc, NULL);
    }

    /* initialize these to values that aren't possible */
    BB_OldWidth(new) = -1;
    BB_OldHeight(new) = -1;
}

static void
destroy(Widget w)
{
}

static Boolean
set_values(Widget old,
	   Widget request,
	   Widget new,
	   ArgList args,
	   Cardinal *num_args)
{
    XmBulletinBoardWidget ow = (XmBulletinBoardWidget) old;
    XmBulletinBoardWidget nw = (XmBulletinBoardWidget) new;
    Boolean need_refresh = False;

    XdbDebug(__FILE__, new, "SetValues\n");
    if (BB_DefaultButton(ow) != BB_DefaultButton(nw))
    {
	/* the default button has changed, so set the old one's showAsDefault
	   to 0, and make the new one display as the default. */

	_XmBBUpdateDynDefaultButton(new);

	need_refresh = True; /* do we need to ?*/
    }

    if (XtIsSubclass(XtParent(new), xmDialogShellWidgetClass)) {
/* Shouldn't this be in synthetic resource handlers ? FIX ME */
	if (XmStringCompare(ow->bulletin_board.dialog_title, nw->bulletin_board.dialog_title) != 0){
	    char	*p;
	    if (XmStringGetLtoR(nw->bulletin_board.dialog_title, XmFONTLIST_DEFAULT_TAG, &p)) {
		XtVaSetValues(XtParent(new), XtNtitle, p, NULL);
		XtFree(p);
	    }
	    XmStringFree(ow->bulletin_board.dialog_title);
	    nw->bulletin_board.dialog_title = XmStringCopy(nw->bulletin_board.dialog_title);
	}
/*
 * If we're a dialog, refuse to be placed away from 0,0
 */
	if (XtX(new) != 0) {
	    XtX(new) = 0;
	    need_refresh = True;
	}

	if (XtY(new) != 0) {
	    XtY(new) = 0;
	    need_refresh = True;
	}
    }

    return need_refresh;
}

static XtGeometryResult 
query_geometry(Widget w, XtWidgetGeometry *proposed, XtWidgetGeometry *answer)
{
    XdbDebug(__FILE__, w, "QueryGeometry => YES\n");

/* FIX ME */
    if (answer)
	*answer = *proposed;
    return XtGeometryYes;
}

static void
handle_resize(Widget w, XmGeoCreateProc mat_make)
{
    Dimension wd, ht;
    XmGeoMatrix geo;

    wd = XtWidth(w);
    ht = XtHeight(w);

    geo = mat_make(w, NULL, NULL);

    _XmGeoMatrixGet(geo, XmGET_PREFERRED_SIZE);

    _XmGeoArrangeBoxes(geo, 0, 0, &wd, &ht);

    _XmGeoMatrixSet(geo);

    if (XtIsRealized(w)) {
	_XmClearShadowType(w, BB_OldWidth(w), BB_OldHeight(w),
			   BB_OldShadowThickness(w), 0);

	_XmDrawShadows(XtDisplay(w), XtWindow(w),
		       MGR_TopShadowGC(w), MGR_BottomShadowGC(w),
		       0, 0, XtWidth(w), XtHeight(w),
		       MGR_ShadowThickness(w), BB_ShadowType(w));
    }

    _XmGeoMatrixFree(geo);

    BB_OldWidth(w) = XtWidth(w);
    BB_OldHeight(w) = XtHeight(w);
    BB_OldShadowThickness(w) = MGR_ShadowThickness(w);
}

static void
resize(Widget w)
{
    XmBulletinBoardClassRec *bb = (XmBulletinBoardClassRec *)XtClass(w);
    Widget p;

    XdbDebug(__FILE__, NULL, "resize\n");

    if (bb->bulletin_board_class.geo_matrix_create) {
	handle_resize(w, bb->bulletin_board_class.geo_matrix_create);
	return;
    }

    /* some call with args (w, BB_MarginWidth(w), BB_MarginHeight(w)); */

    _XmClearShadowType(w, BB_OldWidth(w), BB_OldHeight(w),
		       BB_OldShadowThickness(w), 0);

    BB_OldShadowThickness(w) = 0;

    if (XtIsComposite(w))
	p = w;
    else
	p = XtParent(w);

    if (XtIsRealized(p) || XtWidth(w) == 0 || XtHeight(w) == 0) {

	_XmGMDoLayout(w, BB_MarginWidth(w), BB_MarginHeight(w),
		      BB_ResizePolicy(w), True);
    }

    if ((XtWidth(w) < BB_OldWidth(w) || XtHeight(w) < BB_OldHeight(w)) &&
	XtIsRealized(w)) {
	_XmDrawShadows(XtDisplay(w), XtWindow(w),
		       MGR_TopShadowGC(w), MGR_BottomShadowGC(w),
		       0, 0, XtWidth(w), XtHeight(w),
		       MGR_ShadowThickness(w), BB_ShadowType(w));
    }

    BB_OldWidth(w) = XtWidth(w);
    BB_OldHeight(w) = XtHeight(w);
    BB_OldShadowThickness(w) = MGR_ShadowThickness(w);
}

static void
CalculateNewSize(Widget w, Dimension *wid, Dimension *hei)
{
    XmBulletinBoardWidget	bb = (XmBulletinBoardWidget) w;
    Widget			child;
    int				i;
    Dimension			ww, hh;

    *wid = 0;
    *hei = 0;

    for (i=0; i<bb->composite.num_children; i++) {
	child = bb->composite.children[i];

	if (!XtIsManaged(child)) continue;

	ww = XtX(child) + XtWidth(child) +
		2 * XtBorderWidth(child) +
		BB_MarginWidth(bb);
	hh = XtY(child) + XtHeight(child) +
		2 * XtBorderWidth(child) +
		BB_MarginHeight(bb);

	if (*wid < ww)
	    *wid = ww;

	if (*hei < hh)
	    *hei = hh;
    }
}

/*
 * DoResize() : compute the BB's size depending on XmNresizePolicy
 *	and the geometry of its children.
 *
 *	Note : XmNnoResize has nothing to do with this.
 */
static void
DoResize(Widget w)
{
    XmBulletinBoardWidget	bb = (XmBulletinBoardWidget)w;
    Dimension	wid, ht;
    XtWidgetGeometry	request;
    XtGeometryResult	result;

    if (BB_ResizePolicy(bb) == XmRESIZE_NONE) {
	XdbDebug(__FILE__, w, "DoResize: XmRESIZE_NONE\n", XtName(w));
        return;
    }

    request.width = XtWidth(bb);
    request.height = XtHeight(bb);

    CalculateNewSize(w, &wid, &ht);

    if (BB_ResizePolicy(bb) == XmRESIZE_ANY) {
	request.width = wid;
	request.height = ht;
    }

    if (BB_ResizePolicy(bb) == XmRESIZE_GROW) {
        if (wid > XtWidth(bb))
	    request.width = wid;
        if (ht > XtHeight(bb))
	    request.height = ht;
    }

/* Deal with initial size */
    if (! XtIsRealized(w)) {
	if (XtWidth(bb) != 0) request.width = XtWidth(bb);
	if (XtHeight(bb) != 0) request.height = XtHeight(bb);
    }

    XdbDebug(__FILE__, w, "DoResize => %d %d\n", request.width, request.height);

/* Now try to resize yourself - if parent permits ... */
    if (request.width != XtWidth(bb) || request.height != XtHeight(bb)) {
	request.request_mode = CWWidth | CWHeight;
	do {
	    result = XtMakeGeometryRequest(w, &request, &request);

	    XdbDebug(__FILE__, w, "DoResize: XtMakeGeometryRequest => %s\n",
		XdbGeometryResult2String(result));
	} while (result == XtGeometryAlmost);
    }
}

/*
 * Geometry Manager is always called by a child of BB.
 *	It asks to get a different geometry. We may allow this.
 *	Also we may have to resize ourself because of this.
 *
 * As we are a simplistic widget, we'll allow everything.
 * FIXME : should this thing implement XmNallowOverlap ?
 * 101995 - MLM -- yes.  It should also check for MarginWidth/MarginHeight
 */
static XtGeometryResult
geometry_manager(Widget w, XtWidgetGeometry *desired, XtWidgetGeometry *allowed)
{
    XmBulletinBoardWidget           bb = (XmBulletinBoardWidget) XtParent(w);
    Position x, y;
    Dimension width, height;

#define Wants(x)        (desired->request_mode & x)

    XdbDebug2(__FILE__, (Widget)bb, w, "GeometryManager (desired ");
    if (Wants(CWX)) XdbDebug0(__FILE__, (Widget)bb, "X %d ", desired->x);
    if (Wants(CWY)) XdbDebug0(__FILE__, (Widget)bb, "Y %d ", desired->y);
    if (Wants(CWWidth)) XdbDebug0(__FILE__, (Widget)bb, "W %d ", desired->width);
    if (Wants(CWHeight)) XdbDebug0(__FILE__, (Widget)bb, "H %d ", desired->height);
    XdbDebug0(__FILE__, w, ")\n");

    if (Wants(CWX))
	x = desired->x;
    else
	x = XtX(w);
    if (Wants(CWY))
	y = desired->y;
    else
	y = XtY(w);
    if (Wants(CWWidth))
	width = desired->width;
    else
	width = XtWidth(w);
    if (Wants(CWHeight))
	height = desired->height;
    else
	height = XtHeight(w);

    /* if we don't fit in the MarginWidth/MarginHeight bounding box, say no */
    if (x < BB_MarginWidth(bb) || x + width > XtWidth(bb) - BB_MarginWidth(bb)) {
	XdbDebug(__FILE__, w, "Not in Margin* bounding box => NO\n");
	return XtGeometryNo;
    }
    if (y < BB_MarginHeight(bb) || y + height > XtHeight(bb) - BB_MarginHeight(bb)) {
	XdbDebug(__FILE__, w, "Not in Margin* bounding box => NO\n");
	return XtGeometryNo;
    }

    /* check for overlap */
    if (BB_AllowOverlap(bb) == False) {
	Boolean overlap = False;
	int i;

	/* FIXME -- this algorithm only works with rectangles -- is this
	 * right? */
	for (i = 0; i < bb->composite.num_children; i++) {
	    Widget cw = bb->composite.children[i];

	    if (cw == w)
		continue;

	    if (XtY(cw) + XtHeight(cw) <= y) /* above us */
		continue; /* then quit */

	    if (XtY(cw) >= y + height) /* below us */
		continue; /* then quit */

	    if (XtX(cw) + XtWidth(cw) <= x) /* to the left */
		continue; /* then quit */

	    if (XtX(cw) >= x + width) /* to the right */
		continue; /* then quit */

	    /* all your edges aren't above, below, left, or right of us;
	     * you must intersect/overlay/are-contained-by us */
	    overlap = True;
	    break;
	}
	if (overlap) {
	    XdbDebug(__FILE__, w, "Overlap => NO\n");
	    return XtGeometryNo;
	}
    }

    if (Wants(XtCWQueryOnly)) {
	XdbDebug(__FILE__, w, "QueryOnly => YES\n");
	return XtGeometryYes;
    }

    XdbDebug(__FILE__, w, "GeometryManager => ");
    if (Wants(CWWidth)) {
	XtWidth(w) = desired->width;
	XdbDebug0(__FILE__, w, "W %d ", desired->width);
    }
    if (Wants(CWHeight)) {
	XtHeight(w) = desired->height;
	XdbDebug0(__FILE__, (Widget)bb, "H %d ", desired->height);
    }
    if (Wants(CWX)) {
	XtX(w) = desired->x;
	XdbDebug0(__FILE__, w, "X %d ", desired->x);
    }
    if (Wants(CWY)) {
	XtY(w) = desired->y;
	XdbDebug0(__FILE__, w, "Y %d ", desired->y);
    }
    if (Wants(CWBorderWidth)) {
	XtBorderWidth(w) = desired->border_width;
	XdbDebug0(__FILE__, w, "BW %d ", desired->border_width);
    }
    XdbDebug0(__FILE__, w, "=> YES\n");

    DoResize((Widget)bb);

    return XtGeometryYes;
#undef  Wants
}

static void
handle_change_managed(Widget w, XmGeoCreateProc mat_make)
{
    Dimension wd, ht, retw, reth;
    XmGeoMatrix geo;
    XtGeometryResult result;

    if (!XtIsRealized(w))
	wd = ht = 0;

    else if (BB_ResizePolicy(w) != XmNONE)
	wd = ht = 0;

    else {
	wd = XtWidth(w);
	ht = XtHeight(w);
    }

    geo = mat_make(w, NULL, NULL);

    _XmGeoMatrixGet(geo, XmGET_PREFERRED_SIZE);

    _XmGeoArrangeBoxes(geo, 0, 0, &wd, &ht);

    if (BB_ResizePolicy(w) == XmRESIZE_GROW) {
	/* check the return against the original.  If the procedure would
	 * like the BB to shrink, call again */
	if (wd < XtWidth(w) || ht < XtHeight(w)) {
	    wd = XtWidth(w);
	    ht = XtHeight(w);
	    _XmGeoArrangeBoxes(geo, 0, 0, &wd, &ht);
	}
    }

    if (wd == XtWidth(w) && ht == XtHeight(w)) {
	_XmGeoMatrixFree(geo);
	_XmNavigChangeManaged(w);
	return;
    }

    retw = wd;
    reth = ht;
    do {
	result = XtMakeResizeRequest((Widget)w, retw, reth, &retw, &reth);
    } while (result == XtGeometryAlmost);

    if (retw != wd || reth != ht)
	_XmGeoArrangeBoxes(geo, 0, 0, &retw, &reth);

    _XmGeoMatrixSet(geo);

    if (XtIsRealized(w)) {
	_XmClearShadowType(w, BB_OldWidth(w), BB_OldHeight(w),
			   BB_OldShadowThickness(w), 0);

	_XmDrawShadows(XtDisplay(w), XtWindow(w),
		       MGR_TopShadowGC(w), MGR_BottomShadowGC(w),
		       0, 0, XtWidth(w), XtHeight(w),
		       MGR_ShadowThickness(w), BB_ShadowType(w));
    }

    _XmGeoMatrixFree(geo);

    BB_OldWidth(w) = XtWidth(w);
    BB_OldHeight(w) = XtHeight(w);
    BB_OldShadowThickness(w) = MGR_ShadowThickness(w);

    _XmNavigChangeManaged(w);
}

static void
change_managed(Widget w)
{
    Widget p;
    XmBulletinBoardClassRec *bb = (XmBulletinBoardClassRec *)XtClass(w);

    XdbDebug(__FILE__, NULL, "ChangeManaged\n");

    if (bb->bulletin_board_class.geo_matrix_create) {
	handle_change_managed(w, bb->bulletin_board_class.geo_matrix_create);
	return;
    }

    /* some call with args (w, BB_MarginWidth(w), BB_MarginHeight(w)); */

    _XmClearShadowType(w, BB_OldWidth(w), BB_OldHeight(w),
		       BB_OldShadowThickness(w), 0);

    BB_OldShadowThickness(w) = 0;

    if (XtIsComposite(w))
	p = w;
    else
	p = XtParent(w);

    if (XtIsRealized(p) || XtWidth(w) == 0 || XtHeight(w) == 0) {
	_XmGMDoLayout(w, BB_MarginWidth(w), BB_MarginHeight(w),
		      BB_ResizePolicy(w), False);
    }

    if ((XtWidth(w) < BB_OldWidth(w) || XtHeight(w) < BB_OldHeight(w)) &&
	XtIsRealized(w)) {
	_XmDrawShadows(XtDisplay(w), XtWindow(w),
		       MGR_TopShadowGC(w), MGR_BottomShadowGC(w),
		       0, 0, XtWidth(w), XtHeight(w),
		       MGR_ShadowThickness(w), BB_ShadowType(w));
    }

    BB_OldWidth(w) = XtWidth(w);
    BB_OldHeight(w) = XtHeight(w);
    BB_OldShadowThickness(w) = MGR_ShadowThickness(w);

    _XmNavigChangeManaged(w);
}

Widget 
XmCreateBulletinBoard(Widget parent,
		      char *name,
		      Arg *arglist,
		      Cardinal argcount)
{
    return XtCreateWidget(name,
			  xmBulletinBoardWidgetClass,
			  parent,
			  arglist,
			  argcount);
}

Widget 
XmCreateBulletinBoardDialog(Widget parent,
			    char *name,
			    Arg *arglist,
			    Cardinal argcount)
{
    Widget  d;
    char    *s;

    s = _XmMakeDialogName(name);

    d = XtCreateWidget(s, xmDialogShellWidgetClass, parent, arglist, argcount);
    XtFree(s);

    return XtCreateWidget(name, xmBulletinBoardWidgetClass, d, arglist, argcount);
}

static void
ConstraintInitialize(Widget request, Widget new, Arg *args, Cardinal *nargs)
{
        XdbDebug(__FILE__, new, "ConstraintInitialize\n");
}

static Boolean
ConstraintSetValues(Widget old, Widget request, Widget new, Arg *args, Cardinal *nargs)
{
        XmBulletinBoardWidget   bb = (XmBulletinBoardWidget) XtParent(new);

        XdbDebug2(__FILE__, (Widget)bb, new, "ConstraintSetValues\n");

        return False;
}

static void
handle_realize(Widget w, XmGeoCreateProc mat_make)
{
    Dimension wd, ht, retw, reth;
    XmGeoMatrix geo;
    XtGeometryResult	result;

    wd = XtWidth(w);
    ht = XtHeight(w);

    geo = mat_make(w, NULL, NULL);

    _XmGeoMatrixGet(geo, XmGET_PREFERRED_SIZE);

    _XmGeoArrangeBoxes(geo, 0, 0, &wd, &ht);

    if (BB_ResizePolicy(w) == XmRESIZE_GROW) {
	/* check the return against the original.  If the procedure would
	 * like the BB to shrink, call again */
	if (wd < XtWidth(w) || ht < XtHeight(w)) {
	    wd = XtWidth(w);
	    ht = XtHeight(w);
	    _XmGeoArrangeBoxes(geo, 0, 0, &wd, &ht);
	}
    }

    if (wd == XtWidth(w) && ht == XtHeight(w)) {
	_XmGeoMatrixFree(geo);
	return;
    }

    retw = wd;
    reth = ht;
    do {
	result = XtMakeResizeRequest((Widget)w, retw, reth, &retw, &reth);
    } while (result == XtGeometryAlmost);

    if (retw != wd || reth != ht)
	_XmGeoArrangeBoxes(geo, 0, 0, &retw, &reth);

    _XmGeoMatrixSet(geo);

    if (XtIsRealized(w)) {

	_XmClearShadowType(w, BB_OldWidth(w), BB_OldHeight(w),
			   BB_OldShadowThickness(w), 0);

	_XmDrawShadows(XtDisplay(w), XtWindow(w),
		       MGR_TopShadowGC(w), MGR_BottomShadowGC(w),
		       0, 0, XtWidth(w), XtHeight(w),
		       MGR_ShadowThickness(w), BB_ShadowType(w));
    }

    _XmGeoMatrixFree(geo);

    BB_OldWidth(w) = XtWidth(w);
    BB_OldHeight(w) = XtHeight(w);
    BB_OldShadowThickness(w) = MGR_ShadowThickness(w);
}

static void 
realize(Widget w,
        XtValueMask *value_mask, 
        XSetWindowAttributes *attributes)
{
    XmBulletinBoardClassRec *bb = (XmBulletinBoardClassRec *)XtClass(w);
    Widget p;

    XdbDebug(__FILE__, w, "Realize\n");

#define superclass (&xmManagerClassRec)
    (*superclass->core_class.realize)(w, value_mask, attributes);
#undef superclass

    if (bb->bulletin_board_class.geo_matrix_create) {
	handle_realize(w, bb->bulletin_board_class.geo_matrix_create);
	return;
    }

    /* some call with args (w, BB_MarginWidth(w), BB_MarginHeight(w)); */

    _XmClearShadowType(w, BB_OldWidth(w), BB_OldHeight(w),
		       BB_OldShadowThickness(w), 0);

    BB_OldShadowThickness(w) = 0;

    if (XtIsComposite(w))
	p = w;
    else
	p = XtParent(w);

    if ((XtWidth(w) < BB_OldWidth(w) || XtHeight(w) < BB_OldHeight(w)) &&
	XtIsRealized(w)) {
	_XmDrawShadows(XtDisplay(w), XtWindow(w),
		       MGR_TopShadowGC(w), MGR_BottomShadowGC(w),
		       0, 0, XtWidth(w), XtHeight(w),
		       MGR_ShadowThickness(w), BB_ShadowType(w));
    }

    BB_OldWidth(w) = XtWidth(w);
    BB_OldHeight(w) = XtHeight(w);
    BB_OldShadowThickness(w) = MGR_ShadowThickness(w);
}

static void
expose(Widget w, XEvent *event, Region region) {
    _XmDrawShadows(XtDisplay(w), XtWindow(w),
                  MGR_TopShadowGC(w), MGR_BottomShadowGC(w),
                  0, 0, XtWidth(w), XtHeight(w),
                  MGR_ShadowThickness(w), BB_ShadowType(w));

    _XmRedisplayGadgets(w, event, region);
}

static void
_XmBbButton(Widget w, XtPointer client, XtPointer call)
{
	XmBulletinBoardWidget bb = (XmBulletinBoardWidget) XtParent(w);

	XdbDebug2(__FILE__, (Widget)bb, w, "XmBbButton\n");

/*
 * How do we make sure the APPLY or HELP buttons don't
 * trigger unmanaging ??  
 *
 * Somebody already caught this.  See comments in SelectionBox.c (look
 * for auto_unmanage).
 */
	if (BB_AutoUnmanage(bb)) {
		if (XtIsSubclass(XtParent((Widget)bb), xmDialogShellWidgetClass)) {
			XtUnmanageChild((Widget)bb);
			XdbDebug2(__FILE__, (Widget)bb, w, "AutoUnmanage\n");
		}
	}
}

/*
 * Keep track of button children
 * again, this shouldn't be done, but a call to XmGMEnforceMargins should be
 * made in the relevant places (such as change_managed).
 */
static void
InsertChild(Widget w)
{
    Widget p = XtParent(w);
    XmBulletinBoardWidgetClass bbclass;

#define	superclass	(&xmManagerClassRec)
    (*superclass->composite_class.insert_child)(w);
#undef	superclass

    if (XtX(w) < BB_MarginWidth(p))
	XtX(w) = BB_MarginWidth(p);
    if (XtY(w) < BB_MarginHeight(p))
	XtY(w) = BB_MarginHeight(p);

    _XmConfigureObject(w, XtX(w), XtY(w), XtWidth(w), XtHeight(w),
		       XtBorderWidth(w));

    /* this can't be right, but it does work */
    if (_XmIsFastSubclass(XtClass(w), XmPUSH_BUTTON_GADGET_BIT) ||
	_XmIsFastSubclass(XtClass(w), XmPUSH_BUTTON_BIT)) {
	XtAddCallback(w, XmNactivateCallback, _XmBbButton, NULL);
	bbclass = (XmBulletinBoardWidgetClass)XtClass(XtParent(w));
	if (bbclass->bulletin_board_class.focus_moved_proc)
	    _XmBulletinBoardSetDefaultShadow(w);
    }
}

void
_XmBbMap(Widget w)
{
    XdbDebug(__FILE__, w, "Map\n");
    XtCallCallbackList(w, BB_MapCallback(w), NULL);
    if (BB_DefaultPosition(w)
	&& XtIsSubclass(XtParent(w), xmDialogShellWidgetClass)) {
	Position	px, py;
	Widget	p;

	p = XtParent(XtParent(w));

	px = XtX(p) + (XtWidth(p) - XtWidth(w)) / 2;
	py = XtY(p) + (XtHeight(p) - XtHeight(w)) / 2;
	XdbDebug(__FILE__, w, "def. pos. %d %d\n", px, py);
	XtMoveWidget(XtParent(w), px, py);
    }
}
 
void
_XmBbUnmap(Widget w)
{
    XdbDebug(__FILE__, w, "Unmap\n");
    XtCallCallbackList(w, BB_UnmapCallback(w), NULL);
}

Widget
_XmBB_CreateButtonG(Widget bb, XmString l_string, char *name)
{
    Widget button;
    Arg args[2];

    XtSetArg(args[0], XmNlabelString, l_string);
#ifdef USE_WIDGETS
    button = XmCreatePushButton(bb, name, args, 1);
#else
    button = XmCreatePushButtonGadget(bb, name, args, 1);
#endif
    _XmBulletinBoardSetDefaultShadow(button);
    return button;
}

Widget
_XmBB_CreateLabelG(Widget bb, XmString l_string, char *name)
{
    Widget label;
    Arg args[2];

    XtSetArg(args[0], XmNlabelString, l_string);
#ifdef USE_WIDGETS
    label = XmCreateLabel(bb, name, args, 1);
#else
    label = XmCreateLabelGadget(bb, name, args, 1);
#endif
    return label;
}

void
_XmBulletinBoardSizeUpdate(Widget wid)
{
}

void
_XmBulletinBoardFocusMoved(Widget wid, XtPointer client_data, XtPointer data)
{
    _XmBulletinBoardSetDynDefaultButton(wid, (Widget)data);
}

void
_XmBulletinBoardMap(Widget wid, XEvent *event,
		    String *params, Cardinal *numParams)
{
}

void
_XmBulletinBoardReturn(Widget wid, XEvent *event,
		       String *params, Cardinal *numParams)
{
}

void
_XmBulletinBoardCancel(Widget wid, XEvent *event,
		       String *params, Cardinal *numParams)
{
}

void
_XmBulletinBoardSetDefaultShadow(Widget button)
{
    Dimension st, dbst;
    Arg args[2];

    if (_XmIsFastSubclass(XtClass(button), XmPUSH_BUTTON_GADGET_BIT) ||
	_XmIsFastSubclass(XtClass(button), XmPUSH_BUTTON_BIT)) {

	XtSetArg(args[0], XmNdefaultButtonShadowThickness, &dbst);
	XtSetArg(args[1], XmNshadowThickness, &st);
	XtGetValues(button, args, 2);
	if (st > 1)
	    st >>= 1;
	XtSetArg(args[0], XmNdefaultButtonShadowThickness, st);
	XtSetValues(button, args, 1);
    }
}

void
_XmBulletinBoardSetDynDefaultButton(Widget wid, Widget newDefaultButton)
{
    Arg argl[2];

    if (BB_DynamicDefaultButton(wid) == newDefaultButton)
	return;
    if (BB_DynamicDefaultButton(wid) != NULL) {
	if (!BB_DynamicDefaultButton(wid)->core.being_destroyed) {
	    XtSetArg(argl[0], XmNshowAsDefault, 0);
	    XtSetValues(BB_DynamicDefaultButton(wid), argl, 1);
	}
    }

    if (newDefaultButton == NULL) {
	BB_DynamicDefaultButton(wid) = NULL;
	return;
    }

    if (_XmIsFastSubclass(XtClass(newDefaultButton), XmPUSH_BUTTON_GADGET_BIT) ||
	_XmIsFastSubclass(XtClass(newDefaultButton), XmPUSH_BUTTON_BIT)) {
	BB_DynamicDefaultButton(wid) = newDefaultButton;
	XtSetArg(argl[0], XmNshowAsDefault, 1);
	XtSetValues(BB_DynamicDefaultButton(wid), argl, 1);
    }
}

void
_XmBBUpdateDynDefaultButton(Widget bb)
{
    if (BB_DynamicDefaultButton(bb))
	_XmBulletinBoardSetDynDefaultButton(bb, BB_DynamicDefaultButton(bb));
    else
	_XmBulletinBoardSetDynDefaultButton(bb, BB_DefaultButton(bb));
}

static Boolean
_XmBBParentProcess(Widget widget, XmParentProcessData data)
{
    return False;
}

