/*
** revnetgroup.c - creates the reverse netgroup list
**
** Copyright (c) 1996 Thorsten Kukuk
**
** This file is part of the NYS YP Server.
**
** The NYS YP Server is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License as
** published by the Free Software Foundation; either version 2 of the
** License, or (at your option) any later version.
**
** The NYS YP Server is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** General Public License for more details.
** 
** You should have received a copy of the GNU General Public
** License along with the NYS YP Server; see the file COPYING.  If
** not, write to the Free Software Foundation, Inc., 675 Mass Ave,
** Cambridge, MA 02139, USA.
**
** Author: Thorsten Kukuk <kukuk@uni-paderborn.de>
*/

static char rcsid[] = "$Id: revnetgroup.c,v 1.0.4.1 1996/07/14 15:10:45 kukuk Exp $";

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "system.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if defined(HAVE_GETOPT_H) && defined(HAVE_GETOPT_LONG)
#include <getopt.h>
#else
#include <compat/getopt.h>
#endif

#include "version.h"
#include "hash.h"

#ifndef HAVE_GETOPT_LONG
#include <compat/getopt.c>
#include <compat/getopt1.c>
#endif

extern void rev_setnetgrent(const char *);
extern void rev_endnetgrent(void);
extern int rev_getnetgrent(char **, char **, char **);

static int insert_netgroup(data_t **liste, const char *key, const char *val);
static void usage(int exit_code);

data_t *input = NULL;
static data_t **output = NULL;
static data_t **empty = NULL;

int main(int argc, char **argv)
{
#define BUFSIZE 1024
  char *host, *user, *domain, *key, *val;
  char buffer[BUFSIZE+1];
  int hosts;
  data_t *work;
  
  /* Only -u or -h are allowed, --version will exit the program */
  /* --users = -u, --hosts = -h */
  hosts = -1;
  
  if (argc < 2)
    usage(1);

  while(1) 
    {
      int c;
      int option_index = 0;
      static struct option long_options[] =
      {
        {"version", no_argument, NULL, '\255'},
        {"hosts", no_argument, NULL, 'h'},
	{"users", no_argument, NULL, 'u'},
        {"help", no_argument, NULL, '\254'},
        {"usage", no_argument, NULL, '\254'},
        {NULL, 0, NULL, '\0'}
      };
  
      c=getopt_long(argc, argv, "uh",long_options, &option_index);
      if (c==EOF) break;
      switch (c) 
	{
	case 'u':
	  hosts = 0;
	  break;
	case 'h':
	  hosts = 1;
	  break;
	case '\255':
          fprintf(stderr,"revnetgroup - NYS YP Server version %s\n",version);
	  exit(0);
	case '\254':
	  usage(0);
	  break;
	default:
	  usage(1);
	  break;
	}
    }


  /* Put the netgroup names in a list: */
  while (fgets(buffer, BUFSIZE, stdin)) 
    {
      char *cptr;
      
      if (buffer[0] == '#')
	continue;

      /* Replace first '\n' with '\0' */
      if((cptr = strchr(buffer,'\n'))!=NULL)
	*cptr = '\0';

      while ( buffer[strlen(buffer)-1] == '\\')
	{
	  fgets(&buffer[strlen(buffer) -1], 
		sizeof(buffer) - strlen(buffer), stdin);
	  if((cptr = strchr(buffer,'\n'))!=NULL)
	    *cptr = '\0';
	}
      
      val = (char *)(strpbrk(buffer, " \t"));
      if(val == NULL)
	continue;
      key = (char *)&buffer;
      *val = '\0';
      val++;
      insert_netgroup(&input, key, val);
    }
	
  /*
   * Find all members of each netgroup and keep track of which
   * group they belong to.
   */
  empty = hash_malloc();
  output = hash_malloc();
  work = input;
  while(work != NULL)
    {
      rev_setnetgrent(work->key);
      while(rev_getnetgrent(&host, &user, &domain) != 0) 
	{
	  static char star[] = "*";
	  char *key, *dat;
	  
	  dat=hosts ? host : user;
	  if(dat == NULL) dat = star;
	  if(strcmp(dat,"-") != 0)
	    {
	      if(domain == NULL)
		{
		  key = malloc(strlen(dat)+3);
		  sprintf(key,"%s.*", dat);
		}
	      else
		{
		  key = malloc(strlen(dat)+strlen(domain)+2);
		  sprintf(key,"%s.%s",dat,domain);
		}
	      if(strcmp(dat,"*") == 0 )
		{
		  dat = hash_search(empty,key);
		  if(dat != NULL)
		    {
		      char *buf, *buf2;
		      int found;
		      
		      buf2 = malloc(strlen(dat)+1);
		      strcpy(buf2,dat);
		      buf = strtok(buf2,",");
		      found = 0;
		      while(buf != NULL && !found)
			{
			  found = (0 == strcmp(buf,work->key));
			  buf = strtok(NULL,",");
			}
		      free(buf2);
		      if(!found)
			{
			  buf = malloc(strlen(work->key)+strlen(dat)+2);
			  sprintf(buf,"%s,%s",dat,work->key);
			  hash_delkey(empty,key);
			  hash_insert(empty,key,buf);
			  free(buf);
			}
		    }
		  else
		    hash_insert(empty,key,work->key);
		  free(key);
		}
	      else
		{
		  dat = hash_search(output,key);
		  if(dat != NULL)
		    {
		      char *buf;
		      
		      buf = malloc(strlen(work->key)+strlen(dat)+2);
		      sprintf(buf,"%s,%s",dat,work->key);
		      hash_delkey(output,key);
		      hash_insert(output,key,buf);
		      free(buf);
		    }
		  else
		    hash_insert(output,key,work->key);
		  free(key);
		}
	    }
	}
      work=work->next;
      /* Release resources used by the getnetgrent code. */
      rev_endnetgrent();
    }

  /* Print the results. */
  work = hash_first(output);
  while(work != NULL)
    {
      printf("%s\t%s\n",work->key,work->val);
      work=hash_next(output,work->key);
    }
  
  work = hash_first(empty);
  while(work != NULL)
    {
      printf("%s\t%s\n",work->key,work->val);
      work=hash_next(empty,work->key);
    }
  
#if 0
  remove_netgroup(&input);
  remove_netgroup(&output);
  remove_netgroup(&empty);
#endif

  return 0;
}

static void usage(int exit_code)
{
  fprintf (stderr,"usage: revnetgroup -u|-h\n");
  fprintf (stderr,"       revnetgroup --version\n");
  exit(exit_code);
}

int insert_netgroup (data_t **liste, const char *key, const char *val)
{
  data_t *work;

  work = malloc(sizeof(data_t));
  work->next = *liste;
  work->key = malloc(strlen(key)+1);
  work->val = malloc(strlen(val)+1);
  strcpy(work->key,key);
  strcpy(work->val,val);
  *liste = work;
 
  return 0;
}
