/* Copyright (C) 1994 
            Olav Woelfelschneider (wosch@rbg.informatik.th-darmstadt.de)

 This library is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <X11/cursorfont.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include "McAlloc.h"
#include <X11/Xatom.h>
#include "McApp.h"
#include "McGadget.h"
#include "McDebug.h"
#include "McMenubar.h"

/**************************************************************************/

#define MENUBAR ((McMenubar *)(gadget->specialInfo))
#define DISTANCE (10*BW)

/**************************************************************************/

static int  MenubarEvent(McGadget *gadget, XEvent *event);
static void MenubarCleanup(McGadget *gadget);
static void MenubarCreateMenus(McMenubar *menubar,
			       void (*callback)(int));
static void MenubarCreateWidths(McMenubar *menubar);
static int  MenubarRelease(McGadget *gadget, int x, int y);
static int  HotkeyHandler(McGadget *gadget, XEvent *event);

/**************************************************************************/

McSpecialInfo *McCreateMenubar(McGadget *gadget,
			       unsigned int flags, XFontStruct *font,
			       McMenubarInit *items, int size,
			       void (*callback)(int)) {

  McMenubar *menubar = (McMenubar *) McAlloc(sizeof(McMenubar));
  McWindow *mcw=gadget->mcw;

  if (!mcw->app->hotkeyHandler) {
    mcw->app->hotkeyHandler=&menubar->hotkeyHandler;
    menubar->hotkeyHandler.callback=HotkeyHandler;
    menubar->hotkeyHandler.gadget=gadget;
  }

  menubar->mcw = mcw;
  menubar->flags = flags;
  menubar->specialInfo.updateProc = McMenubarUpdate;
  menubar->specialInfo.eventProc = MenubarEvent;
  menubar->specialInfo.cleanupProc = MenubarCleanup;
  menubar->specialInfo.releaseProc = MenubarRelease;
  menubar->size = size;
  menubar->items = items;
  menubar->font = font;
  menubar->widths = NULL;
  menubar->current = -1;
  menubar->lastcurrent = -1;

  MenubarCreateWidths(menubar);

  MenubarCreateMenus(menubar, callback);

  return (McSpecialInfo *)menubar;
}

/**************************************************************************/

static void MenubarCreateMenus(McMenubar *menubar, void (*callback)(int)) {
  int i;
  menubar->menu=(McMenuList **)calloc(menubar->size, sizeof(McMenu));
  for (i=0; i<menubar->size; i++) {
    menubar->menu[i]=McCreateMenu(menubar->mcw,
				  menubar->items[i].items,
				  menubar->items[i].count, 0);
    menubar->menu[i]->callback=callback;
  }
}

/**************************************************************************/

static void MenubarCleanup(McGadget *gadget) {
  McWindow *mcw;

  McMenubar *menubar = (McMenubar *)gadget->specialInfo;
  if (menubar->menu) {
    int i;
    for (i=0; i<menubar->size; i++) {
      if (menubar->menu[i]) McFreeMenuList(menubar->menu[i]);
    }
    free(menubar->menu);
  }
  if (menubar->widths) free(menubar->widths);

  mcw=gadget->mcw;
  if (mcw->app->hotkeyHandler==&menubar->hotkeyHandler)
    mcw->app->hotkeyHandler=NULL;

  McFree(gadget->specialInfo);
}

/**************************************************************************/

void McMenubarUpdate(McGadget *gadget,int busy,int all) {
  McMenubar *menubar = (McMenubar *) gadget->specialInfo;
  McWindow *mcw=gadget->mcw;
  McApp *app=mcw->app;
  int i,x,y,h,_3d;
  unsigned char *data;
  XRectangle rect;
  GC gc1 = XCreateGC(app->display, mcw->window, 0, NULL);
  GC gc2 = XCreateGC(app->display, mcw->window, 0, NULL);
  GC gc3, gc4;

  if (gadget->flags & GAD_SELECTED) {
    XCopyGC(app->display, app->gc[GC_SELECTED_BITMAP], ~0, gc1);
    XCopyGC(app->display, app->gc[GC_NORMAL], ~0, gc2);
    gc3=app->gc[GC_CLEAR];
    gc4=app->gc[GC_SELECTED];
    _3d=_3D_OUT;
  } else {
    XCopyGC(app->display, app->gc[GC_NORMAL], ~0, gc1);
    XCopyGC(app->display, app->gc[GC_SELECTED_BITMAP], ~0, gc2);
    gc3=app->gc[GC_SELECTED];
    gc4=app->gc[GC_CLEAR];
    _3d=_3D_IN;
  }
  XSetFont(app->display, gc1, menubar->font->fid);
  XSetFont(app->display, gc2, menubar->font->fid);
  rect.x=gadget->x+BW; rect.y=gadget->y;
  rect.width=gadget->width-BW-BW;
  rect.height=gadget->height;
  McSetClipRectangle(mcw, gc1, &rect);
  McSetClipRectangle(mcw, gc2, &rect);

  y=gadget->y+BW+BW+BW; x=gadget->x+(DISTANCE>>1)+BW; h=gadget->height-BW-BW;
  for (i=0; i<menubar->size; i++) {
    data=menubar->items[i].title;
    if (i==menubar->current) {
      XFillRectangle(app->display, mcw->window, gc3,
		     x-(DISTANCE>>1)+BW, y-BW,
		     menubar->widths[i]+DISTANCE-BW-BW, h-(BW<<1));
      McAppDrawbox(mcw, mcw->window,
		   x-(DISTANCE>>1)+BW, y-BW,
		   menubar->widths[i]+DISTANCE-BW-BW, h-(BW<<1), _3d);
      XDrawImageString(app->display, mcw->window, gc2,
		       x, menubar->font->ascent+y, data, strlen(data));
      
    } else if (all || (i==menubar->lastcurrent)) {
      XFillRectangle(app->display, mcw->window, gc4,
		     x-(DISTANCE>>1), y-BW-BW,
		     menubar->widths[i]+DISTANCE, h);
      XDrawImageString(app->display, mcw->window, gc1,
		       x, menubar->font->ascent+y, data, strlen(data));
    }
    
    x+=menubar->widths[i]+DISTANCE;
  }
  menubar->lastcurrent=-1;

  XFreeGC(app->display, gc1);
  XFreeGC(app->display, gc2);
}

/**************************************************************************/

static void MenubarCreateWidths(McMenubar *menubar) {
  int i;

  menubar->widths=calloc(menubar->size, sizeof(int));

  for(i=0; i<menubar->size; i++) {
    int ascent, descent, dummy;
    XCharStruct overall;

    XTextExtents(menubar->font,
		 menubar->items[i].title,
		 strlen(menubar->items[i].title),
		 &dummy, &ascent, &descent, &overall);
    
    menubar->widths[i]=overall.width;
  }

}

/**************************************************************************/

static int IsInMenu(McGadget *gadget, int x) {
  McMenubar *menubar = (McMenubar *)gadget->specialInfo;
  int prev, next, i;

  if (x>=0) {
    next=BW;
    for (i=0; i<menubar->size; i++) {
      prev=next;
      next+=menubar->widths[i]+DISTANCE;
      if ((x>=(prev+BW)) && (x<=(next-BW))) {
	if (i!=menubar->current) return i; else return -2;
      }
    }
  }

  return -1;
}

static void MenubarOff(McGadget *gadget) {
  McMenubar *menubar = (McMenubar *)gadget->specialInfo;
  if (menubar->current!=-1) {
    menubar->lastcurrent=menubar->current;
    menubar->current=-1;
    McMenubarUpdate(gadget, 0, 0);
  }
  gadget->flags&=~GAD_PRESSED;
}

/**************************************************************************/

static void ZapMenubar(McMenubar *menubar) {
  if (menubar->currentMenu) {
    McZapMenus(menubar->currentMenu);
    menubar->currentMenu=NULL;
  }
}

static void PopdownMenu(McGadget *gadget, int this) {
  McMenubar *menubar = (McMenubar *)gadget->specialInfo;
  McApp *app=gadget->mcw->app;
  int x, xr, yr, i;
  Window nonsense;

  ZapMenubar(menubar);

  if (this>=0) {
    x=gadget->x+BW;
    for (i=0; i<this; i++) x+=menubar->widths[i]+DISTANCE;

    XTranslateCoordinates(app->display, gadget->mcw->window,
			  RootWindow(app->display, app->screen),
			  x, gadget->y+gadget->height-BW,
			  &xr, &yr, &nonsense);

    menubar->currentMenu=McShowMenu(app, gadget, menubar->menu[this], xr, yr);
  }
}

/**************************************************************************/

static int MenubarRelease(McGadget *gadget, int x, int y) {
  int i;
  McMenubar *menubar = (McMenubar *)gadget->specialInfo;

  if (gadget->flags&GAD_PRESSED) {
    if (!x && !y) {
      ZapMenubar(menubar);
      MenubarOff(gadget);
      return 1;
    }

    if (McInRectangle(x, y,
		      gadget->x, gadget->y, gadget->width, gadget->height)) {
      i=IsInMenu(gadget, x-gadget->x);
      if (i<-1) return 0;
      menubar->lastcurrent=menubar->current;
      menubar->current=i;
      McMenubarUpdate(gadget, 0, 0);
      PopdownMenu(gadget, i);
      return 1;
    }
    return 0;
  }

  ZapMenubar(menubar);
  return 1;
}

/**************************************************************************/

static int HandleHotkeyInItems(McMenuItem *item, unsigned char key);

static int HotkeyHandler(McGadget *gadget, XEvent *event) {
  McMenubar *menubar;
  int n, i;
  unsigned char buf[16];

  n=XLookupString((XKeyEvent *)event,buf,15,
		  &gadget->mcw->keysym, &gadget->mcw->compose);
  if (n!=1) return 0;
  *buf=tolower(*buf);

  menubar = (McMenubar *)gadget->specialInfo;
  for (i=0; i<menubar->size; i++) {
    if (HandleHotkeyInItems(menubar->menu[i]->first, *buf)) return 1;
  }
  return 0;
}

static int HandleHotkeyInItems(McMenuItem *item, unsigned char key) {
  while(item) {
    if (item->subMenu) {
      if (HandleHotkeyInItems(item->subMenu->first, key)) return 1;
    } else {
      if (tolower(item->hotkey)==key) {
	if (item->flags & ITEM_TOGGLE) item->flags^=ITEM_CHECKED;
	if (item->callback) (*(item->callback))(item->id);
	return 1;
      }
    }
    item=item->next;
  }
  return 0;
}

/**************************************************************************/

static int MenubarEvent(McGadget *gadget, XEvent *event) {
  McMenubar *menubar = (McMenubar *)gadget->specialInfo;
  int this;
  
  switch(event->type) {
  case MotionNotify:
  case ButtonPress:
    if (McInRectangle(event->xbutton.x, event->xbutton.y,
		      gadget->x, gadget->y,
		      gadget->width, gadget->height)) {
      gadget->flags|=GAD_PRESSED;
      this=IsInMenu(gadget, event->xbutton.x-gadget->x);
      if (this>-1) {
	menubar->lastcurrent=menubar->current;
	menubar->current=this;
	McMenubarUpdate(gadget, 0, 0);
	PopdownMenu(gadget, this);
      }
      return 1;
    }

  case ButtonRelease:
    if (gadget->flags & GAD_PRESSED) {
      MenubarOff(gadget);
      return 1;
    }
  }

  return 0;
}

/**************************************************************************/

McGadget *MakeMenubar(McWindow *mcw, McMenubarInit *items, int size,
		      void (*callback)(int), int width) {
  McGadget *gadget;
  McMenubar *bar;

  gadget = McCreateGadget(mcw, GAD_H3D|GAD_3D|GAD_ACTIVE,
			  MENUBARGADGET, BW, BW, mcw->w-BW-BW,
			  mcw->app->defaultFont->ascent+
			    mcw->app->defaultFont->descent+3+(BW<<2));

  bar = (McMenubar *)McCreateMenubar(gadget, 0, mcw->app->defaultFont,
				     items, size, callback);

  gadget->specialInfo = (McSpecialInfo *)bar;
  gadget->bottomRightGravity=NorthEastGravity;

  if (width) McResizeGadget(gadget, width, gadget->height);

  return gadget;
}

/**************************************************************************/


