/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include "mvmebug.h"
#include "loader.h"
#include "vmelilo.h"

#include "mvmeboot.data"
#include "mvmeloader.data"

/*--------------------------------------------------------------------------*/

void
mvme_init_boot_block
(	void			*buf
)
{	BOOTBLOCK		*bb = (BOOTBLOCK *) buf;
	u_long			n1;
	u_long			n2;
	DSKRD_CMND		*cdb;
	const FILEMAP	*frag;

	/* set boot block id */
	strcpy(bb->boot_id, BOOT_BLOCK_ID);

	if (sizeof(mvmeboot_code) > sizeof(bb->boot_code))
	{
		die("%d bytes too much boot code\n",
			sizeof(bb->boot_code) - sizeof(mvmeboot_code));
	}

	/* put boot code into boot block */
	memcpy(bb->boot_code, mvmeboot_code, sizeof(mvmeboot_code));

	bb->sp               = 0x21000;
	bb->pc               = 0x30;
	bb->boot_start_block = 0;
	bb->boot_block_count = 2;
	bb->boot_load_addr   = 0x20000;
	bb->zero             = 0;

	if (MAP_NUMFRAGS(loader_map) > MAX_LDR_FRAGS)
	{
		n1 = MAP_NUMFRAGS(loader_map) - MAX_LDR_FRAGS;

		die("Boot loader file is too fragmented for boot block "
			"(%ld fragment%s too many)\n",
					n1, (n1 == 1) ? "" : "s");
	}

	if (f_verbose)
	{
		n1 = MAP_NUMFRAGS(loader_map);
		n2 = MAX_LDR_FRAGS - n1;

		printf(	"Boot loader file has %ld fragment%s, "
				"enough space for %lu more\n",
				  	n1, (n1 == 1) ? "" : "s", n2);
	}

	/* put file map of loader into boot block */
	frag = MAP_FIRSTFRAG(loader_map);
	cdb  = bb->cdb;
	n2   = 0x10000;
	for (n1 = 0; n1 < MAP_NUMFRAGS(loader_map); n1++, frag++, cdb++)
	{
		cdb->clun    = 0;
		cdb->dlun    = 0;
		cdb->status  = 0xdead;
		cdb->address = n2;
		cdb->block   = frag->offset >> 8;
		cdb->count   = frag->length >> 8;
		cdb->flags   = 0;
		cdb->mod     = 0;
		n2          += frag->length;
	}
	while (n1 < MAX_LDR_FRAGS)
	{
		cdb->clun = 0xff;
		cdb++;
		n1++;
	}
}

/*--------------------------------------------------------------------------*/

int
mvme_valid_boot_block
(	void		*buf
)
{	BOOTBLOCK	*bb = (BOOTBLOCK *) buf;
	
	return (strcmp(bb->boot_id, BOOT_BLOCK_ID) == 0);
}

/*--------------------------------------------------------------------------*/

LILO_DRIVER mvme_lilo_driver = {
	"MVMEBug",
	mvmeloader_code,
	sizeof(mvmeloader_code),
	sizeof(BOOTBLOCK),
	mvme_init_boot_block,
	mvme_valid_boot_block
};

/*-----------------------------< end of file >------------------------------*/
